/******************************
 Javascript Console
 Init by A. Krywaniuk, Jan 2006
 Copyright Fortinet, inc.
 All rights reserved
 ******************************/

// EXPORT_SYMBOL on_customize_clicked
// EXPORT_SYMBOL jsconn_test1
// EXPORT_SYMBOL on_clear_clicked
// EXPORT_SYMBOL print_status
// EXPORT_SYMBOL set_alert_string
// EXPORT_SYMBOL console_window

/* The Javascript viewer is divided into multiple subcomponents. To use it, you
should include all of the following JS files:

jsconsole.js - general stuff (this file)
jsconsole_kbd.js - keyboard handlers
jsconsole_ajax.js - Ajax IO functions
jsconsole_term.js - terminal functionality
jsconsole_init.js - initialization & cleanup functions

Some sample code to include in your C module:

"<script type=\"text/javascript\" src=\"/jsconsole.js?q=" CONFIG_GUI_NO "\"></script>\n"
"<script type=\"text/javascript\" src=\"/jsconsole_kbd.js?q=" CONFIG_GUI_NO "\"></script>\n"
"<script type=\"text/javascript\" src=\"/jsconsole_ajax.js?q=" CONFIG_GUI_NO "\"></script>\n"
"<script type=\"text/javascript\" src=\"/jsconsole_term.js?q=" CONFIG_GUI_NO "\"></script>\n"
"<script type=\"text/javascript\" src=\"/jsconsole_init.js?q=" CONFIG_GUI_NO "\"></script>\n"
*/


// STRIP-FUNCTION: debug_msg
// STRIP-FUNCTION: debug_kbd
 
 
// Turn off verbose_debug to get an idea of the true runtime performance.
//var verbose_debug = true;
var verbose_debug = false;

// Whether to enable debugging of keyboard events (with debug_kbd)
//var keyboard_debug = true;
var keyboard_debug = false;



/******************************
      Accessor Functions
 ******************************/

/* Small changes to the HTML layout can have big effects on the DOM
 representation (e.g. putting the console window in a separate frame
 required a lot of changes). So it is better to access some global
 elements via these centralized interfaces. */
 

// Hmmm... doesn't seem possible to cache this reference. JS is like
// Java - maybe we need a wrapper class or something.
var console_document = null; // TEST: cache the document for easy access.
var console_window = null;


// getConsoleDoc - this is the inner document (consoleframe.html).
function getConsoleDoc()
{
    if (window.frames.console_iframe.document)
	return window.frames.console_iframe.document;
    else
	return console_window.document;
}

function getConsoleWin()
{
    if (window.frames.console_iframe.document)
	return window.frames.console_iframe.window;
    else
	return console_window;
}

// getConsoleTable - this is the console <table> in the inner document.
function getConsoleTable()
{
	try
	{
		var cons_elem = getConsoleDoc().getElementById("history_txt");
		return cons_elem;
	}
	catch (e)
	{
		return null;
	}
}


// getConsoleFrame - this is the <iframe> in the outer document.
function getConsoleFrame()
{
	try
	{
	    if (window.frames.console_iframe.document)
		return window.frames.console_iframe;
	    else
		return console_window;
	}
	catch (e)
	{
		return null;
	}
}


// getConsoleTbody - this is what most functions that manipulate the history 
// buffer need.
function getConsoleTbody()
{
	// This shouldn't fail. Let it throw an exception if it does.
	var console_elem = getConsoleTable();
	var tbody = get_last_child_of_type(console_elem, "TBODY");
	return tbody;
}


// getCursor - get the <span> tag associated with the cursor (mostly so we can blink it).
function getCursor()
{
	try
	{
		var cons_elem = getConsoleDoc().getElementById("history_cursor");
		return cons_elem;
	}
	catch (e)
	{
		return null;
	}
}

// Set the checkbox value to true or false on logo frame
// name_t:   checkbox id on logo frame
// value_t:  true or false
function set_logo_checkbox(name_t, value_t)
{
	var opener_hdl = window;
	if(opener_hdl.opener)
		opener_hdl = opener_hdl.opener;
	var oElem = opener_hdl.top.document.getElementById(name_t);
	oElem.checked = value_t;
}

// Set the text box value on logo frame
// name_t:   text box id on logo frame
// value_t:  value
function set_logo_text(name_t, value_t)
{
	var opener_hdl = window;
	if(opener_hdl.opener)
		opener_hdl = opener_hdl.opener;
	var oElem = opener_hdl.top.document.getElementById(name_t);
	oElem.value = value_t;
}


/******************************
       User Preferences
 ******************************/

// acon_prefs - various preferences affecting the user experience.
// Most of these can be overridden from the CMDB or cookie.
var acon_prefs = {
	acon_fgColor : 'FFFFFF', // white
	acon_bgColor : '000000', // black
	acon_font_family : 'monospace',
	acon_font_size : '10pt',
	acon_consbuflen : 500, // length of the console scroll buffer
	acon_histbuflen : 50, // buffer that we save in the logo frame
	acon_showcmd : false,
	use_connect_btns : false, // whether to include support connect/disconnect buttons
	dummy : 0
};

/* acon_histbuflen basicaly does the same thing as acon_consbuflen, except that the strings
 are stored in a JS array instead of a DOM table and then they are later kept in the logo 
 frame. It will probably adds a small speed penalty to the JS code. 
 
 use_connect_btns=true represents the original design. use_connect_btns=false represents 
 the changes requested by Michael X. after the feature was complete. Whether or not to have
 the explicit connect & disconnect buttons is a matter of taste. If the disconnect option is
 not provided, the CLI session will idle out or be cleaned up when the connection limit is reached.
 One possibly annoying thing about removing the disconnect button is that the console history
 (scroll buffer) will not be accessible after the disconnect.
 
*/
 

// override_default_console_prefs - override the global defaults in acon_prefs
// with the specific defaults from the CMDB user.
function override_default_console_prefs()
{
	try
	{
		if (default_console_prefs)
		{
			clone_console_prefs(acon_prefs, default_console_prefs);
		}
	}
	catch (e)
	{}
}


// clone_console_prefs - deep copy function for console prefs.
function clone_console_prefs(dst, src)
{
	// Most of these are specified in jsconsole.c. If they aren't then we need
	// to make sure they aren't undefined before copying them.
	try
	{
		// Copying fields 1-by-1 leads to less catastrophic failure if the structure
		// definition in the C files gets out of sync w/ the above.
		for(prop in src)
			dst[prop] = src[prop];
	}
	catch (e)
	{
		debug_msg("Unable to copy default console prefs: " + e);
	}
}


// apply_console_prefs - update the onscreen screen display to reflect
// any changes to the global variables.
function apply_console_prefs(cnsl_doc)
{
	if (!cnsl_doc.styleSheets)
	{
		debug_msg("Failed to change console colours (no stylesheets)");
		return;
	}
		
	var my_css = new Array();
	
	if (cnsl_doc.styleSheets[0].cssRules)
	{
		// Mozilla case
		my_css = cnsl_doc.styleSheets[0].cssRules;
	}
	else
	{
		// IE case
		my_css = cnsl_doc.styleSheets[0].rules;
	}
	
	my_css[0].style.backgroundColor = acon_prefs.acon_bgColor;
	my_css[1].style.backgroundColor = acon_prefs.acon_bgColor;

	my_css[0].style.color = acon_prefs.acon_fgColor;
	my_css[1].style.color = acon_prefs.acon_fgColor;
	
	my_css[0].style.fontFamily = acon_prefs.acon_font_family;
	my_css[1].style.fontFamily = acon_prefs.acon_font_family;

	my_css[0].style.fontSize = acon_prefs.acon_font_size;
	my_css[1].style.fontSize = acon_prefs.acon_font_size;
	
	// Set the state of the command div.
	// TODO: one idea here would be to fade the command box div instead 
	// of hiding it (on the console customize page only). 
	// However, this proved to be more difficult than
	// expected. Mozilla supports style.opacity=<N>, and everything works
	// great. IE uses style.filter="alpha(opacity=<N>)" which seems to only
	// work on inline elements. I couldn't make it work with spans or divs.
	var elem_d = document.getElementById("jsconn_cmd_div");
	if (elem_d)
	{
		var st = (acon_prefs.acon_showcmd) ? "" : "none";
		elem_d.style.display = st;
	}
}


var min_sane_cmd_histlen = 10;
var max_sane_cmd_histlen = 100;

var min_sane_consbuf_len = 10;
var max_sane_consbuf_len = 99999;


// parse_console_cookie - Parse the cookie for the console preferences.
function parse_console_cookie()
{
	// The feature to persistently store the preferences in a cookie was cancelled.
	// But we still use the cookie to apply the changes for this session when they
	// are changed.
	
	
	// The cookie is generated in consolecustomize.js. As of 04/05/06...
	// The cookie format is: "fnjsconsole=version&fgColor&bgColor&hist_len&
	//    &font_family&font_size&cons_buf_size&show_command;"
	
	try 
	{
		var i = document.cookie.lastIndexOf("fnjsconsole=");
		
		if (i<0)
		{
			// No cookie found.
			debug_msg("No JS console cookie found.");
			return false;
		}
		
		// Initialize str at each stage, since it will be displayed
		// in the message if there is an error.
		var str = document.cookie;

		// for bug 270809, comment these 3 lines
		// current the cookie value doesn't include ';', only "fnjsconsole=XX&XX&XX"		
//		var j = str.indexOf(';', i);
//		if (j < 0) throw(1);
//		str = str.substring(i,j);

		i = str.indexOf('=');
		if (i<0) throw(7);
		
		str = str.substring(i+1);
		
		var vals = str.split('&');
		
		if (vals.length < 8) throw(2);
		if (vals[1].length != 6) throw(3);
		if (vals[2].length != 6) throw(4);
		
		var new_histlen = parseInt(vals[3]);
		if (new_histlen < min_sane_cmd_histlen) throw 5;
		if (new_histlen > max_sane_cmd_histlen) throw 6;
		
		var new_conslen = parseInt(vals[6]);
		if (new_conslen < min_sane_consbuf_len) throw 5;
		if (new_conslen > max_sane_consbuf_len) throw 6;
		
		var old_showcmd = acon_prefs.acon_showcmd;
		
		acon_prefs.acon_fgColor = vals[1];
		acon_prefs.acon_bgColor = vals[2];
		acon_prefs.acon_histbuflen = new_histlen;
		acon_prefs.acon_font_family = vals[4];
		acon_prefs.acon_font_size = vals[5];
		acon_prefs.acon_consbuflen = new_conslen;
		acon_prefs.acon_showcmd = (vals[7] == 1);
		
		if (old_showcmd != acon_prefs.acon_showcmd)
			wnd_size_st.need_resync = true;
	}
	catch (e)
	{
		debug_msg("error in parsing cookie - invalid format: " + e + str);
		return false;
	}
	
	return true;
}

function parse_connection_cookie()
{
	try 
	{
		var i = document.cookie.lastIndexOf("connection=");
		
		if (i<0)
		{
			// No cookie found.
			debug_msg("No JS cons cookie found.");
			return false;
		}
		
		// Initialize str at each stage, since it will be displayed
		// in the message if there is an error.
		var str = document.cookie;
		
		var j = str.indexOf(';', i);
		if (j < 0) throw(1);

		str = str.substring(i,j);
		
		i = str.indexOf('=');
		if (i<0) throw(2);
		
		str = str.substring(i+1);
		
		var vals = str.split('&');
		
		if (vals.length < 1) throw(3);
		
		if(vals[0] == 'true')
			cons_connected = true;
		else
			cons_connected = false;
	}
	catch (e)
	{
		debug_msg("error in parsing cookie - invalid format: " + e + str);
		return false;
	}
	
	return true;
}

function on_customize_clicked()
{
	// Add approximate height/width, and we will do the auto-resize at runtime.
	try {
		var open_hdl = window.opener?window.opener:window;
		var win_hdl = open_hdl.top.window;
		if(win_hdl.hPopupCustomize)
			// Focus on the existing page
			win_hdl.hPopupCustomize.focus();
		else
			// Open a new page and store the window handle in logo page
	    	win_hdl.hPopupCustomize = window.open("/system/consolecustomizedlg","","width=450,height=428,top=100,left=200,resizable=1");
	}
	catch (e)
	{}
}


/******************************
     Window Size functions
 ******************************/

// Variables related to window size calculations. Not all of these are used
// by the current code. We may want to implement some fancy features later,
// such as saving the tweaked size (or the current size & location) in
// a cookie.
var wnd_size_st = {
	already_resized: false,
	need_resync: false,
	
	// Window size
	orig_w: 0,
	orig_h: 0,
	auto_w: 0,
	auto_h: 0,
	cur_w: 0,
	cur_h: 0,
	
	// Frame size
	orig_fw: 0,
	orig_fh: 0,
	
	// Div size
	orig_dw: 0,
	orig_dh: 0,
	auto_dw: 0,
	auto_dh: 0
};

// jsconn_popup_min_sz - minimum size of the popup window.
// TODO: these minimum size settings are fairly arbitrary.
var jsconn_popup_min_sz = {
	h: 200,
	w: 400
};

// sz_obj - global object to hold size calculations.
// Used to hold the return value of calc_client_sz, calc_elem_sz, etc.
var sz_obj = {
	w:0,
	h:0
};

function calc_client_sz(rv)
{
  var x = window; 
  var myW = 0, myH = 0, d = x.document.documentElement, b = x.document.body;
	  if( x.innerWidth ) { myW = x.innerWidth; myH = x.innerHeight; }
	  else if( d && d.clientWidth ) { myW = d.clientWidth; myH = d.clientHeight; }
	  else if( b && b.clientWidth ) { myW = b.clientWidth; myH = b.clientHeight; }
	
	rv.w = myW;
	rv.h = myH;
}

function calc_elem_sz(id, rv)
{
	var elem = document.getElementById(id);
	rv.w = parseInt(elem.style.width);
	rv.h = parseInt(elem.style.height);
}

function calc_autodiv_sz(rv)
{
	calc_elem_sz('autosizing_div', rv);
}


/* Window resizing event handlers:

 We want to do two things when the window is resized:
 
 1. Adjust the size of the console window to match the outer window.
 2. Prevent the user from resizing the window beyond a certain minimum size.
 
 Unfortunately, this is amazingly complicated to do (not surprisingly, pretty much
 every API function works differently on each browser). For the most part, the
 size adjustment of the console window works well. The difficulties come in adjusting
 for the case where the window goes below the minimum size.
 
 Complications:
 
 1. I didn't see any way to retrieve both the new size and the old size at the same
 time, so we store the previous values in a global variable.
 2. IE seems to receive this event twice. The second call is basically sprurious.
 3. We have to deal with the client area of the window, which is different from the
 size of the window frame.
 4. If we call resizeTo() or resizeBy() from the onresize handler, we have to deal
 with various amounts of recursion.
 
 Issues 3 & 4 means that we can't just resize exactly to the desired final size.
 Basically, we have to take an iterative approach and be happy if the final size is
 within close enough limits. (Also, we layout the page in such a way that most of
 the realignment happens automatically.)

 There is also a strange bug that I could only reproduce on old Mozilla. If the user
 resizes the window below the minimum reasonable size (at which it can display all
 the content), then scrolls the client area (using the mouse drag), then resizes
 the window again (to make it bigger), the clipping area won't match the client area
 and there will be garbage displayed on the screen. We use an extra scrollTo to
 correct that problem.
 
 Various error messages come up in the Javascript console about failure to parse the
 value for attribute height or width. I'm not sure exactly when these errors are
 generated, but they don't appear to come from the JS in this file. (It is possibly due 
 to an attempt to read the values before the window is redrawn.
 
*/

// resize_handler - window.onresize event handler
// Adjust the div & iframe elements to have the appropriate size.
function resize_handler(evt)
{
	// save old size
	var old_w = wnd_size_st.cur_w;
	var old_h = wnd_size_st.cur_h;

	// store new size
	calc_client_sz(sz_obj);
	wnd_size_st.cur_w = sz_obj.w;
	wnd_size_st.cur_h = sz_obj.h;
	
	// reset the semaphore
	wnd_size_st.need_resync = false;

	// Don't do anything else if we are still in the auto-size phase.
	// already_resized might be a bad indicator of whether this is
	// the first call, so check old_w/h as well.
	if (!wnd_size_st.already_resized || !old_w || !old_h)
	{
		return;
	}
	
	debug_msg("resizing: width " + old_w + "->" + wnd_size_st.cur_w);
	debug_msg("resizing: height " + old_h + "->" + wnd_size_st.cur_h);
	
	// Base all the adjustments relative to the initial (auto-calculated)
	// window & div size.
	var dw = wnd_size_st.cur_w - wnd_size_st.auto_w;
	var dh = wnd_size_st.cur_h - wnd_size_st.auto_h;
	
	try
	{
		var em_ifrm = document.getElementById("console_iframe");
		if (em_ifrm)
		{
			em_ifrm.style.width = "" + (wnd_size_st.orig_fw + dw) + "px";
			em_ifrm.style.height = "" + (wnd_size_st.orig_fh + dh) + "px";
		}
	}
	catch (e)
	{
		// This fails in IE when you try to resize the frame to be larger
		// than the current dimensions of the window. It should correct
		// itself iteratively.
	}
	
	try
	{
		em_div = document.getElementById('autosizing_div');
		if (em_div)
		{
			em_div.style.width = "" + (wnd_size_st.auto_dw + dw) + "px";
			em_div.style.height = "" + (wnd_size_st.auto_dh + dh) + "px";
		}
	}
	catch (e)
	{
		// This fails on IE for some reason. However, the layout generally
		// looks ok.
	}
	
	// Don't let the user set the size below a certain minimum.
	// We can't prevent them from doing it, but we can immediately revert
	// the change.
	var rsz_w = 0;
	var rsz_h = 0;
	
	// Resize a minimum of 50 pixels at a time - we are comparing the client area 
	// with the frame size here, so the values aren't going to match exactly.
	if (wnd_size_st.cur_w < jsconn_popup_min_sz.w)
		rsz_w = Math.max(jsconn_popup_min_sz.w - wnd_size_st.cur_w, 50);
	if (wnd_size_st.cur_h < jsconn_popup_min_sz.h)
		rsz_h = Math.max(jsconn_popup_min_sz.h - wnd_size_st.cur_h, 50);

	// The reason we use resizeBy rather than resizeTo is because the outer window
	// contains various non-client elements (e.g. status bar), and it's difficult
	// for us to predict the size. If we got it wrong, we could get into an endless
	// loop. With resizeBy, we are guaranteed to terminate.
	if (rsz_w || rsz_h)
	{
		var new_cmd = "window.resizeBy(" + rsz_w + "," + rsz_h + ");";
		setTimeout(new_cmd, 5);
	}
	
	
	// Fix the old-Mozilla clipping area bug.
	// We don't want to run this on IE. (Possibly there are bad side effects.)
	if (evt)
	{
		try
		{
			setTimeout("window.scrollTo(0,0);", 2);
		}
		catch (e)
		{
		}
	}
}

// store_orig_size - store the size of the window prior to the auto-resize.
// (this is probably not needed)
function store_orig_size()
{
	calc_client_sz(sz_obj);
	wnd_size_st.orig_w = sz_obj.w;
	wnd_size_st.orig_h = sz_obj.h;
	
	calc_autodiv_sz(sz_obj);
	wnd_size_st.orig_dw = sz_obj.w;
	wnd_size_st.orig_dh = sz_obj.h;
	
	calc_elem_sz("console_iframe", sz_obj)
	wnd_size_st.orig_fw = sz_obj.w;
	wnd_size_st.orig_fh = sz_obj.h;
}

// store_auto_size - store the size of the window after the auto-resize.
// (all future adjustments are calculated relative to this size)
function store_auto_size()
{
	calc_client_sz(sz_obj);
	wnd_size_st.auto_w = sz_obj.w;
	wnd_size_st.auto_h = sz_obj.h;

	calc_autodiv_sz(sz_obj);
	wnd_size_st.auto_dw = sz_obj.w;
	wnd_size_st.auto_dh = sz_obj.h;
}


// do_auto_resize - called when the popup window is loaded, or
// when the layout changes.
function do_auto_resize()
{
	if (!wnd_size_st.already_resized)
	{
		store_orig_size();
		auto_resize_popup();
		store_auto_size();
		wnd_size_st.already_resized = true;
	}
}


/******************************
       Misc functions
 ******************************/

// in_hist_wnd - does the history window have the focus?
var in_hist_wnd = false;

// on_console_focus - track this so we can enable the cursor.
function on_console_focus(evt)
{
//debug_msg(" on_console_focus ");
	in_hist_wnd = true;
	
	// If the connect buttons are disabled, then clicking the history
	// window is the only way to establish the connection.
	if (!acon_prefs.use_connect_btns)
	{
		if (!cons_connected)
			on_connect_clicked(); 
	}
	
	// Change the input focus to the shunt. Don't do this if there is selected
	// text, since then the user would lose the selection. All that means is
	// that Ctrl-V won't work when there is a selection.
	if (!isConsoleDocTxtSelected())
	{
		focus_on_cons_input_shunt();
	}
}

// on_console_blur - track this so we can disable the cursor.
function on_console_blur(evt)
{
//debug_msg(" on_console_blur ");
	in_hist_wnd = false;
}


// rescroll_history - make sure the history window is scrolled down to the
// bottom whenever new text appears. 
function rescroll_history()
{
	try
	{
		var cbody = getConsoleDoc().body;
		var st_y = Math.max(cbody.scrollHeight - cbody.clientHeight, 0);

		var console_elem = getConsoleFrame();
		console_elem.scrollTo(0, st_y);
	}
	catch (e)
	{
		debug_msg("error in rescroll_history: " + e);
	}
}

// reset_gui_idle_timer - this will prevent the JS idle timer from closing the
// window. Normally the timer is reset when the user submits a form or clicks
// a link, but with Ajax in a popup window, that doesn't happen anymore.
function reset_gui_idle_timer()
{
	try
	{
    	// Use the function from Util.js.
		resetInactiveCountDown();
	}
	catch (e)
	{
		// Ignore any errors, since not every page has the idle timer.
	}
}


// BEGIN-STRIP-BLOCK
/******************************
        Debug output
 ******************************/

// debug_msg - print to the debug window on the bottom right.
// (Don't enable this in production builds.)
function debug_msg(e)
{
	if (!verbose_debug) 
		return;
		
	debug_msg_i(e);
}


// debug_kbd - same as debug_msg, but with different flag to enable it.
function debug_kbd(e)
{
	if (!keyboard_debug) 
		return;
		
	debug_msg_i(e);
}


// debug_msg_i - internal implementation of debug_msg and debug_kbd.
function debug_msg_i(e)
{
	try
	{
		oElem = document.getElementById("jsconn_debugmsg");
		
		if (oElem)
		{
			var txt = oElem.value;
			txt += e + "\n";

			/* When the debug buffer gets too big, the JS slows down to a snail's
			crawl. If we wanted to be clever, we could use a second console
			window for debug output. But there is too much risk of that conflicting
			with the operation of the console. So we settle for this crude
			algorithm to restrict the length. */
			if (txt.length > 2000)
			{
				txt = txt.substring(txt.length - 2000);
			}
			
			oElem.value = txt;
		}
		else window.status = e;
	}
	catch (e2)
	{
		alert(e2);
	}
	
	alert(e);
//	toJavaScriptConsole(e);
/*
	var consoleService = Components.classes['@mozilla.org/consoleservice;1']
		.getService(Components.interfaces.nsIConsoleService);
	consoleService.logStringMessage(e);*/
//	Components.utils.reportError(e);
}

// print_status - print some variables (for debugging)
function print_status()
{
	try
	{
	alert(""
	  + " connecting = " + acon_connecting
	  + "\n cmd_history.length = " + cmd_history.length
	  + "\n cmd_hist_pos = " + cmd_hist_pos
	  + "\n history len = " + cnsl_buffer_cur_len
	  + "\n cursor_x_pos = " + cursor_x_pos
	  + "\n inBuf = " + inBuf
	  + "\n startup_tries = " + startup_tries
	  );
	}
	catch (e)
	{
		alert("Error in print_status.");
	}
}


// jsconn_test1 - test function (set this to do whatever you want).
function jsconn_test1()
{
	try
	{
	alert(""
	  + " Original Wnd = (" + wnd_size_st.orig_w + "," + wnd_size_st.orig_h + ")"
	  + "\nAuto Wnd = (" + wnd_size_st.auto_w + "," + wnd_size_st.auto_h + ")"
	  + "\nOriginal Div = (" + wnd_size_st.orig_dw + "," + wnd_size_st.orig_dh + ")"
	  + "\nAuto Div = (" + wnd_size_st.auto_dw + "," + wnd_size_st.auto_dh + ")"
	  + "\nCurrent Wnd = (" + wnd_size_st.cur_w + "," + wnd_size_st.cur_h + ")"
	  );
	}
	catch (e)
	{
		alert("Error in test function: " + e);
	}
}


// END-STRIP-BLOCK

// on_clear_clicked - clear the bottom right debug window.
function on_clear_clicked()
{
	// Clear debug buffer
	oElem = document.getElementById("jsconn_debugmsg");
	if (oElem)
	{
		oElem.value = "";
	}
}


/******************************
        String Tables
 ******************************/

var alert_strings_obj = {
	conn_msg           : "Click here to connect...",
	conn_failed        : "Unable to connect.", 
	conn_lost          : "Connection lost.",
	cookie_save_failed : "Failed to save cookie: ",
	page_init_failed   : "Failed to initialize console page.",
	excp               : "exception: ",
	setting_failed     : "Failed to change the console setting.",
	connected_st       : "(connected)",
	disconnected_st    : "(not connected)",
	detached_st        : "(detached)",
	title_name         : "CLI Console"
};

// set_alert_string - used to overwrite the above strings for
// multi-language support.
function set_alert_string(key, val)
{
	try {
		alert_strings_obj[key] = val;
	}
	catch (e)
	{}
}


