/******************************
 Javascript Console
 Init by A. Krywaniuk, Jan 2006
 Copyright Fortinet, inc.
 All rights reserved
 ******************************/

// EXPORT_SYMBOL on_customize_clicked
// EXPORT_SYMBOL jsconn_test1
// EXPORT_SYMBOL on_clear_clicked
// EXPORT_SYMBOL print_status
// EXPORT_SYMBOL set_alert_string
// EXPORT_SYMBOL console_window

/* The Javascript viewer is divided into multiple subcomponents. To use it, you
should include all of the following JS files:

jsconsole.js - general stuff (this file)
jsconsole_kbd.js - keyboard handlers
jsconsole_ajax.js - Ajax IO functions
jsconsole_term.js - terminal functionality
jsconsole_init.js - initialization & cleanup functions

Some sample code to include in your C module:

"<script type=\"text/javascript\" src=\"/jsconsole.js?q=" CONFIG_GUI_NO "\"></script>\n"
"<script type=\"text/javascript\" src=\"/jsconsole_kbd.js?q=" CONFIG_GUI_NO "\"></script>\n"
"<script type=\"text/javascript\" src=\"/jsconsole_ajax.js?q=" CONFIG_GUI_NO "\"></script>\n"
"<script type=\"text/javascript\" src=\"/jsconsole_term.js?q=" CONFIG_GUI_NO "\"></script>\n"
"<script type=\"text/javascript\" src=\"/jsconsole_init.js?q=" CONFIG_GUI_NO "\"></script>\n"
*/


// STRIP-FUNCTION: debug_msg
// STRIP-FUNCTION: debug_kbd
 
 
// Turn off verbose_debug to get an idea of the true runtime performance.
//var verbose_debug = true;
var verbose_debug = false;

// Whether to enable debugging of keyboard events (with debug_kbd)
//var keyboard_debug = true;
var keyboard_debug = false;



/******************************
      Accessor Functions
 ******************************/

/* Small changes to the HTML layout can have big effects on the DOM
 representation (e.g. putting the console window in a separate frame
 required a lot of changes). So it is better to access some global
 elements via these centralized interfaces. */
 

// Hmmm... doesn't seem possible to cache this reference. JS is like
// Java - maybe we need a wrapper class or something.
var console_document = null; // TEST: cache the document for easy access.
var console_window = null;


// getConsoleDoc - this is the inner document (consoleframe.html).
function getConsoleDoc()
{
    if (window.frames.console_iframe.document)
	return window.frames.console_iframe.document;
    else
	return console_window.document;
}

function getConsoleWin()
{
    if (window.frames.console_iframe.document)
	return window.frames.console_iframe.window;
    else
	return console_window;
}

// getConsoleTable - this is the console <table> in the inner document.
function getConsoleTable()
{
	try
	{
		var cons_elem = getConsoleDoc().getElementById("history_txt");
		return cons_elem;
	}
	catch (e)
	{
		return null;
	}
}


// getConsoleFrame - this is the <iframe> in the outer document.
function getConsoleFrame()
{
	try
	{
	    if (window.frames.console_iframe.document)
		return window.frames.console_iframe;
	    else
		return console_window;
	}
	catch (e)
	{
		return null;
	}
}


// getConsoleTbody - this is what most functions that manipulate the history 
// buffer need.
function getConsoleTbody()
{
	// This shouldn't fail. Let it throw an exception if it does.
	var console_elem = getConsoleTable();
	var tbody = get_last_child_of_type(console_elem, "TBODY");
	return tbody;
}


// getCursor - get the <span> tag associated with the cursor (mostly so we can blink it).
function getCursor()
{
	try
	{
		var cons_elem = getConsoleDoc().getElementById("history_cursor");
		return cons_elem;
	}
	catch (e)
	{
		return null;
	}
}


/******************************
       User Preferences
 ******************************/

// acon_prefs - various preferences affecting the user experience.
// Most of these can be overridden from the CMDB or cookie.
var acon_prefs = {
	acon_fgColor : 'FFFFFF', // white
	acon_bgColor : '000000', // black
	acon_font_family : 'monospace',
	acon_font_size : '10pt',
	acon_consbuflen : 500, // length of the console scroll buffer
	acon_histbuflen : 50, // buffer that we save in the logo frame
	acon_showcmd : false,
	use_connect_btns : false, // whether to include support connect/disconnect buttons
	dummy : 0
};

/* acon_histbuflen basicaly does the same thing as acon_consbuflen, except that the strings
 are stored in a JS array instead of a DOM table and then they are later kept in the logo 
 frame. It will probably adds a small speed penalty to the JS code. 
 
 use_connect_btns=true represents the original design. use_connect_btns=false represents 
 the changes requested by Michael X. after the feature was complete. Whether or not to have
 the explicit connect & disconnect buttons is a matter of taste. If the disconnect option is
 not provided, the CLI session will idle out or be cleaned up when the connection limit is reached.
 One possibly annoying thing about removing the disconnect button is that the console history
 (scroll buffer) will not be accessible after the disconnect.
 
*/
 

// override_default_console_prefs - override the global defaults in acon_prefs
// with the specific defaults from the CMDB user.
function override_default_console_prefs()
{
	try
	{
		if (default_console_prefs)
		{
			clone_console_prefs(acon_prefs, default_console_prefs);
		}
	}
	catch (e)
	{}
}


// clone_console_prefs - deep copy function for console prefs.
function clone_console_prefs(dst, src)
{
	// Most of these are specified in jsconsole.c. If they aren't then we need
	// to make sure they aren't undefined before copying them.
	try
	{
		// Copying fields 1-by-1 leads to less catastrophic failure if the structure
		// definition in the C files gets out of sync w/ the above.
		for(prop in src)
			dst[prop] = src[prop];
	}
	catch (e)
	{
		debug_msg("Unable to copy default console prefs: " + e);
	}
}


// apply_console_prefs - update the onscreen screen display to reflect
// any changes to the global variables.
function apply_console_prefs(cnsl_doc)
{
	if (!cnsl_doc.styleSheets)
	{
		debug_msg("Failed to change console colours (no stylesheets)");
		return;
	}
		
	var my_css = new Array();
	
	if (cnsl_doc.styleSheets[0].cssRules)
	{
		// Mozilla case
		my_css = cnsl_doc.styleSheets[0].cssRules;
	}
	else
	{
		// IE case
		my_css = cnsl_doc.styleSheets[0].rules;
	}
	
	my_css[0].style.backgroundColor = acon_prefs.acon_bgColor;
	my_css[1].style.backgroundColor = acon_prefs.acon_bgColor;

	my_css[0].style.color = acon_prefs.acon_fgColor;
	my_css[1].style.color = acon_prefs.acon_fgColor;
	
	my_css[0].style.fontFamily = acon_prefs.acon_font_family;
	my_css[1].style.fontFamily = acon_prefs.acon_font_family;

	my_css[0].style.fontSize = acon_prefs.acon_font_size;
	my_css[1].style.fontSize = acon_prefs.acon_font_size;
	
	// Set the state of the command div.
	// TODO: one idea here would be to fade the command box div instead 
	// of hiding it (on the console customize page only). 
	// However, this proved to be more difficult than
	// expected. Mozilla supports style.opacity=<N>, and everything works
	// great. IE uses style.filter="alpha(opacity=<N>)" which seems to only
	// work on inline elements. I couldn't make it work with spans or divs.
	var elem_d = document.getElementById("jsconn_cmd_div");
	if (elem_d)
	{
		var st = (acon_prefs.acon_showcmd) ? "" : "none";
		elem_d.style.display = st;
	}
}


var min_sane_cmd_histlen = 10;
var max_sane_cmd_histlen = 100;

var min_sane_consbuf_len = 10;
var max_sane_consbuf_len = 99999;


// parse_console_cookie - Parse the cookie for the console preferences.
function parse_console_cookie()
{
	// The feature to persistently store the preferences in a cookie was cancelled.
	// But we still use the cookie to apply the changes for this session when they
	// are changed.
	
	
	// The cookie is generated in consolecustomize.js. As of 04/05/06...
	// The cookie format is: "fnjsconsole=version&fgColor&bgColor&hist_len&
	//    &font_family&font_size&cons_buf_size&show_command;"
	
	try 
	{
		var i = document.cookie.lastIndexOf("fnjsconsole=");
		
		if (i<0)
		{
			// No cookie found.
			debug_msg("No JS console cookie found.");
			return false;
		}
		
		// Initialize str at each stage, since it will be displayed
		// in the message if there is an error.
		var str = document.cookie;
		
		var j = str.indexOf(';', i);
		if (j < 0) throw(1);

		str = str.substring(i,j);
		i = str.indexOf('=');
		if (i<0) throw(7);
		
		str = str.substring(i+1);
		
		var vals = str.split('&');
		
		if (vals.length < 8) throw(2);
		if (vals[1].length != 6) throw(3);
		if (vals[2].length != 6) throw(4);
		
		var new_histlen = parseInt(vals[3]);
		if (new_histlen < min_sane_cmd_histlen) throw 5;
		if (new_histlen > max_sane_cmd_histlen) throw 6;
		
		var new_conslen = parseInt(vals[6]);
		if (new_conslen < min_sane_consbuf_len) throw 5;
		if (new_conslen > max_sane_consbuf_len) throw 6;
		
		var old_showcmd = acon_prefs.acon_showcmd;
		
		acon_prefs.acon_fgColor = vals[1];
		acon_prefs.acon_bgColor = vals[2];
		acon_prefs.acon_histbuflen = new_histlen;
		acon_prefs.acon_font_family = vals[4];
		acon_prefs.acon_font_size = vals[5];
		acon_prefs.acon_consbuflen = new_conslen;
		acon_prefs.acon_showcmd = (vals[7] == 1);
		
		if (old_showcmd != acon_prefs.acon_showcmd)
			wnd_size_st.need_resync = true;
	}
	catch (e)
	{
		debug_msg("error in parsing cookie - invalid format: " + e + str);
		return false;
	}
	
	return true;
}

function parse_connection_cookie()
{
	try 
	{
		var i = document.cookie.lastIndexOf("connection=");
		
		if (i<0)
		{
			// No cookie found.
			debug_msg("No JS cons cookie found.");
			return false;
		}
		
		// Initialize str at each stage, since it will be displayed
		// in the message if there is an error.
		var str = document.cookie;
		
		var j = str.indexOf(';', i);
		if (j < 0) throw(1);

		str = str.substring(i,j);
		
		i = str.indexOf('=');
		if (i<0) throw(2);
		
		str = str.substring(i+1);
		
		var vals = str.split('&');
		
		if (vals.length < 1) throw(3);
		
		if(vals[0] == 'true')
			cons_connected = true;
		else
			cons_connected = false;
	}
	catch (e)
	{
		debug_msg("error in parsing cookie - invalid format: " + e + str);
		return false;
	}
	
	return true;
}

function on_customize_clicked()
{
	// Add approximate height/width, and we will do the auto-resize at runtime.
	try {
		var open_hdl = window.opener?window.opener:window;
		var win_hdl = open_hdl.top.window;
		if(win_hdl.hPopupCustomize)
			// Focus on the existing page
			win_hdl.hPopupCustomize.focus();
		else
			// Open a new page and store the window handle in logo page
	    	win_hdl.hPopupCustomize = window.open("/system/consolecustomizedlg","","width=450,height=428,top=100,left=200,resizable=1");
	}
	catch (e)
	{}
}


/******************************
     Window Size functions
 ******************************/

// jsconn_popup_min_sz - minimum size of the popup window.
// TODO: these minimum size settings are fairly arbitrary.
var popup_min_sz = {
	h: 200,
	w: 400
};

// console_resize_handler - window.onresize event handler
// Adjust the div & iframe elements to have the appropriate size.
function console_resize_handler(evt)
{
	return resize_handler(evt, "console_iframe");
}

// resize_console_to_container - resize the console window to fill up the
// containing window. (This would be an alternate way to implement our
// current resizing functions, but currently we only use it for the
// widgetized console.)
function resize_console_to_container()
{
    // Not needed for standard dashboard, only widget container.
    if (!window.wij_mode) return;

    // Calculate the available window size.
    calc_client_sz(sz_obj);
    var client_ht = sz_obj.h;
    var client_wd = sz_obj.w;


    // Calculate the current height of the console frame.
    var frm = $("console_iframe");
    var frm_ht = frm.offsetHeight;

    // Calculate the maximum extent that we are currently using.
    var v_extent;
    var cmd_h_extent;

    if (acon_prefs.acon_showcmd)
    {
        var eb = $("jsconn_cmd");
        var pos = Position.cumulativeOffset(eb);
        v_extent = pos[1] + eb.offsetHeight;
        cmd_h_extent = pos[0] + eb.offsetWidth;
    }
    else
    {
        v_extent = frm_ht;
    }

    // Gap is the current margin, which is probably bigger than the desired margin.
    var gap = client_ht - v_extent;
    var ideal_margin = 16; // TODO: any logic behind this?
    frm.style.height = (frm_ht + gap - ideal_margin) + "px";


    if (acon_prefs.acon_showcmd)
    {
        // Resize the command window to have 10% border on each side. Gap is
        // multiplied by 2 because the H-alignment is center.
        var eb = $("jsconn_cmd");
        var desired_h_extent = (client_wd * 0.9);
        gap = (desired_h_extent - cmd_h_extent);
        eb.style.width = (eb.offsetWidth + gap*2) + "px";
    }
}


/******************************
       Misc functions
 ******************************/

// in_hist_wnd - does the history window have the focus?
var in_hist_wnd = false;

// on_console_focus - track this so we can enable the cursor.
function on_console_focus(evt)
{
//debug_msg(" on_console_focus ");
	in_hist_wnd = true;
	
	// If the connect buttons are disabled, then clicking the history
	// window is the only way to establish the connection.
	if (!acon_prefs.use_connect_btns)
	{
		if (!cons_connected)
			on_connect_clicked(); 
	}
	
	// Change the input focus to the shunt. Don't do this if there is selected
	// text, since then the user would lose the selection. All that means is
	// that Ctrl-V won't work when there is a selection.
	if (!isConsoleDocTxtSelected())
	{
		focus_on_cons_input_shunt();
	}
}

// on_console_blur - track this so we can disable the cursor.
function on_console_blur(evt)
{
//debug_msg(" on_console_blur ");
	in_hist_wnd = false;
}


// rescroll_history - make sure the history window is scrolled down to the
// bottom whenever new text appears. 
function rescroll_history()
{
	try
	{
		var cbody = getConsoleDoc().body;
		var st_y = Math.max(cbody.scrollHeight - cbody.clientHeight, 0);

		var console_elem = getConsoleFrame();
		console_elem.scrollTo(0, st_y);
	}
	catch (e)
	{
		debug_msg("error in rescroll_history: " + e);
	}
}

// reset_gui_idle_timer - this will prevent the JS idle timer from closing the
// window. Normally the timer is reset when the user submits a form or clicks
// a link, but with Ajax in a popup window, that doesn't happen anymore.
function reset_gui_idle_timer()
{
	try
	{
    	// Use the function from Util.js.
		resetInactiveCountDown();
	}
	catch (e)
	{
		// Ignore any errors, since not every page has the idle timer.
	}
}


// BEGIN-STRIP-BLOCK
/******************************
        Debug output
 ******************************/

// debug_msg - print to the debug window on the bottom right.
// (Don't enable this in production builds.)
function debug_msg(e)
{
	if (!verbose_debug) 
		return;
		
	debug_msg_i(e);
}


// debug_kbd - same as debug_msg, but with different flag to enable it.
function debug_kbd(e)
{
	if (!keyboard_debug) 
		return;
		
	debug_msg_i(e);
}


// debug_msg_i - internal implementation of debug_msg and debug_kbd.
function debug_msg_i(e)
{
	try
	{
		oElem = document.getElementById("jsconn_debugmsg");
		
		if (oElem)
		{
			var txt = oElem.value;
			txt += e + "\n";

			/* When the debug buffer gets too big, the JS slows down to a snail's
			crawl. If we wanted to be clever, we could use a second console
			window for debug output. But there is too much risk of that conflicting
			with the operation of the console. So we settle for this crude
			algorithm to restrict the length. */
			if (txt.length > 2000)
			{
				txt = txt.substring(txt.length - 2000);
			}
			
			oElem.value = txt;
		}
		else window.status = e;
	}
	catch (e2)
	{
		alert("Unable to access debug console: " + e2);
	}
	
//	alert(e);
//	toJavaScriptConsole(e);
/*
	var consoleService = Components.classes['@mozilla.org/consoleservice;1']
		.getService(Components.interfaces.nsIConsoleService);
	consoleService.logStringMessage(e);*/
//	Components.utils.reportError(e);
}

// print_status - print some variables (for debugging)
function print_status()
{
	try
	{
	alert(""
	  + " connecting = " + acon_connecting
	  + "\n cmd_history.length = " + cmd_history.length
	  + "\n cmd_hist_pos = " + cmd_hist_pos
	  + "\n history len = " + cnsl_buffer_cur_len
	  + "\n cursor_x_pos = " + cursor_x_pos
	  + "\n inBuf = " + inBuf
	  + "\n startup_tries = " + startup_tries
	  );
	}
	catch (e)
	{
		alert("Error in print_status.");
	}
}


// jsconn_test1 - test function (set this to do whatever you want).
function jsconn_test1()
{
	try
	{
	alert(""
	  + " Original Wnd = (" + wnd_size_st.orig_w + "," + wnd_size_st.orig_h + ")"
	  + "\nAuto Wnd = (" + wnd_size_st.auto_w + "," + wnd_size_st.auto_h + ")"
	  + "\nOriginal Div = (" + wnd_size_st.orig_dw + "," + wnd_size_st.orig_dh + ")"
	  + "\nAuto Div = (" + wnd_size_st.auto_dw + "," + wnd_size_st.auto_dh + ")"
	  + "\nCurrent Wnd = (" + wnd_size_st.cur_w + "," + wnd_size_st.cur_h + ")"
	  );
	}
	catch (e)
	{
		alert("Error in test function: " + e);
	}
}


// END-STRIP-BLOCK

// on_clear_clicked - clear the bottom right debug window.
function on_clear_clicked()
{
	// Clear debug buffer
	oElem = document.getElementById("jsconn_debugmsg");
	if (oElem)
	{
		oElem.value = "";
	}
}


/******************************
        String Tables
 ******************************/

var alert_strings_obj = {
	conn_msg           : "Click here to connect...",
	conn_failed        : "Unable to connect.", 
	conn_lost          : "Connection lost.",
	cookie_save_failed : "Failed to save cookie: ",
	page_init_failed   : "Failed to initialize console page.",
	excp               : "exception: ",
	setting_failed     : "Failed to change the console setting.",
	connected_st       : "(connected)",
	disconnected_st    : "(not connected)",
	detached_st        : "(detached)",
	title_name         : "CLI Console"
};

// set_alert_string - used to overwrite the above strings for
// multi-language support.
function set_alert_string(key, val)
{
	try {
		alert_strings_obj[key] = val;
	}
	catch (e)
	{}
}




/******************************
 Javascript Console
 Init by A. Krywaniuk, Jan 2006
 Copyright Fortinet, inc.
 All rights reserved
 ******************************/

// STRIP-FUNCTION: debug_msg
// STRIP-FUNCTION: debug_kbd



/******************************
      Keyboard Events
 ******************************/

var last_keydown = 0;
var last_keypress = 0;
var cachedCtrl = false;
var cancel_keypress = false;
var key_repeat_count = 0;


// kbevent_dbg_timer - timer for update_kbevent_debugging
var kbevent_dbg_timer = null;

var reset_welcome_ctime = null;

// Judge whether there is text selection on console window
function isConsoleDocTxtSelected()
{
	try {
		// Mozilla supports window.getSelection
		var cslWin = getConsoleWin();
		if(cslWin && cslWin.getSelection) {
			if(cslWin.getSelection().toString().length > 0)
				return true;
			return false;
		}
		
		var cslDoc = getConsoleDoc();
		if(cslDoc) {
			// Mozilla also supports document.getSelection
			if(cslDoc.getSelection) {
				if(cslDoc.getSelection().length > 0)
					return true;
				return false;
			}
			// IE supports document.selection
			if(cslDoc.selection) {
				if(cslDoc.selection.createRange().text.length > 0)
					return true;
				return false;
			}
			return false;
		}
	}
	catch (e)
	{}
}

// Judge whether there is text selection on console window
function isCmdTxtSelected()
{
	try {
		// TODO: Does not work on Mozilla
		if (window.getSelection)
		{
			if(window.getSelection().toString().length > 0)
				return true;
			return false;
		}
		else if (document.getSelection)
		{
			if(document.getSelection().length > 0)
				return true;
			return false;
		}
		else if(document.selection) {
			if(document.selection.createRange().text.length > 0)
				return true;
			return false;
		}
		return false;
	}
	catch (e)
	{}
}

// update_kbevent_debugging - update the outbuf and last key debug windows.
// (only used when verbose_debug is on)
function update_kbevent_debugging()
{
	var outbuf = document.getElementById("txt_outbuf");
	if (outbuf) outbuf.value = glob_outbuf;

	// Also update "last key" debug window
	var lkelem = document.getElementById("jsconn_lastkey");
	if (lkelem) lkelem.value = last_keydown + ";" + last_keypress;
}


// trigger_acon_timers - setup a timer to transmit the buffer. Make sure there
// is an active poll request as well.
function trigger_acon_timers()
{
	if (tx_loop_timer) clearTimeout(tx_loop_timer);
	tx_loop_timer = setTimeout("acon_tx_loop()", 10);

	// As a safety measure, make sure there is an active RX and if not, 
	// then make sure it will happen quickly.
	if (!acon_rx.in_progress)
	{
		if (rx_loop_timer) clearTimeout(rx_loop_timer);
		rx_loop_timer = setTimeout("acon_rx_loop()", 10);
	}
}

// act_on_keypress - send the data and update the debug windows
function act_on_keypress()
{
//debug_msg(" act_on_keypress ");
	// This timeout determines the responsiveness vs. bandwidth tradeoff. If the
	// user types very quickly, we will batch the input and delay sending it until
	// there is an N millisecond pause.
	
	// Set a timeout to send the buffer.
	trigger_acon_timers();
}


// send_one_keypress - will add one keypress from the console into the global buffer.
function send_one_keypress(charCode)
{
	glob_outbuf += String.fromCharCode(charCode);
	act_on_keypress();
}

// send_one_character - will add one ASCII character into the global buffer.
function send_one_character(ch)
{
	glob_outbuf += ch;
	act_on_keypress();
}


var vt100_escseq_cmd = "\033["; // 033 = 27 = Esc

// send_esc_sequence - implemented as a virtual keypress.
function send_esc_sequence(ch)
{
	glob_outbuf += vt100_escseq_cmd;
	glob_outbuf += ch;
	act_on_keypress();
}


// get_cons_kbd_event - Resolve the IE/Mozilla incompatibility in accessing 
// the keyboard event object.
function get_cons_kbd_event(evt_p)
{
	if (evt_p)
	{
		// Mozilla case - event passed as a parameter.
		return evt_p;
	}
	
	// IE case - get event from window object.
	evt = getConsoleFrame().event;
	if (evt) return evt;
	
debug_msg("event is null in console key handler");
	throw("event is null in console key handler");
}


// get_cmd_kbd_event - same as get_cons_kbd_event, but uses different document object
function get_cmd_kbd_event(evt_p)
{
	if (evt_p)
	{
		// Mozilla case - event passed as a parameter.
		return evt_p;
	}
	
	// IE case - get event from window object.
	evt = window.event;
	if (evt) return evt;
	
debug_msg("event is null in command key handler");
	throw("event is null in command key handler");
}


// crack_kbd_event - Returns the keypress code associated with the event.
function crack_kbd_event(evt)
{
//debug_kbd("crack_kbd_event: which= " + evt.which + ", keyCode= " + evt.keyCode + ", charCode=" + evt.charCode);

	if (evt.which)
	{
		// Mozilla - standard case (e.g. alpha-numeric keys)
		return evt.which;
	}
	else if (evt.keyCode)
	{
		// IE - standard case (all keys)
		// Mozilla - a few special keys (esc, tab)
		return evt.keyCode;
	}
	else if (evt.charCode)
	{
		// I never saw this case happen - but found others who used it on the web.
		// Basically, on Mozilla, "charCode" is always the same as "which",
		// except that that charCode of CR is 0, while which is 13.
		return evt.charCode;
	}
	
debug_msg("No key code in keypress");
	throw("No key code in keypress");
}


/* DHTML keyboard event processing.

Keycode cracking is a bizarre and difficult chore. The keycodes correspond partially to
ASCII, but there is a lot of additional interpretation required to interpret them.

A table of ASCII codes is at www.lookuptables.com.

A-Z: 65;65 (65='A')
a-z: 65;97 (65='A', 97='a')
0-9: 48;48 (48='0')
tab: 9;9 (9=tab) - keydown only
del: 46;46 (46='.') - we translate this to ASCII DEL, which is effectively BS
ins: 45;45 (45='-') - we ignore it
home: 36;36
end: 35;35
alt: 18;18 (device control) - ignore it
sysreq: 44;44 (44=',') - ignore it
caps: 20;20 (device control) - no special handling
!@#$%^&*() [top row]: (unpredictable) - no special handling
F1-F12: 112;112 (112='p') - ignore it

Each time the user presses a key, we receive one keydown events, one (or more or zero) 
keypress events, and one key up event. Generally, it is 1-1-1, but some of the extended
keys are stolen by the OS and don't translate into a keypress. Also, if the user holds
a key down then auto-repeat kicks in and you may get multiple keypresses for one keydown.

We can detect the difference between tab & shift-tab, but there is no way to translate this
into ASCII. It would be nice to do the command iteration in reverse with shift-tab, but I
checked on the regular console and it appears to treat shift-tab just like tab as well.

Caps lock is only detected on Linux - good thing we don't need it.
SysRq is also only captured on Linux.

A few keys can only be captured in keydown: arrow keys, tab, BS, alt. 
This can even vary on a per-platform basis. For IE, PgUp, PgDown, and Del are also not
translated into keypress events, although they are on Mozilla.

*/


// cons_keydown - on_keypress handler for the console window.
function cons_keydown(evt_p) 
{
debug_kbd(" cons_keydown ");

	try
	{
		var evt = get_cons_kbd_event(evt_p);
		var key_code = crack_kbd_event(evt);
	}
	catch (e)
	{
		// Unable to handle keydown.
		debug_msg("Unable to crack console keydown");
		return true;
	}
	
	last_keydown = key_code;
	last_keypress = 0;
	key_repeat_count = 0;
	
	// Reset the cmd window's last keydown flag as well. Typing in the console
	// window should interrupt a repeated tab sequence.
	last_cmd_keydown = 0;

	// Set the editline buffer as possibly dirty, which means that we will have to
	// send the "clear line" command if the user switches to the command window.
	editline_buffer_maybe_dirty = true;
	
	// Not sure if we really need this. (I guess the user could release the
	// control key during a repeated keypress sequence.)
	cachedCtrl = evt.ctrlKey;  
  
	// If this is a special key, handle it here, since we won't necessarily
	// receive the keypress event.
	if (!handle_kd_special_key(evt, last_keydown))
	{
		cancel_keypress = true;
		return false;
	}
	
	// The standard case - handle the event in the keydown handler.
	cancel_keypress = false;
	return true ;
}

var editline_buffer_maybe_dirty = false;
// cons_keypress - on_keypress handler for the console window.
function cons_keypress(evt_p)
{
debug_kbd(" cons_keypress ");
	try
	{
		var evt = get_cons_kbd_event(evt_p);
		var key_code = crack_kbd_event(evt);
		last_keypress = key_code;
	}
	catch (e)
	{
		// Unable to handle keypress.
		// Returning true here may have some effect on event bubbling (need to check it)
		debug_msg("Unable to crack console keypress");
		return true;
	}
	
	// DEBUG: display the contents of the buffer on the GUI
	// Do this prior to possibly ignoring the keypress & returning false below.
	if (verbose_debug)
	{
		try
		{
			// No one wants to see this info when they are typing quickly.
			// Use a timeout for a) rate-limiting, and b) to give the Tx event priority.
			if (kbevent_dbg_timer) clearTimeout(kbevent_dbg_timer);
			kbevent_dbg_timer = setTimeout("update_kbevent_debugging()", 100);
		}
		catch (e)
		{
		}
	}

	// Cancel the keypress if it was handled in keydown.
	if (cancel_keypress)
	{
		key_repeat_count++;
		
		if (key_repeat_count > 1)
		{
debug_kbd("kp - repeat special keypress: " + last_keypress);
			if (handle_kd_special_key(evt, last_keydown))
			{
debug_kbd("unexpected result of repeating special keypress");
			}
		}
		else
		{
debug_kbd("kp - ignore cancelled keypress: " + last_keypress);
		}

		return false;
	}
	
	// A variety of keystrokes have the same keypress and keydown values
	if (last_keypress == last_keydown)
	{
		// F1 to F12 - ignore them
		if (last_keypress >= 112 && last_keypress <= 123)
		{
debug_kbd("kp - special Fn key case: " + last_keypress);
			return false;
		}
		
		switch (last_keypress)
		{
		case 33: // PgUp
		case 34: // PgDown
			// Return true, which will allow the event to bubble on Mozilla.
			// (IE can only capture this event in keydown, so it will be
			// bubbled there instead.)
			return true;
			
		case 44: // SysRq/Print Screen
debug_kbd("kp - ignore SysRq");
			return false;
			
		case 45: // Ins
			return false;
			
		// The next 4 should be trapped in keydown.
		case 18: // Alt
		case 35: // End
		case 36: // Home
		case 46: // Del
			debug_msg("kp - key was not trapped in keydown: " + last_keypress);
			return false;
		}
	}
	
	// Sending Ctrl characters can do various bad things, such as
	// possibly hanging the browser or generating VT sequences that
	// we haven't considered. So we only allow a few "known safe"
	// ones.
	if (cachedCtrl)
	{
		// Only consider ctrl-<Alpha char>
		// Use last keydown, which should have the uppercase version of the key.
		if (last_keydown >= 65 && last_keydown <= 90)
		{
			var ch = String.fromCharCode(last_keydown);

			switch(ch)
			{
				case 'C':
					// Our handling of Ctrl-C is schizophrenic by design. If there is some selected
					// text in the console window, we allow the OS to steal the keypress
					// (for copy & paste). Otherwise, we send it to the console. (Calling
					// send_one_keypress is unneccessary because it's already done in
					// cons_keydown.)
					if(isConsoleDocTxtSelected())
						return true;
					break;
					
				case 'V':
					// We handle Ctrl-V in handle_kd_special_key (because IE doesn't seem
					// to support it here). So no need to handle it twice.
					return true;
					
				// These are a few emacs editing commands that are known to be safe.
				case 'B':
				case 'E':
				case 'W':
					// Convert to ctrl-character.
					send_one_keypress(last_keydown - 64);
					return false;
			}
		}

		return false;
	}

	// Some sanity checks - extended ASCII (don't know how this would happen)
	if (last_keypress > 127)
		return true;
		
	send_one_keypress(last_keypress);

	return false;
}


// cons_keyup - we don't do anything important here currently.
function cons_keyup(evt)
{
// Don't reset these here... they are used for debugging.
//	last_keypress = 0;
//	last_keydown = 0;
	return true ;
}


// send_pasted_data - called when the keyboard routines detect a ctrl-v in the 
// console input shunt.
function send_pasted_data(evt)
{
debug_msg("send_pasted_data called");
	var xdoc = getConsoleDoc();
	var elem = xdoc.getElementById("cons_input_shunt");
	
	var str = elem.value;
	glob_outbuf += str;
	elem.value = "";
	
	act_on_keypress();
}

// handle_kd_special_key - handling for extended keys that can't be processed
// in the keypress handler.
function handle_kd_special_key(evt, key_code)
{
	switch (key_code)
	{
	case 8: // BS
		send_one_keypress(127); // DEL
		return false;
	
	case 9: // TAB
		// It is possible to detect shift-tab using (evt.shiftKey), but there is
		// no way to encode that in ASCII anyway.
debug_msg("kd - tab: " + last_keydown);
		send_one_keypress(last_keydown);
		return false;

	// CR - The Unix standard return character is \n, not \r so to avoid interoperability
	// problems we had better translate.
	case 13:
		send_one_character('\n');
		return false;
		
	case 18: // ALT - ignore it
debug_kbd("kd - alt: " + last_keydown);
		return false;
		
	case 38 :
debug_kbd("kd - up-arrow: " + last_keypress);
		send_esc_sequence('A') ;
		return false ;

	case 40:
		send_esc_sequence('B') ;
		return false ;

	case 39:
		send_esc_sequence('C') ;
		return false ;

	case 37:
		send_esc_sequence('D') ;
		return false ;
		
	case 35: // End
		// Ctrl-E: Emacs command to go to the end of the line.
		send_one_keypress(5);
		return false;

	case 36: // Home
		send_esc_sequence('H') ;
		return false;
			
	case 46: // Del
		// We treat this as backspace, which is the same as
		// the serial console.
		send_one_keypress(127); // ASCII DEL = 127
		return false;

	case 67: // c
		if(cachedCtrl) {// Ctrl-c
			if(isConsoleDocTxtSelected())
				return true;
			// There is nothing selected on console window, we add the 
			// cmd to global buffer
			send_one_keypress(key_code - 64);
			return false;
		}
		return true;
		
	case 85: // u
		if(cachedCtrl) {// Ctrl-u
			send_one_keypress(key_code - 64);
			return false;
		}
		return true;
		
	case 86: // v
		if(cachedCtrl)
		{
			align_shunt_to_window();
			setTimeout("send_pasted_data()",1);
			return true;
		}
		return true;

	default:
		return true;
	}
}


/* Predefined VT-100/Emacs escape sequences. */

// Clear line - used to clear the current line in case the user starts typing in
// the console window and switches to the command window later.
// Another one that works is "\033[H\013" ^[H (VT-100 home) + Ctrl-K (Emacs kill to EOL).

var vt100_clearline_cmd = "\025"; // Ctrl-U (Emacs kill full line).


/*

Most standard alphanumeric and system keys can be captured & handled in keypress,
and it is preferable to handle them there.

However, on IE, a number of the ones we are interested in, such as the arrow keys 
and TAB key can only be caught in keydown.

TAB requires special processing in the command edit window, since we need to copy the
results of the completion into the window after we receive them. Also, the CLI
processing of tab relies on internal variables, and not just the current buffer state.
So if we want to send multiple consecutive tabs, we *cannot* resend the whole
current line.

*/


// Command window tab processing state
var tab_proc_st = {
	pre_cmd : "",
	in_proc : false,
	recent_typing : false,
	stage : 1
};


var last_cmd_keydown = 0;


// cmd_keydown - keydown handler for the command window.
function cmd_keydown(evt_p)
{
//debug_msg(" cmd_keydown ");

	try
	{
		var evt = get_cmd_kbd_event(evt_p);
		var key_code = crack_kbd_event(evt);
	}
	catch (e)
	{
		// Unable to handle keypress.
		// Returning true here may have some effect on event bubbling (need to check it)
		return true;
	}
	
	var rv = false; // false if we handle the event.
	
	cachedCtrl = evt.ctrlKey;
	tab_proc_st.recent_typing = true;
	
	if(cachedCtrl && key_code == 67) { // Ctrl-c
		if(!isCmdTxtSelected())
			// Send "Ctrl-c" to the global buffer
			send_one_keypress(key_code - 64);
		return false;
	}
	else if (key_code == 38) // up arrow
	{
//debug_kbd("keydown:handle up arrow");
		var str = cmd_hist_up();
		
		if (str)
		{
//debug_kbd("up arrow - using: " + str);
			var cmd_elem = document.getElementById("jsconn_cmd")
			cmd_elem.value = str;
		}
	}
	else if (key_code == 40) // down arrow
	{
//debug_kbd("keydown:handle down arrow");
		var str = cmd_hist_down();
		var cmd_elem = document.getElementById("jsconn_cmd");
		cmd_elem.value = str;
debug_kbd("down arrow - using: " + str);
	}
	else if (key_code == 9) // TAB command completion
	{
debug_kbd("keydown:handle TAB");
		// We need to do some kind of kludgy command window fixup, so save the
		// current state of the command window & input buffer.
		var cmd_elem = document.getElementById("jsconn_cmd");
		tab_proc_st.pre_cmd = cmd_elem.value;
		tab_proc_st.stage = 1;
		tab_proc_st.recent_typing = false;
		tab_proc_st.in_proc = true;
		
		// Again, this is a little kludgy. We use buffer_cmd to trigger a TX
		// event, but then overwrite the buffer later. For consecutive tabs
		// we need a special way of handling the event since we couldn't just
		// resend the whole buffer.
		if (last_cmd_keydown == 9)
		{
			buffer_cmd(false, false);
			glob_outbuf = '\t';
		}
		else
		{
			buffer_cmd(false, false);
			glob_outbuf = vt100_clearline_cmd + glob_outbuf + '\t';
		}
	}
	else
	{
		rv = true;
	}

	// IE's way of cancelling the event.
	evt.returnValue = rv;
	
	// Setting this *after* processing the tab key.
	last_cmd_keydown = key_code;
	
	return rv;
}

// cmd_keypress - keypress handler for the command window.
function cmd_keypress(evt_p)
{
//debug_kbd(" cmd_keypress ");

	try
	{
		var evt = get_cmd_kbd_event(evt_p);
		var key_code = crack_kbd_event(evt);
	}
	catch (e)
	{
		// Unable to handle keypress.
		// Returning true here may have some effect on event bubbling (need to check it)
		return true;
	}
	
	var rv = false; // false if we handle the event.
	
	if (key_code == 13) // CR
	{
debug_kbd("keydown:handle CR");
		if(editline_buffer_maybe_dirty)
			glob_outbuf += vt100_clearline_cmd;
		buffer_cmd(true, true);
	}
	else if (key_code == 63) // ? command completion
	{
debug_kbd("keydown:handle ?");
		buffer_cmd(false, false);
		glob_outbuf = vt100_clearline_cmd + glob_outbuf + '?';
	}
	else if (key_code == 9) // TAB command completion
	{
		// On Mozilla, this will still be called, even if we try
		// to cancel the event in cmd_keydown. We need to cancel
		// it again here (this is specific to old Mozilla, not Firefox).
	}
	else
	{
		rv = true;
	}

	// IE's way of cancelling the event.
	evt.returnValue = rv;
	
	return rv;
}


// cmd_keyup - not really needed.
function cmd_keyup(evt_p)
{
//debug_kbd("cmd_keyup");
}


var saved_glob_outbuf;

// buffer_cmd - Add the contents of the command window to the global
// buffer and trigger a send event. In some cases, we are doing ? or
// tab completion, so we don't want to add the CR and reset the
// command window afterwards.
function buffer_cmd(add_cr, clear_line)
{
	var cmd_elem = document.getElementById("jsconn_cmd");
	var new_str = cmd_elem.value;
	
	// Preserve anything remaining in the buffer if the console is
	// connected. Otherwise, just overwrite it.
	if (cons_connected)
	{
		glob_outbuf += new_str;
	}
	else
	{
		glob_outbuf = new_str;
	}
	
	
	// Clear the edit box when sending the text (differs between tab 
	// completion and regular cases).
	if (clear_line)
	{
		cmd_elem.value = "";
	}
	
	// CR-terminate the line (also differs w/ tab completion).
	if (add_cr)
	{
		add_to_cmd_history(new_str);
		glob_outbuf += "\n";
		editline_buffer_maybe_dirty = false;
	}
	else
	{
		// We are transmitting a partial command (e.g. command completion), which
		// means that the edit buffer will now be dirty.
		editline_buffer_maybe_dirty = true;
	}
	
	
	// DEBUG: resync the debug buffer display
	if (verbose_debug)
	{
		var outbuf = document.getElementById("txt_outbuf");
		if (outbuf) outbuf.value = glob_outbuf;
	}
	
	
	// If the console is connected, just send the buffer. If we are not
	// connected, delay sending the buffer for 1 second (a bit of a
	// kludge, but good enough for this case).
	if (cons_connected)
	{
		// Set a timer to send the buffer.
		trigger_acon_timers();
	}
	else
	{
		// Save the glob_outbuf, since tx_buffer will clear it during the connection
		// attempt.
		saved_glob_outbuf = glob_outbuf;
		
		// Connect, and then send the command.
		on_connect_clicked();
		
		// Blindly waiting for 1 second isn't ideal, but it solves the general problem.
		setTimeout("send_delayed_cmd();", 1000);
		
	}
}


// send_delayed_cmd - wait until the console is connected, then send a command.
function send_delayed_cmd()
{
	// Make sure the connection completed.
	if (!cons_connected)
		return;
		
	// Sanity check: make sure the user isn't typing right now.
	if (glob_outbuf.length > 0)
		return;
	
	// Clear the current line before sending.
	glob_outbuf = vt100_clearline_cmd + saved_glob_outbuf;
	saved_glob_outbuf = "";
	
	trigger_acon_timers();
}


// buffer_and_submit - called when the "go" button is clicked. Not currently used.
function buffer_and_submit()
{
	buffer_cmd(true, true);
}



/******************************
 Javascript Console
 Init by A. Krywaniuk, Jan 2006
 Copyright Fortinet, inc.
 All rights reserved
 ******************************/

// STRIP-FUNCTION: debug_msg
// STRIP-FUNCTION: debug_kbd
 
// EXPORT_SYMBOL on_connect_clicked
// EXPORT_SYMBOL on_disconnect_clicked
// EXPORT_SYMBOL acon_tx_buffer()


/******************************
      Ajax IO functions
 ******************************/

var glob_outbuf = "";

var tx_loop_timer = null;
var rx_loop_timer = null;

var acon_connecting = false;
var acon_disconnecting = false;
var acon_tx_success = false;

// Throttling counter.
var rx_since_last_tx = 0;


// on_acon_tx_done - called when the Ajax tx is complete.
function on_acon_tx_done()
{
//debug_msg("on_acon_tx_done");

	var reply_txt = this.xmlhttp.responseText;

	this.cleanup();
	
	// In case anyone is listening, notify them of the successful transaction
	if (acon_connecting)
	{
		acon_tx_success = true;
		handle_connection_result(reply_txt);
	}
	else if (acon_disconnecting)
	{
debug_msg("Handle successful disconnect.");
		acon_tx_success = true;
		handle_disconnect_event(connection_lost);
	}
}


// acon_tx_buffer - transmit the global buffer.
// Returns 0 if the call did nothing, 1 if the request was sent, and 2 if the 
// object is busy and you should retry later.
function acon_tx_buffer()
{
//debug_msg(" acon_tx_buffer ");
	// Check & set the semaphore/watchdog.
	if (acon_tx.in_progress)
		return 2;
	
	acon_tx.set_watchdog();

	// Choose the command action and URL format based on our current state.
	var str_body;
	if (acon_connecting)
	{
		str_body = "act=connect";
	}
	else if (acon_disconnecting)
	{
		str_body = "act=disconnect&sid=" + console_session_id;
	}
	else if (glob_outbuf.length > 0)
	{
		str_body = "act=xmit&sid=" + console_session_id + "&cmd=" + encodeURIComponent(glob_outbuf);
	}
	else
	{
		// If the buffer is empty, just send a keepalive.
//debug_msg("Send keepalive packet");
		str_body = "act=keepalive&sid=" + console_session_id;
	}
	
	var xml_obj = get_xmlhttp();
	acon_tx.xmlhttp = xml_obj;
	
	// Not sure if the line below actually does anything (fails on IE)
//	xml_obj.async = use_async;
	
	xml_obj.onreadystatechange = acon_tx.readystatechg_fn_i;
	
	// AFAICT, string method will always be POST here.
	var str_method = ((str_body == "") ? "GET" : "POST");
	
	try
	{
		// True means to use an asynchronous connection. Always do this, since
		// blocking connections may hang the browser if they fail. 
		xml_obj.open(str_method, "/httpclirqst", true);
		
		// Set the request header to make sure the browser won't use cache content 
		xml_obj.setRequestHeader("Pragma", "no-cache");
		xml_obj.setRequestHeader("Cache-Control", "no-store, no-cache, must-revalidate");
		xml_obj.setRequestHeader("If-Modified-Since", "Sat, 1 Jan 2000 00:00:00 GMT");
		
		xml_obj.send(str_body);
	}
	catch (e)
	{
		acon_tx.clear_watchdog();
debug_msg("Fatal error in acon_tx_buffer()");
	}

	// DEBUG: write pending text to display textarea
	if (verbose_debug)
	{
		var outbuf = document.getElementById("txt_outbuf");
		if (outbuf) outbuf.value = glob_outbuf;
	}

	// Delay to the end of the file to reset glob_outbuf because we may
	// want to display it in the debug window above.
	glob_outbuf = "";
	
	// Reset the rx throttling counter.
	rx_since_last_tx = 0;

	return 1;
}


// acon_rx_poll - called to begin an Ajax poll.
function acon_rx_poll()
{
	// Check & set the semaphore/watchdog.
	if (acon_rx.in_progress)
	{
//debug_msg("ignore poll");
		return;
	}
		
	acon_rx.set_watchdog();
	
	var xml_obj = get_xmlhttp();
	acon_rx.xmlhttp = xml_obj;

	xml_obj.onreadystatechange = acon_rx.readystatechg_fn_i;
	
// debug_msg("acon: poll");
	try
	{
		// Use "get" instead of "post" for the polling requests.
		// Use "true" as the 2nd parameter for async.
		var str_body = "/httpclirqst?act=poll&sid=" + console_session_id;
		xml_obj.open("GET", str_body, true);
		
		// Set the request header to make sure the browser won't use cache content 
		xml_obj.setRequestHeader("Pragma", "no-cache");
		xml_obj.setRequestHeader("Cache-Control", "no-store, no-cache, must-revalidate");
		xml_obj.setRequestHeader("If-Modified-Since", "Sat, 1 Jan 2000 00:00:00 GMT");
		
		xml_obj.send("");
	}
	catch (e)
	{
		acon_rx.clear_watchdog();
debug_msg("Fatal error in acon_rx_poll()");
	}
}


// acon_tx_loop - timer callback to send any data in the queue.
function acon_tx_loop()
{
	tx_loop_timer = null;
	
	if (!cons_connected)
		return;
		
	try
	{
		if (glob_outbuf.length > 0)
		{
			var rv = acon_tx_buffer();
			// Reset GUI idle timer when there is activity at the console.
			reset_gui_idle_timer();
			
			// If we were blocked because the acon_tx object is busy, then
			// set another timer to retry later. (No need to make this timer
			// too quick, since the user is probably typing at the same time.)
			if (rv == 2)
			{
				tx_loop_timer = setTimeout("acon_tx_loop()", 50);
				return;
			}
		}
		else
		{
			// Since glob_outbuf.length is 0, this will send a keepalive.
			acon_tx_buffer();
		}
	}
	catch (e)
	{
	}

	// We used to run this as a loop, hence the name. But now we will trigger a
	// tx whenever the user enters some data, so it is unnecessary to poll for
	// tx data here. We just set a very long timeout (30 seconds) for sending
	// keepalive packets.
	tx_loop_timer = setTimeout("acon_tx_loop()", 30000);
}


// acon_rx_loop - Make sure there is always a polling request active 
// whenever we are connected. This is important for for error recovery.
function acon_rx_loop()
{
	// Reset variables relating to the loop timer.
	if (rx_loop_timer)
	{
		clearTimeout(rx_loop_timer);
		rx_loop_timer = null;
	}
	
	// Check if we're in a state where polling is not necessary.
	if (!cons_connected || acon_disconnecting)
		return;
	
	// Run the poll.
	try
	{
		if (!acon_rx.in_progress)
			acon_rx_poll();
	}
	catch (e)
	{
	}

	// Make sure there is always another poll request pending.
	if (!rx_loop_timer)
		rx_loop_timer = setTimeout("acon_rx_loop()", 60000);
}


/******************************
    Ajax Class Implementation
 ******************************/

// acon_obj::acon_cleanup - delete the ajax objects.
function acon_cleanup()
{
	this.clear_watchdog();

	if (this.xmlhttp)
	{
		try
		{
			this.xmlhttp.abort();
			delete this.xmlhttp;
			this.xmlhttp = null;
		}
		catch (e)
		{
			debug_msg("Failed to cleanup old xmlhttp obj: " + e);
		}
	}
}

// acon_obj::acon_set_watchdog - set the watchdog timer.
function acon_set_watchdog()
{
	this.in_progress = true;
	this.timeout = setTimeout(this.watchdog_fn_i, 30000);
}
 

// acon_obj::acon_clear_watchdog - clear the watchdog timer.
function acon_clear_watchdog()
{
	this.in_progress = false;
	
	if (this.timeout)
	{
		clearTimeout(this.timeout);
		this.timeout = null;
	}
}

// acon_tx_watchdog_fn - called when the TX watchdog times out.
function acon_tx_watchdog_fn()
{
	this.cleanup();
	
	if (acon_connecting)
	{
debug_msg("Hard timeout during connect attempt.");
		handle_connection_result("");
	}
	else if (acon_disconnecting)
	{
debug_msg("Hard timeout during disconnect attempt.");
		handle_disconnect_event(connection_lost);
	}
	else
	{
debug_msg("Hard timeout during TX request.");
		handle_disconnect_event(connection_lost);
	}
}

// acon_rx_watchdog_fn - called when the RX watchdog times out.
function acon_rx_watchdog_fn()
{
	// no longer a fatal error.
//	handle_disconnect_event();
//	acon_obj.clear_watchdog();

	// Currently we have no error handling here.
	// Probably we need to retry the poll.
}

// acon_obj::acon_onreadystatechg - even filter for onreadystatechange.
function acon_onreadystatechg(ev)
{
//debug_msg("acon_onreadystatechg: state = " + this.xmlhttp.readyState);
	if (this.xmlhttp.readyState == 4) // 4 == complete
	{
		// On Mozilla, the callback will be called repeatedly w/ state 4 if
		// you try to abort() on a successful transaction (which we do, just
		// for safety's sake). So unset the event handler to avoid this bug.
		try
		{
			this.xmlhttp.onreadystatechange = null;
		}
		catch (e)
		{
			// This will fail on IE, but it doesn't matter since IE doesn't
			// exhibit the above behaviour.
		}
		
		this.buffer_ready_fn();
	}
}

// For event handlers, we couldn't set them to dynamically created functions since those
// are anonymous. libPrototype has some solution for this, although I can't figure it out.
function on_tx_readystatechg(ev)
{
	acon_tx.readystatechg_fn(ev);
}

function on_rx_readystatechg(ev)
{
	acon_rx.readystatechg_fn(ev);
}


// acon_obj::acon_obj_init - initialization function for acon_obj
function acon_obj_init()
{
	this.in_progress = false;
	this.timeout = null;
	this.xmlhttp = null;
	
	this.clear_watchdog = acon_clear_watchdog;
	this.set_watchdog = acon_set_watchdog;
	this.watchdog_fn = this.is_rx ? acon_rx_watchdog_fn : acon_tx_watchdog_fn;
	this.watchdog_fn_i = "" + this.obj_name + ".watchdog_fn();";
	this.buffer_ready_fn = this.is_rx ? on_acon_buffer_ready : on_acon_tx_done;
	this.readystatechg_fn = acon_onreadystatechg;
	this.readystatechg_fn_i = this.is_rx ? on_rx_readystatechg : on_tx_readystatechg;
	this.cleanup = acon_cleanup;	
}


// acon_obj::acon_obj_bootstrap - convert an object into an acon_obj.
function acon_obj_bootstrap(acon_obj, obj_name, is_rx)
{
	acon_obj.obj_name = obj_name;
	acon_obj.is_rx = is_rx; // default;
	
	acon_obj.init = acon_obj_init;
	acon_obj.init();
}


/* These are the global acon_obj objects and the global initialization function.
Almost everything else will be done via callbacks. 

Javascript is not as OO as Java. Basically, there is no low level class support.
It's a bit like perl. Basically we just create two objects and then bless them into 
acon_obj variables by initializing them to have the same structure. 

*/

var acon_rx = new Object();
var acon_tx = new Object();

function init_acon_objs()
{
debug_msg("init_acon_objs");
	acon_obj_bootstrap(acon_rx, "acon_rx", true);
	acon_obj_bootstrap(acon_tx, "acon_tx", false);
}


/******************************
      Connect/Disconnect
 ******************************/

var cons_connected = false;

// Session id so we can distinguish between multiple sessions w/ the same cookie.
var console_session_id = 0;

// Default string to display when the connection is lost. (For any other
// type of event, we are guaranteed to get a string from the server.)
var connection_lost = "Connection lost.";


// begin_acon_polling - called at startup and whenever we reconnect in
// order to start the Ajax threads. 
function begin_acon_polling()
{
	acon_tx.clear_watchdog();
	acon_rx.clear_watchdog();
	
	// Start the polling loop.
	acon_rx_loop();
	
	// The tx loop (to send keepalives) won't be started until we transmit
	// some data first.
}


// stop_ajax - abort and kill all pending Ajax objects. This is particularly
// necessary on IE, since they have a tendency to hang the browser.
function stop_ajax()
{
	// TODO: we didn't used to kill the watchdog timers here... any reason?
	acon_rx.cleanup();
	acon_tx.cleanup();

	// Kill the polling interval timer.
	if (rx_loop_timer)
	{
		clearTimeout(rx_loop_timer);
		rx_loop_timer = null;
	}

	// Kill the tx timer if it is active.
	if (tx_loop_timer)
	{
		clearTimeout(tx_loop_timer);
		tx_loop_timer = null;
	}
}

var con_active_icon_path = "/images/connect_active.gif";
var con_inactive_icon_path = "/images/connect_inactive.gif";
var discon_active_icon_path = "/images/disconnect_active.gif";
var discon_inactive_icon_path = "/images/disconnect_inactive.gif";

// set_connect_button_state
// param bConnected is 0 if disconnected (enable connect button),
// 1 if connected (enable disconnect button), 
// or -1 while connecting (disable both buttons).
function set_connection_button_states(bConnected)
{
	// Not sure how Boolean<->int equivalence would be handled by the
	// switch statement below, so do an explicit conversion.
	if (bConnected == true) bConnected = 1;
	if (bConnected == false) bConnected = 0;
	
	// Disable buttons during the connection attempt.
	try
	{
		var em_connect = document.getElementById("jsconn_connect_btn");
		var em_disconnect = document.getElementById("jsconn_disconnect_btn");
		
		switch (bConnected)
		{
		case -1: // connecting
			em_connect.disabled = true;
			em_connect.src = con_inactive_icon_path;
			em_disconnect.disabled = true;
			em_disconnect.src = discon_inactive_icon_path;
			break;
		case 0: // disconnected
			em_connect.disabled = false;
			em_connect.src = con_active_icon_path;
			em_disconnect.disabled = true;
			em_disconnect.src = discon_inactive_icon_path;
			break;
		case 1: // connected
			em_connect.disabled = true;
			em_connect.src = con_inactive_icon_path;
			em_disconnect.disabled = false;
			em_disconnect.src = discon_active_icon_path;
			break;
		}
	}
	catch (e)
	{
		debug_msg("failed to update button state.");
	}
}


// on_connect_clicked - called when the user presses the "connect" button.
// Also called automatically when the user opens the console page.
function on_connect_clicked()
{
debug_msg("on_connect_clicked");
	if (cons_connected)
		return;
		
	// Disable buttons during the connection attempt.
	set_connection_button_states(-1);

	// Try to connect with a simple query.
	try
	{
		// Reset the success flag so we may test it later.
		acon_connecting = true;
		acon_disconnecting = false;
		acon_tx_success = false;
		
		acon_tx_buffer();
		// Reset GUI idle timer when there is activity at the console.
		reset_gui_idle_timer();
	}
	catch (e)
	{
debug_msg("on_connect_clicked: connect failed");
		handle_connection_result("");
		
		// Will this work?
		throw(e);
	}
}


// handle_connection_result - callback when the Ajax connect either
// succeeds or fails.
function handle_connection_result(reply_txt)
{
debug_msg(" handle_connection_result ");

	// The global variable acon_tx_success is reset when we attempt the connection
	// and will be set to true if we received a reply.
	acon_connecting = false;
	
	// Check if the operation failed at the connection level. (This results in an
	// alert box rather than an inline... sort of a more alarming notification.)
	if (!acon_tx_success)
	{
		set_connection_button_states(false);
		setTimeout("alert('" + alert_strings_obj.conn_failed + "');", 1);
		return;
	}
	
	// Parse the message into status code, display text and session id (connect messages only).
	var status_code = reply_txt.charAt(0);
	var display_msg = "";
	
	var i = reply_txt.indexOf(":");
	
	if (i >= 0)
	{
		console_session_id = parseInt(reply_txt.substring(1,i));
		set_logo_text('saved_session_id', console_session_id);
		display_msg = reply_txt.substring(i+1);
	}
	else
	{
		// Shouldn't happen (invalid response from server).
		console_session_id = -1;
		display_msg = reply_txt.substring(1);
	}
	
	try
	{
		// Clear the timeout in disconnection that shows the welcome message
		if(reset_welcome_ctime) {
			clearTimeout(reset_welcome_ctime);
			reset_welcome_ctime = null;
		}
		
		var tbody = getConsoleTbody();
		
		// Finalize the last row before adding a new one (otherwise we may
		// end up with multiple cursors).
		inBuf = "";
		finalize_row(tbody, inBuf);
		
		// Add the response from the server. If we couldn't parse the reply, just add
		// some boilerplate text.
		if (display_msg.length > 0) 
		{
			append_complete_row(tbody, display_msg);
		}
		else if (status_code == '0')
		{
			append_complete_row(tbody, "(Connected.)")
		}
		else
		{
			append_complete_row(tbody, "(Connection failed.)")
		}
		
		// Add a blank line - as a matter of form, we should also initialize
		// the next partial row as well.
		append_complete_row(tbody, "")
		append_partial_row(tbody, "")
		
		set_connection_button_states((status_code == '0'));
		setTimeout("rescroll_history();", 1);
	}
	catch (e)
	{
	}
	
	if (status_code == '9')
	{
		//logout (e.g. cookie expired)
		on_close_console_logo();
		setTimeout("window.location=\"/logout\"", 50);
		return;
	}

	// Check if the server rejected the login (e.g. too many concurrent connections)
	if (status_code != '0')
	{
		return;
	}


debug_msg("Connection succeeded.");
	cons_connected = true;
	store_connection_cookie();
	update_connected_status_notifier();
	// Set the focus to the console window.
	set_console_window_focus(this);
	
	// Store the connection state in the logo frame.
	try
	{
		set_logo_checkbox('saved_cnsl_enabled', true);
		if (!window.opener)
		{
			oElem = top.document.getElementById('saved_cnsl_hist');
			oElem.value = "";
		}
	}
	catch (e)
	{
		// Failure here probably indicates that this is not the dashboard frame.
	}
	
	begin_acon_polling();
}

// handle_disconnect_event - called when the user disconnects or when
// there is a TX failure.
function handle_disconnect_event(msg)
{
	if (!cons_connected)
		return;
		
	cons_connected = false;
	set_logo_checkbox('saved_cnsl_enabled', false);
	store_connection_cookie();
	
	try
	{
		// Destroy the ajax objects.
		stop_ajax();
	}
	catch (e)
	{
debug_msg("failed to clear loop timers.");
	}
	
	// Update the buttons & cmd window
	set_connection_button_states(0);
	
	// Display "Connection lost" on GUI.
	try
	{
		var tbody = getConsoleTbody();
		finalize_row(tbody, inBuf);
		append_complete_row(tbody, "")
		append_complete_row(tbody, msg)
		append_partial_row(tbody, "")
		setTimeout("rescroll_history();", 1);
	}
	catch (e)
	{
		setTimeout("alert('"+ alert_strings_obj.conn_lost + "');", 1);
	}
	
	update_connected_status_notifier();
	
	
	// If we don't have the connect/disconnect buttons, we need to always
	// display the "click here" message. Give the user 4 seconds to read
	// any error message and then redirect to that.
	if (!acon_prefs.use_connect_btns)
	{
		reset_welcome_ctime = setTimeout("reset_welcome_text()", 4000);
	}
}


// on_disconnect_clicked - called when the user presses the disconnect
// button. We will always report the connection as lost, but we try
// to do it gracefully if possible.
function on_disconnect_clicked()
{
debug_msg("on_disconnect_clicked.");

	// If we are already connecting but haven't succeeded yet, we should
	// be able to use the watchdog to simulate a connection failure.
	// (We might also just be xmitting here.)
	if (acon_tx.in_progress)
	{
debug_msg("on_disconnect_clicked: abort current connection attempt.");
		if (acon_tx.xmlhttp)
			acon_tx.xmlhttp.abort();
		acon_tx.watchdog_fn();
		return;
	}
	
	// Store the dashboard connection state in the logo frame.
	try
	{
		if (!window.opener)
		{
			var oElem = top.document.getElementById('saved_cnsl_enabled');
			oElem.checked = false;
			
			oElem = top.document.getElementById('saved_cnsl_hist');
			oElem.value = "";
		}
	}
	catch (e)
	{
		// Failure here probably indicates that this is not the dashboard frame.
	}

	
	// Try to disconnect with a simple query.
	try
	{
		// Reset the success flag so we may test it later.
		acon_disconnecting = true;
		acon_tx_success = false;

		acon_tx_buffer();
		store_connection_cookie();
	}
	catch (e)
	{
debug_msg("on_disconnect_clicked: disconnect failed");
		handle_disconnect_event(connection_lost);
		
		// Will this work?
		throw(e);
	}
}


// update_connected_status_notifier - update the window title (or the
// widget title on the dashboard) to reflect the connection state.
function update_connected_status_notifier()
{
	var window_name = alert_strings_obj.title_name;
	var st_str = cons_connected ? 
		alert_strings_obj.connected_st : alert_strings_obj.disconnected_st;

	// Switch to an alternate message when the console is detached.
	try
	{
		if (!window.opener)
		{
			var hdr_doc = top.document;
			var oElem_d = hdr_doc.getElementById('cnsl_detached');
			
			if (oElem_d.checked)
				st_str = alert_strings_obj.detached_st;
		}
	}
	catch (e)
	{
	}

	// Now display the message on either the widget title or the window title.
	try
	{
		if (window.opener)
		{
			// For the popup console, update the title bar.
			document.title = window_name + " " + st_str;
		}
		else
		{
			// For the dashboard console, update the widget header.
			var oElem = document.getElementById("jsconn_state");
			oElem.innerHTML = st_str;
		}
	}
	catch (e)
	{
		debug_msg("Failed to update connection status bar");
	}
}



/******************************
 Javascript Console
 Init by A. Krywaniuk, Jan 2006
 Copyright Fortinet, inc.
 All rights reserved
 ******************************/

// STRIP-FUNCTION: debug_msg
// STRIP-FUNCTION: debug_kbd



/* Editline octal sequences:

The CLI can store Asian characters, but it doesn't interpret them. Normally, this is not a
problem for displaying them (aside from the encoding conversion hassle), but it does cause
problems with editline. Editline doesn't treat MB characters as 2 byte characters. Instead, it
encodes them as escaped octal sequences (e.g. \303\374). When performing editing commands, this
causes great confusion, since you may receive the VT-100 sequence ^[8D to delete a single Asian
character. To solve this problem, we pad the inBuf with special non-printable characters in
order to keep the cursor position in sync.

This works some of the time, but not all of the time. Sometimes editline will send us a sequence
such as BS3 (backspace 3). The intention may be to replace the character \303\374 with \303\373.
However, we have no means of processing this. The interpretation of this character is strictly
bound to the MB character set that is used on the server side, so once we have converted the
message to UTF-8, we can no longer perform byte-wise editing operations. The solution is to add
MBC awareness to the CLI. Or just rewrite the GUI in UTF-8.

*/

// mbc_octseq_marker - this special character is sent in the input stream to mark the place where
// an octal sequence (generated by editline) has been converted into binary. In order to keep the
// cursor positions in sequence, we must expand the marker into a series of padding bytes which are
// not displayed to the user. 
var mbc_octseq_marker = '\x1F';

// chr_spacing - use this special character (0xF1 = 31 decimal) to pad Asian characters in the
// command line so their length matches the encoded length in the CLI. Then we strip them
// out before displaying them. Note that we currently use the same character for the sequence marker
// and the padding bytes, but this is not a requirement.
var chr_spacing = '\x1F';
var chr_spacing7 = chr_spacing.concat(chr_spacing,chr_spacing,chr_spacing,chr_spacing,chr_spacing,chr_spacing);
var strip_sc_regexp = /\x1F/g;


/******************************
  Main window display functions
 ******************************/

// Since we now put the console buf within a separate <iframe>, we must use the correct
// document reference when creating elements. (This is required by IE, but not Mozilla.)

// create_hist_elem - create an HTML element within the appropriate document.
function create_hist_elem(type_str)
{
	return getConsoleDoc().createElement(type_str);
}

// create_hist_elem - create a text node within the appropriate document.
function create_hist_text_node(xstr)
{
	return getConsoleDoc().createTextNode(xstr);
}

 
// clean_last_td - Remove all text and subelements from the last TR in the table, and
// populate it with a single empty TD. (Create a new TR if the table is empty.)
function clean_last_td(tbody)
{
	var last_tr = get_last_child_of_type(tbody, "TR");
	
	if (last_tr)
	{
		while(last_tr.childNodes.length)
		{
			last_tr.removeChild(last_tr.childNodes[0]);
		}
	}
	else
	{
		last_tr = create_hist_elem("TR");
		tbody.appendChild(last_tr);
	}
	
	var new_td = create_hist_elem("TD");
	last_tr.appendChild(new_td);
	
	// Use a <PRE> block to solve the problem of trailing/consecutive spaces being
	// ignored by HTML. Possibly this could be fixed in a better way(?)
	
	var new_pre = create_hist_elem("PRE");
	new_td.appendChild(new_pre);
	
	return new_pre;
}


/* Various display problems may occur if we create an empty <td> using
createTextNode. Either the row will be added with 0 height, or sometimes it
will initially be displayed correctly but then fail later when we copy
the innerHTML. So make sure we always add a blank space when we
detect an empty line.
*/


// display_row - main function to display the active line of the console
function display_row(new_td, txt)
{
	var str1 = "";
	var str2 = "";
	var str3 = "";
	
	try
	{
		if (cursor_x_pos > 0)
		{
			if (cursor_x_pos < txt.length)
			{
				str1 = txt.substr(0, cursor_x_pos);
			}
			else
			{
				str1 = txt;
			}
		}
		
		if (txt.length && cursor_x_pos < txt.length)
		{
			str2 = txt.substr(cursor_x_pos, 1);
		}
		else if (txt.length)
		{
			// Need to have a space so the cursor blink looks correct.
			str2 = " ";
		}
		else
		{
			// Various formatting problems may occur if we add an empty <td>.
			// But there should never be a cursor at the beginning of the line,
			// so better to put the empty space in str1.
			str1 = " ";
		}
		
		if (cursor_x_pos + 1 < txt.length)
		{
			str3 = txt.substr(cursor_x_pos+1);
		}
	}
	catch (e)
	{
		debug_msg("split text failed, cursor_x_pos = " + cursor_x_pos + ", len = " + txt.length);
	}
	
	var txt1 = create_hist_text_node(str1);
	var txt2 = create_hist_text_node(str2);
	var txt3 = create_hist_text_node(str3);


	// Implement cursor
	var new_span = create_hist_elem("SPAN");
	new_span.appendChild(txt2);
	new_span.id = 'history_cursor';
	// set on/off state based on global variable
	update_cursor_style(new_span);
	
	new_td.appendChild(txt1);
	new_td.appendChild(new_span);
	new_td.appendChild(txt3);
}


// strip_spacing_chars - strip out special spacing chars that are only used in keeping the cursor
// position in sync w/ the CLI when using Asian character sets.
function strip_spacing_chars(txt)
{
//debug_msg("strip_spacing_chars - before=" + txt);
	var rslt = txt.replace(strip_sc_regexp, "");
//debug_msg("strip_spacing_chars - after=" + rslt);
	return rslt;
}

// overwrite_last_row - called to update an existing row. Also displays
// the cursor.
function overwrite_last_row(tbody, txt)
{
//debug_msg("overwrite_last_row: " + txt);
	var td = clean_last_td(tbody);
	txt = strip_spacing_chars(txt);
	
	display_row(td, txt);
}

// Store the console history in an array. Keep the history in one hidden 
// textarea variable in logo page when the URL goes away from status page.

var console_history = new Array();

function add_to_console_history(str)
{
	console_history.push(str);
	
	if (console_history.length > acon_prefs.acon_histbuflen)
	{
		console_history.shift();
	}
//debug_msg("console_history = " + console_history);
}

function store_console_hist()
{
	var logo_doc = get_logo_frame().document;
	var oElem = logo_doc.getElementById('saved_cnsl_hist');
	if(!oElem) return;
	
	var str="";
	for ( i = 0; i < console_history.length; i++ ) {
		str += console_history[i] + '\n';
	}
	str += inBuf;
	
	oElem.value = str;
}

// finalize_row - called after we receive a CR. The cursor is no longer on
// this row, so just output a single td.
function finalize_row(tbody, txt)
{
	var td = clean_last_td(tbody);
	
	txt = strip_spacing_chars(txt);
	
	// Various formatting problems may occur if we add an empty string.
	if (txt.length == 0)
		txt = " ";
	
	var oTxt = create_hist_text_node(txt);
	
	add_to_console_history(txt);
	td.appendChild(oTxt);
}

function setup_new_td(tbody)
{
	var new_tr = create_hist_elem("TR");
	var new_td = create_hist_elem("TD");
	var new_pre = create_hist_elem("PRE");
	
	tbody.appendChild(new_tr);
	new_tr.appendChild(new_td);
	new_td.appendChild(new_pre);
	
	return new_pre;
}

// append_complete_row - add a new row and finalize it right away.
// NOTE: the table must always end with a partial row, so after adding
// a complete row, you should generally create a partial row as well.
function append_complete_row(tbody, txt)
{
	add_to_console_history(txt);
	var td = setup_new_td(tbody);
	
	txt = strip_spacing_chars(txt);
	
	// Various formatting problems may occur if we add an empty string.
	if (txt.length == 0)
		txt = " ";
	
	var oTxt = create_hist_text_node(txt);
	
	td.appendChild(oTxt);
	
	cnsl_buffer_cur_len++;
}

// append_partial_row - append a "partial row". The difference between
// a partial row and a complete row is that a partial row displays the
// cursor (and it will be updated later).
function append_partial_row(tbody, txt)
{
//debug_msg("append_partial_row: " + txt);
	txt = strip_spacing_chars(txt);

	// Presumably, this will be the prompt + whatever we have typed so far.
	cursor_x_pos = txt.length;
	
	var new_td = setup_new_td(tbody);
	display_row(new_td, txt);
	
	cnsl_buffer_cur_len++;
}

// append_welcome_msg - similar to append_complete_row, but we don't append
// the empty row afterwards, so this message will be overridden by the next
// new text from the console. (We also don't add it to the console history.)
function append_welcome_msg(tbody, txt)
{
	var td = setup_new_td(tbody);
	var oTxt = create_hist_text_node(txt);
	td.appendChild(oTxt);
}


/******************************
     Console Buffer Length
 ******************************/

// set low for testing - need to make this configurable
var cnsl_buffer_cur_len = 0; 

// prune_histbuf - truncate to cnsl_buffer_max_len lines
function prune_histbuf()
{
	if (cnsl_buffer_cur_len <= acon_prefs.acon_consbuflen)
		return;
		
	var tbody = getConsoleTbody();
	
	if (cnsl_buffer_cur_len != tbody.childNodes.length)
	{
debug_msg("console buffer length got out of sync: " + cnsl_buffer_cur_len + " vs. " + tbody.childNodes.length);
		cnsl_buffer_cur_len = tbody.childNodes.length;
	}
	
	while (cnsl_buffer_cur_len > acon_prefs.acon_consbuflen)
	{
		tbody.removeChild(tbody.childNodes[0]);
		cnsl_buffer_cur_len--;
	}
}

// Store connection cookie for connection status
function store_connection_cookie()
{
	try
	{
		var str = "connection=" + cons_connected
			+ "; path=/frames";
			
		document.cookie = str;
	}
	catch (e)
	{
		alert(alert_strings_obj.cookie_save_failed + e);
	}
}



/******************************
     Command History Buffer 
 ******************************/

// The command history is stored as an array with the most recent entries
// stored at the end. New entries are added at the end of the array.
// When the array overflows, old ones are removed from the beginning.

var cmd_history = new Array();
var cmd_hist_pos = 0;
var cmd_hist_maxlen = 10;

function add_to_cmd_history(str)
{
	// Ignore empty CRs.
	if (str.length == 0)
		return;
		
	cmd_history.push(str);
	
	if (cmd_history.length > cmd_hist_maxlen)
	{
		cmd_history.shift();
	}
	
	// Setting the history position to one above the last index
	// ensures that a single up-arrow takes the user back to the
	// command they just typed.
	
	cmd_hist_pos = cmd_history.length;
debug_msg("cmd_history = " + cmd_history + " pos = " + cmd_hist_pos);
}

function cmd_hist_up()
{
//debug_msg("cmd_history = " + cmd_history + " pos = " + cmd_hist_pos);
	if (cmd_hist_pos - 1 >= 0)
	{
		cmd_hist_pos--;
//debug_msg("cmd_history[cmd_hist_pos] = " + cmd_history[cmd_hist_pos]);
		return cmd_history[cmd_hist_pos];
	}
	
	// Resync, just in case.
	cmd_hist_pos = 0;
	
	// Return null, meaning no action.
	return null;
}


function cmd_hist_down()
{
//debug_msg("cmd_history = " + cmd_history + " pos = " + cmd_hist_pos);
	if (cmd_hist_pos + 1 < cmd_history.length)
	{
		cmd_hist_pos++;
		
//debug_msg("cmd_history[cmd_hist_pos] = " + cmd_history[cmd_hist_pos]);
		return cmd_history[cmd_hist_pos];
	}
	
	// Resync, just in case.
	cmd_hist_pos = cmd_history.length;
	
	// Return empty string, meaning clear the input box.
	return "";
}


/******************************
     Main parsing functions
 ******************************/

// rawBuf is the unparsed response buffer (that may contain VT escape sequences)
var rawBuf = "";

// inBuf is the buffer of parsed input that is displayed in the last (unfinalized)
// row of the table.
var inBuf = "";

// cursor_x_pos is the offset of the curser within inBuf.
var cursor_x_pos = 0;

// Number of consecutive empty polling responses from the server (typically indicates
// that two consoles are competing for the same data... or this code has a bug and it 
// left multiple polling objects running.
var empty_response_cnt = 0;



// acon_obj::on_acon_buffer_ready - called indirectly by the onreadystatechange handler
function on_acon_buffer_ready(ev)
{
//debug_msg("xx - handle_buffer_ready");
	       
	// Check & reset semaphore.
	if (!this.in_progress)
		return;
		
	var err_code = 0;

	// Preparse the message format. Extract the first byte, which is the
	// error code.
	var reply_len = this.xmlhttp.responseText.length;
	var status_code = this.xmlhttp.responseText.charAt(0);
	// Copy the reply text so we can be done with the xmlhttp obj.
	var new_text = this.xmlhttp.responseText.substring(1);
	
	// Delete the object and cancel the watchdog.
	this.cleanup();
	
	// Update the cached console document in case it changed somehow.
	console_document = getConsoleDoc();
	
	if (reply_len > 0)
	{
		// Reset the empty response counter... although it is still theoretically
		// possible that we could get a valid poll message with an empty string
		// (which would be an implementation error on the server side).
		empty_response_cnt = 0;
		
		if (status_code != '0') 
		{
			err_code = status_code;
			handle_disconnect_event(new_text);
		}
	}
	else
	{
		// Empty response - used to mean that the socket had closed abnormally,
		// although now we should send a specific error message in that case.
		err_code = 999;
		empty_response_cnt++;
		
		// This is to detect infinite loops of connection attempts when the server
		// disconnects right away.
		if (empty_response_cnt > 10)
		{
			debug_msg("Comm error - " + empty_response_cnt + " consecutive empty poll RXs.");
			handle_disconnect_event(connection_lost);
			empty_response_cnt = 0;
		}
	}
	
	if (err_code)
	{
debug_msg("Comm error in handle_buffer_ready: " + err_code);
		return;
	}
	
	/* Reschedule RX timer here. Currently it is being queued via a timer so we don't
	 request any new data until we have finished displaying the old data. It might be
	 possible to reduce latency by sending the request immediately (and displaying
	 the old data while we are waiting), however I have experimentally confirmed that
	 this will choke under stress (it hangs the browser when I run "dia sniff packet"
	 on IE, although it ok on Mozilla...).

	 So anyway... if we wanted to avoid the timeout, we would have to be very careful
	 about implementing some kind of throttling algorithm to introduce some delays
	 when under stress. E.g. if the user is actively typing at the console, it may be
	 acceptable to call acon_rx_loop() directly, but if they are just monitoring the
	 output from the console, then we need to use the timeout.
	 */

	rx_since_last_tx++;
	if (rx_since_last_tx > 4)
	{
		if (rx_loop_timer)
		{
			clearTimeout(rx_loop_timer);
			rx_loop_timer = null;
		}
		
		rx_loop_timer = setTimeout("acon_rx_loop()", 1);
	}
	else
	{
		acon_rx_loop();
	}

	// Process the (non-empty buffer).
	if (new_text.length > 0)
	{
//		debug_msg("received message: " + new_text);
		
		// Append to rawBuf since there could be some partially complete multi-byte
		// sequences left over from the last round of parsing.
		rawBuf += unescape(new_text);

		try
		{
			process_rx_buffer();
		}
		catch (e)
		{
			debug_msg("Failed in process_rx_buffer: " + e);
			
			// If we get an exception here, we had better do something to cleanup the
			// parsing state. Otherwise rawBuf could grow without bound.
			rawBuf = "";
			
			// Try to display some kind of message on the console.
			try
			{
				var tbody = getConsoleTbody();
				finalize_row(tbody, inBuf);
				inBuf = "";
				append_complete_row(tbody, "(Connection error.)");
				append_partial_row(tbody, "");
			}
			catch (e)
			{
			}
		}
	}
}


// fixup_tab_response - when the user presses tab in the command window, we need to
// interpret the reply as best we can. I don't think anyone would disagree that this
// is a kludge. If the user is typing quickly while pressing tab, the results might
// be unpredictable. We could probably do something more reliable by hacking into 
// the CLI internals.
function fixup_tab_response()
{
debug_msg("fixup_tab_response, inbuf=" + inBuf);
	try
	{
		var cmd_elem = document.getElementById("jsconn_cmd");

		if (tab_proc_st.recent_typing)
		{
			// User was typing while we were waiting for the tab response.
			throw("user continued typing");
		}
		
		if (parse_st.added_rows)
		{
			throw("rows added during tab command");
		}
		
    	if (inBuf.length == 0)
        {
        	// This is an intermediate state caused by the Asian character octal
            // fixup. Don't abort the completion. We expect the current line to 
            // be resent in the next message.
            return;
        }
		
		// Strip the command-line prompt from inBuf.
		// There are two basic types of prompts, so find the first one.
		var prompt_offset1 = inBuf.indexOf("# ");
		var prompt_offset2 = inBuf.indexOf("$ ");

		// If either one isn't found, just set them both the same
		// (so min will work).
		if (prompt_offset1 == -1) prompt_offset1 = prompt_offset2;
		if (prompt_offset2 == -1) prompt_offset2 = prompt_offset1;

		var prompt_offset = Math.min(prompt_offset1, prompt_offset2);

		if (prompt_offset == -1)
		{
			throw("Unable to locate command prompt.");
		}

		var new_cmd = inBuf.substring(prompt_offset += 2);

		// Set the command box to contain the current command buffer.
		// (It's a bit of a leap of faith - we are basically relying on 
		// there not being extraneous characters in the output.)
		cmd_elem.value = strip_spacing_chars(new_cmd);
		tab_proc_st.stage++;
		
		// Tab completion results sometimes take more than one message to
		// complete (e.g. we might still receive some local echo character
		// before receiving the completion). But if we don't finish the
		// completion within 3 packets then give up.
		if (tab_proc_st.stage > 5)
		{
			tab_proc_st.in_proc = false;
		}
	}
	catch (e)
	{
		// This is usually not an error unless tab_proc_st.stage == 1. It is 
		// difficult for us to predict heuristically when the response is
		// complete, so we can't distingish completion from failure.
		debug_msg("couldn't fixup_tab_response: " + e);
		tab_proc_st.in_proc = false;
	}
}



// process_vt100_esc_sequence - handle an escape sequence
// The escape character '^' is not included in buf.
// This function directly manipulates the parser state.
// It returns the number of characters processed (or 0 if
// the buffer is incomplete, or -1 if we can't interpret
// the sequence).
function process_vt100_esc_sequence(buf)
{
debug_msg("begin vt100: ^" + buf);
	// Escape sequences often have a repeat count option. (Use 1 if it is omitted.)
	var repeat_cnt = 1;
	
	var j = 0;
	
	if (buf.length < 2)
	{
		// Incomplete buffer.
		return 0;
	}
	
	// In general, we support escape sequences of the form '^[<n><C>', where <n> is the
	// repeat count, and <C> is the command.
	ch = buf.charAt(j); 
	if (ch == 'O') // Esc-O
	{
		// This represents some kind of reverse font (i.e. white on black) that we currently
		// don't implement. (Can be generated with 'dia sys top').
		return j+1;
	}
	else if (ch != '[')
	{
		debug_msg("1c Unrecognized escape sequence header: ^" + rawBuf.charAt(j));
		return -1;
	}
	
	j++;

	// The first character may be either the command or the repeat count.
	ch = buf.charAt(j);
	
	// Check if this is a repeated command.
	if (ch >= '0' && ch <= '9')
	{
		// Parse repeated command.
		repeat_cnt = parseInt(buf.substr(j));

		if (isNaN(repeat_cnt))
		{
			debug_msg("cnt is NaN in esc sequence.");
			return -1;
		}

//debug_msg("cnt = " + cnt + ", substr = " + rawBuf.substr(j));

		if (repeat_cnt >= 10)
			j += 2;
		else
			j += 1;

		// Return 0 if the buffer is partially complete.
		if (j >= buf.length)
			return 0;

		// Now the next character should be the command.
		ch = buf.charAt(j);
	}
	
	j++;

	// Parse sequence with multiple integer parameters (';' is delimiter).
	if (ch == ';')
	{
		if (j >= buf.length)
			return 0;
		
		ch = buf.charAt(j);
		
		// Just ignore the 2nd argument.
		while (ch >= '0' && ch <= '9')
		{
			j++;
			if (j >= buf.length)
				return 0;
				
			ch = buf.charAt(j);
		}
	}
	
	// This is not a complete VT-100 terminal implementation. It only covers
	// the characters that the CLI is known to implement.
	switch (ch)
	{
	// Up/down arrow
	case 'A':
	case 'B':
		// Move cursor up? How (and why) are we supposed to do that?
		debug_msg("VT100 - don't know how to handle ^[" + ch + " (up/down arrow)");
		return -1;
	
	// right/left arrow
	case 'C': 
		cursor_x_pos += repeat_cnt;
		parse_st.moved_cursor = true;
		break;
	case 'D':
		cursor_x_pos -= repeat_cnt;
		parse_st.moved_cursor = true;
		break;
		
	case 'K':
		// Erase from cursor to EOL
		if (cursor_x_pos <= 0)
		{
			inBuf = "";
		}
		else
		{
			inBuf = inBuf.substr(0, cursor_x_pos);
		}
		break;
		
	case '@':
		// Insert N (empty) characters
		var str1 = inBuf.substr(0, cursor_x_pos);
		var x1;
		for (x1 = 0; x1<repeat_cnt; x1++)
		{
			str1 += ' ';
		}
		str1 += inBuf.substr(cursor_x_pos);
		inBuf = str1;
		break;
	case 'H':
		// set cursor home
		cursor_x_pos = 0;
		parse_st.moved_cursor = true;
		break;
	case 'J':
		// erase to end of screen (inclusive)
debug_msg("VT100 - don't handle ^[" + ch + " for now");
		break;
	case 'P':
		// erase one character backwards
		var str1 = inBuf.substr(0, cursor_x_pos);
		var str2 = inBuf.substr(cursor_x_pos + 1);
		inBuf = str1 + str2;
		break;
	case 'm':
		// This implements some kind of text colour feature that we don't currently
		// support. (Can be generated with "dia sys top". Also with the debug shell
		// on the Faz.)
		j++;
		break;
	default:
		debug_msg("1a Unrecognized escape sequence: ^" + buf.substring(0,j-1));
		return -1;
	}

	return j;
}


// Parsing state - used by process_rx_buffer and process_vt100_esc_sequence.
// Reset every time process_rx_buffer is called.
var parse_st = {
	added_rows : false,
	moved_cursor : false
};


// process_rx_buffer - main parsing function. Handle VT-100 char codes here.
function process_rx_buffer()
{
	// DEBUG: display rawBuf in the tmp buffer window.
	if (verbose_debug)
	{
		var outbuf = document.getElementById("txt_outbuf");
		if (outbuf) outbuf.value = rawBuf;
	}


//alert("length = " + rawBuf.length + ", rawBuf = " + rawBuf);


	var tbody = getConsoleTbody();
	if (!tbody)
	{
		debug_msg("Failed to get history tbody.");
		
		// We can't just return when there is an error, since then
		// rawbuf won't be cleaned up.
		throw("internal error 33k");
	}
	
	var ch;
	var i;
	var j = 0;
	
	// Reception of CRs triggers rescrolling & history pruning.
	parse_st.added_rows = false;
	
	// Moving the cursor should force it on immediately.
	parse_st.moved_cursor = false;
	
	
	// Ignore what we have already printed to the table. The global variables inBuf and
	// cursor_x_pos form the master copy of the working buffer for the active line.
	
	for (i=0; i<rawBuf.length; i++)
	{
		// Adjust for any multi-byte sequence in the last pass.
		if (j)
		{
			i += j;
			j = 0;
		
			if (i >= rawBuf.length)
				break;
		}
		
		
		// Begin processing the next character in the raw buffer.
		ch = rawBuf.charAt(i);
		
		if (ch == '\n')
		{
			finalize_row(tbody, inBuf);
			inBuf = "";
			
			// It would be a little more efficient to not add the new row until we
			// have some data (that's what I originally did), but it makes the code
			// a fair bit more complicated and it resulted in several bugs (e.g. in
			// displaying the cursor).
			
			append_partial_row(tbody, "");
			parse_st.added_rows = true;
			continue;
		}
		else if (ch == '\r')
		{
			// Not needed, I guess. Just let the '\n' handler do its work.
			continue;
		}
		else if (ch == '\x1b') // 0x1b = 27 = \033 = ESC
		{
			// Implement a VT100 virtual terminal.
			
			j = process_vt100_esc_sequence(rawBuf.substring(i+1));
			
			if (j > 0)
			{
				// Received complete ESC sequence. The variable 'i' will be adjusted
				// at the top of the loop.
				continue; 
			}
			
			if (j == 0)
			{
				// It is possible that the read operation would break in the middle
				// of the sequence. Handle the whole thing later.
				i--;
				break;
			}
			else
			{
				// If we can't interpret the esc sequence, just ignore
				// the escape and print out the characters on the
				// screen.
				ch = '^';
				j = 0;
			}
		}
		else if (ch == '\b') // backspace = 0x08
		{
			// VT-100 backspace doesn't actually erase the character. It just moves
			// one place to the left. Then it will send ^[K to clear the line and
			// echo back any new characters after that.
			cursor_x_pos--;
			if (cursor_x_pos < 0)
				cursor_x_pos = 0;
			
			continue;
		}
		else if (ch == '\x07') // bell = 0x07
		{
			// Happens when you backspace past the beginning of a line.
			// We could play a sound here, I suppose.
			continue;
		}
		else if (ch == mbc_octseq_marker)
		{
			// When we receive the special chr_spacing char, it indicates that the httpcli proxy
			// has compressed an 8 character octal sequence from the CLI into a 2 byte UTF-8
			// string + 1 bytes chr_spacing marker. Since JS will treat the 2 byte UTF-8 
			// string as a single character, we must then add 7 bytes of padding characters in
			// order to keep the cursor positions in sync. (These padding characters will be
			// stripped out when 
			var str1 = inBuf.substr(0, cursor_x_pos);
			var str2 = inBuf.substr(cursor_x_pos+7);
			inBuf = str1 + chr_spacing7 + str2;
			cursor_x_pos += 7;
			continue;
		}
		
		// Insert the new character at the cursor position.
		var str1 = inBuf.substr(0, cursor_x_pos);
		var str2 = inBuf.substr(cursor_x_pos+1);
		inBuf = str1 + ch + str2;
		cursor_x_pos++;
	} 

	if (i >= rawBuf.length)
	{
		rawBuf = "";
	}
	else
	{
		// save for next time (can this happen?)
		rawBuf = rawBuf.substr(i);
	}
    
	// Handle any response to Tab in the command window.
	if (tab_proc_st.in_proc)
	{
		fixup_tab_response();
	}
	
	// If we moved the cursor, make sure it is initially on.
	if (in_hist_wnd && parse_st.moved_cursor)
	{
		accel_cursor_on();
	}

	// Display remainder of unterminated buffer.
	overwrite_last_row(tbody, inBuf);


	// DEBUG: Update cursor pos display window
	if (verbose_debug)
	{
		var cp_elem = document.getElementById("jsconn_cursorpos");
		if (cp_elem) cp_elem.value = cursor_x_pos;
	}

	// If we added new rows, scroll the console window to the bottom, and 
	// also check if we exceeded the maximum console buffer length.
	if (parse_st.added_rows)
	{
		prune_histbuf();
		
		// Scroll the history window so the new line is in view.
		rescroll_history();
	}
}


/******************************
       Cursor functions
 ******************************/

var cursor_on = false;
var cursor_blink_timer = null;

// update_cursor_style - used by cursor_blink_loop, but included as a separate
// function so display_row can also set the initial state.
function update_cursor_style(celem)
{
	if (cursor_on)
	{
		celem.style.color = acon_prefs.acon_bgColor;
		celem.style.backgroundColor = acon_prefs.acon_fgColor;
	}
	else
	{
		celem.style.color = acon_prefs.acon_fgColor;
		celem.style.backgroundColor = acon_prefs.acon_bgColor;
	}
	
	celem.style.fontFamily = acon_prefs.acon_font_family;
	celem.style.fontSize = acon_prefs.acon_font_size;
}


// cursor_blink_loop - called all the time to blink the cursor, but it will
// only do so if the console window has the focus.
function cursor_blink_loop()
{
	try
	{
		// Do the switch whether we find the cursor or not (shouldn't matter?)
		// Only show the cursor when the history window has the focus.
		if (in_hist_wnd)
		{
			cursor_on = !cursor_on;
		}
		else
		{
			cursor_on = false;
		}
		
		var celem = getCursor(); 
		if (celem)
		{
			update_cursor_style(celem);
		}
	}
	catch (e)
	{
		// ignore any errors (but make sure not to abort before resetting the timer)
	}
	
	// Slightly longer duty cycle for cursor on than cursor off.
	if (cursor_on)
	{
		cursor_blink_timer = setTimeout("cursor_blink_loop()", 710);
	}
	else
	{
		cursor_blink_timer = setTimeout("cursor_blink_loop()", 605);
	}
}

// When using the left & right arrow keys, the results can seem sluggish due to the
// blinking cursor. Better to always turn the cursor on right after handing a left/right
// arrow event.

// accel_cursor_on - turn the cursor on more quickly
function accel_cursor_on()
{
	// Actually, display_row is always called by process_rx_buffer after calling this
	// function, so no need to schedule it later. Just turn the cursor on now, 
	// and delay the blinking slightly.
	cursor_on = true;
	
	// TODO: I'm not sure this works correctly.
	if (cursor_blink_timer) clearTimeout(cursor_blink_timer);
	cursor_blink_timer = setTimeout("cursor_blink_loop()", 1000);
}



/******************************
 Javascript Console
 Init by A. Krywaniuk, Jan 2006
 Copyright Fortinet, inc.
 All rights reserved
 ******************************/

// STRIP-FUNCTION: debug_msg
// STRIP-FUNCTION: debug_kbd

// EXPORT_SYMBOL dashboard_detach_console
// EXPORT_SYMBOL jsconsole_init
// EXPORT_SYMBOL on_call_back_clicked
// EXPORT_SYMBOL display_console_window
// EXPORT_SYMBOL on_close_console_logo

/******************************
        Initialization
 ******************************/

// These variables protect against some race conditions that used to happen when
// the initialization was called from the document.onload or window.onload handlers.
// The problems don't seem to occur now that we use the iframe's onload handler.
var jsconn_init_done = false;
var startup_tries = 0;


// test_window_loaded - check whether the window has finished loading. Basically,
// we just want to verify that essential elements and functions are accessible.
function test_window_loaded()
{
	// Tends to fail on all browsers during startup.
	try
	{
		var hist_elem = getConsoleTable();
		if (!hist_elem) return 0;
	
		// Not sure if the test below is useful.
		var xml_test = get_xmlhttp();
		if (!xml_test) return 0;
		
		delete xml_test;
		xml_test = null;

		// Cache the console document so we don't have to look it up 1000 times.
		// (This doesn't seem to work, unfortunately.)
		console_document = getConsoleDoc();
		if (!console_document) return 0;
	}
	catch (e)
	{
		return 0;
	}
	
	return 1;
}


var other_console_elems = new Array("jsconn_customize", "jsconn_popup", "cons_cmd_span");

// set_console_elems_visibility - enable/disable the 4 buttons and the console window in 
// the dashboard widget. If we are disabling the console elements then enable some alternate
// elements (a message stating that the console is detached).
function set_console_elems_visibility(open_hdl, enable)
{
	try 
	{
		var doc = open_hdl.document;
		var i, elem;
		
		// Hide or show the console iframe, and preserve the height to
		// use in the detached message.
		var orig_height = 200; // arbitrary size.
		elem = doc.getElementById("console_iframe");
		if (elem)
		{
			orig_height = elem.offsetHeight;
			elem.style.display = enable ? "" : "none";
		}
		
		// If the console iframe is hidden, show the detached message
		st = enable ? "hidden" : "visible";
		var disp = enable ? "none" : "block";
		
		elem = doc.getElementById("cons_detach_msg");
		if (elem)
		{
			if (!enable)
				elem.style.height = "" + orig_height + "px";
			elem.style.visibility = st;
			elem.style.display = disp;
		}

		// For the customize & detach buttons, these are now enabled/disabled by the D&D dashboard title
		// bar mouseover. The reason this still works is because we set the visibility to visible/hidden,
		// whereas the dashboard sets the display to block/none. So we don't stomp on each others' feet.
		var st = enable ? "visible" : "hidden";
		for (i=0; i < other_console_elems.length; i++)
		{
			elem = doc.getElementById(other_console_elems[i]);
			if (elem)
				elem.style.visibility = st;
		}
		
/*		// The connect buttons are always hidden if the use_connect_btns
		// flag is not set in the preferences.
		// Note: these buttons have been removed now.
		{
			if (!acon_prefs.use_connect_btns)
				st = "hidden";
			else
				st = "visible";
			
			doc.getElementById("jsconn_connect_btn").style.visibility = st;
			doc.getElementById("jsconn_disconnect_btn").style.visibility = st;
		}*/
	}
	catch (e)
	{
		debug_msg("Failed to set console elems visibility" + e);
	}
}


// hide_console_window - hide the console widget on the dashboard.
function hide_console_window(open_hdl)
{
	set_console_elems_visibility(open_hdl, false);
}

// show_console_window - show the console widget on the dashboard.
function show_console_window(open_hdl)
{
	set_console_elems_visibility(open_hdl, true);
}

// Set focus on console window. If the connection is on, focuses on cmd or 
// iframe according to cmd box is displayed or not. Otherwise, focuses on 
// connection button if there is the button
function set_console_window_focus(open_hdl)
{
	if(open_hdl.cons_connected) {
		if (open_hdl.acon_prefs.acon_showcmd)
			open_hdl.document.getElementById("jsconn_cmd").focus();
		else
			open_hdl.document.getElementById("console_iframe").contentWindow.focus();
	}
	else if(acon_prefs.use_connect_btns) {
		open_hdl.document.getElementById("jsconn_connect_btn").focus();
	}
	else
		return;
	
	// set in_hist_wnd to true to make cursor blink
	in_hist_wnd = true;
}

/* Part of the jsconsole_init initialization function is written as a co-routine.
 Some steps cannot be completed right away, so we keep track of what we have done
 and skip those steps when the function is called a second time. Basically, the
 independent stages are: 
 
 1. loading the preferences
 2. auto-resizing the window
 3. everything else
 
 Stage 3 is considered the general initialization routine. It is run after the
 window state has been verified, and it fails atomically.
*/

// load_prefs_done is the state flag for coroutine section 1 above.
var load_prefs_done = false;


// jsconsole_init - initialize the JS console iframe.
// Note: All the parameters to this function are currently incomplete. We ought to 
// change it to receive a single parameter, which is default_acon_prefs.
function jsconsole_init()
{
	// Many segements of the initialization depend on whether this is the dashboard console
	// or the detached console.
	var is_popup = (window.opener != null);

	// Load our preferences first (since they may affect the window layout & size)
	// We no longer want to use this semaphore in the D&D dashboard, since it would result in the
	// preferences not being reapplied after a drag.
//	if (!load_prefs_done)
	{
		override_default_console_prefs();

		try
		{
			// Override any of the global prefs values if the user has a prefs cookie from this
			// session. If two users login from the same account and both change the settings, we can
			// handle it in two ways. Either the change takes place immediately, or each user's custom
			// settings are in effect until they restart the browser.
//			parse_console_cookie();

			// Apply the settings from the CMDB/cookie.
			var cnsl_doc = getConsoleDoc();
			apply_console_prefs(cnsl_doc);

		}
		catch (e)
		{
			debug_msg("Failed to load settings from cookies.");
		}
		
		load_prefs_done = true;
	}
	
	// Auto-resize the popup window. Only do this once... resizing may affect the result 
	// of test_window_loaded, so it is protected internally as a coroutine segment.
	if (is_popup)
	{
		// In the popup case we initially resize the parent window to the contents.
		do_auto_resize("console_iframe");
	}
	else
	{
		// In the widget case, we want to resize the contents to the parent window.
		resize_console_to_container();
	}
	

	// There are some race conditions that occur during startup, so we want to make sure
	// that all the essential elements are loaded before initializing.
	// TODO: Not sure if this still happens... the code below doesn't look like it will work.
	if (!test_window_loaded())
	{
		startup_tries++;
		
		if (startup_tries < 5)
		{
			debug_msg("Window not ready... retry");
			setTimeout("jsconsole_init();", 100);
		}
		
		if (startup_tries == 5)
		{
			alert(alert_strings_obj.page_init_failed);
		}
		return;
	}

    // Call get_logo_frame to initalize the cached pointer to the
    // logo frame. (The reason we do this is that if the user closes the
    // opener window later on then we will still be able to locate the
    // logo frame.)
    get_logo_frame();

	create_console_input_shunt();
	
	init_acon_objs();
	
	init_cnsl_event_handlers();

	
	// Load our connection state & preferences.
	try
	{
		// Read the connection state from the connections cookie.
		parse_connection_cookie();
	}
	catch (e)
	{
		debug_msg("Failed to load settings from cookies.");
	}
	
	// Initialize the console frame with the saved connection & history state.
	// The implementation is different, depending on whether this is the dashboard
	// console or the detached console. (This is for historical reasons.)
	if (is_popup)
	{
		// If the dashboard console is being detached, copy the current state 
		// of the console from the opener window.
		try
		{
			var opener_hdl = window.opener;
			
			// Stop any current activity on the dashboard... otherwise we will compete.
			opener_hdl.stop_ajax();
			
			// Currently we can only be popped up from the dashboard, so we assume 
			// that we can get access to all the variables from the dashboard console.
			cons_connected = opener_hdl.cons_connected;
			console_session_id = opener_hdl.console_session_id;
			cursor_x_pos = opener_hdl.cursor_x_pos;
			inBuf = opener_hdl.inBuf;
			editline_buffer_maybe_dirty = opener_hdl.editline_buffer_maybe_dirty;
			for ( i = 0; i < opener_hdl.console_history.length; i++ ) 
				console_history.push(opener_hdl.console_history[i]);

			// Right now, we always copy the contents of the dashboard console (including the
			// "please connect" message if we are not connected.
			var openerelem = opener_hdl.getConsoleDoc().getElementById("cons_span");
			var curelem = getConsoleDoc().getElementById("cons_span");
			
			curelem.innerHTML = openerelem.innerHTML;
		}
		catch (e)
		{
			debug_msg("Failed to transfer console state.");
		}
	}
	else
	{
	    if (!jsconn_init_done) // whole page reload
		// If we are in the main window, and browsing back to the status page, try
		// to restore the saved state from the logo frame.
		try
		{
			var hdr_doc = top.document;
			var oElem_c = hdr_doc.getElementById('saved_cnsl_enabled');
			var oElem_d = hdr_doc.getElementById('cnsl_detached');
			
			// cons_connected is a local variable to say whether *this frame* is 
			// connected, so check if we are detached as well.
			cons_connected = oElem_c.checked && !oElem_d.checked;

			// 3 cases: detached, attached & connected, attached & not connected.
			if (oElem_d.checked)
			{
				hide_console_window(window);
			}
			else if (cons_connected)
			{
				cursor_x_pos = hdr_doc.getElementById('saved_cursor_pos').value;
				//inBuf = hdr_doc.getElementById('saved_inBuf').value;
				editline_buffer_maybe_dirty = 
					hdr_doc.getElementById('saved_editline_buf_dirty').checked;
				oElem = hdr_doc.getElementById('saved_session_id');
				console_session_id = oElem.value;
				oElem = hdr_doc.getElementById('saved_cnsl_hist');
				rawBuf = oElem.value;
				process_rx_buffer();
			}
			else
			{
				reset_welcome_text();
			}
		}
		catch (e)
		{
			debug_msg("Failed to load saved console state.");
		}
	    // iframe reload()
	    else {
		if (cons_connected) {
		    if (cons_span)
			getConsoleDoc().getElementById('cons_span').innerHTML = cons_span;
		    process_rx_buffer();
		}
		else
		    reset_welcome_text();
	    }
	}


	// If we are already connected, start the polling.
	if (cons_connected)
	{
		// Update the buttons & cmd window
		set_connection_button_states(1);

		begin_acon_polling();
	}
	else
		set_connection_button_states(0);

	
	// If we are detaching then update the variable in the header frame and send a message 
	// to the dashboard window so it can update its state (e.g. hide the console widget).
	if (is_popup)
	{
		// Make the opener window blank
		try
		{
			var opener_hdl = window.opener;
			
			var oElem = opener_hdl.top.document.getElementById('cnsl_detached');
			if (oElem)
				oElem.checked = true;
				
			// Click the hidden button on the opener window to send a notification message.
			var oElem = opener_hdl.document.getElementById('call_back');
			if(oElem)
				oElem.click();
			
			
			// Set the connect/disconnect button visibility.
			var st = acon_prefs.use_connect_btns ? "" : "none";
			document.getElementById("jsconn_cnct_btn_span").style.display = st;
		}
		catch (e)
		{
		}
	}

	// Setup the event handlers for closing the page.
	if (is_popup)
	{
		window.onbeforeunload = on_close_console_popup;
		window.onunload = on_close_console_final;
	}
	else
		window.onbeforeunload = on_leave_page;

	
	cursor_blink_loop();
	update_connected_status_notifier();
	
	// Set a couple of things to happen after the window is loaded - (the timeout
	// values here are arbitrary fudge values based on experimentation, and
	// may be incorrect.)
	setTimeout("rescroll_history();", 300);
	
	// If we are detached, set the input focus. (If this is on the dashboard,
	// don't presume that the user is interested in the console specifically.)
	if (is_popup)
	{
		setTimeout("set_console_window_focus(this);", 400);
	}
	
	debug_msg("Initialization succeeded.");
	jsconn_init_done = true;
}

function is_status_page(opener_hdl)
{
	if(opener_hdl.location.href.indexOf("status") == -1)
		return false;
	return true;
}

function is_logo_page(opener_hdl)
{
	var loc = opener_hdl.location.href;
	var pos = loc.indexOf("logo");
	if(pos == -1)
		return false;
	// Need to check it's a real "logo" page or "logo..." page
	if( (pos + 4) == loc.length)
		return true;
	return false;
}

// on_close_console_popup - onbeforeclose handler for the popup console.
// The current behaviour is to reattach the console to the dashboard and
// preserve the current state.
function on_close_console_popup()
{
    // window.opener shouldn't normally be null here. If the main window is closed, we
    // would be automatically closed before it happened. However, if the console was
    // originally detached from inside a widget, then we may very well lose our back
    // pointer to the opener (which was an iframe). Thus we always keep a cached reference
    // to the logo frame just for this reason.
    // TODO: the cached reference to the logo frame doesn't work in IE if the parent
    // window is closed.
    var opener_hdl = window.opener;
    var logo_doc = get_logo_frame().document;


    // restore_console_to_opener - copy the JS console state to a frame that is currently
    // displaying the "console is currently detached" message.
    function restore_console_to_frame(opener_hdl)
    {
        // Copy the resource to opener window
        var openerelem = opener_hdl.getConsoleDoc().getElementById("cons_span");
        var curelem = getConsoleDoc().getElementById("cons_span");

        openerelem.innerHTML = curelem.innerHTML;
        opener_hdl.cons_span = curelem.innerHTML; // also save history in opener for restore if iframe reload
        opener_hdl.cons_connected = cons_connected;
        opener_hdl.console_session_id = console_session_id;
        opener_hdl.cursor_x_pos = cursor_x_pos;
        opener_hdl.editline_buffer_maybe_dirty = editline_buffer_maybe_dirty;
        opener_hdl.inBuf = inBuf;
        opener_hdl.console_history.length = 0;
        for (var i = 0; i < console_history.length; i++)
            opener_hdl.console_history.push(console_history[i]);
    }

    // save_console_to_logo_frame - if we can't restore the console directly to
    // a visible frame then save the state in the logo frame.
    function save_console_to_logo_frame()
    {
        store_console_hist();

        logo_doc.getElementById('saved_cnsl_enabled').checked = cons_connected;
        logo_doc.getElementById('saved_editline_buf_dirty').checked = editline_buffer_maybe_dirty;
        logo_doc.getElementById('saved_cursor_pos').value = cursor_x_pos;
        logo_doc.getElementById('saved_session_id').value = console_session_id;
        //logo_doc.getElementById('saved_inBuf').value = inBuf;
    }


    try
    {
        // If the user is currently viewing the dashboard page (with the "console is detached"
        // message, then restore the console immediately).
        // TODO: this doesn't cover the widgetization case - we don't detect if the console is
        // being displayed on some other page. (In which case opener_hdl will probably
        // be null and we would need to detect the page in some other way.)
        if (opener_hdl && is_status_page(opener_hdl))
        {
            // Since the opener page won't be reloaded here, we need to copy the innerHTML,
            // connection status, current cursor position, inBuf and console history to
            // the opener page.
            restore_console_to_frame(opener_hdl);
        }
        else
        {
            // If we mean to close the popup window when the current page is not status
            // we need to store the history, connection status, cursor position and
            // to temporary logo variables so that the status page can get all the
            // information when it's loaded next time.
            // TODO: this may not work on IE from within a widgetized page. IE doesn't seem
            // to allow us to keep the reference to the logo frame after the parent
            // iframe is closed.
            save_console_to_logo_frame();
        }
    }
    catch(e)
    {
//        logo_frm.console.log("error in on_close_console_popup: " + e);
    }

    try
    {
        oElem = logo_doc.getElementById('cnsl_detached');
        if(oElem)
            oElem.checked = false;

        // Try to click one hidden button on opener window to begin polling
        // TODO: this doesn't look like it will be compatible with widgetization.
        var oElem = opener_hdl.document.getElementById('call_back');
        if(oElem)
            oElem.click();

        // TODO: Yanna said that setting hPopupConsole = null in the logo frame is needed here
        // but not tested...

        // Close myself
        // TODO: this shouldn't be necessary... this function is only called in response to a close event.
        // window.close();
    }
    catch(e)
    {
//        logo_frm.console.log("error2 in on_close_console_popup: " + e);
    }
}


// display_console_window - bad name... actually the console window is displayed by default,
// and this function may only hide it.
function display_console_window()
{
	// If elem_d is null, that means the logo frame is not loaded... probably we refreshed
	// the frameset, so the default should be to show the console window.
	var elem_d = top.document.getElementById('cnsl_detached');
	
	if (elem_d && elem_d.checked)
	{
		hide_console_window(window);
	}
	else
	{
		show_console_window(window);
	}
}


// reset_welcome_text - blank out the window and set the text back to the default.
function reset_welcome_text()
{
	var tbody = getConsoleTbody();
	
	// Remove all console history.
	while (tbody.childNodes.length)
	{
		tbody.removeChild(tbody.childNodes[0]);
	}
	
	cnsl_buffer_cur_len = 0;

	
	var msg = alert_strings_obj.conn_msg;
	
	if (acon_prefs.use_connect_btns)
	{
		msg = "Click \"Connect\" to begin...";
	}
	
	// We actually want this message to be overwritten... so don't
	// add an empty row afterwards.
	append_welcome_msg(tbody, msg)
//	append_complete_row(tbody, "")
}


// Unsure whether document.onload or window.onload is the better one. They both work, and they
// both seem to be called multiple times. MS documentation says document.onload doesn't exist,
// so window.onload seems better. (Anyway, don't set this here... let the HTML file call it.)
//window.onload = custom_startit;
//document.onload = custom_startit;

// add_console_focus_listeners - watch the console window for focus/blur events
function add_console_focus_listeners()
{
	try
	{
		var xdoc = getConsoleDoc();

		if (xdoc.addEventListener)
		{
			// Mozilla case (doesn't work on IE).
			//xdoc.addEventListener("focus", on_console_focus, true);
			xdoc.addEventListener("blur", on_console_blur, true);
			xdoc.addEventListener("click", on_console_focus, true);
		}
		else
		{
			// IE case (doesn't work on Mozilla).
			xdoc.onclick = on_console_focus;
			var oFrame = document.getElementById("console_iframe");
			//oFrame.onfocus = on_console_focus;
			oFrame.onblur = on_console_blur;
		}
	}
	catch (e)
	{
		debug_msg("Failed to initialize console focus listeners.");
	}
}

// init_cnsl_event_handlers - initialize the event handlers for keyboard
// and focus events.
function init_cnsl_event_handlers()
{
debug_msg("in init_cnsl_event_handlers");

	// Watch the command window for keypresses (inexplicably, this appears to work 
	// on both IE and Mozilla)
	try
	{
		// Initialize the command edit box listeners.
		// (This component is optional.) 
		var oCmd = document.getElementById("jsconn_cmd");
		if (oCmd)
		{
			oCmd.onkeyup     = cmd_keyup;
			oCmd.onkeydown   = cmd_keydown;
			oCmd.onkeypress  = cmd_keypress;
		}
	}
	catch (e)
	{
		debug_msg("Failed to initialize keyboard event listeners.");
	}

	// Watch the console window for keypresses. The console document object automatically 
	// receives all these events when the console iframe has the focus.
	try
	{
		var xdoc = getConsoleDoc();
		xdoc.onkeypress = cons_keypress;
		xdoc.onkeydown = cons_keydown;
		xdoc.onkeyup = cons_keyup;
	
		// Most sample pages include the below code... Allegedly it does something
		// on Mozilla, although I don't see it being called.
		if (xdoc.layers)
		{
			debug_msg("Initializing captureEvents for layers");
			xdoc.captureEvents(Event.KEYDOWN);
			xdoc.captureEvents(Event.KEYPRESS);
			xdoc.captureEvents(Event.KEYUP);
		}
	}
	catch (e)
	{
		debug_msg("Failed to initialize keyboard event listeners.");
	}

	// Watch the console window for focus/blur events.
	add_console_focus_listeners();
	
	window.onresize = console_resize_handler;
}

/******************************
         Cleanup
 ******************************/

// ie_pause - pause function (only works on IE).
function ie_pause(numberMillis) 
{
	// This function implements a pause on IE browsers by creating a hidden modal
	// dialog. It will throw an exception on Mozilla because the showModalDialog 
	// function doesn't exist (it can be implemented by calling openDialog
	// with the modal flag, but that feature is only available to trusted scripts).
	var dialogScript = 
	   'window.setTimeout(' +
	   ' function () { window.close(); }, ' + numberMillis + ');';
	var result = 
	 window.showModalDialog(
	   'javascript:document.writeln(' +
		'"<script>' + dialogScript + '<' + '/script>")');
}

 
/* 

 The logic when the user closes the popup window is a bit tricky... 
 we would like to send a disconnect
 message before closing the window, but we won't be around to process
 the reply. That's not really important... we just want the browser to
 finish sending the message before ending the script. So we try this:
 
 1. Send the disconnect message from the onbeforeunload handler and then
 stop ajax from the onunload handler. This adds a small window of opportunity
 for the browser to finish sending the request.
 
 2. On IE, sleep for half a second to give the browser time to send
 the message. Then stop Ajax in the onunload handler.
 
 Unfortunately, the disconnect feature is fairly unreliable on Mozilla
 due to the cases mentioned above.

*/

 
// on_close_console - onbeforeunload and logout handler for the logo frame.
function on_close_console_logo()
{
	var topdoc = top.document;
	var oElem = topdoc.getElementById('saved_cnsl_enabled');
	cons_connected = oElem.checked;
	oElem = topdoc.getElementById('saved_session_id');
	console_session_id = oElem.value;
	
	if (cons_connected)
	{
		// Because logo frame never initializes the acon objects, we need to do
		// it here
		init_acon_objs();
		
		// Must set the cons_connected flag to false so the rx_loop doesn't
		// retry the poll request while we are in the process of closing.
		cons_connected = false;

		// Stop all other ajax calls before sending the disconnect event. We want 
		// to maximize the probability of the event getting through.
		try
		{
			stop_ajax();
			on_disconnect_clicked();
		}
		catch (e)
		{
		}
		
		try
		{
			// Sleep long enough to give the browser a chance to send the
			// ajax disconnect message.
			ie_pause(500);
		}
		catch (e)
		{
			// If ie_pause throws an exception, this is presumably not IE.
			// The disconnect event may not always be called before the
			// window closes.
		}
	}
}


/* 

 IE seems to have a bug where it will continue to process XML requests
 after the user has closed the window. Thus, it is imperative that we
 always cleanup any active requests before closing. (The symptom of the
 bug is that if you open the window and close it 2 times in the same
 browser session then the 3rd attempt to open the window will cause the
 browser to hang. 
 
*/


// on_close_console_final - onunload handler for the popup window.
function on_close_console_final()
{
	// On IE, it is imperative that we call stop_ajax() any time
	// the window is closed. Hopefully, the disconnect message has
	// been sent by now.
	try
	{
	   stop_ajax();
	}
	catch (e)
	{
		// In practice, this function will often be called after the page
		// has already been unloaded, in which case the acon_tx/rx
		// objects may already be out of scope.
	}
}


// on_leave_page - onbeforeunload handler for the dashboard window.
// (Called when the user closes the window or browses to another page...
// probably to difficult for us to tell the difference, so we will
// assume they are just leaving the page. Let the logo's onbeforeunload
// callback handle the case where the user closes the whole window.)
function on_leave_page()
{
	// Kill the ajax objects in order to avoid the IE bug.
	// (But don't disconnect, since this is the dashboard console.)
	try
	{
		stop_ajax();
	}
	catch (e)
	{
		// In practice, this function may be called after the page
		// has already been unloaded, in which case the acon_tx/rx
		// objects may already be out of scope.
	}

	// Store the connection state and console history in the logo frame.
	try
	{
		var logo_doc = get_logo_frame().document;

		// If the console window is detached but the current window is not
		// on the popup one, we do nothing.
		if(logo_doc.getElementById('cnsl_detached').checked)
			return;
		
		var oElem = logo_doc.getElementById('saved_cnsl_enabled');
		if(oElem) 
			oElem.checked = cons_connected;
		var oElem = logo_doc.getElementById('saved_session_id');
		if(oElem) 
			oElem.value = console_session_id;
		var oElem = logo_doc.getElementById('saved_editline_buf_dirty');
		if(oElem) 
			oElem.checked = editline_buffer_maybe_dirty;
			
		if (cons_connected)
		{
			store_console_hist();
		}
	}
	catch (e)
	{
		// This may sometimes trigger an error during logout (maybe because the logo_doc
		// contents are no longer valid?), however we don't need to care about saving the
		// session variables if the user is logging out anyway.
	}
}


// dashboard_detach_console - detach the console from the dashboard and display
// it in a new window.
function dashboard_detach_console()
{
	// Launch the popup window using the approximate final size.
//	var new_wnd = popup_resizable('/system/jsconsole', 'js_console', 600, 425);
	var new_wnd = popup_nonscrollable('/system/jsconsole', 'js_console', 600, 425);

	// Test: store reference to this window in the logo frame
	top.window.hPopupConsole = new_wnd;
}



// Call back function to be called in the opener window when something happens
// on the popup window.
// flag: 0 -- the consolecustomize page is called, we need to parse the cookie
//       1 -- when the detached window is attached back, we need to enable/disable
//            the connection/disconnection buttons and begin polling
function on_call_back_clicked(flag)
{
	switch(flag) {
		case 0: // the console setting is changed
			debug_msg("Notification from customize page: reload settings");
			parse_console_cookie();
			
			var cnsl_doc = getConsoleDoc();
			apply_console_prefs(cnsl_doc);
			
			// If the layout has potentially changed, then re-layout the popup window.
			if (window.opener && wnd_size_st.need_resync)
			{
				wnd_size_st.already_resized = false;
				do_auto_resize();
			}
			break;
			
		case 1: // called on both detach and re-attach.
			debug_msg("Notification from jsconsole popup: the window was attached/detached.");
		
			var logo_frm = get_logo_frame();
			var oElem = logo_frm.document.getElementById('cnsl_detached');
			var is_detached = oElem.checked;
			
			// If the console is being reattached, update the button state and begin
			// the polling if we are connected.
			if (!is_detached)
			{
				// Update the buttons & cmd window
				set_connection_button_states(cons_connected);
				
				// cons_connected was copied over by the popup window's JS code during
				// reattachment.
				if (cons_connected)
					begin_acon_polling();
					
				setTimeout("rescroll_history();", 100);
			}
			
			// Show/hide the console widget elements
			if (is_detached)
			{
				hide_console_window(this);
			}
			else
			{
				show_console_window(this);
				//set_console_window_focus(this);
			}

			// Now update the connected status notifier in the widget header. 
			update_connected_status_notifier();
			
			break;
	}
}


// create_console_input_shunt - Create the console input shunt text area. Whenever the
// user focuses on the console window, divert the focus here. That way we can capture
// the input from paste events.
function create_console_input_shunt()
{
	var cnsl_doc = getConsoleDoc();
	
	var elem = cnsl_doc.createElement("textarea");
	elem.id = "cons_input_shunt";
	
	with (elem.style)
	{
		// Set the position to be at the desired initial scroll position.
		position = "absolute";
		left = 0;
		top = 0;
		
		// IE won't let us set the focus on a hidden element, so make sure the element
		// is visible but with 0 size.
		width = 0;
		height = 0;
		lineHeight = 0;
		border = "none"
	}
	
	cnsl_doc.body.appendChild(elem);
	debug_msg("Added console input shunt.");
}

// focus_on_cons_input_shunt - set the focus to the input shunt, but make sure to
// preserve the scroll position.
function focus_on_cons_input_shunt()
{
	align_shunt_to_window();
	
	// For some reason, we need to let the repositioning take effect before setting
	// the focus to the shunt.
	setTimeout("shunt_focus_part2()",1);
}

// align_shunt_to_window - place the shunt so it is in the top left of the
// visible portion of the window.
function align_shunt_to_window()
{
	var xdoc = getConsoleDoc();
	var elem = xdoc.getElementById("cons_input_shunt");
	
	var st_x = xdoc.body.scrollLeft;
	var st_y = xdoc.body.scrollTop - 1;
	
	debug_msg("set scroll height = " + st_y + ", width = " + st_x);

	with (elem.style)
	{
		// When we set the position, the browser will scroll the window to put the
		// element in view. So we need to make sure that the input shunt is always
		// at the appropriate screen location before setting the focus.
		left = "" + st_x + "px";
		top = "" + st_y + "px";
	}
}

function shunt_focus_part2()
{
	var xdoc = getConsoleDoc();
	var elem = xdoc.getElementById("cons_input_shunt");
	elem.focus();
}



/******************************
 wij_detach.js - Detaching & auto-resizing related functions
 Init by A. Krywaniuk, Jan 2006
 Split from jsconsole.js and jsconsole_init.js by J. Thompson
 Copyright Fortinet, inc.
 All rights reserved
 ******************************/

//EXPORT_SYMBOL get_logo_obj

// get_logo_frame - access the logo (header) frame, taking into account that:
// a) this may be a popup window or inline
// b) the parent may be a widget rather than a standalone page.
function get_logo_frame()
{
    var wnd = window;
    var opener = wnd.opener || wnd.parent.opener;

    if (opener)
    {
        wnd = opener.parent || opener;
    }

    return wnd.top;
}

function get_logo_obj(id)
{
    var wdg = get_logo_frame().widget_states;
    if (!wdg)
    {
        // Note that this is really an error, as if the widget_states object
        // is not located, it means get_logo_frame found the wrong window.
        // Using an empty object will allow modules that depend on this to
        // run as expected, but may have some side-effects.
        wdg = new Object();
    }

    var key = '' + id;
    var obj = wdg[key];

    if (!obj)
    {
        obj = wdg[key] = new Object();
    }

    return obj;
}

// Set the checkbox value to true or false on logo frame
// name_t:   checkbox id on logo frame
// value_t:  true or false
function set_logo_checkbox(name_t, value_t)
{
	var logo_frm = get_logo_frame();
	var logo_doc = logo_frm.document;
	var oElem = logo_doc.getElementById(name_t);
	oElem.checked = value_t;
}

// Set the text box value on logo frame
// name_t:   text box id on logo frame
// value_t:  value
function set_logo_text(name_t, value_t)
{
	var logo_frm = get_logo_frame();
	var logo_doc = logo_frm.document;
	var oElem = logo_doc.getElementById(name_t);
	oElem.value = value_t;
}


/******************************
     Window Size functions
 ******************************/

// wnd_size_st - This object is used by the popup window to cache
// various size values. Currently, only the sessions widget &
// console use it. If it is determined that they both require a 
// local copy, this should be moved into the individual modules and
// use references below.
var wnd_size_st = {
	already_resized: false,
	need_resync: false,
	
	// Window size
	orig_w: 0,
	orig_h: 0,
	auto_w: 0,
	auto_h: 0,
	cur_w: 0,
	cur_h: 0,
	
	// Frame size
	orig_fw: 0,
	orig_fh: 0,
	
	// Div size
	orig_dw: 0,
	orig_dh: 0,
	auto_dw: 0,
	auto_dh: 0
};

// sz_obj - global object to hold size calculations.
// Used to hold the return value of calc_client_sz, calc_elem_sz, etc.
var sz_obj = {
	w:0,
	h:0
};

function calc_client_sz(rv)
{
  var x = window; 
  var myW = 0, myH = 0, d = x.document.documentElement, b = x.document.body;
	  if( x.innerWidth ) { myW = x.innerWidth; myH = x.innerHeight; }
	  else if( d && d.clientWidth ) { myW = d.clientWidth; myH = d.clientHeight; }
	  else if( b && b.clientWidth ) { myW = b.clientWidth; myH = b.clientHeight; }
	
	rv.w = myW;
	rv.h = myH;
}

function calc_elem_sz(id, rv)
{
	var elem = document.getElementById(id);
	rv.w = parseInt(elem.style.width);
	rv.h = parseInt(elem.style.height);
}

function calc_autodiv_sz(rv)
{
	calc_elem_sz('autosizing_div', rv);
}


/* Window resizing event handlers:

 We want to do two things when the window is resized:
 
 1. Adjust the size of the console window to match the outer window.
 2. Prevent the user from resizing the window beyond a certain minimum size.
 
 Unfortunately, this is amazingly complicated to do (not surprisingly, pretty much
 every API function works differently on each browser). For the most part, the
 size adjustment of the console window works well. The difficulties come in adjusting
 for the case where the window goes below the minimum size.
 
 Complications:
 
 1. I didn't see any way to retrieve both the new size and the old size at the same
 time, so we store the previous values in a global variable.
 2. IE seems to receive this event twice. The second call is basically sprurious.
 3. We have to deal with the client area of the window, which is different from the
 size of the window frame.
 4. If we call resizeTo() or resizeBy() from the onresize handler, we have to deal
 with various amounts of recursion.
 
 Issues 3 & 4 means that we can't just resize exactly to the desired final size.
 Basically, we have to take an iterative approach and be happy if the final size is
 within close enough limits. (Also, we layout the page in such a way that most of
 the realignment happens automatically.)

 There is also a strange bug that I could only reproduce on old Mozilla. If the user
 resizes the window below the minimum reasonable size (at which it can display all
 the content), then scrolls the client area (using the mouse drag), then resizes
 the window again (to make it bigger), the clipping area won't match the client area
 and there will be garbage displayed on the screen. We use an extra scrollTo to
 correct that problem.
 
 Various error messages come up in the Javascript console about failure to parse the
 value for attribute height or width. I'm not sure exactly when these errors are
 generated, but they don't appear to come from the JS in this file. (It is possibly due 
 to an attempt to read the values before the window is redrawn.
 
*/

// resize_handler - window.onresize event handler
// Adjust the div & iframe elements to have the appropriate size.
function resize_handler(evt, iframe_id)
{
	// save old size
	var old_w = wnd_size_st.cur_w;
	var old_h = wnd_size_st.cur_h;

	// store new size
	calc_client_sz(sz_obj);
	wnd_size_st.cur_w = sz_obj.w;
	wnd_size_st.cur_h = sz_obj.h;
	
	// reset the semaphore
	wnd_size_st.need_resync = false;

	// Don't do anything else if we are still in the auto-size phase.
	if (!wnd_size_st.already_resized)
	{
		return;
	}

//	debug_msg("resizing: width " + old_w + "->" + wnd_size_st.cur_w);
//	debug_msg("resizing: height " + old_h + "->" + wnd_size_st.cur_h);
	
	// Base all the adjustments relative to the initial (auto-calculated)
	// window & div size.
	var dw = wnd_size_st.cur_w - wnd_size_st.auto_w;
	var dh = wnd_size_st.cur_h - wnd_size_st.auto_h;
	
	try
	{
		var em_ifrm = document.getElementById(iframe_id);
		if (em_ifrm)
		{
			em_ifrm.style.width = "" + (wnd_size_st.orig_fw + dw) + "px";
			em_ifrm.style.height = "" + (wnd_size_st.orig_fh + dh) + "px";
		}
	}
	catch (e)
	{
		// This fails in IE when you try to resize the frame to be larger
		// than the current dimensions of the window. It should correct
		// itself iteratively.
	}
	
	try
	{
		em_div = document.getElementById('autosizing_div');
		if (em_div)
		{
			em_div.style.width = "" + (wnd_size_st.auto_dw + dw) + "px";
			em_div.style.height = "" + (wnd_size_st.auto_dh + dh) + "px";
		}
	}
	catch (e)
	{
		// This fails on IE for some reason. However, the layout generally
		// looks ok.
	}
	
	// Don't let the user set the size below a certain minimum.
	// We can't prevent them from doing it, but we can immediately revert
	// the change.
	var rsz_w = 0;
	var rsz_h = 0;
	
	// Resize a minimum of 50 pixels at a time - we are comparing the client area 
	// with the frame size here, so the values aren't going to match exactly.
	if (wnd_size_st.cur_w < popup_min_sz.w)
		rsz_w = Math.max(popup_min_sz.w - wnd_size_st.cur_w, 50);
	if (wnd_size_st.cur_h < popup_min_sz.h)
		rsz_h = Math.max(popup_min_sz.h - wnd_size_st.cur_h, 50);

	// The reason we use resizeBy rather than resizeTo is because the outer window
	// contains various non-client elements (e.g. status bar), and it's difficult
	// for us to predict the size. If we got it wrong, we could get into an endless
	// loop. With resizeBy, we are guaranteed to terminate.
	if (rsz_w || rsz_h)
	{
		var new_cmd = "window.resizeBy(" + rsz_w + "," + rsz_h + ");";
		setTimeout(new_cmd, 5);
	}
	
	
	// Fix the old-Mozilla clipping area bug.
	// We don't want to run this on IE. (Possibly there are bad side effects.)
	if (evt)
	{
		try
		{
			setTimeout("window.scrollTo(0,0);", 2);
		}
		catch (e)
		{
		}
	}
}

// store_orig_size - store the size of the window prior to the auto-resize.
// (this is probably not needed)
function store_orig_size(iframe_id)
{
	calc_client_sz(sz_obj);
	wnd_size_st.orig_w = sz_obj.w;
	wnd_size_st.orig_h = sz_obj.h;
	
	calc_autodiv_sz(sz_obj);
	wnd_size_st.orig_dw = sz_obj.w;
	wnd_size_st.orig_dh = sz_obj.h;
	
	calc_elem_sz(iframe_id, sz_obj)
	wnd_size_st.orig_fw = sz_obj.w;
	wnd_size_st.orig_fh = sz_obj.h;
}

// store_auto_size - store the size of the window after the auto-resize.
// (all future adjustments are calculated relative to this size)
function store_auto_size()
{
	calc_client_sz(sz_obj);
	wnd_size_st.auto_w = sz_obj.w;
	wnd_size_st.auto_h = sz_obj.h;

	calc_autodiv_sz(sz_obj);
	wnd_size_st.auto_dw = sz_obj.w;
	wnd_size_st.auto_dh = sz_obj.h;
}


// do_auto_resize - called when the popup window is loaded, or
// when the layout changes.
function do_auto_resize(iframe_id)
{
	if (!wnd_size_st.already_resized)
	{
		store_orig_size(iframe_id, wnd_size_st);
		auto_resize_popup();
		store_auto_size(wnd_size_st);
		wnd_size_st.already_resized = true;
	}
}



