/******************************
 Javascript Console
 Init by A. Krywaniuk, Jan 2006
 Copyright Fortinet, inc.
 All rights reserved
 ******************************/

// STRIP-FUNCTION: debug_msg
// STRIP-FUNCTION: debug_kbd



/* Editline octal sequences:

The CLI can store Asian characters, but it doesn't interpret them. Normally, this is not a
problem for displaying them (aside from the encoding conversion hassle), but it does cause
problems with editline. Editline doesn't treat MB characters as 2 byte characters. Instead, it
encodes them as escaped octal sequences (e.g. \303\374). When performing editing commands, this
causes great confusion, since you may receive the VT-100 sequence ^[8D to delete a single Asian
character. To solve this problem, we pad the inBuf with special non-printable characters in
order to keep the cursor position in sync.

This works some of the time, but not all of the time. Sometimes editline will send us a sequence
such as BS3 (backspace 3). The intention may be to replace the character \303\374 with \303\373.
However, we have no means of processing this. The interpretation of this character is strictly
bound to the MB character set that is used on the server side, so once we have converted the
message to UTF-8, we can no longer perform byte-wise editing operations. The solution is to add
MBC awareness to the CLI. Or just rewrite the GUI in UTF-8.

*/

// mbc_octseq_marker - this special character is sent in the input stream to mark the place where
// an octal sequence (generated by editline) has been converted into binary. In order to keep the
// cursor positions in sequence, we must expand the marker into a series of padding bytes which are
// not displayed to the user. 
var mbc_octseq_marker = '\x1F';

// chr_spacing - use this special character (0xF1 = 31 decimal) to pad Asian characters in the
// command line so their length matches the encoded length in the CLI. Then we strip them
// out before displaying them. Note that we currently use the same character for the sequence marker
// and the padding bytes, but this is not a requirement.
var chr_spacing = '\x1F';
var chr_spacing7 = chr_spacing.concat(chr_spacing,chr_spacing,chr_spacing,chr_spacing,chr_spacing,chr_spacing);
var strip_sc_regexp = /\x1F/g;


/******************************
  Main window display functions
 ******************************/

// Since we now put the console buf within a separate <iframe>, we must use the correct
// document reference when creating elements. (This is required by IE, but not Mozilla.)

// create_hist_elem - create an HTML element within the appropriate document.
function create_hist_elem(type_str)
{
	return getConsoleDoc().createElement(type_str);
}

// create_hist_elem - create a text node within the appropriate document.
function create_hist_text_node(xstr)
{
	return getConsoleDoc().createTextNode(xstr);
}

 
// clean_last_td - Remove all text and subelements from the last TR in the table, and
// populate it with a single empty TD. (Create a new TR if the table is empty.)
function clean_last_td(tbody)
{
	var last_tr = get_last_child_of_type(tbody, "TR");
	
	if (last_tr)
	{
		while(last_tr.childNodes.length)
		{
			last_tr.removeChild(last_tr.childNodes[0]);
		}
	}
	else
	{
		last_tr = create_hist_elem("TR");
		tbody.appendChild(last_tr);
	}
	
	var new_td = create_hist_elem("TD");
	last_tr.appendChild(new_td);
	
	// Use a <PRE> block to solve the problem of trailing/consecutive spaces being
	// ignored by HTML. Possibly this could be fixed in a better way(?)
	
	var new_pre = create_hist_elem("PRE");
	new_td.appendChild(new_pre);
	
	return new_pre;
}


/* Various display problems may occur if we create an empty <td> using
createTextNode. Either the row will be added with 0 height, or sometimes it
will initially be displayed correctly but then fail later when we copy
the innerHTML. So make sure we always add a blank space when we
detect an empty line.
*/


// display_row - main function to display the active line of the console
function display_row(new_td, txt)
{
	var str1 = "";
	var str2 = "";
	var str3 = "";
	
	try
	{
		if (cursor_x_pos > 0)
		{
			if (cursor_x_pos < txt.length)
			{
				str1 = txt.substr(0, cursor_x_pos);
			}
			else
			{
				str1 = txt;
			}
		}
		
		if (txt.length && cursor_x_pos < txt.length)
		{
			str2 = txt.substr(cursor_x_pos, 1);
		}
		else if (txt.length)
		{
			// Need to have a space so the cursor blink looks correct.
			str2 = " ";
		}
		else
		{
			// Various formatting problems may occur if we add an empty <td>.
			// But there should never be a cursor at the beginning of the line,
			// so better to put the empty space in str1.
			str1 = " ";
		}
		
		if (cursor_x_pos + 1 < txt.length)
		{
			str3 = txt.substr(cursor_x_pos+1);
		}
	}
	catch (e)
	{
		debug_msg("split text failed, cursor_x_pos = " + cursor_x_pos + ", len = " + txt.length);
	}
	
	var txt1 = create_hist_text_node(str1);
	var txt2 = create_hist_text_node(str2);
	var txt3 = create_hist_text_node(str3);


	// Implement cursor
	var new_span = create_hist_elem("SPAN");
	new_span.appendChild(txt2);
	new_span.id = 'history_cursor';
	// set on/off state based on global variable
	update_cursor_style(new_span);
	
	new_td.appendChild(txt1);
	new_td.appendChild(new_span);
	new_td.appendChild(txt3);
}


// strip_spacing_chars - strip out special spacing chars that are only used in keeping the cursor
// position in sync w/ the CLI when using Asian character sets.
function strip_spacing_chars(txt)
{
//debug_msg("strip_spacing_chars - before=" + txt);
	var rslt = txt.replace(strip_sc_regexp, "");
//debug_msg("strip_spacing_chars - after=" + rslt);
	return rslt;
}

// overwrite_last_row - called to update an existing row. Also displays
// the cursor.
function overwrite_last_row(tbody, txt)
{
//debug_msg("overwrite_last_row: " + txt);
	var td = clean_last_td(tbody);
	txt = strip_spacing_chars(txt);
	
	display_row(td, txt);
}

// Store the console history in an array. Keep the history in one hidden 
// textarea variable in logo page when the URL goes away from status page.

var console_history = new Array();

function add_to_console_history(str)
{
	console_history.push(str);
	
	if (console_history.length > acon_prefs.acon_histbuflen)
	{
		console_history.shift();
	}
//debug_msg("console_history = " + console_history);
}

function store_console_hist(open_hdl)
{
	var topdoc = open_hdl.top.document;
	var oElem = topdoc.getElementById('saved_cnsl_hist');
	if(!oElem) return;
	
	var str="";
	for ( i = 0; i < console_history.length; i++ ) {
		str += console_history[i] + '\n';
	}
	str += inBuf;
	
	oElem.value = str;
}

// finalize_row - called after we receive a CR. The cursor is no longer on
// this row, so just output a single td.
function finalize_row(tbody, txt)
{
	var td = clean_last_td(tbody);
	
	txt = strip_spacing_chars(txt);
	
	// Various formatting problems may occur if we add an empty string.
	if (txt.length == 0)
		txt = " ";
	
	var oTxt = create_hist_text_node(txt);
	
	add_to_console_history(txt);
	td.appendChild(oTxt);
}

function setup_new_td(tbody)
{
	var new_tr = create_hist_elem("TR");
	var new_td = create_hist_elem("TD");
	var new_pre = create_hist_elem("PRE");
	
	tbody.appendChild(new_tr);
	new_tr.appendChild(new_td);
	new_td.appendChild(new_pre);
	
	return new_pre;
}

// append_complete_row - add a new row and finalize it right away.
// NOTE: the table must always end with a partial row, so after adding
// a complete row, you should generally create a partial row as well.
function append_complete_row(tbody, txt)
{
	add_to_console_history(txt);
	var td = setup_new_td(tbody);
	
	txt = strip_spacing_chars(txt);
	
	// Various formatting problems may occur if we add an empty string.
	if (txt.length == 0)
		txt = " ";
	
	var oTxt = create_hist_text_node(txt);
	
	td.appendChild(oTxt);
	
	cnsl_buffer_cur_len++;
}

// append_partial_row - append a "partial row". The difference between
// a partial row and a complete row is that a partial row displays the
// cursor (and it will be updated later).
function append_partial_row(tbody, txt)
{
//debug_msg("append_partial_row: " + txt);
	txt = strip_spacing_chars(txt);

	// Presumably, this will be the prompt + whatever we have typed so far.
	cursor_x_pos = txt.length;
	
	var new_td = setup_new_td(tbody);
	display_row(new_td, txt);
	
	cnsl_buffer_cur_len++;
}

// append_welcome_msg - similar to append_complete_row, but we don't append
// the empty row afterwards, so this message will be overridden by the next
// new text from the console. (We also don't add it to the console history.)
function append_welcome_msg(tbody, txt)
{
	var td = setup_new_td(tbody);
	var oTxt = create_hist_text_node(txt);
	td.appendChild(oTxt);
}


/******************************
     Console Buffer Length
 ******************************/

// set low for testing - need to make this configurable
var cnsl_buffer_cur_len = 0; 

// prune_histbuf - truncate to cnsl_buffer_max_len lines
function prune_histbuf()
{
	if (cnsl_buffer_cur_len <= acon_prefs.acon_consbuflen)
		return;
		
	var tbody = getConsoleTbody();
	
	if (cnsl_buffer_cur_len != tbody.childNodes.length)
	{
debug_msg("console buffer length got out of sync: " + cnsl_buffer_cur_len + " vs. " + tbody.childNodes.length);
		cnsl_buffer_cur_len = tbody.childNodes.length;
	}
	
	while (cnsl_buffer_cur_len > acon_prefs.acon_consbuflen)
	{
		tbody.removeChild(tbody.childNodes[0]);
		cnsl_buffer_cur_len--;
	}
}

// Store connection cookie for connection status
function store_connection_cookie()
{
	try
	{
		var str = "connection=" + cons_connected
			+ "; path=/frames";
			
		document.cookie = str;
	}
	catch (e)
	{
		alert(alert_strings_obj.cookie_save_failed + e);
	}
}



/******************************
     Command History Buffer 
 ******************************/

// The command history is stored as an array with the most recent entries
// stored at the end. New entries are added at the end of the array.
// When the array overflows, old ones are removed from the beginning.

var cmd_history = new Array();
var cmd_hist_pos = 0;
var cmd_hist_maxlen = 10;

function add_to_cmd_history(str)
{
	// Ignore empty CRs.
	if (str.length == 0)
		return;
		
	cmd_history.push(str);
	
	if (cmd_history.length > cmd_hist_maxlen)
	{
		cmd_history.shift();
	}
	
	// Setting the history position to one above the last index
	// ensures that a single up-arrow takes the user back to the
	// command they just typed.
	
	cmd_hist_pos = cmd_history.length;
debug_msg("cmd_history = " + cmd_history + " pos = " + cmd_hist_pos);
}

function cmd_hist_up()
{
//debug_msg("cmd_history = " + cmd_history + " pos = " + cmd_hist_pos);
	if (cmd_hist_pos - 1 >= 0)
	{
		cmd_hist_pos--;
//debug_msg("cmd_history[cmd_hist_pos] = " + cmd_history[cmd_hist_pos]);
		return cmd_history[cmd_hist_pos];
	}
	
	// Resync, just in case.
	cmd_hist_pos = 0;
	
	// Return null, meaning no action.
	return null;
}


function cmd_hist_down()
{
//debug_msg("cmd_history = " + cmd_history + " pos = " + cmd_hist_pos);
	if (cmd_hist_pos + 1 < cmd_history.length)
	{
		cmd_hist_pos++;
		
//debug_msg("cmd_history[cmd_hist_pos] = " + cmd_history[cmd_hist_pos]);
		return cmd_history[cmd_hist_pos];
	}
	
	// Resync, just in case.
	cmd_hist_pos = cmd_history.length;
	
	// Return empty string, meaning clear the input box.
	return "";
}


/******************************
     Main parsing functions
 ******************************/

// rawBuf is the unparsed response buffer (that may contain VT escape sequences)
var rawBuf = "";

// inBuf is the buffer of parsed input that is displayed in the last (unfinalized)
// row of the table.
var inBuf = "";

// cursor_x_pos is the offset of the curser within inBuf.
var cursor_x_pos = 0;

// Number of consecutive empty polling responses from the server (typically indicates
// that two consoles are competing for the same data... or this code has a bug and it 
// left multiple polling objects running.
var empty_response_cnt = 0;



// acon_obj::on_acon_buffer_ready - called indirectly by the onreadystatechange handler
function on_acon_buffer_ready(ev)
{
//debug_msg("xx - handle_buffer_ready");
	       
	// Check & reset semaphore.
	if (!this.in_progress)
		return;
		
	var err_code = 0;

	if(this.xmlhttp.status != 200)
	{
		this.cleanup();
		handle_disconnect_event(connection_lost);
		return;
	}

	// Preparse the message format. Extract the first byte, which is the
	// error code.
	var reply_len = this.xmlhttp.responseText.length;
	var status_code = this.xmlhttp.responseText.charAt(0);
	// Copy the reply text so we can be done with the xmlhttp obj.
	var new_text = this.xmlhttp.responseText.substring(1);
	
	// Delete the object and cancel the watchdog.
	this.cleanup();
	
	// Update the cached console document in case it changed somehow.
	console_document = getConsoleDoc();
	
	if (reply_len > 0)
	{
		// Reset the empty response counter... although it is still theoretically
		// possible that we could get a valid poll message with an empty string
		// (which would be an implementation error on the server side).
		empty_response_cnt = 0;
		
		if (status_code != '0') 
		{
			err_code = status_code;
			handle_disconnect_event(new_text);
		}
	}
	else
	{
		// Empty response - used to mean that the socket had closed abnormally,
		// although now we should send a specific error message in that case.
		err_code = 999;
		empty_response_cnt++;
		
		// This is to detect infinite loops of connection attempts when the server
		// disconnects right away.
		if (empty_response_cnt > 10)
		{
			debug_msg("Comm error - " + empty_response_cnt + " consecutive empty poll RXs.");
			handle_disconnect_event(connection_lost);
			empty_response_cnt = 0;
		}
	}
	
	if (err_code)
	{
debug_msg("Comm error in handle_buffer_ready: " + err_code);
		return;
	}
	
	/* Reschedule RX timer here. Currently it is being queued via a timer so we don't
	 request any new data until we have finished displaying the old data. It might be
	 possible to reduce latency by sending the request immediately (and displaying
	 the old data while we are waiting), however I have experimentally confirmed that
	 this will choke under stress (it hangs the browser when I run "dia sniff packet"
	 on IE, although it ok on Mozilla...).

	 So anyway... if we wanted to avoid the timeout, we would have to be very careful
	 about implementing some kind of throttling algorithm to introduce some delays
	 when under stress. E.g. if the user is actively typing at the console, it may be
	 acceptable to call acon_rx_loop() directly, but if they are just monitoring the
	 output from the console, then we need to use the timeout.
	 */

	rx_since_last_tx++;
	if (rx_since_last_tx > 4)
	{
		if (rx_loop_timer)
		{
			clearTimeout(rx_loop_timer);
			rx_loop_timer = null;
		}
		
		rx_loop_timer = setTimeout("acon_rx_loop()", 1);
	}
	else
	{
		acon_rx_loop();
	}

	// Process the (non-empty buffer).
	if (new_text.length > 0)
	{
//		debug_msg("received message: " + new_text);
		
		// Append to rawBuf since there could be some partially complete multi-byte
		// sequences left over from the last round of parsing.
		rawBuf += unescape(new_text);

		try
		{
			process_rx_buffer();
		}
		catch (e)
		{
			debug_msg("Failed in process_rx_buffer: " + e);
			
			// If we get an exception here, we had better do something to cleanup the
			// parsing state. Otherwise rawBuf could grow without bound.
			rawBuf = "";
			
			// Try to display some kind of message on the console.
			try
			{
				var tbody = getConsoleTbody();
				finalize_row(tbody, inBuf);
				inBuf = "";
				append_complete_row(tbody, "(Connection error.)");
				append_partial_row(tbody, "");
			}
			catch (e)
			{
			}
		}
	}
}


// fixup_tab_response - when the user presses tab in the command window, we need to
// interpret the reply as best we can. I don't think anyone would disagree that this
// is a kludge. If the user is typing quickly while pressing tab, the results might
// be unpredictable. We could probably do something more reliable by hacking into 
// the CLI internals.
function fixup_tab_response()
{
debug_msg("fixup_tab_response, inbuf=" + inBuf);
	try
	{
		var cmd_elem = document.getElementById("jsconn_cmd");

		if (tab_proc_st.recent_typing)
		{
			// User was typing while we were waiting for the tab response.
			throw("user continued typing");
		}
		
		if (parse_st.added_rows)
		{
			throw("rows added during tab command");
		}
		
    	if (inBuf.length == 0)
        {
        	// This is an intermediate state caused by the Asian character octal
            // fixup. Don't abort the completion. We expect the current line to 
            // be resent in the next message.
            return;
        }
		
		// Strip the command-line prompt from inBuf.
		// There are two basic types of prompts, so find the first one.
		var prompt_offset1 = inBuf.indexOf("# ");
		var prompt_offset2 = inBuf.indexOf("$ ");

		// If either one isn't found, just set them both the same
		// (so min will work).
		if (prompt_offset1 == -1) prompt_offset1 = prompt_offset2;
		if (prompt_offset2 == -1) prompt_offset2 = prompt_offset1;

		var prompt_offset = Math.min(prompt_offset1, prompt_offset2);

		if (prompt_offset == -1)
		{
			throw("Unable to locate command prompt.");
		}

		var new_cmd = inBuf.substring(prompt_offset += 2);

		// Set the command box to contain the current command buffer.
		// (It's a bit of a leap of faith - we are basically relying on 
		// there not being extraneous characters in the output.)
		cmd_elem.value = strip_spacing_chars(new_cmd);
		tab_proc_st.stage++;
		
		// Tab completion results sometimes take more than one message to
		// complete (e.g. we might still receive some local echo character
		// before receiving the completion). But if we don't finish the
		// completion within 3 packets then give up.
		if (tab_proc_st.stage > 5)
		{
			tab_proc_st.in_proc = false;
		}
	}
	catch (e)
	{
		// This is usually not an error unless tab_proc_st.stage == 1. It is 
		// difficult for us to predict heuristically when the response is
		// complete, so we can't distingish completion from failure.
		debug_msg("couldn't fixup_tab_response: " + e);
		tab_proc_st.in_proc = false;
	}
}



// process_vt100_esc_sequence - handle an escape sequence
// The escape character '^' is not included in buf.
// This function directly manipulates the parser state.
// It returns the number of characters processed (or 0 if
// the buffer is incomplete, or -1 if we can't interpret
// the sequence).
function process_vt100_esc_sequence(buf)
{
debug_msg("begin vt100: ^" + buf);
	// Escape sequences often have a repeat count option. (Use 1 if it is omitted.)
	var repeat_cnt = 1;
	
	var j = 0;
	
	if (buf.length < 2)
	{
		// Incomplete buffer.
		return 0;
	}
	
	// In general, we support escape sequences of the form '^[<n><C>', where <n> is the
	// repeat count, and <C> is the command.
	ch = buf.charAt(j); 
	if (ch == 'O') // Esc-O
	{
		// This represents some kind of reverse font (i.e. white on black) that we currently
		// don't implement. (Can be generated with 'dia sys top').
		return j+1;
	}
	else if (ch != '[')
	{
		debug_msg("1c Unrecognized escape sequence header: ^" + rawBuf.charAt(j));
		return -1;
	}
	
	j++;

	// The first character may be either the command or the repeat count.
	ch = buf.charAt(j);
	
	// Check if this is a repeated command.
	if (ch >= '0' && ch <= '9')
	{
		// Parse repeated command.
		repeat_cnt = parseInt(buf.substr(j));

		if (isNaN(repeat_cnt))
		{
			debug_msg("cnt is NaN in esc sequence.");
			return -1;
		}

//debug_msg("cnt = " + cnt + ", substr = " + rawBuf.substr(j));

		if (repeat_cnt >= 10)
			j += 2;
		else
			j += 1;

		// Return 0 if the buffer is partially complete.
		if (j >= buf.length)
			return 0;

		// Now the next character should be the command.
		ch = buf.charAt(j);
	}
	
	j++;

	// Parse sequence with multiple integer parameters (';' is delimiter).
	if (ch == ';')
	{
		if (j >= buf.length)
			return 0;
		
		ch = buf.charAt(j);
		
		// Just ignore the 2nd argument.
		while (ch >= '0' && ch <= '9')
		{
			j++;
			if (j >= buf.length)
				return 0;
				
			ch = buf.charAt(j);
		}
	}
	
	// This is not a complete VT-100 terminal implementation. It only covers
	// the characters that the CLI is known to implement.
	switch (ch)
	{
	// Up/down arrow
	case 'A':
	case 'B':
		// Move cursor up? How (and why) are we supposed to do that?
		debug_msg("VT100 - don't know how to handle ^[" + ch + " (up/down arrow)");
		return -1;
	
	// right/left arrow
	case 'C': 
		cursor_x_pos += repeat_cnt;
		parse_st.moved_cursor = true;
		break;
	case 'D':
		cursor_x_pos -= repeat_cnt;
		parse_st.moved_cursor = true;
		break;
		
	case 'K':
		// Erase from cursor to EOL
		if (cursor_x_pos <= 0)
		{
			inBuf = "";
		}
		else
		{
			inBuf = inBuf.substr(0, cursor_x_pos);
		}
		break;
		
	case '@':
		// Insert N (empty) characters
		var str1 = inBuf.substr(0, cursor_x_pos);
		var x1;
		for (x1 = 0; x1<repeat_cnt; x1++)
		{
			str1 += ' ';
		}
		str1 += inBuf.substr(cursor_x_pos);
		inBuf = str1;
		break;
	case 'H':
		// set cursor home
		cursor_x_pos = 0;
		parse_st.moved_cursor = true;
		break;
	case 'J':
		// erase to end of screen (inclusive)
debug_msg("VT100 - don't handle ^[" + ch + " for now");
		break;
	case 'P':
		// erase one character backwards
		var str1 = inBuf.substr(0, cursor_x_pos);
		var str2 = inBuf.substr(cursor_x_pos + 1);
		inBuf = str1 + str2;
		break;
	case 'm':
		// This implements some kind of text colour feature that we don't currently
		// support. (Can be generated with "dia sys top". Also with the debug shell
		// on the Faz.)
		j++;
		break;
	default:
		debug_msg("1a Unrecognized escape sequence: ^" + buf.substring(0,j-1));
		return -1;
	}

	return j;
}


// Parsing state - used by process_rx_buffer and process_vt100_esc_sequence.
// Reset every time process_rx_buffer is called.
var parse_st = {
	added_rows : false,
	moved_cursor : false
};


// process_rx_buffer - main parsing function. Handle VT-100 char codes here.
function process_rx_buffer()
{
	// DEBUG: display rawBuf in the tmp buffer window.
	if (verbose_debug)
	{
		var outbuf = document.getElementById("txt_outbuf");
		if (outbuf) outbuf.value = rawBuf;
	}


//alert("length = " + rawBuf.length + ", rawBuf = " + rawBuf);


	var tbody = getConsoleTbody();
	if (!tbody)
	{
		debug_msg("Failed to get history tbody.");
		
		// We can't just return when there is an error, since then
		// rawbuf won't be cleaned up.
		throw("internal error 33k");
	}
	
	var ch;
	var i;
	var j = 0;
	
	// Reception of CRs triggers rescrolling & history pruning.
	parse_st.added_rows = false;
	
	// Moving the cursor should force it on immediately.
	parse_st.moved_cursor = false;
	
	
	// Ignore what we have already printed to the table. The global variables inBuf and
	// cursor_x_pos form the master copy of the working buffer for the active line.
	
	for (i=0; i<rawBuf.length; i++)
	{
		// Adjust for any multi-byte sequence in the last pass.
		if (j)
		{
			i += j;
			j = 0;
		
			if (i >= rawBuf.length)
				break;
		}
		
		
		// Begin processing the next character in the raw buffer.
		ch = rawBuf.charAt(i);
		
		if (ch == '\n')
		{
			finalize_row(tbody, inBuf);
			inBuf = "";
			
			// It would be a little more efficient to not add the new row until we
			// have some data (that's what I originally did), but it makes the code
			// a fair bit more complicated and it resulted in several bugs (e.g. in
			// displaying the cursor).
			
			append_partial_row(tbody, "");
			parse_st.added_rows = true;
			continue;
		}
		else if (ch == '\r')
		{
			// Not needed, I guess. Just let the '\n' handler do its work.
			continue;
		}
		else if (ch == '\x1b') // 0x1b = 27 = \033 = ESC
		{
			// Implement a VT100 virtual terminal.
			
			j = process_vt100_esc_sequence(rawBuf.substring(i+1));
			
			if (j > 0)
			{
				// Received complete ESC sequence. The variable 'i' will be adjusted
				// at the top of the loop.
				continue; 
			}
			
			if (j == 0)
			{
				// It is possible that the read operation would break in the middle
				// of the sequence. Handle the whole thing later.
				i--;
				break;
			}
			else
			{
				// If we can't interpret the esc sequence, just ignore
				// the escape and print out the characters on the
				// screen.
				ch = '^';
				j = 0;
			}
		}
		else if (ch == '\b') // backspace = 0x08
		{
			// VT-100 backspace doesn't actually erase the character. It just moves
			// one place to the left. Then it will send ^[K to clear the line and
			// echo back any new characters after that.
			cursor_x_pos--;
			if (cursor_x_pos < 0)
				cursor_x_pos = 0;
			
			continue;
		}
		else if (ch == '\x07') // bell = 0x07
		{
			// Happens when you backspace past the beginning of a line.
			// We could play a sound here, I suppose.
			continue;
		}
		else if (ch == mbc_octseq_marker)
		{
			// When we receive the special chr_spacing char, it indicates that the httpcli proxy
			// has compressed an 8 character octal sequence from the CLI into a 2 byte UTF-8
			// string + 1 bytes chr_spacing marker. Since JS will treat the 2 byte UTF-8 
			// string as a single character, we must then add 7 bytes of padding characters in
			// order to keep the cursor positions in sync. (These padding characters will be
			// stripped out when 
			var str1 = inBuf.substr(0, cursor_x_pos);
			var str2 = inBuf.substr(cursor_x_pos+7);
			inBuf = str1 + chr_spacing7 + str2;
			cursor_x_pos += 7;
			continue;
		}
		
		// Insert the new character at the cursor position.
		var str1 = inBuf.substr(0, cursor_x_pos);
		var str2 = inBuf.substr(cursor_x_pos+1);
		inBuf = str1 + ch + str2;
		cursor_x_pos++;
	} 

	if (i >= rawBuf.length)
	{
		rawBuf = "";
	}
	else
	{
		// save for next time (can this happen?)
		rawBuf = rawBuf.substr(i);
	}
    
	// Handle any response to Tab in the command window.
	if (tab_proc_st.in_proc)
	{
		fixup_tab_response();
	}
	
	// If we moved the cursor, make sure it is initially on.
	if (in_hist_wnd && parse_st.moved_cursor)
	{
		accel_cursor_on();
	}

	// Display remainder of unterminated buffer.
	overwrite_last_row(tbody, inBuf);


	// DEBUG: Update cursor pos display window
	if (verbose_debug)
	{
		var cp_elem = document.getElementById("jsconn_cursorpos");
		if (cp_elem) cp_elem.value = cursor_x_pos;
	}

	// If we added new rows, scroll the console window to the bottom, and 
	// also check if we exceeded the maximum console buffer length.
	if (parse_st.added_rows)
	{
		prune_histbuf();
		
		// Scroll the history window so the new line is in view.
		rescroll_history();
	}
}


/******************************
       Cursor functions
 ******************************/

var cursor_on = false;
var cursor_blink_timer = null;

// update_cursor_style - used by cursor_blink_loop, but included as a separate
// function so display_row can also set the initial state.
function update_cursor_style(celem)
{
	if (cursor_on)
	{
		celem.style.color = acon_prefs.acon_bgColor;
		celem.style.backgroundColor = acon_prefs.acon_fgColor;
	}
	else
	{
		celem.style.color = acon_prefs.acon_fgColor;
		celem.style.backgroundColor = acon_prefs.acon_bgColor;
	}
	
	celem.style.fontFamily = acon_prefs.acon_font_family;
	celem.style.fontSize = acon_prefs.acon_font_size;
}


// cursor_blink_loop - called all the time to blink the cursor, but it will
// only do so if the console window has the focus.
function cursor_blink_loop()
{
	try
	{
		// Do the switch whether we find the cursor or not (shouldn't matter?)
		// Only show the cursor when the history window has the focus.
		if (in_hist_wnd)
		{
			cursor_on = !cursor_on;
		}
		else
		{
			cursor_on = false;
		}
		
		var celem = getCursor(); 
		if (celem)
		{
			update_cursor_style(celem);
		}
	}
	catch (e)
	{
		// ignore any errors (but make sure not to abort before resetting the timer)
	}
	
	// Slightly longer duty cycle for cursor on than cursor off.
	if (cursor_on)
	{
		cursor_blink_timer = setTimeout("cursor_blink_loop()", 710);
	}
	else
	{
		cursor_blink_timer = setTimeout("cursor_blink_loop()", 605);
	}
}

// When using the left & right arrow keys, the results can seem sluggish due to the
// blinking cursor. Better to always turn the cursor on right after handing a left/right
// arrow event.

// accel_cursor_on - turn the cursor on more quickly
function accel_cursor_on()
{
	// Actually, display_row is always called by process_rx_buffer after calling this
	// function, so no need to schedule it later. Just turn the cursor on now, 
	// and delay the blinking slightly.
	cursor_on = true;
	
	// TODO: I'm not sure this works correctly.
	if (cursor_blink_timer) clearTimeout(cursor_blink_timer);
	cursor_blink_timer = setTimeout("cursor_blink_loop()", 1000);
}

