/******************************
 api_notify.js
 Init by J. Tary, Jan 2011
 FOS Auto Profiling Monitor
 Copyright Fortinet, inc.
 All rights reserved
 ******************************/

var Notify = (function($) {
    var default_container = "div#notify-container";
    var container = [];
    var notify_list;
    var timeouts = {
        "info" : 5000,
        "warn" : 0,
        "error" : 0
    };
    var django_severities = {
        10 : "info", // debug
        20 : "info",
        25 : "info", // success
        30 : "warn",
        40 : "error"
    };

    function get_django_message_cookie_data(cookie_key) {
        cookie_key = cookie_key || "messages";
        if (window.getCookie) {
            // see django.contrib.messages.storage.cookie.CookieStorage._decode
            var octal_escape_chars = /\\0([0-7]{2})/g,
                // Django sets some octal escape chars, but JSON doesn't allow them
                // messages_prep changes the octal escape chars to hex
                messages_prep = getCookie(cookie_key)
                    .replace(octal_escape_chars,
                        function(m, n){return "\\u"+
                            ("0000"+parseInt(n,8).toString(16)).slice(-4);}),
                messages = $.parseJSON(messages_prep);
            return $.parseJSON(messages.substr(messages.indexOf("$")+1));
        }
    }

    function get_django_messages() {
        if (window.hasCookie && hasCookie("messages")) {
            var message_obj = get_django_message_cookie_data();
            if ($.isArray(message_obj)) {
                for (var i=0;i<message_obj.length;i++) {
                    // see django.contrib.messages.constants for severity levels
                    post(message_obj[i][2], django_severities[message_obj[i][1]]);
                }
            }
            removeCookie("messages","/");
        }
    }

    function init(cont_selector) {
        if (cont_selector) {
            container = $(cont_selector);
        } else {
            container = $(default_container);
        }

        if (container.length < 1) {
            container = $("<div id=\"notify-container\"></div>").appendTo("body");
        } else {
            //make sure the div isn't inside of another element
            $(container).remove().appendTo("body");
        }

        notify_list = $("<ul class=\"notify-list\"></ul>").appendTo(container);
        update_container_display();
    }

    function create_notice(message, severity) {
        var notice = $("<li class=\"notify-item notify-" + severity + "\"></li>")
            .append("<span class=\"notify-label\">" + message + "</span>")
            .hide();
        var close = $("<a href=\"#\" class=\"tool_sprite tool_cancel notify-close\">close</a>")
            .click(function() { remove_notice(notice); return false; })
            .prependTo(notice);

        // automatically remove info notices after a few seconds
        if (timeouts[severity] > 0) {
            setTimeout(function() { remove_notice(notice); }, timeouts[severity]);
        }

        return notice;
    }

    function create_prompt(message, buttons) {
        var notice = $("<li class=\"notify-item notify-prompt\"></li>").hide();

        notice.append("<span class=\"notify-body\">" + message + "</span>");

        var btn_defaults = {
            "position": "right",
            "label": "MISSING_LABEL",
            "callback": $.noop
        };

        var btn_cont = {
            "left": $("<ul class=\"notify-button-list\" />")
                .addClass("notify-button-list")
                .addClass("notify-button-list-left")
                .appendTo(notice),

            "right": $("<ul class=\"notify-button-list\" />")
                .addClass("notify-button-list")
                .addClass("notify-button-list-right")
                .appendTo(notice)
        };

        if ($.isArray(buttons)) {
            for (var idx = 0; idx < buttons.length; ++idx) {
                var btn_info = $.extend({}, btn_defaults, buttons[idx]);

                var button = $("<input type=\"button\" class=\"button\" \>")
                    .val(btn_info["label"])
                    .click(btn_info["callback"]);

                button.appendTo(btn_cont[btn_info["position"]]).wrap("<li />");
            }
        }

        return notice;
    }

    function update_container_display(state) {
        if (typeof state == "undefined") {
            state = ($("li.layout-notify", notify_list).length > 0);
        }

        notify_list.css("display", state ? "" : "none");
    }

    function append_notice(notice) {
        notify_list.show();
        notify_list.prepend(notice); // newest notice on top
        notice.fadeIn("slow");
    }

    function remove_notice(notice) {
        $(notice).fadeOut("slow", function() {
            $(notice).remove();

            if (notify_list.find("li").length < 1) {
                notify_list.hide();
            }
        });
    }

    function post(message, severity) {
        if (container.length < 1) {
            init(default_container);
        }

        if (typeof(severity) == "undefined") {
            severity = "info";
        }

        var notice = create_notice(message, severity);
        append_notice(notice);
    }

    /*
     * Shows a message once using Notify.post() and sets a
     * cookie indicating that the message has been shown.
     * Subsequent calls to .once() will do notihing.
     *
     * arguments:
     * message - string containing message to be displayed
     * severity - string containing message severity
     * cookie_name - unique cookie name (or message name) that
     *               will be used to check if the message has
     *               already been displayed
     *
     * example usage:
     * Notify.once("Section view is currently disabled",
     *             "info",
     *             "section_view_disabled");
     */
    function once(message, severity, cookie_name) {
        if (getCookie(cookie_name)) {
            return;
        }

        post(message, severity);
        setCookie(cookie_name, "true");
    }

    function notify_confirm(message, cb_true, cb_false) {
        var ret = confirm(message);

        if (ret && cb_true) {
            cb_true(ret);
        } else if (!ret && cb_false) {
            cb_false(ret);
        }
    }

    /*
     * Shows a message without a timeout, which can include
     * one of more buttons defining actions the user can take
     *
     * arguments:
     * message - string containing message to be displayed
     * buttons - array of objects defining buttons to be added.
     *
     * example usage:
     * Notify.prompt("Do you wish to continue", [
     *     {"label": "ok", "callback": fn_ok},
     *     {"label": "cancel", "callback": fn_cancel}
     * ]);
     */
    function notify_prompt(message, buttons) {
        if (container.length < 1) {
            init(default_container);
        }

        var notice = create_prompt(message, buttons);
        append_notice(notice);
    }

    function clear() {
        $("li.notify-item", notify_list).remove();
        update_container_display();
    }

    return {
        "init" : init,
        "post" : post,
        "once" : once,
        "prompt" : notify_prompt,
        "confirm" : notify_confirm,
        "clear" : clear,
        "check_messages" : get_django_messages
    };
})(jQuery);
jQuery(function(){Notify.check_messages();});

