/******************************
 forti_chart.js
 Init by J. Tary, Nov 2010
 FOS Unified Monitors Project
 Copyright Fortinet, inc.
 All rights reserved
 ******************************/

var FortiChart = (function($) {
    var defaults = {
        "height" : 640,
        "width" : 480
    };

    // private prototype class
    function Chart() {
        this.callbacks = {};
        this.height = defaults.height;
        this.width = defaults.width;

        this.appendTo = appendTo;
        this.constructImage = constructImage;
        this.constructImageMap = constructImageMap;

        function appendTo(elem) {
            var img_html;
            var img_elem;
            var map_id = "fc_map_" + Math.floor(Math.random()*9999 + 1);
            var i;

            if (elem) {
                // To prevent scrollbar on elem, -1 (found on IE)
                this.height = $(elem).height();
                this.width = $(elem).width()-1;

                img_html = this.constructImage(map_id);

                if (typeof this.breadcrumbs == "object") {
                    if (typeof this.menu == "undefined") {
                        this.menu = [];
                    }

                    if (this.breadcrumbs.length > 1) {
                        var crumbs = this.breadcrumbs.clone();
                        crumbs.pop(); // drop the last crumb, since it's the current level

                        var last_crumb = crumbs[crumbs.length - 1];
                        this.menu.push({
                                "class" : "db_sprite tool_prev",
                                "title" : last_crumb.title,
                                "position" : "left",
                                "callback" : last_crumb.callback,
                                "context_menu" : crumbs
                        });
                    }
                }

                if (typeof this.menu == "object") {
                    if ($(elem).css("position") == "static") {
                        $(elem).css("position", "relative");
                    }

                    var menu = $(constructMenu(this.menu));
                    $(elem).append(menu);
                }

                chart_elem = $(img_html).appendTo(elem);

                if (typeof this.constructImageMap == "function") {
                    this.constructImageMap(elem, map_id);
                }
            }
        }

        function constructMenu(menu) {
            var item;
            var label;
            var menu_cont;
            var menu_list;
            var menu_timer;

            menu_cont = $("<div class=\"fc_menu_container\">&nbsp</div>");
            left_list = $("<ul class=\"fc_menu_list fc_menu_left\"></ul>").appendTo(menu_cont);
            right_list = $("<ul class=\"fc_menu_list fc_menu_right\"></ul>").appendTo(menu_cont);

            for (var i in menu) {
                item = menu[i];

                if (typeof(item) == "object") {
                    var menu_class = typeof item["class"] != "undefined" ? item["class"] : "";
                    var menu_item = $("<li class=\"fc_menu_item " + menu_class + "\" title=\"" + item.title + "\"><a href=\"#\"></a></li>")
                        .click(item.callback);

                    if (typeof item.context_menu == "object") {
                        menu_item.bind("contextmenu", show_context_menu(item.context_menu));
                    }

                    if (typeof item.position == "undefined" || item.position == "right") {
                        menu_item.appendTo(right_list);
                    } else {
                        menu_item.appendTo(left_list);
                    }
                }
            }

            $(menu_cont).mouseover(show_menu).mouseout(hide_menu);

            toggle_menu(false);

            return menu_cont;

            function toggle_menu(state) {
                $("*", menu_cont).toggleClass("fc_hidden", !state);

                if (!state) {
                    $("ul.fc_context_menu", menu_cont).hide();
                }
            }

            function show_menu() {
                toggle_menu(true);

                if (typeof(menu_timer) !== "undefined") {
                    clearTimeout(menu_timer);
                }
            }

            function hide_menu() {
                menu_timer = setTimeout(function() { toggle_menu(false); }, 500);
            }

            function show_context_menu(items) {
                var display_timer;
                var context_menu = $("<ul class=\"fc_context_menu\"></ul>").hide().appendTo(menu_cont);

                for (var i=0; i<items.length; i++) {
                    var item = items[i];
                    var click_cb = item.callback;
                    var label = item.label;
                    var title = item.title || "";

                    $("<li><a href=\"#\" title=\"" + title + "\">" + label + "</a></li>").click(click_cb).appendTo(context_menu);
                }

                function _show_context_menu() {
                    clearTimeout(display_timer);
                    $(context_menu)
                        .one("mouseleave", _hide_context_menu)
                        .show();
                }

                function _hide_context_menu() {
                    display_timer = setTimeout(function() { context_menu.hide(); }, 500);
                }

                return function() { _show_context_menu(); return false; };
            }
        }

        function constructImage(map_id) {
            var url = this.constructURL(this.height, this.width);

            return "<img src=\"" + url + "\" usemap=\"#" + map_id + "\" />";
        }

        function constructImageMap(elem, map_id) {
            if (typeof this.constructImageMapAreas != "function") {
                return;
            }

            var registerCallbacks = this.registerCallbacks;

            this.constructImageMapAreas(this.height, this.width, function( area_html ) {
                var map_html = "<map name=\"" + map_id + "\">" + area_html + "</map>";

                var map_elem = $(map_html).appendTo(elem);

                if (typeof registerCallbacks == "function") {
                    registerCallbacks(map_elem);
                }
            });

        }
    }

    function Bar(params) {
        var data = params.data || [];
        var x_labels = params.x_labels || [];
        this.breadcrumbs = params.breadcrumbs;
        this.menu = params.menu;

        this.constructURL = constructURL;
        this.constructImageMapAreas = constructImageMapAreas;
        this.registerCallbacks = registerCallbacks;

        function constructURL(height, width) {
            var args = [];
            var i;
            var x_label;
            var value;

            args.push("h=" + height);
            args.push("w=" + width);

            for (i=0; i<data.length; i++) {
                x_label = data[i].label;
                value = data[i].value;

                args.push("l=" + encodeURIComponent(x_label));
                args.push("v=" + value); 
            }

            if (params.style) {
                args.push("bt=" + params.style);
            }

            if (params.type) {
                args.push("t=" + params.type);
            }

            if (params.title) {
                args.push("ct=" + encodeURIComponent(params.title));
            }

            if (params.y_caption) {
                args.push("yc=" + encodeURIComponent(params.y_caption));
            }

            if (params.flags) {
                args.push("f=" + params.flags);
            }

            if (params.baseline) {
                args.push("bl=" + params.baseline);
            }

            return "/chart/bar?" + args.join("&");
        } 

        function constructImageMapAreas(height, width, callback) {
            var url = this.constructURL(height, width) + "&coords";

            $.ajax({
                "async" : true,
                "url" : url,
                "dataType" : "json",
                "success" : function(cdata) {
                    var i;
                    var cat;
                    var data_val;
                    var title;
                    var values = {};
                    var html = [];

                    // create a map of values to speed up for title generation
                    for (i = 0; i < data.length; i++) {
                        values[data[i].label] = data[i].value;
                    }

                    for (var i=0; i<cdata.length; i++) {
                        var coords = [ cdata[i].left, cdata[i].top, cdata[i].right, cdata[i].bottom ];

                        if (params.formatTitle) {
                            title = params.formatTitle(cdata[i]["x-label"], values[cdata[i]["x-label"]]);
                        } else {
                            title = cdata[i]["x-label"] + ": " + values[cdata[i]["x-label"]];
                        }

                        html.push( "<area title=\"" + title + "\" coords=\"" + coords.join(",") + "\" x-label=\"" + cdata[i]["x-label"] + "\" data=\"" + data_val + "\" />" );
                    }

                    callback( html.join("") );
                }
            });
        }

        function registerCallbacks(elem) {
            if (params.onclick) {
                $("area", elem).bind("click", function(evt) {
                    var x_label = $(evt.target).attr("x-label");
                    var data_val = $(evt.target).attr("data");

                    params.onclick(x_label, data_val);

                    return false;
                });

                $("area", elem).attr("href", "#");
            }
        }
    }

    function Stacked(params) {
        var data = params.data || [];

        this.breadcrumbs = params.breadcrumbs;
        this.menu = params.menu;

        this.constructURL = constructURL;
        this.constructImageMapAreas = constructImageMapAreas;
        this.registerCallbacks = registerCallbacks;

        function constructURL(height, width) {
            var args = [];
            var i,j;
            var x_label;
            var y_label;
            var value;

            args.push("h=" + height);
            args.push("w=" + width);

            for (i=0; i<data.length; i++) {
                x_label = data[i].label;

                for (j=0; j<data[i].value.length; j++) {
                    y_label = data[i].value[j].label;
                    value = data[i].value[j].value;

                    if (j == 0) {
                        args.push("l=" + encodeURIComponent(x_label));
                    }

                    if (i == 0) {
                        args.push("s=" + encodeURIComponent(y_label));
                    }

                    args.push("v=" + value); 
                }
            }

            if (params.style) {
                args.push("bt=" + params.style);
            }

            if (params.type) {
                args.push("t=" + params.type);
            }

            if (params.title) {
                args.push("ct=" + encodeURIComponent(params.title));
            }

            if (params.y_caption) {
                args.push("yc=" + encodeURIComponent(params.y_caption));
            }

            if (params.flags) {
                args.push("f=" + params.flags);
            }

            return "/chart/stacked?" + args.join("&");
        } 

        function constructImageMapAreas(height, width, callback) {
            var url = this.constructURL(height, width) + "&coords";

            $.ajax({
                "async" : true,
                "url" : url,
                "dataType" : "json",
                "success" : function(cdata) {
                    var day;
                    var cat;
                    var data_val;
                    var title;
                    var values = {};
                    var html = [];

                    // create a map of values to speed up for title generation
                    for (day = 0; day < data.length; day++) {
                        values[data[day].label] = {};

                        for (cat = 0; cat < data[day].value.length; cat++) {
                            values[data[day].label][data[day].value[cat].label] = data[day].value[cat].value;
                        }
                    }

                    for (var i=0; i<cdata.length; i++) {
                        var coords = [ cdata[i].left, cdata[i].top, cdata[i].right, cdata[i].bottom ];

                        if (params.formatTitle) { 
                            title = params.formatTitle(cdata[i]["x-label"], cdata[i]["y-label"], values[cdata[i]["x-label"]][cdata[i]["y-label"]]);
                        } else {
                            title = cdata[i]["x-label"] + " " + cdata[i]["y-label"] + ": " + values[cdata[i]["x-label"]][cdata[i]["y-label"]];
                        }

                        data_val = values[cdata[i]["x-label"]][cdata[i]["y-label"]];

                        html.push( "<area title=\"" + title + "\" coords=\"" + coords.join(",") + "\" x-label=\"" + cdata[i]["x-label"] + "\" y-label=\"" + cdata[i]["y-label"] + "\" data=\"" + data_val + "\" />" );
                    }

                    callback( html.join("") );
                }
            });
        }

        function registerCallbacks(elem) {
            if (params.onclick) {
                $("area", elem).bind("click", function(evt) {
                    var x_label = $(evt.target).attr("x-label");
                    var y_label = $(evt.target).attr("y-label");
                    var data_val = $(evt.target).attr("data");

                    params.onclick(x_label, y_label, data_val);

                    return false;
                });

                $("area", elem).attr("href", "#");
            }
        }
    }

    function Pie(params) {
        var data = params.data || [];

        this.breadcrumbs = params.breadcrumbs;
        this.menu = params.menu;

        this.constructURL = constructURL;
        this.constructImageMapAreas = constructImageMapAreas;
        this.registerCallbacks = registerCallbacks;

        function constructURL(height, width) {
            var args = [];
            var i;
            var x_label;
            var y_label;
            var value;
            var t_val;

            t_val = 0;

            for (i=0; i<data.length; i++) {
                x_label = data[i].label;
                value = data[i].value;
                t_val += data[i].value;

                args.push("l=" + encodeURIComponent(x_label));
                args.push("v=" + value); 
            }

            // if the total values equal zero, empty the chart so that we'll see "no data"
            if (t_val == 0) {
                args = [];
            }

            args.push("h=" + height);
            args.push("w=" + width);

            if (params.title) {
                args.push("ct=" + encodeURIComponent(params.title));
            }

            if (params.y_caption) {
                args.push("yc=" + encodeURIComponent(params.y_caption));
            }

            if (params.flags) {
                args.push("f=" + params.flags);
            }

            return "/chart/pie?" + args.join("&");
        } 

        function constructImageMapAreas(height, width, callback) {
            var url = this.constructURL(height, width) + "&coords";

            $.ajax({
                "async" : true,
                "url" : url,
                "dataType" : "json",
                "success" : function(cdata) {
                    var coords = [];
                    var title;
                    var values = {};
                    var i;
                    var label;
                    var data;
                    var html = [];

                    // create a map of values to speed up for title generation
                    for (i=0; i<params.data.length; i++) {
                        values[params.data[i].label] = params.data[i].value;
                    }

                    for (i=0; i<cdata.length; i++) {
                        label = cdata[i]["name"];
                        data = values[label];
                        title = label + ": " + data;

                        for (var j=0; j<cdata[i].points.length; j++) {
                            coords.push(cdata[i].points[j].x); 
                            coords.push(cdata[i].points[j].y); 
                        }

                        html.push( "<area label=\"" + label + "\" data=\"" + data + "\" title=\"" + title + "\" coords=\"" + coords.join(",") + "\" shape=\"poly\" />" );
                    }

                    callback( html.join("") );
                }
            });
        }

        function registerCallbacks(elem) {
            if (params.onclick) {
                $("area", elem).bind("click", function(evt) {
                    var label = $(evt.target).attr("label");
                    var data_val = $(evt.target).attr("data");

                    params.onclick(label, data_val);

                    return false;
                });

                $("area", elem).attr("href", "#");
            }
        }
    }

    function Table(params)
    {
        var columns = params.columns || [];
        var fields = params.fields || [];
        var data = params.data || [];
        var formats = params.formats || [];

        // table
        var tbl = $j("<table class=list cellspacing=1 cellpadding=0 width=100%>");

        var row = $j("<tr class=heading></tr>");
        for (var i = 0; i < columns.length; i++)
            row.append("<th>"+($j.getInfo(columns[i]) || "")+"</th>");
        tbl.append(row);

        var r, fldlen = fields.length || data.length;
        for (var i = 0; (r = data[i]); i++)
        {
            row = $j("<tr"+(i%2 ? " class=odd" : "")+"></tr>");
            for (var j = 0; j < fldlen; j++)
            {
                var v = r[fields[j] || j];
                if (formats[j]) v = formats[j](v, r);
                $j("<td>" + (typeof(v)!="undefined" ? v : "") + "</td>").appendTo(row);
            }
            row.appendTo(tbl);
        }
        return tbl;
    }

    function Line(params) {
        var segments = params.segments || [];
        var x_labels = params.x_labels || [];

        this.breadcrumbs = params.breadcrumbs;
        this.menu = params.menu;

        this.constructURL = constructURL;

        function constructURL(height, width) {
            var args = [];
            var i;
            var value;

            args.push("h=" + height);
            args.push("w=" + width);

            for (i=0; i<segments.length; i++) {
                args.push("s=" + segments[i].name);
                args.push("v=" + segments[i].values.join(","));
            }

            for (i=0; i<x_labels.length; i++) {
                args.push("l=" + x_labels[i]);
            }

            if (params.title) {
                args.push("ct=" + encodeURIComponent(params.title));
            }

            if (params.y_caption) {
                args.push("yc=" + encodeURIComponent(params.y_caption));
            }

            if (params.flags) {
                args.push("f=" + params.flags);
            }

            if (params.baseline) {
                args.push("bl=" + params.baseline);
            }

            if (params.type) {
                args.push("t=" + params.type);
            }

            if (params.hide_no_data) {
                args.push("hnd=true");
            }

            return "/chart/line?" + args.join("&");
        }
    }

    Bar.prototype = new Chart();
    Stacked.prototype = new Chart();
    Pie.prototype = new Chart();
    Line.prototype = new Chart();    

    return {
        "Bar" : Bar,
        "Stacked" : Stacked,
        "Pie" : Pie,
        "Table": Table,
        "Line" : Line
    };
})(jQuery);
