/******************************
 jquery.multilist.js
 Init by J. Tary, March 2011

 JQuery plugin for handling lists
 of values.

 Copyright Fortinet, inc.
 All rights reserved
 ******************************/
/*global escapeHTML, prompt*/
(function($) {
    /* jquery multi-list multiple selection support
       we need to override some ui function in order to get
       ui.autocomplete work with multiple selection.
     */
    var namespaced = fweb.util.namespacifyEvents('fortinetMulticomplete');
    var namespaced_click = namespaced('click');
    var _unique_id = 0;
    // NOTE: if events are bound outside of frames, we'll need to use an counter
    //       bound to window.top or just use a sufficiently large random number
    // alternatively, use _.uniqueId()
    var get_unique_id = function() {
        return _unique_id++;
    };
    /* multicomplete is tightly coupled to multilist, and standalone use is not recommended  */
    $.widget("fortinet.multicomplete", $.ui.autocomplete, {
        options: {
            'multiselect': true
        },
        _create: function() {
            $.ui.autocomplete.prototype._create.call(this);

            // unbind blur and override it later
            // TODO: now that showing/hiding the multiList no longer relies on
            //       listening to the blur event, is this still needed?
            this._off(this.element, 'blur');

            // override default menu click handler
            this.menu._off(this.menu.element, 'click');
            this._on(this.menu.element, {
                'dbclick .ui-menu-item a': function(event) {
                    // NOTE: possible regression: this handler seems to never get called
                    var target = this._get_ctrl(event);
                    if (!target.length) {
                        return;
                    }
                    return this._menu_select(event, target);
                },
                'click .ui-menu-item a': '_on_multicomplete_click'
            });
            // make sure that old dom references aren't kept.
            this._on({
                // NOTE: this currently doesn't trigger; multiList needs to call close on me
                'autocompleteclose': function() {
                    this.last_selected_el = $();
                },
                // 'autocompleterendercomplete': function() {
                'autocompleteopen': function() {
                    // TODO: the `open` event doesn't fire reliably
                    this.last_selected_el = $();
                    this._refreshMultilistOkButton();
                }
            });
            // track the most recent selection for shift clicking.
            this.last_selected_el = $();
        },
        _get_ctrl: function(event) {
            return $(event.target).closest(".ui-menu-item a");
        },
        _menu_select: function(event, target) {
            this.menu.focus(event, target.parent('.ui-menu-item'));
            this.menu.select(event);
            return false;
        },
        _on_multicomplete_click: function(event) {
            var target = this._get_ctrl(event);
            if (!target.length || target.parent().is('.disabled')) {
                return;
            }

            if (!this.options.multiselect) {
                return this._menu_select(event, target);
            }

            var last = this.last_selected_el;
            last = last.not(target);

            // Here is the click behaviour:
            // - single click to choose immediately
            // - ctrl/shift click to add multiple
            // TODO: although usable, this doesn't follow standard ctrl+shift behaviour
            if (!event.shiftKey && !event.ctrlKey) {
                last.removeClass("selected");
                last = $();
                target.addClass("selected");
                return this._menu_select(event, target);
            } else if (event.ctrlKey) {
                target.toggleClass("selected");
            } else {
                var start = $(last[last.length - 1]);
                if (start.length === 0) {
                    // this is the first shift click, count it like the
                    // first ctrl click
                    target.toggleClass("selected");
                } else {
                    if (start == target) {
                        return;
                    }
                    var items = [start.closest("li"), target.closest("li")]
                        .sort(function(a, b) {
                            var a_ctrl = a.data("ui-autocomplete-item");
                            var b_ctrl = b.data("ui-autocomplete-item");
                            if (a_ctrl && b_ctrl) {
                                return a_ctrl.index - b_ctrl.index;
                            }
                            return true;
                        });
                    // select all of the elements between the previously
                    //  clicked item and the currently clicked item
                    items[0].nextUntil(items[1], ".ui-menu-item").each(function() {
                        var elem = $("a:first", this).addClass("selected");
                        last = last.add(elem);
                    });
                    target.addClass("selected");
                }
            }

            if (target.hasClass("selected")) {
                last = last.add(target);
            }

            this.last_selected_el = last;
            this._refreshMultilistOkButton();
        },
        _refreshMultilistOkButton: function() {
            // traverse control for find OK button
            var $inputContainer = this.menu.element.parents('.multiList-input-container');
            if ($inputContainer.length <= 0) { $inputContainer = this.element.parents('.multiList-input-container'); }
            $inputContainer.find('.multiList-add')
                // TODO: .multiList-add should have display:inline-block in
                //       css, and jQuery#toggle(bool) should be called
                .css('display', this.last_selected_el.length > 0 ? 'inline-block' : 'none');
        },
        _renderItem: function(ul, item) { // override
            // make html/dom supported for appending as label
            var li = document.createElement("li");
            var a = document.createElement("a");
            li.appendChild(a);
            a.innerHTML = item.label;
            $(li).data("ui-autocomplete-item", item)
                .addClass("ui-menu-item")
                .toggleClass("disabled", !!item.disabled)
                .attr("role", "menuitem");
            ul[0].appendChild(li);
        },
        _normalize: function() {
            var normalized = $.ui.autocomplete.prototype._normalize.apply(this, arguments);
            $.each(normalized, function(index, item) { item.index = index; });
            return normalized;
        },
        stopRender: function() {
            if (this.renderMenuLoop != null) { this.renderMenuLoop.interrupt(); }
        },
        // TODO: make sure that dom is detached when doing this render
        _renderMenuByCategories: function(ul, items) {
            var that = this, category;

            var $no = ul.siblings('.no-options').addClass('ui-autocomplete-loading');
            var no_oldtext = $no.text();
            $no.text($.getInfo('loading'));

            var reattach = fweb.util.$detached(ul);

            this.renderMenuLoop = fweb.util.asyncEach(items, function(index, item) {
                // keep track of the current category; generate a new category header
                // if it changes (see module_js/firewall/policy.js)
                if (item.category != category) {
                    category = item.category;
                    if (category) {
                        var li = '<li class="ui-autocomplete-category">' + category + '</li>';
                        ul.append(li);
                    }
                }
                that._renderItem(ul, item);
            }).done(function() {
                reattach();
                $no.removeClass('ui-autocomplete-loading');
                $no.text(no_oldtext);
                that.renderMenuLoop = null;
                that._trigger('rendercomplete');
            });

        },
        _renderMenu: function(ul, items) { // override
            this._renderMenuByCategories(ul, items);
        },
        complete: function(event) {
            var vals = [];
            $.each($(".selected", this.menu.element), function() {
                var item = $(this).closest("li").data("ui-autocomplete-item");
                vals.push(item.value);
            });
            this._trigger( "select", event, { item: { "value": vals } } );
            this.close(event);
        }
    });

    function setOptionsFromSelect(config) {
        var opts = {}, src = [], opt, i, val;

        for (i = 0; (opt = this.options[i]); i++) {
            val = opt.value;
            opts[val] = opt;
            src.push(val);
        }

        config.source = src;
        config.option = opts;

        return config;
    }

    function getStaticSourceFunction(get_items, config, input) {
        return function(request, response) {
            var term = request.term.toLowerCase();
            var items = get_items();

            response($.map(config.source, function(v, i) {
                // hide from selection
                // if it's empty or already selected
                var str = v.toString() || '';
                var in_arr = $.inArray(str, items) !== -1;
                if (!str ||
                    (in_arr && config.selector_type === 'multi')) {
                    return null;
                }

                var disp = config.display.call(input[0], v, {});

                if (typeof(disp) == "string") {
                    disp = {label: disp};
                }

                var elmt = $.extend({index: i, value: v}, disp);

                // hide those no label
                if (!elmt.label) {
                    return null;
                }

                if (config.selector_type === 'single' && in_arr) {
                    elmt.label = '<span class="selected">' +
                        elmt.label + '</span>';
                }
                if (!term) {
                    return elmt;
                }

                function _search_contains() {
                    return elmt.value.toLowerCase().indexOf(term) >= 0 ||
                        $("<div>" + elmt.label + "</div>").text().toLowerCase().indexOf(term) >= 0;
                }
                function _search_start_with() {
                    return elmt.value.toLowerCase().indexOf(term) === 0 ||
                        $("<div>" + elmt.label + "</div>").text().toLowerCase().indexOf(term) === 0;
                }
                var found = true;
                if (config.search_method === 'auto') {
                    found = term.length === 1 ? _search_start_with() : _search_contains();
                } else if (config.search_method === 'contains') {
                    found = _search_contains();
                } else if (config.search_method === 'start') {
                    found = _search_start_with();
                }


                return found ? elmt : null;
            }));
        };
    }

    // TODO: refactor to use the jquery widget factory, like the above multicomplete.
    $.fn.multiList = function(params) {
        // handle commands
        if (typeof params === 'string') {
            if (params === 'refreshOptions') {
                $(this).each(function() {
                    var config = this.config,
                        input = $(this),
                        get_items;
                    // retrieves the items that are currently selected
                    if (this.tagName == "SELECT") {
                        if (config == null) {
                            $.fn.multiList.error(
                                'cannot call refreshOptions on multiList prior to initialization');
                        }
                        if (input.prop('multiple')) {
                            // assume that the value is an array
                            get_items = function() { return input.val(); };
                        } else {
                            get_items = function() { return [input.val()]; };
                        }
                        config = setOptionsFromSelect.call(this, config);
                        $(this).closest('.multiList-container').find('.multiList-input')
                            .multicomplete('option', 'source',
                                getStaticSourceFunction(get_items, config, input));
                    }
                });
            }
        }

        // add multilist style sheet
        //$.addStyle("/" + fweb.CONFIG_GUI_NO + "/css/jquery.multilist.css");
        //$.addStyle("/" + fweb.CONFIG_GUI_NO + "/css/jquery.ui.css");
        $.addStyle("/css/jquery.multilist.css");
        $.addStyle("/css/jquery.ui.css");

        var defaults = {
            'cls': '.multiList-container',
            'class': "",  // extra class assign to multiList root
            'source': [], // url or array
            'max': 0, // max number of selection, 1 means single selection
            'min': 0, // minimal number of selection, 1 means can not be empty
            'readonly': false,  // display only, no change allow
            // undefined/'multi'/'single' - the type of the selector
            'selector_type': undefined,
            // set to false if you want the qlist to take up space when expanded
            'floating': true,
            // 'contains': use 'contains' method to search string
            // 'start': use 'start with' method to search string
            // 'auto': search for one letter, uses 'start with' method;
            //         otherwise, use 'contains' method.
            'search_method': 'auto',
            // callback on how to display data item
            // this == dom node that the multilist was applied to
            // v == value ??
            'display': function(v) {
                return v && v.toString() ||
                    (this.config.selector_type === 'single' &&
                     this.config.lang['select']) ||
                    this.config.lang['default'];
            },
            // Allow creation of new entries (in an inline fashion unless callback
            // is provided)
            'create': {
                'enable': false,
                'callback': null,
                // icon class for the Create button
                'class': "multiList-new",
                // label for the Create button
                'label': $.getInfo('create')
            },
            // Enable inline editing mode. Removes the list selection and allows
            // for only user defined values which can be added & edited in an
            // inline fashion.
            'inline_edit': false,
            // "afterInsert": callback when new entry append
            // "validate": data input validator
            'separator': ",",
            'lang': {
                'heading': $.getInfo('select'),
                'default': $.getInfo('click_add'),
                'select': $.getInfo('click_set'),
                'no_options': $.getInfo('no_options')
            },
            'tooltips': {
                'new': $.getInfo('create'),
                'add': $.getInfo('ok'),
                'remove': $.getInfo('remove')
            }
            // 'width': will use input/select width if not defined
            // 'max_height': the mex height of the multilist selector
        };

        return $(this).each(function() {
            // flag used to prevent firing of the click handler as the event bubbles
            var _mouseHandled = false;
            // Are we currently editing an inline item?
            var _editing_inline = false;
            var input = $(this);

            var config = $.extend(true, {}, defaults, this.config, {
                "readonly": input.prop("readonly"),
                "disabled": input.prop("disabled"),
                "disabled_dropdown": false,
                "width": this.style.width || undefined,
                "uid": 'multilist' + get_unique_id()
            });

            // preset config for html <select>
            if (this.tagName == "SELECT") {
                if (!this.config) {
                    $.extend(config, {
                        'selector_type': input.attr("multiple") ?
                            'multi' : 'single'
                    });
                }
                config = setOptionsFromSelect.call(this, config);
                if (!this.config || !this.config.display) {
                    config.display = function(v) {
                        // use case: <option value=''>No Certificate</option>
                        // then the following will return 'No Certificate'
                        // Later in refreshList(), display() may be called
                        // with '' value when get_items returns []
                        return this.config.option[v] && this.config.option[v].text || v ||
                            config.lang['select'] ||
                            config.lang['default'];
                   };
                }
            }

            config = $.extend(true, {}, config, params);
            // Overwrite old source
            if (params && params.source) {
                config.source = params.source;
            }

            if (!config.selector_type) {
                config.selector_type = config.max === 1 ? 'single' : 'multi';
            } else if (config.selector_type === 'single') {
                config.max = 1;
            }

            // update input control according new config
            input.prop("readonly", config.readonly);
            input.prop("disabled", config.disabled);

            // save config associated
            // TODO: !!! store as jquery.data rather than directly on the dom element.
            //       there may be circular references here that require extra work
            //       for the garbage collector
            this.config = config;

            // nothing more need to do with type="hidden"
            if (input.attr("type") == "hidden") {
                return;
            }

            var wrap = input.closest(config.cls);

            if (wrap.length > 0) {
                // remove old wrap before re-generate
                wrap.before(input);
                wrap.remove();
            }

            var container_wrap = config.validate? $('<form />') : $('<div />');
            var container_class = config.cls.substr(1);
            // list selected item usually have list icon, thus around 16px extra width
            var width_diff = this.tagName == "SELECT" ? 22 : 18;
            container_wrap.addClass(container_class).width(
                config.width || (input.outerWidth() + width_diff));
            var container = input.wrap(container_wrap).closest(config.cls);

            var $list = $('<ul class="multiList-list"></ul>');
            $list.addClass(config["class"]);
            // use prepend just because form.validate
            // putting error label after input element
            $list.prependTo(container);

            var $inputContainer = $('<li class="multiList-input-container" />')
                .hide()
                .appendTo($list);

            if (config.floating) {
                $inputContainer.addClass("multiList-input-container-float");
            }

            var head_div = $('<div class="multiList-heading" />')
                .append('<span>' + config.lang['heading'] + '</span>')
                .append('<a class="multiList-btn multiList-close" />')
                .appendTo($inputContainer);

            var opts_html = '<div class="multiList-options" onselectstart="return false;" />';
            var $opts_div = $(opts_html)
                .append('<div class="no-options">' + config.lang['no_options'] + '</div>')
                .appendTo($inputContainer);
            var $tool_div = $('<div class="multiList-footer" />')
                .appendTo($inputContainer);

            var $input_mc = $('<input type="text" ' +
                          'class="multiList-input tool_sprite tool_search2 search_box" />')
                .appendTo($tool_div);

            $("a.multiList-close", container)
                .off(namespaced_click)
                .on(namespaced_click, closeList);

            function create_item(event) {
                var num_items = get_items().length;
                var create_callback = config.create.callback;
                var value = '';
                var updated_value;
                // Use event to determine if this was called from inside multicomplete
                // (can simply pass truthy argument to mimic that behaviour)
                if (event) {
                    _mouseHandled = true;
                    value = $input_mc.val() || '';
                }
                if (typeof create_callback == 'function') {
                    updated_value = create_callback.call(input[0], value);
                    if(updated_value) {
                        $input_mc.val(updated_value);
                        addItem();
                    }
                    closeList();
                } else {
                    closeList();
                    if (config.max === 1 || num_items === 0) {
                        editInline.call($list.find('.multiList-list-item').first(), value);
                    } else if (config.max === num_items) {
                        editInline.call($list.find('.multiList-list-item').last(), value);
                    } else {
                        addInline(value);
                    }
                }
            }

            // TODO: !!! namespace & off/on click handlers below -
            //       explicit_nat_dialog.js:enatdlg_ready will have double-binding problems.
            var button_div = $('<div class="right_btn" />')
                .appendTo($tool_div);
            if (config.create.enable) {
                var _cls = config.create["class"],
                    _label = config.create.label;
                $('<a class="multiList-btn ' + _cls + '" />')
                    .text(_label)
                    .attr('title', _label)
                    .appendTo(button_div)
                    .click(create_item);
            }
            // TODO: rename these css classes appropriately -- add -> ok, show -> add
            // this is the OK button that will appear when ctrl/shift clicking is used
            $('<a class="multiList-btn multiList-add" />')
                .text(config.tooltips["add"])
                .attr('title', config.tooltips["add"])
                .appendTo(button_div)
                .click(function() {
                    _mouseHandled = true;
                    $input_mc.multicomplete('complete');
                })
                .css('display', 'none');

            var add_button = $('<a class="multiList-btn multiList-show" />')
                .attr('title', config.tooltips["add"])
                .click(function() {
                    _mouseHandled = true;
                    if (config.inline_edit) {
                        addInline();
                        return;
                    }
                    $(document).off(namespaced('click', config.uid))
                    .on(namespaced('click', config.uid), documentClickHandler);
                    // show the input box / options list
                    $input_mc.val("");
                    $inputContainer.css({
                        "min-width": $list.width() - 4
                    }).show();
                    // TODO: don't always call search - should require a larger cleanup
                    $input_mc.multicomplete('search');
                    // NOTE: calling focus() _before_ calling multicomplete('search')
                    //       is _much_ slower. The reason for this is currently unknown.
                    $input_mc.focus();
                    var $optlist = $opts_div.find(".ui-autocomplete");
                    if ($optlist.length > 0 && config.max_height) {
                        $optlist.css('max-height', config.max_height);
                    }
                    if ($optlist.length <= 0 ||
                        $optlist[0].scrollHeight <= $optlist[0].clientHeight) {
                        // no scrollbar, search should hide
                        $input_mc.css({
                            "position": "absolute",
                            "margin-left": "-1000000px"
                        });
                        // if ($input_mc.multicomplete('instance').renderMenuLoop != null) { // jQuery UI >=1.11.0
                        if ($input_mc.data('fortinetMulticomplete').renderMenuLoop != null) {
                            $input_mc.one(namespaced('multicompleterendercomplete'), function() {
                                var optlist = $opts_div.find(".ui-autocomplete")[0];
                                if (optlist && optlist.scrollHeight > optlist.clientHeight) {
                                    $input_mc.css({
                                        "position": "static",
                                        "margin-left": "0"
                                    });
                                }
                            });
                        }
                    }

                    if (!input.val()) {
                        // clear hint
                        $(".multiList-list-item", $list).text("");
                    }
                });

            if (!config.readonly) {
                $list.before(add_button);
            }

            if ($.isFunction($.fn.validate) && config.validate) {
                container.validate();
                $input_mc.rules("add", config.validate);
            }

            $inputContainer.on('mouseout', $.debounce(50, function() {
                // IE: make input re-focus for blur to close
                if (!$input_mc.is(":focus") && $inputContainer.is(":visible")) {
                    $input_mc.focus();
                }
            }));

            $input_mc.keypress(function(evt) {
                if (evt.keyCode == 13 && config.create.enable) {
                    create_item(true);
                }
            }).keyup(function(evt) {
                if (evt.keyCode == 27) {
                    if (!$input_mc.val()) {
                        closeList();
                    } else {
                        $input_mc.val('').multicomplete('search');
                    }
                }
            }).blur(function() {
                // close drop-down list when no longer in focus
                var loss_focus = $(document.activeElement).closest(config.cls)[0] != container[0];
                if (loss_focus && $inputContainer.is(":visible")) {
                    setTimeout(closeList, 150);
                }
            });

            // based on a similar handler in jquery.ui.menu.js:_create
            // Clicks outside of the list closes the list
            var documentClickHandler = function(event) {
                if (!_mouseHandled && $inputContainer.is(":visible") &&
                        !$(event.target).closest(".multiList-input-container").length) {
                    closeList();
                }
                // reset the mouseHandled flag
                _mouseHandled = false;
            };

            // retrieves the items that are currently selected
            function get_items() {
                var val = input.val();
                var items = [];
                if (val) {
                    if ($.isArray(val)) {
                        items = val;
                    } else {
                        items = val.split(config.separator);
                    }
                }
                return items;
            }

            function set_items(items) {
                var val = input.val();
                var result = items;
                if (!$.isArray(val)) {
                    result = items.join(config.separator);
                }
                input.val(result);
                // ngModel listens for "change" event, let trigger that
                input.change();
            }

            if (config.source) {
                var source = $.noop();
                if ($.isFunction(config.source)) {
                    source = config.source;
                } else {
                    source = getStaticSourceFunction(get_items, config, input);
                }
                // use multicomplete control
                $input_mc.multicomplete({
                    source: source,
                    multiselect: config.selector_type === 'multi',
                    appendTo: $opts_div,
                    select: function(e, ui) {
                        var item_val = ui.item.value,
                            is_array = $.isArray(item_val);
                        $input_mc.val(is_array ? item_val.join(config.separator) : item_val);
                        addItem();
                        closeList();
                    },
                    close: function(evt) {
                        // ignore close event if still focus
                        if ($(this).is(":focus")) {
                            return;
                        }

                        closeList();

                        if (config.dialog) {
                            config.dialog.dialog("option", "position", config.dialog.dialog("option", "position"));
                        }
                    },
                    open: function(event, ui) {
                        if (config.dialog) {
                            config.dialog.dialog("option", "position", config.dialog.dialog("option", "position"));
                        }
                    },
                    minLength: 0
                });
            }

            input.hide();
            // hook on focus to show red border
            this.focus = function() {
                $list.addClass("focus");
            };

            refreshList();

            function addItem() {
                $.each($input_mc.val().split(config.separator), function() {
                    var itemText = this;
                    var items = get_items();

                    items.push(itemText);
                    if (config.max && items.length > config.max) {
                        items.shift();
                    }

                    set_items(items);

                    if ($.isFunction(config.afterInsert)) {
                        config.afterInsert.call(input[0], itemText);
                    }
                });

                $input_mc.val("");
                refreshList();
            }

            function addInline(value) {
                var $li = $('<li class="multiList-inline-item"></li>');
                var $input = $('<input type="text" />').val(value || '');
                $list.append($li.append($input));
                _editing_inline = true;

                $input.select().on('focusout keypress', function(event) {
                    var items;
                    var value = $input.val();
                    if (event.type === 'focusout' || event.keyCode === $.ui.keyCode.ENTER) {
                        $input.off();
                        if (value) {
                            items = get_items();
                            items.push(value);
                            set_items(items);
                            refreshList();
                        }
                        $li.remove();
                        _editing_inline = false;
                    }
                });
            }

            function editInline(value) {
                var items = get_items();
                var num_items = items.length;
                var $entry = $(this).closest('li');
                var index = $entry.data('index');
                var currentVal = items[index];
                var $li = $('<li class="multiList-inline-item"></li>');
                var $input = $('<input type="text" />');
                if (num_items) {
                    $input.val(value || items[index]);
                } else {
                    $input.val(value || '');
                }
                $li.append($input);
                $entry.after($li).detach();
                _editing_inline = true;

                $input.select().on('focusout keypress', function(event) {
                    var value = $input.val();
                    if (event.type === 'focusout' || event.keyCode === $.ui.keyCode.ENTER) {
                        $input.off();
                        // We need to refresh the list completely if the list
                        // has no items. This is necessary even if the value
                        // hasn't changed so that first item message placeholder
                        // will be restored
                        if (!num_items || (value && value !== currentVal)) {
                            if (items.length) {
                                items[index] = value;
                            } else {
                                items = [value];
                            }
                            set_items(items);
                            $li.remove();
                            refreshList();
                        } else {
                            $li.replaceWith($entry);
                        }
                        _editing_inline = false;
                    }
                });
            }

            function removeItem() {
                var items = get_items();

                // this has no meaning when being called from multiList-close, to be removed.
                var idx = $(this).closest("li").data("index");
                if (idx !== undefined) {
                    var item = items.splice(idx, 1);
                    set_items(items);

                    if ($.isFunction(config.afterRemove)) {
                        config.afterRemove.call(input[0], item);
                    }
                }

                refreshList();
            }

            function addIcon($listItem, icon_type) {
                if ($.inArray(icon_type, ['deletion', 'dropdown']) < 0) {
                    return;
                }
                var cls = icon_type === 'deletion' ?
                              'multiList-close' : 'multiList-dropdown';
                var item_a = $('<a class="multiList-btn ' +
                               cls + '" />');
                $listItem.append(item_a);
            }

            function refreshList() {
                // detach the list from the dom to avoid reflows
                var reattach = fweb.util.$detached($list);

                var items = get_items();

                $list.removeClass("focus");

                $("li.multiList-list-item", $list).remove();
                var len = items.length, disp, $listItem, $shown_item;

                var show_deletion = !config.readonly &&
                                    !config.disabled &&
                                    len > config.min;
                var show_dropdown = !config.readonly &&
                                    !config.disabled &&
                                    !config.disabled_dropdown;

                for (var i = 0; i < len; ++i) {
                    $listItem = $('<li class="multiList-list-item"></li>')
                        .data("index", i);

                    var icon_type = show_deletion ? 'deletion' : (
                                        show_dropdown ? 'dropdown' : '');
                    addIcon($listItem, icon_type);

                    $shown_item = $('<div class="multiList-item" />');
                    disp = config.display.call(input[0], items[i], {});
                    $shown_item.append(disp.label || disp);
                    var tooltip = disp.tooltip;
                    // Using text of the element is unreliable but mostly safe
                    $shown_item.attr('title', tooltip ? tooltip : $shown_item.text());

                    $listItem.append($shown_item)
                        .addClass(config.disabled ? "disabled" : "")
                        .addClass(config.disabled_dropdown ? "disabled_dropdown" : "");

                    $inputContainer.before($listItem);
                }

                if (!len) {
                    disp = config.display.call(input[0], '', {});
                    var lbl = disp.label || disp;
                    $listItem = $('<li class="multiList-list-item"></li>');
                    addIcon($listItem, show_dropdown ? 'dropdown' : '');
                    $shown_item = $('<div class="multiList-item" />');
                    $shown_item.append(lbl);
                    $listItem.append($shown_item)
                        .addClass(config.disabled ? "disabled" : "")
                        .addClass(config.disabled_dropdown ? "disabled_dropdown" : "");
                    $listItem.prependTo($list);
                }

                $("li.multiList-list-item", $list).off(namespaced_click).on(namespaced_click,
                    function(event) {
                        if (!$(this).hasClass("disabled") && !$(this).hasClass("disabled_dropdown")) {
                            if (config.create.enable && config.source.length === 0) {
                                create_item();
                            }
                            // TODO: move this check into the click handler.
                            else if (!$input_mc.data('fortinetMulticomplete').menu.element.is(":visible")) {
                                if (config.inline_edit) {
                                    editInline.call(this);
                                } else {
                                    // TODO: triggering handlers is a terrible way to call our own functions
                                    add_button.triggerHandler('click');
                                }
                            }
                        }
                    });

                $("a.multiList-close", $list).off(namespaced_click).on(namespaced_click, function() {
                    _mouseHandled = true;
                    removeItem.call(this);
                    return false;
                });

                var src = config.source;
                var select_cond = config.create.enable || config.inline_edit ||
                    typeof(src) == "string" || len < src.length;
                add_button.toggle(
                    // not disabled, has value
                    !config.disabled && !!input.val()
                    // not reach to max
                        && (!config.max || len < config.max && config.max > 1)
                    // not select-only or not all options selected
                        && select_cond);

                reattach();
                if (typeof config.post_refresh_callback == 'function') {
                    config.post_refresh_callback.call($list);
                }
            }

            function closeList() {
                $(document).off(namespaced('click', config.uid));
                $inputContainer.hide();
                $input_mc.multicomplete('stopRender');

                if (!input.val() && !_editing_inline) {
                    // make sure that the hint text is re-added
                    refreshList();
                }

                if (config.dialog) {
                    config.dialog.dialog("option", "position", config.dialog.dialog("option", "position"));
                }
            }
        });
    };

    $.fn.multiList.error = function(message) {
        throw new $.fn.multiList.Error(message);
    };
    // TODO: if we override $.error to provide debugging, use the following
    // $.fn.multiList.error = $.error

    $.fn.multiList.Error = function(message) {
        this.name = 'multiList.Error';
        this.message = message;
        this._is_fweb_multiList_error = true;
    };

    $.toArray = function(obj) {
        var arr = [];
        for (var x in obj) {
            if (obj.hasOwnProperty(x)) {
                arr.push(x);
            }
        }
        return arr;
    };
})(jQuery);
