/* global define */
/**
 * The contents of this file define qlist's column sorting feature
 * ANY other mention of sorting in jquery.qlist.js does not apply to this,
 * unless it explicitly says "qlist.sort.js". This composes a single, _cohesive_
 * feature that allows the user to click on a column header, and sort (usually
 * without hitting the server) the qlist according to that column. This does not
 * affect the order of the entries in `config.source`, and instead changes the
 * order of the `display_master`.
 *
 * This extension was not written from scratch, but rather refactored out of
 * jquery.qlist.js. So, there are some non-standard extension conventions:
 *
 * `enabled_config_path` is used, and should _only_ be used here, so that this
 * could still be a managed extension
 *
 * Some state data is stored directly on core.state, rather than independently here.
 * Some pages depend on reading that data.
 *
 * `display_master` could be managed by a different extension, although it originated with sorting.
 *
 * Lots of other configuration is not stored in the standard place. See the
 * extension definition's defaults for more info.
 */

(function(global, body) {
    'use strict';

    var exports = body(jQuery, global);
    if (typeof define === 'function' && define.amd) {
        define('qlist.sort', ['jquery'], function() {
            return exports;
        });
    } else {
        global.fweb = global.fweb || {};
        global.fweb.qlist = global.fweb.qlist || {};
        global.fweb.qlist.sort = exports;
    }
})(this, function($, global) {
    'use strict';

    var try_JSON_parse = global.fweb.util.dom.try_JSON_parse;
    var RegExpCommon = global.RegExpCommon;

    var $cookieStore = {
        remove: global.removeCookie,
        get: function(n) { return try_JSON_parse(global.getCookie(n)); },
        put: function(n, v) { return global.setCookie(n, JSON.stringify(v)); }
    };

    var SortState = function() {};
    SortState.prototype.init = function() {
        this._sorting_cache = {};
    };
    SortState.prototype.preload = function(that, internal) {
        this.core_config = internal.core.config;
        this.config = this.core_config.sort;
        this.core_state = internal.core.state;
        this.el = $(that);

        this.internal_fns = {
            qlistRefreshTable: internal.core.fn.qlistRefreshTable
        };

        if (this.core_state.sorting == null) {
            var sorting_cookie_data;
            if (this.config.save_in_cookie) {
                sorting_cookie_data = clean_sorting_cookie_data(
                    $cookieStore.get(this.cookie_name), this.core_config);
            }

            // sorting is of type [{selector:string, direction:string}]
            this.core_state.sorting = sorting_cookie_data || this.core_config.default_sort;
        }

        this.resortIfNeeded();
    };
    SortState.prototype.postload = function(that, internal) {
        this.head_row = internal.core.head_row;
        this.add_sortable_indicators();
        this.renderInitialSortIndicator();

        this.head_row.off('click.sort');
        this.head_row.on('click.sort', $.proxy(this, 'onHeaderClick'));
        this.el.on('reset_columns.sort', $.proxy(this, 'reset_sorting'));
    };
    /**
     * @return {boolean} true if the display master was reset
     */
    SortState.prototype.resetDisplayMasterIfNeeded = function() {
        var source = this.getSource();
        var displayMaster = this.getDisplayMaster();
        if (displayMaster == null ||
            displayMaster.length !== source.length) {
            // (re)initialize display_master
            // NOTE: won't trigger in the unlikely case that the source has been changed
            // but it's the same length. In this case, the developer can force a sort.
            this.setDisplayMaster(get_new_display_master(source.length));
            return true;
        }
        return false;
    };
    SortState.prototype.destroy = function() {
        this.core_state = null;
        this.core_config = null;
        this.el = null;
        if (this.head_row != null) { this.head_row.off('click.sort'); }
        this.head_row = null;
        this.internal_fns = null;
    };
    SortState.prototype.onHeaderClick = function(event) {
        if (this.core_config.options.fixed_sort) { return; }
        var config = this.core_config;
        var sort_columns = config.options.sort_columns;
        var el = $(event.target).closest('th');
        var selector = el.data('selector');

        if ($.isFunction(config.callbacks.pre_sort)) {
            // call the pre-sort callback function
            config.callbacks.pre_sort(event, el, selector);
        }

        // Column Sorting
        if (config.checkboxes.enabled && !config.options.hide_checkboxes &&
           $('input:checkbox', el).length > 0) {
            return; // don't sort by checkboxes
        }
        if (this.el.find('td.empty_row').length > 0) {
            return; // disable sorting of empty qlists
        }
        if (!config.sort_fn['*'] && !(selector in config.sort_fn)) {
            // if the sort function isn't defined, don't try to sort by this column.
            return;
        }
        event.stopPropagation();
        if (!sort_columns || sort_columns[selector]) {
            this.do_sort_toggle(selector);
        }
    };
    SortState.prototype.prepareFastSort = function(target) {
        this.core_config.fast_sort = [];
        this.core_config.fast_sort_target = target;
    };
    SortState.prototype.disableFastSort = function() {
        this.core_config.fast_sort = undefined;
    };
    SortState.prototype.checkFastSort = function(target) {
        if (this.config.fast_update !== false && !this.core_config.paging.enabled &&
            !(this.core_config.category.q_type !== null &&
              this.core_config.column_map.category != null)) {
            this.prepareFastSort(target || this.el.find('tbody').eq(0));
        } else {
            this.disableFastSort();
        }
    };
    SortState.prototype.fastSortEnabled = function(check) {
        if (check) { this.checkFastSort(); }
        return this.core_config.fast_sort !== undefined;
    };
    SortState.prototype.clear_sorting_cache = function() {
        this.core_state.display_master = get_new_display_master(this.getSource().length);
        this._sorting_cache = {};
        this.checkFastSort();
    };
    SortState.prototype.reset_sorting = function() {
        this.clear_sorting_cache();
        this.source_order_indicator = false;
        this.setSortValue(this.getDefaultSort());

        $cookieStore.remove(this.cookie_name);

        var sortValue = this.getSortValue();
        if (sortValue != null && sortValue.length > 0) {
            this._do_sort();
        }
    };

    // normalize access of data stored in different locations
    SortState.prototype.getSortValue = function() { return this.core_state.sorting; };
    SortState.prototype.setSortValue = function(value) { this.core_state.sorting = value; };
    SortState.prototype.getDisplayMaster = function() { return this.core_state.display_master; };
    SortState.prototype.setDisplayMaster = function(value) {
        this.core_state.display_master = value;
    };
    SortState.prototype.getSource = function() { return this.core_config.source; };
    SortState.prototype.getDefaultSort = function() { return this.core_config.default_sort; };

    SortState.prototype.toggle_source_order_indicator = function() {
        var source_order = this.core_config.options.sort_source_order;
        if (source_order) {
            this.setSortValue([source_order]);
            this.renderInitialSortIndicator();
        }
    };

    SortState.prototype.source_load_complete = function() {
        this.resortIfNeeded();
    };


    SortState.prototype.resortIfNeeded = function() {
        // NOTE: the display master won't be reset in the unlikely case that the source
        // has been changed but it's the same length (server side paging, maybe).
        // In this case, the developer can force a sort.
        var sorting = this.getSortValue();
        if (this.resetDisplayMasterIfNeeded() &&
            sorting != null &&
                sorting.length > 0 && this.getSource() != null &&
                    (sorting !== this.getDefaultSort() ||
                     !this.config.default_sort_is_serverside)) {
            // if we have a sort state saved in the cookie, presort the data.
            // it's placed down here since _do_sort needs that.state to be set.
            this._do_sort();
        }
    };

    SortState.prototype.renderInitialSortIndicator = function() {
        var sortValue = this.getSortValue();
        var source = this.getSource();
        var source_order = this.core_config.options.sort_source_order;

        if ((!sortValue || !sortValue.length) && source_order) {
            sortValue = [source_order];
            this.setSortValue(sortValue);
        }

        // get the current sort to draw the sort indicator on the header.
        // don't get it if there's no data so sort, since clicking won't do anything then.
        // if it's a promise, just assume we'll get data
        var current_sort = (source.length > 0 || 'then' in source) &&
            sortValue != null ? sortValue[0] : undefined;

        // draw the sort indicator on the currently sorted row
        if (current_sort && !this.core_config.options.fixed_sort) {
            this.head_row.find('th').filter(function() {
                return $(this).data('selector') === current_sort.selector;
            }).addClass('sort-' + current_sort.direction);
        }
    };

    SortState.prototype.add_sortable_indicators = function() {
        var config = this.core_config;
        var sort_columns = config.options.sort_columns;

        if (this.el.find('td.empty_row').length) {
            return;
        }

        this.head_row.find('th').each(function() {
            var $this = $(this);
            var selector = $this.data('selector');
            if (config.checkboxes.enabled && !config.options.hide_checkboxes &&
                $this.find('input:checkbox')) {
                return;
            }
            if (!config.sort_fn['*'] && !(selector in config.sort_fn)) {
                return;
            }
            if (sort_columns && !sort_columns[selector]) {
                return;
            }

            $this.addClass('can_sort');
        });
    };

    /**
     * ----------------------------------------------------------------------------
     * The following code was originally generated by CoffeeScript
     * Please improve the readability of any lines you touch
     * ----------------------------------------------------------------------------

    /*
# creates a one-element array of sorting arguments to pass to do_sort
# selector: string
*/

    var argIndexOf, isServerSort, serverSortIndex,
        __indexOf = [].indexOf || function(item) {
            for (var i = 0, l = this.length; i < l; i++) {
                if (i in this && this[i] === item) {
                    return i;
                }
            }
            return -1;
        };

    SortState.prototype.do_sort_toggle = function(selector) {
        var config, current_direction, dirs, new_direction, state, _i, _len, _ref, _ref1;

        config = this.core_config;
        _ref = this.getSortValue();

        for (_i = 0, _len = _ref.length; _i < _len; _i++) {
            state = _ref[_i];
            if (state.selector === selector) {
                current_direction = state.direction;
                break;
            }
        }

        if (config.options.sort_columns &&
            typeof config.options.sort_columns[selector] === 'string') {
            if (_ref.length && _ref[0].selector === selector) {
                // Do nothing...were currently sorted in the only direction we allow
                return;
            } else {
                new_direction = config.options.sort_columns[selector];
            }
        } else {
            dirs = (_ref1 = config.sort_fn[selector]) != null ? _ref1.directions : void 0;
            if (dirs != null) {
                /* use the next direction in manually defined list of directions*/

                /* this is for complex sorts*/

                new_direction = dirs[($.inArray(current_direction, dirs) + 1) % dirs.length];
            } else {
                /* invert the current direction*/

                new_direction = current_direction === 'asc' ? 'desc' : 'asc';
            }
        }

        return this.do_sort([{
            selector: selector,
            direction: new_direction
        }]);
    };

    /*
# basically remove whatever sort_args are already in the sort_state
#    and add the sort_args onto the front.
# sort_args: [{selector:string, direction:<'asc'|'desc'>}]
*/


    SortState.prototype.do_sort = function(sort_args) {
        var config, defer, del_indicies, directionsNotRemoved, el, get_header, i, new_top_state,
            odd, old_sort_state, old_top_state, prevServerSortIndex, selectors, sortValue,
            sorting_event, _i, _j, _len, _len1, _ref, _ref1, _ref2, _ref3, _ref4,
            _this = this;

        config = this.core_config;
        old_sort_state = (_ref = this.getSortValue()) != null ? _ref.slice(0) : void 0;
        prevServerSortIndex = argIndexOf(old_sort_state, this.prevServerSort);

        if (sort_args != null) {
            old_top_state = this.getSortValue()[0];
            new_top_state = sort_args[0];
            /* Utility function to get the <th> for the current selector*/

            get_header = function(selector) {
                var ret;
                ret = null;
                _this.el.find('th').each(function(i, el) {
                    if ($.data(el, 'selector') === selector) {
                        ret = $(el);
                        return false;
                    }
                });
                return ret;
            };
            /* Apply the appropriate CSS class to the header*/

            if (!config.options.fixed_sort) {
                if (old_top_state != null) {
                    if ((_ref1 = get_header(old_top_state.selector)) != null) {
                        _ref1.removeClass('sort-' + old_top_state.direction);
                    }
                }
                if (new_top_state != null) {
                    if ((_ref2 = get_header(new_top_state.selector)) != null) {
                        _ref2.addClass('sort-' + new_top_state.direction);
                    }
                }
            }
            /* Process the sorting arguments*/

            /* Extract a list of just the selectors from the list of arguments*/
            selectors = $.map(sort_args, function(arg) {
                return arg.selector;
            });

            /* Create a list of the indicies that should be removed from the current state*/
            sortValue = this.getSortValue();
            directionsNotRemoved = {};
            del_indicies = $.grep(get_new_display_master(sortValue.length), function(i) {
                var duplicate, st, _ref3, sort_fn;
                st = sortValue[i];
                sort_fn = _this.core_config.sort_fn[st.selector];
                if (sort_fn && isServerSort(st, sort_fn)) {
                    /*
                     * server sorting combined with client side sorting is odd, so treat server
                     * side sorts differently here, only remove duplicates of server sorts and have
                     * both the same selector and direction
                     */
                    duplicate = (
                        _ref3 = st.direction,
                         __indexOf.call(directionsNotRemoved[st.selector] || [], _ref3) >= 0
                    );
                    if (!duplicate) {
                        (directionsNotRemoved[st.selector] ||
                         (directionsNotRemoved[st.selector] = [])).push(st.direction);
                    }
                    return duplicate;
                }
                return $.inArray(st.selector, selectors) >= 0;
            });

            /* Remove the indicies from the current state*/
            _ref3 = del_indicies.reverse();
            for (_i = 0, _len = _ref3.length; _i < _len; _i++) {
                i = _ref3[_i];
                sortValue.splice(i, 1);
            }

            /* Add the new sorting arguments to the current sorting state*/
            this.setSortValue(clean_sorting_cookie_data(
                sort_args.slice(0).concat(this.getSortValue()), this.core_config
            ));
        }

        /* Trigger an event for possible canceling & other notification*/
        sorting_event = $.Event('sorting');
        this.el.trigger(sorting_event, [old_sort_state]);
        if (this.config.save_in_cookie) {
            $cookieStore.put(config.ck_name.sorting, this.getSortValue());
        }
        if (sorting_event.isDefaultPrevented()) {
            return false;
        }
        /* Now that all the plumbing is done, call our actual sorting function,
           which will sort the display master according to the given sorting arguments
           */

        defer = this._do_sort(sort_args != null ? sort_args.length : void 0, prevServerSortIndex);
        if (defer) {
            return defer;
        }
        /* Reorder/render the html elements based on the new sorting order*/

        if (this.core_state.fast_sort != null) {
            /* if we can do a "fast sort", then do so. This just rearranges the <tr>s
               instead of completely redrawing the table
               */

            _ref4 = this.getDisplayMaster();
            for (_j = 0, _len1 = _ref4.length; _j < _len1; _j++) {
                i = _ref4[_j];
                if ((el = this.core_state.fast_sort[i]) != null) {
                    el.appendTo(this.core_state.fast_sort_target);
                    if (odd) {
                        el.addClass('odd');
                    } else {
                        el.removeClass('odd');
                    }
                    odd = !odd;
                }
            }
        } else {
            this.internal_fns.qlistRefreshTable();
        }
        this.el.trigger('sort', [old_sort_state]);
        return true;
    };

    isServerSort = function(arg, fn) {
        var sort_fn;
        var value = false;
        if (fn.kind === 'server' || (fn.kind === 'complex' && fn.server)) {
            value = true;
        } else if ((sort_fn = fn[arg.direction] || fn['*'])) {
            value = sort_fn.kind === 'server';
        }
        return value;
    };

    serverSortIndex = function(args) {
        var arg, i, r, _i, _len;
        r = -1;
        for (i = _i = 0, _len = args.length; _i < _len; i = ++_i) {
            arg = args[i];
            if (isServerSort(arg, arg.fn)) {
                r = i;
                break;
            }
        }
        return r;
    };

    argIndexOf = function(args, target) {
        var arg, i, r, _i, _len;
        r = -1;
        if (target == null) {
            return r;
        }
        for (i = _i = 0, _len = args.length; _i < _len; i = ++_i) {
            arg = args[i];
            if (target.selector === arg.selector && target.direction === arg.direction) {
                r = i;
                break;
            }
        }
        return r;
    };

    SortState.prototype.isNewServerSort = function(arg) {
        var p;
        p = this.prevServerSort;
        return ((arg != null) &&
                ((p == null) || arg.selector !== p.selector || arg.direction !== p.direction));
    };

    SortState.prototype.isGlobalServerSort = function() {
        return isServerSort({
            selector: '*',
            direction: '*'
        }, this.core_config.sort_fn['*']);
    };

    SortState.prototype._doServerSort = function(index) {
        var sortArg, sortFn, sortFns;
        if (index == null) {
            index = 0;
        }
        sortFns = this.core_config.sort_fn;
        sortArg = this.getSortValue()[index];
        /* TODO: handle a returned promise*/

        /* promise =*/

        sortFn = sortFns[sortArg.selector] || sortFns['*'];
        if (sortFn.kind === 'complex') {
            sortFn = sortFn[sortArg.direction] || sortFn['*'] || sortFn.server;
        }
        return sortFn({
            current_sort: this.getSortValue(),
            target_sort: sortArg,
            display_master: this.getDisplayMaster()
        });
    };

    /*
# compare to dataTables._fnSort
# count (optional): amount of properties in state.sorting to actually process
*/


    SortState.prototype._do_sort = function(count, prevServerSortIndex) {
        var arg, args, category_map, category_sort_fn, config, currentServerSort,
            currentServerSortIndex, defer, display_current, entry, get_complex_sort_child_test,
            get_simple_test, i, j, prevServerSort, source, tmp_obj, val, _i, _j, _k, _l, _len,
            _len1, _len2, _len3, _ref, _ref1, _ref2, _ref3, _s,
            _this = this;

        if (prevServerSortIndex == null) {
            prevServerSortIndex = -1;
        }

        config = this.core_config;
        if (this.getSortValue().length === 0) {
            /* reset the sort*/

            this.setDisplayMaster(get_new_display_master());
            return;
        }
        /* roughly adapted from jquery.dataTables._fnSort*/

        args = count ? this.getSortValue().slice(0, count) : this.getSortValue().slice(0);
        /*
           # add the sorting function to the arg list for convenience
           # this makes a copy of each so that we don't taint the state
           */

        for (i = _i = 0, _len = args.length; _i < _len; i = ++_i) {
            arg = args[i];
            tmp_obj = {
                fn: config.sort_fn[(_s = arg.selector) in config.sort_fn ? _s : '*']
            };
            args[i] = $.extend(tmp_obj, args[i]);
        }
        if (this.isGlobalServerSort(currentServerSort)) {
            return this._doServerSort() || true;
        }
        prevServerSort = this.prevServerSort;
        currentServerSortIndex = serverSortIndex(args);
        currentServerSort = this.getSortValue()[currentServerSortIndex];
        if ((currentServerSort != null) && (this.isNewServerSort(currentServerSort) ||
                prevServerSortIndex !== -1 && currentServerSortIndex !== prevServerSortIndex)) {
            this.prevServerSort = currentServerSort;
            defer = this._doServerSort(currentServerSortIndex);
            if (currentServerSortIndex === 0) {
                this.setDisplayMaster(get_new_display_master());
            }
            return defer || true;
        }
        if (currentServerSortIndex >= 0) {
            args = args.slice(0, currentServerSortIndex);
        }
        /* backwards compatibility for old style categories
           # this may need to be revised to accomodate new-style categories as well.
           # TODO (maybe): move this into a plugin-style framework (like with sort_fn)
           */

        if (config.category.q_type !== null && (config.column_map.category != null)) {
            category_map = {};
            _ref = this.getSource();
            for (i = _j = 0, _len1 = _ref.length; _j < _len1; i = ++_j) {
                entry = _ref[i];
                if (!(entry.category in category_map)) {
                    category_map[entry.category] = i;
                }
            }
            category_sort_fn = function(a, b) {
                return numeric_sort_fn(category_map[a], category_map[b]);
            };
            /* adds an extra argument to the front of the arguments to
               # maintain the current order of the categories
               */

            args.unshift({
                selector: 'category',
                direction: 'asc',
                fn: category_sort_fn
            });
        }
        /*
           # TODO: remove the above section
           # TODO: hidden sort (if needed):
           # args.unshift.apply(args, @config.hidden_sort) if @hidden_sort?
           */

        /*
           # if there's a 'key' or 'key_row' sort function for the column,
           # make a cache of the translated values
           */

        function gen_sort_cache() {
            var _l, _len3, _results;
            _results = [];
            for (_l = 0, _len3 = source.length; _l < _len3; _l++) {
                val = source[_l];
                _results.push(config.sort_fn[arg.selector](val[arg.selector]));
            }
            return _results;
        }

        function gen_key_cache() {
            var _l, _len3, _results;
            _results = [];
            for (_l = 0, _len3 = source.length; _l < _len3; _l++) {
                val = source[_l];
                _results.push(config.sort_fn[arg.selector](val));
            }
            return _results;
        }

        source = this.getSource();
        for (_k = 0, _len2 = args.length; _k < _len2; _k++) {
            arg = args[_k];
            if (((_ref1 = config.sort_fn[arg.selector]) != null ? _ref1.kind : void 0) === 'key' &&
                !(arg.selector in this._sorting_cache)) {
                this._sorting_cache[arg.selector] = gen_sort_cache(source, config);
            } else if (((_ref2 = config.sort_fn[arg.selector]) != null ? _ref2.kind : void 0) ===
                       'key_row' && !(arg.selector in this._sorting_cache)) {
                this._sorting_cache[arg.selector] = gen_key_cache(source, config);
            }
        }
        /* Set up an array of current positions so that we can do a stable sort if values match*/

        display_current = [];
        _ref3 = this.getDisplayMaster();
        for (i = _l = 0, _len3 = _ref3.length; _l < _len3; i = ++_l) {
            j = _ref3[i];
            display_current[j] = i;
        }
        get_simple_test = function(fn, arg, a, b) {
            if (fn.kind === 'comp_row') {
                return fn(source[a], source[b], default_clientside_sort_comp_fn);
            } else {
                return fn(source[a][arg.selector], source[b][arg.selector],
                          default_clientside_sort_comp_fn);
            }
        };
        get_complex_sort_child_test = function(fn, arg, a, b) {
            /* TODO: (performance) use _sorting_cache*/

            if (fn.kind === 'key') {
                return simple_sort_comp_fn(fn(source[a][arg.selector]),
                                           fn(source[b][arg.selector]));
            } else if (fn.kind === 'key_row') {
                return simple_sort_comp_fn(fn(source[a]), fn(source[b]));
            } else {
                return get_simple_test(fn, arg, a, b);
            }
        };
        /* Do the actual sorting!*/

        this.getDisplayMaster().sort(function(a, b) {
            var test, _len4, _m;
            if ((source[a] != null) && (source[b] != null)) {
                for (_m = 0, _len4 = args.length; _m < _len4; _m++) {
                    arg = args[_m];
                    /* TODO: (performance) move control flow (ifs) out of this
                       function and instead store control flow-less function in each arg
                       */

                    if (arg.fn.kind === 'key' || arg.fn.kind === 'key_row') {
                        test = simple_sort_comp_fn(_this._sorting_cache[arg.selector][a],
                                                   _this._sorting_cache[arg.selector][b]);
                    } else if (arg.fn.kind === 'complex') {
                        if ('asc' in arg.fn && arg.direction === 'asc') {
                            test = get_complex_sort_child_test(arg.fn.asc, arg, a, b);
                        } else if ('desc' in arg.fn && arg.direction === 'desc') {
                            test = get_complex_sort_child_test(arg.fn.desc, arg, a, b);
                        } else if ('ascdesc' in arg.fn && (arg.direction === 'asc' ||
                                                           arg.direction === 'desc')) {
                            test = get_complex_sort_child_test(arg.fn.ascdesc, arg, a, b);
                        } else {
                            return get_complex_sort_child_test(arg.fn[arg.direction], arg, a, b);
                        }
                    } else {
                        test = get_simple_test(arg.fn, arg, a, b);
                    }
                    if (test !== 0) {
                        return (arg.direction === 'asc' ? test : -test);
                    }
                }
            }
            return numeric_sort_fn(display_current[a], display_current[b]);
        });
    };
    /**
     * ----------------------------------------------------------------------------
     *  End CoffeeScript generated code (orginally)
     *  ---------------------------------------------------------------------------
     */

    // Utility functions

    function get_new_display_master(length) {
        var new_display_master = [];
        for (var _i = 0; _i < length; ++_i) {
            new_display_master.push(_i);
        }
        return new_display_master;
    }

    function get_cookie_name(prefix) {
        return prefix + '_sorting_settings';
    }

    function clean_sorting_cookie_data(sorting_cookie_data, config) {
        // checks the sorting_cookie_data so that it conforms to the
        //   [{selector:string, direction:string}] format and doesn't
        //   contain any selectors that aren't a column.
        // note that this function could be generalized somewhere else,
        //   and it could be simplified with some underscore functions.
        var sort_item_template = {
            selector: '',
            direction: ''
        };

        var i, l, p, x;
        if ($.isArray(sorting_cookie_data)) {
            l = sorting_cookie_data.length;
            for (i = 0; i < l; i++) {
                // _.pick(sorting_cookie_data[i], 'selector', 'direction');
                x = sorting_cookie_data[i];
                for (p in x) {
                    if (x.hasOwnProperty(p) && !sort_item_template.hasOwnProperty(p)) {
                        delete x[p];
                    }
                }
            }
            sorting_cookie_data = $.grep(sorting_cookie_data, function(x) {
                for (var p in sort_item_template) {
                    if (sort_item_template.hasOwnProperty(p) && !x.hasOwnProperty(p)) {
                        return false;
                    }
                }
                return config.column_map.hasOwnProperty(x.selector);
            });
        } else {
            sorting_cookie_data = undefined;
        }
        return sorting_cookie_data;

        //// for possible unit testing.
        // assert.deepEqual(clean_sorting_cookie_data([{selector:'device'}]), []);
        // assert.deepEqual(clean_sorting_cookie_data(
        //     [{selector:'device', direction:'asc'}]),
        //     [{selector:'device', direction:'asc'}]);
        // assert.deepEqual(clean_sorting_cookie_data(
        //     [{selector:'device', direction:'asc', fn: null}]),
        //     [{selector:'device', direction:'asc'}]);
        // assert.deepEqual(clean_sorting_cookie_data(
        //     [{selector:'device', direction:'asc', fn: null},
        //      {selector:'user', direction:'asc', fn: null}]),
        //     [{selector:'device', direction:'asc'},
        //      {selector:'user', direction:'asc'}]);
        // assert.deepEqual(clean_sorting_cookie_data(
        //     [{selector:'device', direction:'asc', fn: null},
        //      {selector:'user24', direction:'asc', fn: null}]),
        //     [{selector:'device', direction:'asc'}]);
        // assert.isUndefined(clean_sorting_cookie_data({foo:'bar'}));
    }

    //// SORTING FUNCTIONS ////

    function simple_sort_comp_fn(x, y) {
        return ((x < y) ? -1 : ((x > y) ? 1 : 0));
    }
    simple_sort_comp_fn.kind = 'comp';

    // _digit_regex is limited to 15 digits to be safe: stackoverflow.com/q/307179
    var _digit_regex = /^(\D*)(\d{1,15})/;
    //optimize when comparing latin alphabet only strings
    //localeCompare is SLOW in chrome 26, 27.
    //see http://jsperf.com/operator-vs-localecompage/9
    //if chrome's localeCompare improves this test can be removed
    //add more non-locale specific characters here as needed

    var _no_locale_regex = /^[\w\-.\s]*$/;
    function default_clientside_sort_comp_fn(x, y) {
        function quick_type(obj) {
            var result = typeof obj;
            return result === 'object' ?
                $.type(obj) : result;
        }
        var xtype, x_reg_result, ytype, y_reg_result, ret;
        // if both of the objects are strings, sort them case insensitive
        xtype = quick_type(x);
        ytype = quick_type(y);
        if (xtype === ytype) {
            if (xtype === 'string') {
                x = x.toLowerCase();
                y = y.toLowerCase();
                if (x === y) {
                    return 0;
                }
                // if the string starts with numbers, sort numerically
                if ((x_reg_result = _digit_regex.exec(x)) !== null &&
                    (y_reg_result = _digit_regex.exec(y)) !== null &&
                        x_reg_result[1] === y_reg_result[1]) {
                    ret = parseInt(x_reg_result[2], 10) - parseInt(y_reg_result[2], 10);
                    if (ret !== 0) {
                        return ret;
                    }
                } else {
                    if (_no_locale_regex.test(x) && _no_locale_regex.test(y)) {
                        return x > y ? 1 : (x === y ? 0 : -1);
                    } else {
                        return x.localeCompare(y);
                    }
                }
            } else if (xtype === 'number') {
                return x - y;
            } else if (x == null) {
                return 0;
            }
        }
        if (y == null) { y = (xtype === 'number') ? Infinity : 'z' + y; }
        if (x == null) { x = (ytype === 'number') ? Infinity : 'z' + x; }
        return ((x < y) ? -1 : ((x > y) ? 1 : 0));
    }
    default_clientside_sort_comp_fn.kind = 'comp'; // comparison/comparator
    // alternative is `fn.kind = "key"` for user defined functions,
    //  which will return some kind of key from the data
    //  that can be sorted by this default sort function
    // also can be comp_row or key_row, which will pass the function the entire row to
    //  be compared (not just the value of the column)

    function numeric_sort_fn(x, y) {
        return x - y;
    }

    function ip4_addr_sort(x, skip_test) {
        if (!skip_test && !RegExpCommon.IP_HOST.test(x)) {
            return 4294967296;
        }

        x = x.split('.');
        return parseInt(x[0], 10) * 16777216 +
            parseInt(x[1], 10) * 65536 +
            parseInt(x[2], 10) * 256 +
            parseInt(x[3], 10);
    }

    ip4_addr_sort.kind = 'key';

    function ip6_addr_sort(x, skip_test) {
        if (!skip_test && !RegExpCommon.IP6_HOST.test(x)) { return x; }
        // based on https://github.com/beaugunderson/javascript-ipv6 (MIT license)
        var i, g = [];
        var halves = x.split('::');
        if (halves.length === 2) {
            var first = halves[0].split(':');
            var last = halves[1].split(':');
            if (first.length === 1 && first[0] === '') { first = []; }
            if (last.length === 1 && last[0] === '') { last = []; }
            var remaining = 8 - (first.length + last.length);
            for (i = 0; i < first.length; i++) { g.push(first[i]); }
            for (i = 0; i < remaining; i++) { g.push(0); }
            for (i = 0; i < last.length; i++) { g.push(last[i]); }
        } else if (halves.length === 1) {
            g = x.split(':');
        }
        var s;
        for (i = 0; i < g.length; i++) {
            s = '' + parseInt(g[i], 16);
            g[i] = '00000'.substr(s.length) + s;
        }
        return g.join('');
    }
    ip6_addr_sort.kind = 'key';

    function ip_addr_sort(x) {
        if (RegExpCommon.IP_HOST.test(x)) {
            return ip4_addr_sort(x, true);
        }
        if (RegExpCommon.IP6_HOST.test(x)) {
            // the prepended '1' assures that an ip6 addr is always after a ip4 addr
            return '1' + ip6_addr_sort(x, true);
        }
        // this is 8 65535's with a 2 on the front, to guarantee that it's larger
        // than any ipv6 address string, as in
        // '2' + ip6_addr_sort('ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff')
        return '26553565535655356553565535655356553565535';
    }
    ip_addr_sort.kind = 'key';

    function ip_mask_sort(x) {
        return ip_addr_sort(x.split('/')[0]);
    }
    ip_mask_sort.kind = 'key';

    // legacy access of sort functions. New access should use the exports
    // values
    if (typeof $.qlist === 'undefined') { $.qlist = {}; }
    if (typeof $.qlist.sort_fns === 'undefined') { $.qlist.sort_fns = {}; }
    $.qlist.sort_fns.ip4_addr_sort = ip4_addr_sort;
    $.qlist.sort_fns.ip6_addr_sort = ip6_addr_sort;
    $.qlist.sort_fns.ip_addr_sort = ip_addr_sort;
    $.qlist.sort_fns.ip_mask_sort = ip_mask_sort;

    var extension = $.qlist.ext.sort = {
        config_path: 'sort',
        managed: true,
        enabled_config_path: 'options.sorting',
        staticDepends: ['display_master'],

        // TODO: move all sorting settings, except sort_fn, to be under the 'sort' key
        defaults: {
            // enabled: false
            'fast_update': true,
            'save_in_cookie': true,
            // when performing a sort, 'pretend' like there are these sorts ahead of other sorts
            //  useful for categories, trees, etc
            // NOT IMPLEMENTED: see commented code in _do_sort
            // [{selector:string, direction:<'asc'|'desc'>, fn:default_clientside_sort_comp_fn?}]
            'hidden_sort': [],

            // enabling this will assume that the data from the server is already
            // sorted by the server, and not re-sort it according to the default sort
            'default_sort_is_serverside': false
        },
        core_defaults: {
            'default_sort': [],

            // 'sort': {...}, // see `defaults`, above
            options: {
                // Enable sorting?
                'sorting': false,
                // A mapping to enable sorting on specific columns only with the
                // option of enabling only a specific direction
                // Example:
                //   {
                //      column1: true,  // Supports sorting in both directions
                //      column2: 'asc', // Supports sorting in 'asc' direction only
                //      column3: 'desc'
                //   }
                // All columns left out of the map will have sorting disabled
                'sort_columns': null,
                // TODO: server-side sorting: for server side sorting, we will send
                //       the sort args and receive a new display_master array
                //       - a display_master is just an array of indicies in the data array
                //         in the order that it should be sorted, like [2, 1, 0]
                //         for a reversed list order.
                //       search for `if config.options.sorting_type == 'server'`
                //       and add new code there.

                // do sorting, but don't allow the user to manually sort
                // you should specify a default sort if you set this to true.
                // Also will hide the sort indicator.
                'fixed_sort': false
            },
            /* sorting functions -
             *  - '*' is the default sort.
             *      set it to null or undefined to restrict columns
             *  - for each function, you can set a .kind attribute that will
             *    define what arguments will be passed to the function and what
             *    the sorting function expects it to return.
             *   'comp' (default) - (x, y[, default_fn]): int -
             *                      x and y are the column values to be compared
             *   'comp_row' - (x, y[, default_fn]): int -
             *                x and y are the entire row values to be compared
             *   'key' - (a): any - returns a single value that will be used by a basic sort
             *   'key_row' - (a): any - a is the entire row, returns a single value
             *   'server' - (info): void|{ then: fn(success, fail) }
             *                  (promise or promise-like object)
             *              `info` is an object in the following format: {
             *                current_sort: [{selector:string, direction:<'asc'|'desc'>}, ...]
             *                  (the sort cookie),
             *                target_sort: {selector:string, direction:<'asc'|'desc'>}
             *                  (the top most sort argument),
             *                display_master: [int, ...] (the display master, described elsewhere),
             *                // more information can be added as needed.
             *              }
             *              If the function doesn't return a thenable, then it is
             *              assumed that the qlist will be reinitialized with the
             *              new source, already sorted. Otherwise, (NOT IMPLEMENTED!!!)
             *              the thenable should resolve with an object of the form {
             *                source: [] (new source)
             *                OR
             *                display_master: [] (new display master) // this form will probably
             *                                                        // never be used.
             *              }
             *   'complex' - rather than a function, an object is expected, with the
             *               form of the following:
             *               { kind: 'complex',
             *                 directions: ['asc', 'desc', 'foofirst', 'barfirst'],
             *                 server: true|false|function (default: false)
             *                         (defines a single server function for all directions)
             *                 foofirst: (a function in any of the above forms;
             *                            defines the sort function for a single direction)
             *                 ascdesc: (a normal sort function that will simply be reversed
             *                           for descending order, use this instead of defining
             *                           both asc and desc)
             *
             *                 IDEA (not implemented) - foolast: '!foofirst'
             *                     - like ascdesc, define a sort function as a
             *                       reversed version of another sort function
             *                 ...
             *               }
             */
            sort_fn: {
                '*': default_clientside_sort_comp_fn
            }
        },
        constructor: SortState,

        //// COMMANDS
        // various commands provided by this extension, like 'refresh' and 'config'
        // - remember to chain (return that) if your command doesn't return anything
        // - unlike _ext_call methods, `internal` is _not_ namespaced, it's the
        //   exact object that's passed from the _ext_handle_commands call
        // in jquery.qlist.js: _ext_handle_commands
        commands: {
            'sort': function(that, internal_core, command) {
                var self = internal_core.$self;
                if (self == null) { return; }
                if ($.type(command) === 'string') {
                    if (command === 'clear_cache') {
                        self.clear_sorting_cache();
                    } else if (command === 'reset') {
                        self.reset_sorting();
                    }
                } else {
                    self.do_sort(command);
                }
                return that;
            },
            'getSortValue': function(that, internal_core) {
                // TODO: if needed, create a simpler way to expose instance methods.
                var self = internal_core.$self;
                if (self != null) { return self.getSortValue(); }
            }
            // string: function(that, internal, ...args)
            // context (this) will be this extension (objectSearch), not the commands object
        },
        postconfigure: function(that, internal) {
            internal.core.config.ck_name.sorting = get_cookie_name(internal.core.config.prefix);
        },
        preload: function(that, internal) {
            var self = internal.$self;
            if (self == null) {
                self = new SortState();
                self.init();
            }
            self.cookie_name = get_cookie_name(internal.core.config.prefix);
            // assert(self.cookie_name === internal.core.config.ck_name.sorting)
            self.preload(that, internal);

            // angular defines a $destroy event; use it for better garbage collection
            if (typeof angular !== 'undefined') {
                $(that).one('$destroy', function() {
                    self.destroy();
                    internal.core.state.sort = null;
                });
            }

            return self;
        },
        postload: function(that, internal) {
            var self = internal.$self;
            if (self == null) {
                self = new SortState();
                self.init();
            }
            self.cookie_name = get_cookie_name(internal.core.config.prefix);
            // assert(self.cookie_name === internal.core.config.ck_name.sorting)
            self.postload(that, internal);

            // angular defines a $destroy event; use it for better garbage collection
            if (typeof angular !== 'undefined') {
                $(that).one('$destroy', function() {
                    self.destroy();
                    internal.core.state.sort = null;
                });
            }

            return self;
        },
        source_load_complete: function(that, internal) {
            if (internal.$self == null) { return; }
            internal.$self.source_load_complete();
        },

        load_row_frag: function(that, internal) {
            // TODO: move the contents of this function into $self
            // reload the display master as the length of the source may have changed
            var state = internal.core.state;
            var display_master = state.display_master;
            var len = internal.core.config.source.length;
            if (!display_master || display_master.length !== len) {
                // Nuke the current sorting. This should only happen with server
                // side sorting. Server side sorting should either be display master
                // aware or the display master should be initialized to the server-
                // side paging page length
                display_master = state.display_master = get_new_display_master(len);
                // OR
                // display_master = that.state.display_master = [];
            }
        }
    };

    // expose internal functions for possible testing, etc
    return {
        ext: extension,
        constructor: SortState,
        get_new_display_master: get_new_display_master,
        sort_fns: {
            ip4_addr_sort: ip4_addr_sort,
            ip6_addr_sort: ip6_addr_sort,
            ip_addr_sort: ip_addr_sort,
            ip_mask_sort: ip_mask_sort
        }
    };
});
