/*
  tagCloud.js - CMDB Tag Cloud Javascript module
  init Aug 2010 by J. Thompson 
  Object Tagging project. 
  Copyright Fortinet Inc
*/

// The following TagCloud script was used as a reference when
// determining tag font sizes:
// http://github.com/buyog/MiscJS/blob/master/tagCloud.js

var TagCloud = (function($) {

    var defaults = {
        // maxEm - Maximum size for the most common tags.
        maxEm: 2.0,

        maxCount: 0,

        // maxTags - Maximum number of filtered tags to display in the cloud.
        maxTags: 25,

        minCount: 0,

        minSize: 1,

        readOnly: true,

        // searchBox - A selector for input elements to use as a search filter for tags.
        searchBox: null,

        tagList: [],

        tagContainer: "#tag_list",

        unusedTagList: []
    };

    var config = {};

    function get_tag_size(tagCount)
    {
        var tagSize = config.maxEm * (tagCount - config.minCount);
        var ems = ((tagSize - config.minSize) / (config.maxCount - config.minCount)) + config.minSize;

        return ems;
    }

    function filter_tag_cloud(filterString)
    {
        if (typeof filterString == "undefined")
        {
            filterString = "";
        }

        $(config.tagContainer).empty();

        var tagList = [];

        for (var i=0; i<config.tagList.length && tagList.length < config.maxTags; i++)
        {
            var tag = config.tagList[i];

            if (tag.q_ref <= config.minCount) continue;

            // Allow name based tag filtering.
            if (tag.name.toLowerCase().indexOf( filterString.toLowerCase() ) == -1)
            {
                continue;
            }

            tagList.push( tag );
        }

        if (tagList.length == 0)
        {
            $("<span></span>")
                .addClass("warning")
                .html( fgt_lang["sysconftags"].no_matching )
                .appendTo(config.tagContainer)
            ;
            return;
        }

        show_popular_tags(tagList);
    }

    function show_popular_tags(tagList)
    {
        tagList.sort( function(a, b) {
            return b.name < a.name ? 1 : -1;
        });

        for (var i=0; i<tagList.length; i++)
        {
            var ems = get_tag_size(tagList[i].q_ref);

            $("<span></span>")
                .addClass("tag")
                .appendTo(config.tagContainer)
                .css("font-size", ems + "em")
                .attr("mkey", tagList[i].name)
                .attr("title", tagList[i].name + " x " + tagList[i].q_ref)
                .text(tagList[i].name)

                // A blank space is added after each <span> to allow word-wrapping
                // to break between tag names.
                .after(" ")
            ;
        }
    }

    function set_config(settings)
    {
        config = $.extend({}, config, settings);
    }

    function init_tag_cloud(settings)
    {
        config = $.extend({}, defaults, settings);

        CMDB.fetch("system", "object-tag", {"with_q_ref": true}, function(data) {
            config.tagList = data.results;

            config.tagList.sort(function(a, b) {
                return b.q_ref - a.q_ref;
            });

            // Store the highest reference count.
            if (config.tagList.length > 0)
            {
                config.maxCount = config.tagList[0].q_ref;
            }

            // Keep track of unused tags.
            for (var i=0; i<config.tagList.length; i++) {
                if (config.tagList[i].q_ref > 0) continue;

                config.unusedTagList.push( config.tagList[i].name );
            }

            init_unused_tags();
            filter_tag_cloud();
        });

        $(".object_tag, .tag", config.tagContainer).live("click", function() {
            var mkey = $(this).attr("mkey");
            if (!mkey) return;

            wij_display_modal_dlg("/objusagedlg?type=" + cmdb_obj_id + "&mkey=" + encodeURIComponent(mkey), { "width": 540 });
        });

        $(config.searchBox).bind("keyup change", function() {
            var searchQuery = $(this).val();

            filter_tag_cloud(searchQuery);
        });

        $("#unused_remove").toggle( !config.readOnly );
    }

    function init_unused_tags()
    {
        $("#unused_count").text( config.unusedTagList.length );

        $("#unused_remove").click( function() {
            if (config.unusedTagList.length && confirm( fgt_lang["sysconftags"].warn_remove_unused )) {
                deleteTags( config.unusedTagList );
            }
        });

        if (config.unusedTagList.length == 0) {
            $("#unused_remove").hide();
        }
    }

    function deleteTags(tagList)
    {
        var url = "/delete?";

        var len = tagList.length;
        for (var i=0; i<len; i++)
        {
            url += "&id=" + cmdb_obj_id + "&mkey=" + encodeURIComponent( tagList[i] );
        }

        window.location.href = url;
    }

    return {
        "init": init_tag_cloud,
        "filter": filter_tag_cloud,
        "settings": set_config
    };

})(jQuery);

var TagSelector = (function($) {
    var defaults = {
        "addTagSelector" : "#addTag",
        "appliedTagSelector" : "#appliedTags",
        "tagsToAddSelector" : "#tagsToAdd",
        "tagsToRemoveSelector" : "#tagsToRemove",
        "appliedTags" : [],
        "tagsToAdd" : [],
        "tagsToRemove" : [],
        "maxPopularTags" : 5,
        "popularTagSelector" : "#popularTags",
        "tagList" : [],
        "tagListSelector" : "#tagList",

        "tagRowSelector" : ".object_tag_row",

        "tagNameList" : [],
        "action" : "edit",

        "enabled": true
    };

    // Override plugin defaults.
    var config = {};

    function initTagSelector(settings)
    {
        config = $.extend({}, defaults, settings);

        if (!config.enabled)
        {
            // The object tagging feature can be disabled via a CLI option. If
            // disabled, the plugin will still be initialized but will
            // only hide the visible tagging elements.
            $(config.tagRowSelector).hide();

            // Preserve any applied tags.
            $(config.tagListSelector).val( config.appliedTags.join("#") );

            return;
        }

        if (!config.tagList || !config.tagList.length) {
            CMDB.fetch("system", "object-tag", {"with_q_ref": true}, function(data) {
                config.tagList = data.results.sort( function(a, b) {
                    return b.q_ref - a.q_ref;
                });

                // Keep a separate list of all tag names only.
                for (var i=0; i<config.tagList.length; i++) {
                    config.tagNameList.push( config.tagList[i].name );
                }
            }, false);
        }

        // treat an empty string value for action as the default action
        if (config.action === "") {
            config.action = defaults.action;
        }

        $(config.appliedTagSelector).addClass("tag_list");
        $(config.tagsToAddSelector).addClass("tag_list");
        $(config.tagsToRemoveSelector).addClass("tag_list");

        // TODO: img url + class as config
        $("<img></img>")
            .addClass("tagSelector-addTag")
            .attr("src", "/images/act_add.gif")
            .click( function() {
                addTags( $(config.addTagSelector).val() );
            })
            .insertAfter(config.addTagSelector)
        ;

        $(config.addTagSelector)
            .keypress( function(event) {
                if (event.keyCode == 13)
                {
                    addTags($(this).val());

                    // prevent propagation
                    return false;
                }
            })
            .autocomplete(config.tagNameList, {
                "matchContains": true,
                "scrollHeight": 50,
                "multiple": true
            })
        ;

        $(".object_tag", config.popularTagSelector).live( "click", function() {
            addTag( $(this).text() );
        });

        refreshDisplay();

        $(config.addTagSelector).focus();
    }

    function setTitleByAction(action) {
        $("h1#add_title").toggle(action === "add");
        $("h1#remove_title").toggle(action === "remove");
    }

    function get_tag_html(tag_name, tag_class) {
        var html = "";

        if (typeof(tag_class) == "undefined") {
            tag_class = "";
        }

        html = "<span class=\"object_tag " + tag_class + "\" mkey=\"" + tag_name + "\">"
            + "<span class=\"tag_label\">" + tag_name + "</span>"
            + "<span class=\"tag_tail\"></span>"
            + "</span>"
        ;

        return html;
    }

    function addTags(tagList)
    {
        var tags = tagList.split(",");
        var tmp;

        for (var i=0; i<tags.length; i++) {
            tmp = tags[i].replace(/(^\s+|\s+$)/g,"");

            if (tmp !== "") {
                addTag(tmp);
            }
        }
    }

    function addTag(tagName)
    {
        var idx;

        if (!tagName || !tagName.length)
            return;

        // TODO: Support multiple??

        if ($.inArray(tagName, config.tagNameList) == -1)
        {
            if (config.readOnly) return;
            // TODO: Alternatively this could only be done when the tag list is saved.
            CMDB.append("system", "object-tag", tagName, {"name": tagName});
        }

        if ($.inArray(tagName, config.appliedTags) == -1)
        {
            config.appliedTags.push( tagName );
        }

        if ($.inArray(tagName, config.tagsToAdd) == -1)
        {
            config.tagsToAdd.push( tagName );
        }

        if ((idx = $.inArray(tagName, config.tagsToRemove)) != -1 )
        {
            config.tagsToRemove.splice(idx, 1);
        }

        refreshDisplay();

        $(config.addTagSelector).val("");
    }

    function removeTag(tagName)
    {
        var idx;

        if ((idx = $.inArray(tagName, config.tagsToAdd)) != -1 )
        {
            config.tagsToAdd.splice(idx, 1);
        }

        if ((idx = $.inArray(tagName, config.appliedTags)) != -1 )
        {
            config.appliedTags.splice(idx, 1);
        }

        config.tagsToRemove.push(tagName);

        refreshDisplay();
    }

    function refreshPopularTags()
    {
        var popularCount = 0;

        $(config.popularTagSelector).empty();

        for (var i=0; i<config.tagList.length && popularCount < config.maxPopularTags; i++)
        {
            if ($.inArray(config.tagList[i].name, config.appliedTags) != -1) {
                continue;
            }

            var span = $(get_tag_html(config.tagList[i].name, "object_tag_add"))
                .attr( "title", config.tagList[i].name + " x " + config.tagList[i].q_ref )
                .appendTo( config.popularTagSelector )
                .after(" ");

            popularCount++;
        }
    }

    function refreshTagList(elem, tag_class, list, click_cb) {
        $(elem).empty();

        list.sort( function(a, b) {
            return b.toLowerCase() < a.toLowerCase() ? 1 : -1;
        });

        for (var i=0; i<list.length; i++)
        {
            var span = $(get_tag_html(list[i], tag_class))
                .appendTo(elem)
                .click( function() {
                    click_cb($(this).attr("mkey"));
                })
            ;
        }    
    }

    function refreshAppliedTags() {
        var className = "object_tag_add";

        if (config.action === "edit") {
            className = "object_tag_remove";
        }

        refreshTagList($(config.appliedTagSelector), className, config.appliedTags, removeTag);
    }

    function refreshTagsToAdd() {
        refreshTagList($(config.tagsToAddSelector), "object_tag_remove", config.tagsToAdd, removeTag);
    }

    function refreshTagsToRemove() {
        refreshTagList($(config.tagsToRemoveSelector), "object_tag_remove", config.tagsToRemove, addTag);
    }

    function resizeDialog()
    {
        if (window.parent && window.parent.wij_auto_resize_iframe)
        {
            window.parent.wij_auto_resize_iframe();
        }
    }

    function refreshDisplay()
    {
        setTitleByAction(config.action);

        refreshAppliedTags();
        refreshTagsToRemove();
        refreshTagsToAdd();
        refreshPopularTags();

        $("table#addTags").toggle(config.action === "add");
        $("table#removeTags").toggle(config.action === "remove");

        resizeDialog();

        $(config.tagListSelector).val( config.appliedTags.join("#") );
    }

    function getTagList(list)
    {
        var tagList = [];

        for (var i=0; i<list.length; i++)
        {
            tagList.push( { "name": list[i] } );
        }

        return tagList;
    }

    function getAppliedTags()
    {
        return getTagList(config.appliedTags);
    }

    function getRemovedTags()
    {
        return getTagList(config.tagsToRemove);
    }

    function getAddedTags()
    {
        return getTagList(config.tagsToAdd);
    }

    return {
        "getAppliedTags" : getAppliedTags,
        "getRemovedTags" : getRemovedTags,
        "getAddedTags" : getAddedTags,
        "getAppliedNameList" : function() {return config.appliedTags},
        "init" : initTagSelector
    };

})(jQuery);

