
if (typeof (FOSGUI) === 'undefined') {
    FOSGUI = {};
}

// namespace
FOSGUI.LogFormatter = (function ($) {

var level_idmap = ['low', 'med', 'high', 'crit'];

var log_columns_dict = {
        'log_event': {
            'utmaction': {
                'allow': 'action_allow',
                'block': 'action_block',
                'reset': 'action_reset',
                'traffic-shape': 'action_traffic-shape'
            }
        },
        'gtp_events': {
            // reference: http://manual.snort.org/node185.html
            'msg-type': {
                // version 0
                '0': {
                    1: 'echo_request',
                    2: 'echo_response',
                    3: 'version_not_supported',
                    16: 'create_pdp_context_request',
                    17: 'create_pdp_context_response',
                    18: 'update_pdp_context_request',
                    19: 'update_pdp_context_response',
                    20: 'delete_pdp_context_request',
                    21: 'delete_pdp_context_response',
                    255: 'pdu'
                },
                // version 1
                '1': {
                    // no override, same as version 0
                },
                // version 2
                '2': {
                    32: 'create_session_request',
                    33: 'create_session_response',
                    34: 'modify_bearer_request',
                    35: 'modify_bearer_response',
                    36: 'delete_session_request',
                    37: 'delete_session_response'
                }
            },
            'status': {
                'prohibited': 'action_block',
                'state-invalid': 'action_block'
            }
        }
    },
    get_column_dict = function(col, entry) {
        var log_dict = log_columns_dict[log_type];
        return log_dict && log_dict[col];
    },
    gen_column_content = function(callback) {
        return function(name, row, td) {
            var dict;
            if (td && (dict = get_column_dict(name, row))) {
                return callback(name, row, dict);
            }
            return row[name];
        }
    };

/**
 * Formatting callback for log entries
 * @callback formattingCallback
 * @param  {String} name The selector/key of the column
 * @param  {Object} row The source data of the row
 * @param  {Boolean} [content] If result will be used in displaying the qlist
 * @return {String|jQuery}
 */

//map of smart columns created below:
var smartColumnMap = {
    'app': ['dstport', 'proto', 'appid'],
    'dstip': ['dstport', 'dstcountry', 'dstip_hostname'],
    'srcip': ['srcmac', 'byod_name', 'byod_device', 'user']
};

//transform into a map from column selector to smart column selector
var scKeys = Object.keys(smartColumnMap);
scKeys.forEach(function(k) {
    smartColumnMap[k].forEach(function(col) {
        smartColumnMap[col] = k;
    });
    smartColumnMap[k] = k;
});

/**
 * Map of nonstandard format functions for displaying values in the log.
 * Used in the details window, qlist format_fn, and the archive data viewer
 * @type {Object.<string, formattingCallback>}
 */
var formatter = {
    '_is_archived': function(name, row, content) {
        if (!content) {
            return null;
        }
        var value = row[name];
        if (value) {
            content.addClass('has_attachment').prop('title', $.getInfo('archive'));
        }
        return '';
    },
    'pri': function(name, row, content) {
        var value = row[name];
        var sev_class = 'log_severity_' + value;
        var sev = '';
        if (!content) {
            sev = value + ' ';
        }
        sev += '<span class="log_severity_level ';
        sev += sev_class + '" title="' + value + '"></span>&nbsp;';
        return sev;
    },
    'severity': function(name, row, content) {
        var value = row[name];
        var sev_class = 'sig_severity_' + value;
        var sev = '';
        if (!content) {
            sev = value + ' ';
        }
        sev += '<span class="sig_severity_level ';
        sev += sev_class + '" title="' + value + '"></span>';
        return sev;
    },
    'apprisk': function(name, row, content) {
        var value = row[name];
        return fweb.util.templates.risk_format()(value);
    },
    'app': function(name, row) {
        var value = row[name];
        var id = row['appid'] || 0;
        var app_data = '';
        var address = row['dstip'];
        var dport = row['dstport'];
        var proto = row['proto'];
        var service = row.service;

        if (!value) {
            if (service) {
                value = service.toUpperCase();
            } else {
                value = $j.getInfo('unknown');
            }
        }

        // The destination IP can be pre-resolved by the log extension to
        // show reverse DNS lookups. If the lookup is successful, the
        // 'dstip' field will look like:-
        //
        //     "184.150.182.158 (apis.google.com)"
        //
        // In this case, we want only the IP portion for the FortiFlow query.
        var address_host_idx = address ? address.indexOf(' ') : -1;
        if (address_host_idx != -1) {
            address = address.substring(0, address_host_idx);
        }

        // Add extra data attributes to support FortiFlow queries for
        // unknown applications.
        if (address && dport && proto) {
            app_data = 'data-address="' + address + '" ' +
                       'data-dport="' + dport + '" ' +
                       'data-protocol="' + proto + '" ';
        }

        value = '<span class="tooltip id_' + id + '" ' + app_data + '>' +
                '<span class="app_icon app' + id + '"/>' +
                '<label class="app_label">' + value + '</label></span>';
        return value;
    },
    'appcat': function(name, row) {
        var value = row[name];
        if (value) {
            return fgd_common.gen_app_cat_html(value);
        }
        return '';
    },
    'attack': function(name, row) {
        var value = row[name] || $j.getInfo("unknown");
        var id = row['attackid'] || 0;

        value = '<span class="tooltip id_' + id + '">' +
                '<label class="app_label">' + value + '</label></span>';
        return value;
    },
    'ref': function(name, row) {
        var value = row[name];
        if (value) {
            return '<a target="_blank" href="' + value +'">' + value + '</a>';
        } else {
            return '';
        }
    },
    'vulnref': function(name, row) {
        var value = row[name];
        return '<a target="_blank" href="' + value +'">' + value + '</a>';
    },
    'vuln': function(name, row) {
        var value = row[name];
        var ref = row.vulnref;
        if (ref) {
            return '<a target="_blank" href="' + ref +'">' + value + '</a>';
        }
        return value;
    },
    'virus': function(name, row) {
        var value = row[name];
        var ref = row.ref;
        if (ref) {
            return '<a target="_blank" href="' + ref +'">' + value + '</a>';
        }
        return value;
    },
    'rel_time': function(name, row, content) {
        var value = row[name];
        var dt = fweb.util.localDateSeconds(value);
        // TODO: see why the FortiGate is using local time instead of UTC
        var result = $j.format_date_diff(dt);
        if (!content) {
            result += ' (' + value + ')';
        }

        return '<span title="' + dt.toLocaleString() + '">' + result + '</span>';
    },
    'timestamp': function(name, row) {
        return fweb.util.localDateSeconds(row[name]).toLocaleString();
    },
    'srcip': function(name, row) {
        var values = [], value = row[name];
        var key, keys = ['user', 'unauthuser'];
        var mac = row['srcmac'];

        while (key = keys.shift()) {
            if (row[key]) { break; }
        }
        var id = row['byod_name'];
        if (mac || id) {
            values.push(byod_common.gen_fw_device_icon(mac, row['byod_device'], key ? row[key] : id));
        } else if (key) {
            values.push(formatter['user'](key, row));
        }
        values.push(values.length ? ' (' + value + ')' : value);
        return values.join(' ');
    },
    'dstip': function(name, row) {
        var value = row[name];
        var country = row['dstcountry'];
        var code = row['dstcountry_code'];
        var hostname_attr = '_' + name + '_hostname';
        var hostname = row[hostname_attr];

        if (code)
            value = '<span class="country_flag country_' + code + '" title="' + country + '"></span> ' + value + (hostname ? ' (' + hostname + ')' : '');
        return value;
    },
    'srcmac': function(name, row) {
        var mac = row[name];
        var device = row['byod_device'];
        name = row['byod_name'];
        if (mac || name) {
            return byod_common.gen_fw_device_icon(mac, device, name);
        }
    },
    'url': function(name, row) {
        var value = row[name] || '';
        if ('hostname' in row) {
            var host = row.hostname;
            if (host && host.toLowerCase() != 'n/a') {
                value = host + value;
            }
        }
        return truncate_comment(value, 64);
    },
    'proto': function(name, row) {
        if ('protocol' in row) {
            return row.protocol;
        }
        return row[name];
    },
    'protocol': function(name, row) {
        if ('proto' in row) {
            return row.proto;
        }
        return row[name];
    },
    'av_details': function(name, row) {
        var service = row.service;
        if (service) {
            if (service.indexOf('smtp') != -1) {
                return 'to: ' + escapeHTML(row.to);
            }
            if (service.indexOf('imap') != -1
                || service.indexOf('pop3') != -1) {
                return 'from: ' + escapeHTML(row.from);
            }
            if (service.indexOf('http') != -1) {
                return 'url: ' + escapeHTML(row.url);
            }
        }
        return 'host: ' + row.dstip;
    },
    'sent_rcvd': function(name, row) {
        var value = '';
        if (row.sentbyte && row.rcvdbyte) {
            value = convertNumberToUnits(row.sentbyte) + ' / ' +
                convertNumberToUnits(row.rcvdbyte);
        }
        return value;
    },
    'size': function(name, row) {
        var value = Number(row[name]);
        return convertNumberToUnits(value);
    },
    'action': function(name, row) {
        var value = row[name];
        // TODO: Check if need to format with icon
        // and aggregate icon for each action
        // since each utm log has different value,
        // better consistant with UTM Profile
        // format for forward log, action refers to firewall action
        // format for forward utm logs, action refers to utm action
        return value;
    },
    'user': function(name, row) {
        var value = escapeHTML(row[name]);
        if (value) {
            value = byod_common.gen_fw_icon('device_' + name, value);
        }
        return value;
    },
    'msg-type': gen_column_content(function(name, row, dict) {
        var val = row[name], ver = row['version'];
        if (!dict[ver]) { return val; }
        return dict[ver][val] || dict[0][val] || val;
    }),
    'status': gen_column_content(function(name, row, dict) {
        var val = row[name], cls = dict[val];
        return cls ? '<span class="action_sprite ' + cls + '" title="' + val + '">'
            + $j.getInfo(val) + '</span>' : val;
    }),
    'utmaction': gen_column_content(function(name, row, dict) {
        var val = row[name], cls = dict[val];
        return cls ? '<span class="action_sprite ' + cls + ' utm_action_icon"'
            + ' title="' + val + '">' + $j.getInfo('utmaction_' + val) + '</span>' : val;
    }),
    'threat': function(name, row) {
        var value = row[name],
            html = '';
        for(var i=0, len=value.length; i < len; i++) {
            html += format_smart_col(row, value[i]);
        }
        return html;
    },
    'countav': function(name, row) {
        var tab = name + '-tab';
        return '<a href="#" class="threat_details" data-tab="' +
               tab + '" data-utmref="' + row['utmref'] +
               '">' + row[name] + '</a>';
    },
    'app_details': function(name, row) {
        return format_smart_col(row, row[name]);
    },
    'checksum': function(name, row) {
        return (row[name] || '').toString(16);
    },
    'catdesc': function(name, row) {
        var value = escapeHTML(row[name]);
        return value;
    },
    'clouddetails': function(name, row) {
        var value = '';

        /* Parse application "action" from application "name" field, i.e.:-
         *
         *   "Dropbox_File.Download" => "File Download"
         *
         */
        if (row.app && row.app.indexOf('_') !== -1) {
            var app_segments = row.app.split('_');
            var app_name = app_segments[app_segments.length-1];

            value += app_name.replace('.', ' ');
        }

        /* Add the filename / video name / etc */
        if (row.filename) {
            if (value) {
                value += ': ';
            }
            value += row.filename;
        }
        return value;
    },
    // For FortiView Threat last level Drilldown
    // back import from Widgets.js
    // TODO: Reduce redundancy
    'threatlevel': function(name, row) {
        var level = level_idmap[row[name]];
        if (typeof level === 'undefined') {
            return '';
        }
        return ["<span class='legend ", level, "'>",
                $j.getInfo(level), "</span>"].join('');
    },
    // TODO: need to reexamine and back import from Widgets.js
    //       to show the icon
   'threattype': function(name, row) {
        return $j.getInfo(row[name]);
    },
    // TODO: need to reexamine and back import from Widgets.js
    //       to show the icon
   'threatname': function(name, row) {
        return $j.getInfo(row[name]);
    },
    'filesize': function(name, row) {
        return '<div title="' + row[name] + ' ' + $.getInfo('bytes') +'">' +
            convertNumberToUnits(row[name]) + '</div>';
    },
    'false_positive_mitigation': function(name, row, content) {
	if(content)
		content[0].style.textAlign = 'center';
	var val = row[name] ? row[name] : '';
	var html = '';
	if(val == 'yes')
		html = "<span style='vertical-align: bottom;' ><img src='/images/status_enabled.png'>&nbsp;Enabled</span>";
	else if(val == 'no')
		html = "<span style='vertical-align: bottom;'><img src='/images/status_disabled.png' />&nbsp;Disabled</span>";
	else
		html = "N/A";

        return html;
    },
    'log_type': function(name, row, content) {
	if(content)
		content[0].style.textAlign = 'center';
	var val = row[name] ? row[name] : '';
	var html = '';
	if(val == 'LOG_TYPE_SCORE_SUM')
		html = "<span style='vertical-align: bottom;'><img src='/images/status_sum.png' style='width:16px;height:16px;' title='"+$j.getInfo('yes')+"'>"+$j.getInfo('yes')+"</span>";

        return html;
    },
    'http_retcode': function(name, row) {
	var html = '';
	if(row[name] != '0')	html = row[name];
	return html;
    },
    'src': function(name, row) {
	var ip = row[name] ? row[name]:'';
	var flag = row['country_flag'] ? row['country_flag']:'';
	var html = '<span class="country_flag country_'+flag+'"></span>&nbsp;<span>'+ip+'</span>';
	return html;
    },
    'original_src': function(name, row) {
	var ip = row[name] ? row[name]:'';
	var flag = row['ori_country_flag'] ? row['ori_country_flag']:'';
	var html = '<span class="country_flag country_' + flag + '"></span>&nbsp;<span>' + ip + '</span>';
	return html;
    },
    'threat_level': function(name, row, content) {
	    var tip_value = row[name]?row[name]:'Off';
	    var value = row[name]?row[name].toLowerCase():'information';
	    if(row[name].toLowerCase() == 'off')
		    value = 'information';
	    var sev_class = 'sig_severity_' + value;
	    var sev = '';
	    if (!content) {
		    sev = value + ' ';
	    }
	    sev += '<span class="sig_severity_level ';
	    sev += sev_class + '" title="' + tip_value + '"></span>&nbsp;';
	    return sev;
    },
    'flag': function(name, row, content) {
	    var value = row[name] ? row[name] : "None";
	    var val = value.toLowerCase().split(' ').join('_');
	    var cls = 'log_flag log_flag_' + val;
	    var html = ['<span class="', cls, '" title="', value, '"></span>' ].join('');
	    return html;
    },
    'has_comment': function(name, row, content) {
	    var value = row[name] ? row[name] : "No Comment";
	    var html = '';
	    if(value == 'Has Comment')
		    html = '<span class="log_flag log_flag_comment" title="Has Comment"></span>';
	    return html;
    }
};

formatter['level'] = formatter['pri'];
formatter['start'] = formatter['rel_time'];
formatter['countweb'] = formatter['countav'];
formatter['countapp'] = formatter['countav'];
formatter['countips'] = formatter['countav'];
formatter['countdlp'] = formatter['countav'];
formatter['countemail'] = formatter['countav'];

function format_smart_col(row, obj) {
    var value = '';
    if (obj) {
        var cls = obj['cls'];
        var key = obj['key'];
        var label = obj['label'];
        var suffix = obj['suffix'];
        if (cls) {
            value += '<span class="margin_right icon_fw ' + cls + '">';
        }
        if (label) {
            value += '<span class="fw_icon_overlay">' + label + '</span>';
        }
        if (key in formatter) {
            value += formatter[key](key, row);
        } else {
            value += escapeHTML(row[key]);
        }
        if (suffix) {
            value += ' ' + $j.getInfo(suffix);
        }
        if (cls)
            value += '</span>';
    }
    return value;
}

return {
        smartColumnMap: smartColumnMap,
        getFormatter: function () {
            return formatter;
        },
        format_fn_to: function(td, col, entry) {
            return formatter[col.selector](col.selector, entry, td);
        }
    }
})(jQuery);

