/* global define, require */
define(['angular', 'jquery', 'fweb', 'ng/directives/d3_base', 'fweb.util/formatters'],
function(angular, $, fweb, D3Controller, formatters) {
    'use strict';

    function setupColorAndStroke(that, $scope) {
        var origColor = that.color;
        var origStroke = that.stroke;
        that.color = function() {
            if ($scope.colorFn) {
                return $scope.colorFn.apply(that, arguments);
            } else {
                return origColor.apply(that, arguments);
            }
        };
        that.stroke = function() {
            if ($scope.strokeFn) {
                return $scope.strokeFn.apply(that, arguments);
            } else {
                return origStroke.apply(that, arguments);
            }
        };
    }

    function D3BubbleController($scope, $element, $injector) {
        $injector.invoke(D3Controller, this, {$scope: $scope, $element: $element});
        this.init();
        this.ready.then(function() {
            setupColorAndStroke(this, $scope);
            this.bubble = this.d3.layout.pack()
            .sort(null)
            .size([this.width, this.height])
            .padding(2);
        }.bind(this));
    }

    D3BubbleController.prototype = Object.create(D3Controller.prototype);
    D3BubbleController.prototype.constructor = D3BubbleController;

    D3BubbleController.prototype.update = function(data) {
        this.bubble.size([this.width, this.height]);
        data = this.bubble.nodes({children: data}).filter(function(d) {
            return d.name && d.value && !d.children;
        });
        var node = this.svg.selectAll('.node').data(data, function(d) {
            return d.name;
        });

        var enter = node.enter()
        .append('g')
        .attr('class', 'node')
        .attr('transform', function(d) {
            return 'translate(' + d.x + ',' + d.y + ')';
        });

        enter.append('circle').attr('r', function(d) {
            return d.r;
        }).style({
            'fill': function(d) {
                return this.color(d.name, d);
            }.bind(this),
            'stroke': function(d) {
                return this.stroke(d.name, d);
            }.bind(this),
            'cursor': this.clickSupported ? 'pointer' : 'auto'
        }).on('click', this.clickFn)
        .each(this.tooltipBindFn);

        enter.append('text')
        .attr('dy', '.3em')
        .attr('text-anchor', 'middle')
        .style('cursor', this.clickSupported ? 'pointer' : 'auto')
        .on('click', this.clickFn)
        .text(function(d) {
            var label = String(d.name).substring(0, (d.r / 3) - 2);
            if (label.length <= 2 && d.name.length > 2) {
                label = '';
            }
            return label;
        }).each(this.tooltipBindFn);

        node.select('circle')
        .each(this.tooltipBindFn)
        .transition().duration(1000)
        .attr('r', function(d) {
            return d.r;
        }).style({
            'fill': function(d) {
                return this.color(d.name, d);
            }.bind(this),
            'stroke': function(d) {
                return this.stroke(d.name, d);
            }.bind(this),
            'cursor': this.clickSupported ? 'pointer' : 'auto'
        });

        node.select('text').text(function(d) {
            var label = String(d.name).substring(0, (d.r / 3 - 2));
            if (label.length <= 2 && d.name.length > 2) {
                label = '';
            }
            return label;
        }).each(this.tooltipBindFn);

        node.transition().attr('class', 'node')
        .attr('transform', function(d) {
            return 'translate(' + d.x + ',' + d.y + ')';
        });

        node.exit().remove();
    };

    /**
     * @ngdoc directive
     * @name ng.directive:fD3BasicBubbleChart
     *
     * @description
     * This directive will render a basic one level bubble chart with d3.
     * Arguments (* => required):
     *   * "data": An array of objects where each object must have the properties "name" and
     *       "value".
     *     "click": An function that will be called when a circle is clicked with the respective
     *       entry.
     *     "tooltipFormatter": A function that will be called when a circle is hovered over with the
     *       respective entry. Must return tooltip content.
     *     "colorFn": A color override function. Will be provided the value and the entry.
     *     "strokeFn": A stroke color override function. Will be provided the value and the entry.
     *
     */
    var d3BasicBubbleChart = function() {
        return {
            restrict: 'A',
            scope: {
                data: '=',
                click: '=',
                tooltipFormatter: '=',
                colorFn: '=',
                strokeFn: '='
            },
            controller: D3BubbleController
        };
    };

    function D3CountryController($scope, $element, $injector, $http, $templateCache, $compile,
                                 loader) {
        $injector.invoke(D3Controller, this, {$scope: $scope, $element: $element});
        var countriesJSON = loader.base_path('/geo/countries.json');
        var legendTemplate = loader.base_path('/ng/directives/d3_country_legend.html');
        this.promises.push($http.get(countriesJSON, {cache: true}).then(function(result) {
            this.geoData = result.data;
        }.bind(this)));
        this.promises.push($http.get(legendTemplate, {cache: $templateCache})
        .then(function(template) {
            this.$legend = $compile(template.data)(this.$scope).appendTo(this.$element);
        }.bind(this)));

        this.$scope.legendFormatter = function(value) {
            var result = '';
            if (value) {
                if (angular.isFunction(this.$scope.legendValueFormatter)) {
                    result = this.$scope.legendValueFormatter(value);
                } else {
                    result = value;
                }
            }
            return result;
        }.bind(this);

        // Map to handle the mismatched country names between geographic data and our own internal
        // representations
        this.COUNTRY_MAP = {
            'Bosnia and Herz.': 'Bosnia and Herzegovina',
            'Central African Rep.': 'Central African Republic',
            'Czech Rep.': 'Czech Republic',
            'Dominican Rep.': 'Dominican Republic',
            'Dem. Rep. Congo': 'Congo, The Democratic Republic of the',
            'Eq. Guinea': 'Equatorial Guinea',
            'Falkland Is.': 'Falkland Islands (Malvinas)',
            'Iran': 'Iran, Islamic Republic of',
            'Korea': 'Korea, Republic of',
            'Libya': 'Libyan Arab Jamahiriya',
            'Moldova': 'Moldova, Republic of',
            'Russia': 'Russian Federation',
            'Syria': 'Syrian Arab Republic',
            'S. Sudan': 'South Sudan',
            'Tanzania': 'Tanzania, United Republic of'
        };
        // Dont complain about missing the following. Reasons could be:
        //   - Not a country (continents, cities, overseas regions, etc..)
        //   - Too small to be included in geographic data
        this.COUNTRY_EXEMPT = {
            'Aland Islands': true,
            'American Samoa': true,
            'Andorra': true,
            'Anguilla': true,
            'Anonymous Proxy': true,
            'Antarctica': true,
            'Antigua and Barbuda': true,
            'Aruba': true,
            'Asia/Pacific Region': true,
            'Bahrain': true,
            'Barbados': true,
            'Bermuda': true,
            'Bonaire, Saint Eustatius and Saba': true,
            'Brunei Darussalam': true,
            'Cape Verde': true,
            'Cayman Islands': true,
            'Christmas Island': true,
            'Cook Islands': true,
            'Curacao': true,
            'Dominica': true,
            'Europe': true,
            'Gibraltar': true,
            'Guadeloupe': true,
            'Guam': true,
            'Guernsey': true,
            'Grenada': true,
            'Faroe Islands': true,
            'French Guiana': true,
            'French Polynesia': true,
            'French Southern Territories': true,
            'Hong Kong': true,
            'Isle of Man': true,
            'Jersey': true,
            'Liechtenstein': true,
            'Macao': true,
            'Maldives': true,
            'Malta': true,
            'Mauritius': true,
            'Martinique': true,
            'Marshall Islands': true,
            'Mayotte': true,
            'Micronesia, Federated States of': true,
            'Monaco': true,
            'Montserrat': true,
            'Nauru': true,
            'Niue': true,
            'Norfolk Island': true,
            'Northern Mariana Islands': true,
            'Palau': true,
            'Palestinian Territory': true,
            'Pitcairn': true,
            'Reserved': true,
            'Reunion': true,
            'Saint Bartelemey': true,
            'Saint Helena': true,
            'Saint Kitts and Nevis': true,
            'Saint Martin': true,
            'Saint Pierre and Miquelon': true,
            'Saint Lucia': true,
            'Saint Vincent and the Grenadines': true,
            'Samoa': true,
            'San Marino': true,
            'Sao Tome and Principe': true,
            'Seychelles': true,
            'Singapore': true,
            'Sint Maarten': true,
            'Solomon Islands': true,
            'South Georgia and the South Sandwich Islands': true,
            'Svalbard and Jan Mayen': true,
            'Tokelau': true,
            'Tonga': true,
            'Turks and Caicos Islands': true,
            'Tuvalu': true,
            'United States Minor Outlying Islands': true,
            'Virgin Islands, British': true,
            'Virgin Islands, U.S.': true,
            'Wallis and Futuna': true
        };
        // Keep track of those "countries" we've already warned about
        this.country_warned = {};
        this.scale = 1;
        this.translation = [0, 0];

        // path.centroid gives country centers in the middle of the ocean for some countries.
        var countryCentersJSON = loader.base_path('/geo/country_centers.json');
        this.promises.push($http.get(countryCentersJSON, {cache: true}).then(function(result) {
            this.countryCenters = result.data;
        }.bind(this)));

        this.init();
        this.ready = this.ready.then(function() {
            this.svg = this.svg.append('g');
            // Must keep colors in sync with legend styling in ngstyl.styl
            this.colorScale = this.d3.scale.linear().range(['#FFEF40', '#C7311A']);
            this.color = function(value) {
                if (value === 0) {
                    return '#D9D5D4';
                } else {
                    return this.colorScale(value);
                }
            }.bind(this);
        }.bind(this));
    }

    D3CountryController.prototype = Object.create(D3Controller.prototype);
    D3CountryController.prototype.constructor = D3CountryController;

    D3CountryController.prototype.requireComponents = function() {
        return D3Controller.prototype.requireComponents.call(this, {
            topojson: 'js/topojson'
        });
    };

    D3CountryController.prototype.createPath = function(features) {
        var scale, translate;
        var topLeft = {x: null, y: null};
        var bottomRight = {x: null, y: null};

        // Create unit projection or hidden orthographic projection
        var projection = this.egg ? this.d3.geo.orthographic().scale(1.2).translate([0, 0])
                .clipAngle(90).rotate([100, 0]).precision(0.1) :
            this.d3.geo.mercator().scale(this.scale).translate(this.translation);
        var path = this.d3.geo.path().projection(projection);

        // Calculate the bounds of the map features
        features.forEach(function(feature) {
            var value;
            var bounds = path.bounds(feature);

            // Top left x
            value = bounds[0][0];
            if (value == null || value < topLeft.x) {
                topLeft.x = value;
            }
            // Top left y
            value = bounds[0][1];
            if (value == null || value < topLeft.y) {
                topLeft.y = value;
            }
            // Bottom right x
            value = bounds[1][0];
            if (value == null || value > bottomRight.x) {
                bottomRight.x = value;
            }
            // Bottom right y
            value = bounds[1][1];
            if (value == null || value > bottomRight.y) {
                bottomRight.y = value;
            }
            if (this.countryCenters && this.countryCenters[feature.properties.iso_a2]) {
                feature.center = [this.countryCenters[feature.properties.iso_a2].long,
                this.countryCenters[feature.properties.iso_a2].lat];
            }
        }.bind(this));

        // Calculate scale with a 5% margin
        scale = 0.95 / Math.max(
            (bottomRight.x - topLeft.x) / this.width,
            (bottomRight.y - topLeft.x) / this.height
        );
        // Center map
        translate = [
            (this.width - scale * (topLeft.x + bottomRight.x)) / 2,
            (this.height - scale * (topLeft.y + bottomRight.y)) / 2
        ];

        projection.scale(scale).translate(translate);

        return path;
    };

    D3CountryController.prototype.update = function(data) {
        var features = this.topojson.feature(this.geoData, this.geoData.objects.countries).features;
        var path = this.createPath(features);
        var hit = {};

        this.colorScale.domain([0, data._meta.sum]);
        this.$scope.minNonZeroValue = data._meta.sum === data._meta.minNonZero ? null :
            data._meta.minNonZero;
        this.$scope.sumValue = data._meta.sum;

        var getStats = function(d) {
            var name = d.properties.name;
            name = this.COUNTRY_MAP[name] || name;
            //FWB_CHANGE var value = {name: name, country: name};
            var value = {name: name, srccountry: name};
            if (data[name]) {
                value = data[name];
                hit[name] = true;
            }
            return value;
        }.bind(this);

        var fillFn = function(d) {
            var stats = getStats(d);
            return this.color(stats.value || 0);
        }.bind(this);

        this.tooltipDataGetter = getStats;
        this.tooltipColorKeyGetter = function(d) {
            var name = d.properties.name;
            return data[name] ? data[name].value : 0;
        };

        if (!this.rendered) {
            this.svg.selectAll('path')
            .data(features, function(d) {
                return d.properties.name;
            }).enter().append('path')
            .attr('class', 'country')
            .attr('d', path)
            .style({
                'fill': fillFn,
                'cursor': this.clickSupported ? 'pointer' : 'auto'
            }).on('click', function(d) {
                this.clickFn(getStats(d));
            }.bind(this))
            .each(this.tooltipBindFn);
            this.rendered = true;
        } else {
            this.svg.selectAll('path')
            .attr('d', path)
            .transition().duration(1000)
            .style({
                'fill': fillFn,
                'cursor': this.clickSupported ? 'pointer' : 'auto'
            });
        }

        // Report any countries with mismatched names
        Object.keys(data).forEach(function(country) {
            if (country !== '_meta' && !hit[country] && !this.COUNTRY_EXEMPT[country] &&
                !this.country_warned[country]) {
                fweb.log('Country "' + country + '" doesn\'t exist on map');
                this.country_warned[country] = true;
            }
        }.bind(this));
    };

    /**
     * @ngdoc directive
     * @name ng.directive:fD3CountryChart
     *
     * @description
     * This directive will render a country chart map with d3.
     * Arguments (* => required):
     *   * "data": An array of objects where each object must have the properties "name" and
     *       "value".
     *     "click": An function that will be called when a circle is clicked with the respective
     *       entry.
     *     "tooltipFormatter": A function that will be called when a circle is hovered over with the
     *       respective entry. Must return tooltip content.
     *     "legendValueFormatter": A function to be used for formatting legend values.
     *
     */
    var d3CountryChart = function() {
        return {
            restrict: 'A',
            scope: {
                data: '=',
                click: '=',
                tooltipFormatter: '=',
                legendValueFormatter: '='
            },
            controller: D3CountryController
        };
    };

    function D3ThreatMapController($scope, $element, $injector, $interval, shortcuts) {
        $injector.invoke(D3CountryController, this, {$scope: $scope, $element: $element});
        this.$interval = $interval;

        this.scale = 0.8;
        this.translation = [0, 0.5];
        this.attackData = [];
        this.countryMap = null;
        this.countryHeat = {};
        this.resizeTransition = null;
        this.countryColorRange = ['#36424A', '#7D7E7E'];
        this.threatLevelNames = ['Off', 'Low', 'Medium', 'High', 'Critical'];
        this.threatLevelColors = ['#c0c0c0', '#62a1d7', '#faffa5', '#ffca5c', '#f32e2b'];

        this.api = this.$scope.api = this.$scope.api || {};
        this.api.attackData = this.attackData;
        this.api.processThreats = this.processThreats.bind(this);


        shortcuts.subscribe(shortcuts.SHORTCUTS.CONTRA, function() {
            this.svg.selectAll('*').remove();
            this.egg = true;
            this.rendered = false;
            this.update(this.$scope.data);
        }.bind(this), 'shortcut::easter_egg');
    }

    D3ThreatMapController.prototype = Object.create(D3CountryController.prototype);

    D3ThreatMapController.prototype.update = function(data) {
        var features = this.topojson.feature(this.geoData, this.geoData.objects.countries).features;
        this.countryMap = features.reduce(function(map, f) {
            map[f.properties.iso_a2] = f;
            return map;
        }, {});
        var path = this.path = this.createPath(features);

        if (!data.device) {
            data.device = {latitude: 0, longitude: 0};
        }
        this.$fortigate = $(data.deviceElement);
        this.setupFortigateDragging(data.device);

        var fillFn = function(d) {
            var countryColorFn = this.d3.scale.linear().range(this.countryColorRange);
            return countryColorFn(this.countryHeat[d.properties.iso_a2] || 0);
        }.bind(this);

        if (!this.rendered) {
            this.svg.selectAll('path')
                .data(features, function(d) {
                    return d.properties.name;
                }).enter().append('path')
                .attr('class', 'country')
                .attr('d', path)
                .style({
                    'fill': fillFn,
                    'stroke-opacity': 0.7
                });
            this.redrawNightMap = this.renderNightMap();

            // Setup country threat heat animation that update on an interval
            var updateLocked = false;
            var updateInterval = this.$interval(function() {
                // Prevent updates from queueing when tab in background by locking until the update
                // function is actually run (d3 timer runs when active) and the lock is released.
                if (!updateLocked) {
                    updateLocked = true;
                    this.d3.timer(function() {
                        // chain after existing resizeTransition or target all paths if not resizing
                        // (otherwise resize transition will be interrupted).
                        var target = this.resizeTransition || this.svg.selectAll('path.country');
                        target.transition().ease('linear').duration(10000)
                            .style({'fill': fillFn});
                        this.resizeTransition = null;
                        updateLocked = false;
                        return true;
                    }.bind(this));
                }
            }.bind(this), 1000);

            if (this.egg) {
                var rotate = [71.03, 0.37],
                    velocity = [0.018, -0.00006];
                this.d3.timer(function(elapsed) {
                    if (!this.egg) {
                        return true;
                    }
                    var projection = this.path.projection();
                    projection.rotate([rotate[0] + elapsed * velocity[0],
                                       rotate[1] + elapsed * velocity[1]]);
                    var position = projection([data.device.longitude,
                                               data.device.latitude]);
                    var visible = !!this.path({ type: 'MultiPoint',
                            coordinates: [[data.device.longitude,
                            data.device.latitude]]});
                    this.$fortigate.toggleClass('clipped', !visible);
                    this.$fortigate.css({
                        left: position[0],
                        top: position[1]
                    });
                    this.svg.selectAll('path').attr('d', this.path);
                }.bind(this));
            }
            this.$scope.$on('$destroy', function() {
                this.$interval.cancel(updateInterval);
                this.egg = false;
            }.bind(this));
            this.rendered = true;
        } else {
            this.resizeTransition = this.svg.selectAll('path.country')
                .style({'fill': fillFn})
                .transition().duration(1000)
                .attr('d', path);
            this.redrawNightMap();
        }
    };

    D3ThreatMapController.prototype.setupFortigateDragging = function(device) {
        var position = this.path.projection()([device.longitude, device.latitude]);
        var maxHeight = this.height - this.$fortigate.height();
        this.$fortigate.css({
            left: position[0],
            top: Math.max(Math.min(position[1], maxHeight), 0)
        });
        if (device.draggable) {
            this.$fortigate.draggable();
            this.$fortigate.off('drag').on('drag', function(e, ui) {
                var svgBounds = this.svg.node().getBBox();

                // Constrain left projected features group bounding box
                ui.position.left = Math.max(svgBounds.x + 10, ui.position.left);
                ui.position.left = Math.min(svgBounds.x + svgBounds.width - 10, ui.position.left);
                // Contrain top from 0 to height of the SVG element.
                ui.position.top = Math.max(0, ui.position.top);
                ui.position.top = Math.min(maxHeight, ui.position.top);
            }.bind(this));
            this.$fortigate.off('dragstop').on('dragstop', function(e, ui) {
                if (this.path.projection().invert) {
                    var position = [ui.position.left, ui.position.top];
                    var coords = this.path.projection().invert(position);
                    device.latitude = coords[1];
                    device.longitude = coords[0];
                }
            }.bind(this));
        }
    };

    D3ThreatMapController.prototype.processLogLines = function() {
        var loglines = this.d3.select('#threatlog table').selectAll('tr.log')
            .data(this.filteredData, function(d) {
                return d.threatid;
            });

        // Create new threat lines at the bottom of the page/svg
        var line = loglines.enter()
            .append('tr')
            .attr('class', 'log')
            .style('color', function(d) {
                return this.threatLevelColors[d.level];
            }.bind(this));

        line.append('td')
            .attr('class', 'flag')
            .html(function(d) {
                return '<span class="country_flag country_' + d.countrycode + '"></span>';
            }.bind(this));
        line.append('td')
            .attr('class', 'location')
            .text(function(d) {
                var country = this.countryMap[d.countrycode];
                return country.properties.name;
            }.bind(this));
        line.append('td')
            .attr('class', 'threat')
            .text(function(d) {
                return $.getInfo(d.threat);
            });
        line.append('td')
            .attr('class', 'level')
            .text(function(d) {
                return $.getInfo(this.threatLevelNames[d.level]) + ' (' + d.score + ')';
            }.bind(this));
        line.append('td')
            .attr('class', 'time')
	    .text(function(d) {
		    var dt = fweb.util.datetime.localSecondsToDate(d.time);
		    return dt.toLocaleString();
		    //    var formattedTime = new Date(d.time * 1000).toLocaleString();
            //    return formattedTime;
            });

        var threatlog = this.d3.select('#threatlog .logs').node();
        threatlog.scrollTop = threatlog.scrollHeight;

        // Remove older lines
        loglines.exit().remove();
    };

    D3ThreatMapController.prototype.processThreats = function() {
        var that = this;
        var getFortigatePosition = function() {
            var position = that.$fortigate.position();
            return [position.left, position.top];
        };

        this.filteredData = this.attackData.filter(function(d) {
            d.countrycode = (!!that.countryMap[d.src_country]) ? d.src_country :
                                d.dst_country;
            return (!!that.countryMap[d.countrycode]);
        });

	if(this.filteredData.length > 200)
	{
		this.attackData.shift();
		this.filteredData.shift();
	}

	this.d3.select('#threatlog table').selectAll('tr.log')
		.data(this.filteredData, function(d) {
			return d.threatid;
		}).exit().remove();
	this.svg.selectAll('g.threat').data(this.filteredData, function(d){
	return d.threatid}).exit().remove();

        this.processLogLines();
        var groups = this.svg.selectAll('g.threat')
            .data(this.filteredData)
            .enter()
            .append('g')
            .attr('class', 'threat');

        groups.attr('transform', function(d) {
            var country = this.countryMap[d.countrycode];
            this.countryHeat[d.countrycode] = (this.countryHeat[d.countrycode] || 0) + 1;
            d.origin = (country.center) ? this.path.projection()(country.center) :
                                          this.path.centroid(country);
            return 'translate(' + d.origin + ')';
        }.bind(this))
        .transition().ease('linear').duration(2000)  // animate threat from origin country
        .attr('transform', function(d) {
            // To avoid recalculating in other transforms, keep values in data object state.
            d.fortigatePos = getFortigatePosition();

            // From fortigate to origin country
            d.slopeX = d.origin[0] - d.fortigatePos[0];
            d.slopeY = d.origin[1] - d.fortigatePos[1];

            // Normalize vector to unit length, and resize to end position radius
            d.length = Math.sqrt((d.slopeX * d.slopeX) + (d.slopeY * d.slopeY));
            d.fgtVector = [(d.slopeX / d.length), (d.slopeY / d.length)];

            // Stop 20 pixels away from the center of the fortigate
            var finalVector = [d.fortigatePos[0] + (d.fgtVector[0] * 20),
                               d.fortigatePos[1] + (d.fgtVector[1] * 20)];

            return 'translate(' + finalVector + ')';
        })
        .transition().ease('linear').duration(1000)
        .attr('transform', function(d) { // Bounce back animation.
            var finalVector = [d.fortigatePos[0] + (d.fgtVector[0] * 100),
                               d.fortigatePos[1] + (d.fgtVector[1] * 100)];
            return 'translate(' + finalVector + ')';
        }).transition().delay(10000).each('end', function(d) {
            that.countryHeat[d.countrycode] = (that.countryHeat[d.countrycode] || 1) - 1;
            // Remove children, but keep group to avoid another animation for this data entry
            that.d3.select(this).selectAll('*').remove();
        });

        groups.append('circle')
            .attr('stroke', function(d) {
                return this.threatLevelColors[d.level];
            }.bind(this))
            .attr('fill', function(d) {
                return this.threatLevelColors[d.level];
            }.bind(this))
            .attr('r', 1.5)
            .transition().delay(2000).duration(1000)
                .attr('fill', 'none')
                .attr('r', 20)
                .style('opacity', 0)
                .style('stroke-width', 5);

        // Threats have tails while travelling to the fortigate, length proportional to distance
        groups.append('line')
            .attr('stroke', function(d) {
                return this.threatLevelColors[d.level];
            }.bind(this))
            .attr('stroke-width', 2)
            .attr('x1', 0).attr('y1', 0)
            .attr('x2', 0).attr('y2', 0)
            .transition().ease('linear').duration(200)
            .attr('x2', function(d) {
                return d.fgtVector[0] * (d.length / 10);
            })
            .attr('y2', function(d) {
                return d.fgtVector[1] * (d.length / 10);
            })
            .transition().delay(2000).ease('linear').duration(1)
            .attr('x2', 0)
            .attr('y2', 0);
        // groups.exit().remove();
    };

    D3ThreatMapController.prototype.renderNightMap = function() {
        var radians = Math.PI / 180,
            degrees = 180 / Math.PI,
            that = this;
        var circle = this.d3.geo.circle()
            .angle(90);

        var night = this.svg.append('path')
            .attr('class', 'night')
            .style({
                'stroke': 'black',
                'fill': 'black',
                'fill-opacity': '.3',
                'stroke-opacity': '.2'
            })
            .attr('d', this.path);

        var redraw = function() {
            night.datum(circle.origin(antipode(solarPosition(new Date()))))
                .attr('d', this.path);
        }.bind(this);
        redraw();

        function antipode(position) {
            return [position[0] + 180, -position[1]];
        }

        function solarPosition(time) {
            var centuries = (time - Date.UTC(2000, 0, 1, 12)) / 864e5 / 36525, // since J2000
                longitude = (that.d3.time.day.utc.floor(time) - time) / 864e5 * 360 - 180;
            return [
                longitude - equationOfTime(centuries) * degrees,
                solarDeclination(centuries) * degrees
            ];
        }

        // Equations based on NOAA’s Solar Calculator; all angles in radians.
        // http://www.esrl.noaa.gov/gmd/grad/solcalc/

        function equationOfTime(centuries) {
            var e = eccentricityEarthOrbit(centuries),
                m = solarGeometricMeanAnomaly(centuries),
                l = solarGeometricMeanLongitude(centuries),
                y = Math.tan(obliquityCorrection(centuries) / 2);
            y *= y;
            return y * Math.sin(2 * l) - 2 * e * Math.sin(m) + 4 * e * y * Math.sin(m) *
                    Math.cos(2 * l) - 0.5 * y * y * Math.sin(4 * l) - 1.25 * e * e *
                    Math.sin(2 * m);
        }

        function solarDeclination(centuries) {
            return Math.asin(Math.sin(obliquityCorrection(centuries)) *
                Math.sin(solarApparentLongitude(centuries)));
        }

        function solarApparentLongitude(centuries) {
            return solarTrueLongitude(centuries) - (0.00569 + 0.00478 * Math.sin((125.04 -
                1934.136 * centuries) * radians)) * radians;
        }

        function solarTrueLongitude(centuries) {
            return solarGeometricMeanLongitude(centuries) + solarEquationOfCenter(centuries);
        }

        function solarGeometricMeanAnomaly(centuries) {
            return (357.52911 + centuries * (35999.05029 - 0.0001537 * centuries)) * radians;
        }

        function solarGeometricMeanLongitude(centuries) {
            var l = (280.46646 + centuries * (36000.76983 + centuries * 0.0003032)) % 360;
            return (l < 0 ? l + 360 : l) / 180 * Math.PI;
        }

        function solarEquationOfCenter(centuries) {
            var m = solarGeometricMeanAnomaly(centuries);
            return (Math.sin(m) * (1.914602 - centuries * (0.004817 + 0.000014 * centuries)) +
                    Math.sin(m + m) * (0.019993 - 0.000101 * centuries) + Math.sin(m + m + m) *
                    0.000289) * radians;
        }

        function obliquityCorrection(centuries) {
            return meanObliquityOfEcliptic(centuries) + 0.00256 * Math.cos((125.04 - 1934.136 *
                centuries) * radians) * radians;
        }

        function meanObliquityOfEcliptic(centuries) {
            return (23 + (26 + (21.448 - centuries * (46.8150 + centuries * (0.00059 - centuries *
                0.001813))) / 60) / 60) * radians;
        }

        function eccentricityEarthOrbit(centuries) {
            return 0.016708634 - centuries * (0.000042037 + 0.0000001267 * centuries);
        }
        return redraw;
    };

    /**
     * @ngdoc directive
     * @name ng.directive:fD3ThreatMap
     *
     * @description
     * This directive will render a threat map with d3.
     * Arguments (* => required):
     *   * "data": An array of objects where each object must have the properties "name" and
     *       "value".
     *
     */
    var d3ThreatMap = function() {
        return {
            restrict: 'A',
            scope: {
                data: '=',
                api: '='
            },
            controller: D3ThreatMapController
        };
    };

    function D3EventlineController($scope, $element, $injector) {
        this.scrollable = true;
        $injector.invoke(D3Controller, this, {$scope: $scope, $element: $element});
        this.init();

        this.$q.all(this.promises).then(function() {
            this.chart = this.setupChart(this.d3);
        }.bind(this));
    }

    D3EventlineController.prototype = Object.create(D3Controller.prototype);
    D3EventlineController.prototype.constructor = D3EventlineController;

    D3EventlineController.prototype.requireComponents = function() {
        var deferred = this.$q.defer();
        var d3Requires = D3Controller.prototype.requireComponents.call(this).then(function() {
            require(['js/d3-eventline'], function(eventline) {
                this.d3.eventline = eventline;
                deferred.resolve();
            }.bind(this));

            return deferred.promise;
        }.bind(this));

        return d3Requires;
    };

    D3EventlineController.prototype.setupChart = function(d3) {
        return d3.eventline()
            .labelClick(this.$scope.click);
    };

    D3EventlineController.prototype.update = function(data) {
        var selection = this.svg;

        if (this.chart) {
            this.chart
                .beginning(data.timeframe.start * 1000)
                .ending(data.timeframe.end * 1000)
                .noDataMessage(this.$scope.emptyMessage);

            selection
                .datum(data.events)
                .call(this.chart);

            selection.selectAll('.group-bg')
                .each(this.tooltipBindFn);

            selection.selectAll('.event')
                .each(this.tooltipBindFn);
        }
    };

    /**
     * @ngdoc directive
     * @name ng.directive:fD3EventlineChart
     *
     * @description
     * This directive will render a gantt-style timeline chart.
     * Arguments (* => required):
     *   * "data": An array of objects where each object must have the properties "name" and
     *       "value".
     *     "click": An function that will be called when an event is clicked with the respective
     *       entry.
     *     "tooltipFormatter": A function that will be called when an event is hovered over with the
     *       respective entry. Must return tooltip content.
     *     "emptyMessage": A string that will be displayed whenever the data parameter is empty
     */
    var d3EventlineChart = function() {
        return {
            restrict: 'A',
            scope: {
                data: '=',
                click: '=',
                tooltipFormatter: '=',
                emptyMessage: '@'
            },
            controller: D3EventlineController
        };
    };


    function D3ChordController($scope, $element, $injector) {
        $injector.invoke(D3Controller, this, {$scope: $scope, $element: $element});
        this.init();
        this.$q.all(this.promises).then(function() {
            setupColorAndStroke(this, $scope);
            this.setupChart();
        }.bind(this));
    }

    D3ChordController.prototype = Object.create(D3Controller.prototype);
    D3ChordController.prototype.constructor = D3ChordController;

    D3ChordController.prototype.setupChart = function() {
        this.svg = this.svg.append('g')
            .attr('transform', 'translate(' + this.width / 2 + ',' + this.height / 2 + ')');

        this.chordGroup = this.svg.append('g')
            .attr('class', 'chord');
    };

    D3ChordController.prototype.update = function(data) {
        if (data.matrix.every(function(row) {
            return row.every(function(value) {
                return value === 0;
            });
        })) {
            // If theres no non-zero data to display: delete chart and SVG elements and return.
            this.chart = null;
            this.svg.selectAll('*').remove();
            this.chordGroup = this.svg.append('g')
                .attr('class', 'chord');
            return;
        }

        var that = this;
        var arcGen = this.d3.svg.arc()
            .padAngle(function(d) {
                // add padding to arc angle if this datum is "small", relative to max.
                var smallFactor = 1 - (d.value / d.maxGroupValue);
                return -0.03 * smallFactor;
            });
        var chordGen = this.d3.svg.chord();

        // Returns an event handler for fading a given chord group.
        function fade(opacity, chords) {
            return function(g, i) {
                chords.filter(function(d) {
                    return d.source.index !== i && d.target.index !== i;
                }).transition('fade')
                    .style('opacity', opacity);
            };
        }

        function chordTween(chord) {
            return function(d, i) {
                var tween, previous = chord && chord.chords()[i];
                if (previous) {
                    tween = that.d3.interpolate(previous, d);
                } else {
                    // New chord: interpolate from zero-width chord
                    var emptyChord = {
                        source: angular.extend({}, d.source, {
                            endAngle: d.source.startAngle
                        }),
                        target: angular.extend({}, d.target, {
                            endAngle: d.target.startAngle
                        })
                    };
                    tween = that.d3.interpolate(emptyChord, d);
                }

                return function(t) { return chordGen(tween(t)); };
            };
        }

        function arcTween(chord) {
            return function(d, i) {
                var tween, previous = chord && chord.groups()[i];
                if (previous) {
                    // Interpolate from an existing arc.
                    tween = that.d3.interpolate(previous, d);
                } else {
                    // New arc: interpolate from a zero-width arc
                    var zeroArc = {
                        startAngle: d.startAngle,
                        endAngle: d.startAngle
                    };
                    tween = that.d3.interpolate(zeroArc, d);
                }

                return function(t) { return arcGen(tween(t)); };
            };
        }

        function countFormatter(count) {
            return count;
        }

        var entityToEntity = data.dataMap;

        // Save current chart to tween the updated chart arcs and chords from previous objects
        this.previousChart = this.chart;
        this.chart = this.d3.layout.chord().padding(0.10).sortGroups(this.d3.ascending);
        // Sort larger chord lowest so small chords are visible. This is problematic when
        // animating chords swapping order between updates because z-index can't be interpolated
        // .sortChords(function(a, b) {
        //     return b - a;
        // });

        this.svg.attr('transform', 'translate(' + this.width / 2 + ',' + this.height / 2 + ')');
        var innerRadius = Math.min(this.width, this.height) * 0.39,
            outerRadius = innerRadius * 1.1;
        arcGen.innerRadius(innerRadius).outerRadius(outerRadius);
        chordGen.radius(innerRadius);

        // The indicies in this array match matrix data element indices with source entities:
        // for example using a data element source index: entities[data.source.index] -> entity
        var entities = data.entities;
        // Use entity names for tooltip color keys.
        this.tooltipColorKeyGetter = function(d) {
            return entities[d.target.index];
        };

        this.chart.matrix(data.matrix);

        //
        // Chords
        var chords = this.chordGroup
            .selectAll('path.chord')
            .data(this.chart.chords);

        chords.enter()
            .append('path').attr('class', 'chord')
            .style('fill', 'white').style('stroke', 'white').style('opacity', 0.85);
        chords.each(function(d) {
                // Find original data element from source and target indicies;
                // Two detail objects can be associated with a chord: src -> dst or dst -> src
                var source, destination,
                    chordSource = entities[d.source.index],
                    chordDestination = entities[d.target.index],
                    srcToDest = entityToEntity[chordSource][chordDestination],
                    dstToSrc = entityToEntity[chordDestination][chordSource];

                // Decide which entity to use for fill and stroke colors based on realValues
                if (srcToDest.realValue > dstToSrc.realValue) {
                    source = chordSource;
                    destination = chordDestination;
                } else {
                    source = chordDestination;
                    destination = chordSource;
                }

                d.fillEntity = destination;
                d.strokeEntity = source;
                d.data = angular.copy(entityToEntity[source][destination]);
                if (srcToDest !== dstToSrc) {
                    d.data.reversed = entityToEntity[destination][source];
                }
            }).each(this.tooltipBindFn)
            .on('click', function(d) {
                if (d.source === d.target) {
                    return that.clickFn(d);
                }
                that.$scope.$apply(function() {
                    that.$scope.drillDownMenu.formatters = {
                        bytes: formatters.metric_bytes,
                        packets: countFormatter,
                        bandwidth: formatters.metric_bits_per_second,
                        shaper_drops: formatters.metric_bytes,
                        sessions: countFormatter
                    };
                    that.$scope.drillDownMenu.entry = d.data;
                    that.$scope.drillDownMenu.toggle({
                        top: that.d3.event.clientY,
                        left: that.d3.event.clientX
                    });
                });
            })
            .transition().duration(1000)
            .style({
                'fill': function(d) { return that.color(d.fillEntity, d); },
                'stroke': function(d) { return that.stroke(d.strokeEntity, d); },
                'cursor': this.clickSupported ? 'pointer' : 'auto'
            })
            .attrTween('d', chordTween(this.previousChart));

        chords.exit().remove();

        //
        // Arcs and associated labels.
        var arcGroups = this.svg.selectAll('g.intf')
            .data(this.chart.groups);

        var newArcGroups = arcGroups.enter().append('g').attr('class', 'intf');
        newArcGroups.append('path').attr('class', 'arc')
            .style('fill', 'white').style('stroke', 'white')
            .append('title');

        newArcGroups.append('text').style('fill', 'white');

        arcGroups.select('path title')
            .text(function(d) {
                var entity = entities[d.index],
                    entityExtra = data.entitiesExtra[entity];
                return entityExtra && entityExtra.label || entity;
            });

        arcGroups.select('text')
            .each(function(d) { d.angle = (d.startAngle + d.endAngle) / 2;})
            .text(function(d) {
                var entity = entities[d.index],
                    entityExtra = data.entitiesExtra[entity];
                return entityExtra && entityExtra.label || entity;
            })
            // Text is aligned to the middle of the arc, and flipped upright if angle greater pi
            .style('text-anchor', function(d) { return d.angle > Math.PI ? 'end' : null; })
            .attr('transform', function(d) {
                return 'rotate(' + (d.angle * 180 / Math.PI - 90) + ')' +
                       'translate(' + (outerRadius + 10) + ',0)' +
                       ((d.angle > Math.PI) ? 'rotate(180)' : '');
            })
            .transition().duration(1000).style('fill', 'black');

        var maxGroupValue = Math.max.apply(null, this.chart.groups().map(function(group) {
            return group.value;
        }));
        arcGroups.select('path.arc').each(function(d) { d.maxGroupValue = maxGroupValue; })
            .transition().duration(1000)
            .style('fill', function(d) { return that.color(entities[d.index], d); })
            .style('stroke', function(d) { return that.stroke(entities[d.index], d); })
            .attrTween('d', arcTween(this.previousChart));

        arcGroups.on('mouseover', fade(0.1, chords))
                 .on('mouseout',  fade(0.85, chords))
                 .on('');

        arcGroups.exit().remove();
    };

    /**
     * @ngdoc directive
     * @name ng.directive:fD3ChordChart
     *
     * @description
     */
    var d3ChordChart = function() {
        return {
            restrict: 'A',
            scope: {
                data: '=',
                click: '=',
                tooltipFormatter: '=',
                drillDownMenu: '='
            },
            controller: D3ChordController
        };
    };

    return function(providers) {
        providers.$compile.directive('fD3BasicBubbleChart', d3BasicBubbleChart);
        providers.$compile.directive('fD3CountryChart', d3CountryChart);
        providers.$compile.directive('fD3ThreatMap', d3ThreatMap);
        providers.$compile.directive('fD3EventlineChart', d3EventlineChart);
        providers.$compile.directive('fD3ChordChart', d3ChordChart);
    };
});
