/* global define, require */
define(['angular', 'jquery'], function(angular, $) {
    'use strict';
    var TOOLTIP_X_OFFSET = 15;
    var TOOLTIP_Y_OFFSET = 15;
    var TOOLTIP_BUFFER = 5;
    var DEBOUNCE_INTERVAL = 200;

    function D3Controller($scope, $element, $q, $window) {
        this.id = 'd3Chart' + Math.floor(1 + Math.random() * 0x10000);
        this.$scope = $scope;
        this.$element = $element;
        this.$q = $q;
        this.$window = $window;

        this.promises = [this.requireComponents().then(function() {
            var parent = this.d3.select(this.$element[0]);
            if (this.scrollable) {
                parent = parent.append('div')
                    .attr('class', 'd3-chart-scrollbox');
            }
            this.svg = parent.append('svg');
            if (!this.scrollable) {
                this.svg.attr('class', 'no-scroll');
            }
            this.color = this.d3.scale.category20c();
            this.stroke = function(value, data) {
                var color = this.d3.rgb(this.color(value, data));
                return color.darker(0.5);
            }.bind(this);
        }.bind(this))];
    }

    D3Controller.prototype.init = function() {
        this.width = this.$element.width();
        this.height = this.$element.height();

        this.$element.addClass('d3-chart');
        this.$loading = angular.element('<div class="d3-chart-loading">' +
            '<f-icon class="fa-loading icon-xxxl"></f-icon></div>').appendTo(this.$element);

        this.setupClick();
        if (angular.isFunction(this.$scope.tooltipFormatter)) {
            this.tooltipBindFn = this.createTooltipBindFn();
        } else {
            this.tooltipBindFn = angular.noop;
        }
        this.ready = this.$q.all(this.promises).then(function() {
            this.setupWatch();
        }.bind(this));
    };

    D3Controller.prototype.requireComponents = function(extra) {
        var required = {d3: 'js/d3'};
        required = angular.extend(required, extra);

        var requiredModules = Object.keys(required);
        var requiredPaths = requiredModules.map(function(module) {
            return required[module];
        });

        var deferred = this.$q.defer();

        require(requiredPaths, function() {
            for (var i = 0; i < arguments.length; i++) {
                this[requiredModules[i]] = arguments[i];
            }
            deferred.resolve();
        }.bind(this));

        return deferred.promise;
    };

    D3Controller.prototype.setupClick = function() {
        this.clickSupported = false;

        this.clickFn = function(d) {
            if (this.clickSupported) {
                this.$scope.$apply(function() {
                    this.$scope.click(d.data ? d.data : d);
                }.bind(this));
            }
        }.bind(this);

        this.$scope.$watch('click', function(click) {
            this.clickSupported = angular.isFunction(click);
            this.render(this.$scope.data);
        }.bind(this));
    };

    D3Controller.prototype.tooltipDataGetter = function(d) {
        return d.data ? d.data : d;
    };

    D3Controller.prototype.tooltipColorKeyGetter = function(d) {
        return d.data ? d.data.name : d.name;
    };

    D3Controller.prototype.createTooltipBindFn = function(content) {
        var that = this;
        var $tooltip = $('<div class="d3-tooltip hidden"></div>').appendTo(this.$element);

        var setPosition = function(event) {
            var height, width, top, left, left_over;
            var offset = that.$element.offset();
            height = $tooltip.height();
            width = $tooltip.width();
            top = event.pageY - offset.top - TOOLTIP_Y_OFFSET - height;
	    /*
            if (top < TOOLTIP_BUFFER) {
                top = TOOLTIP_BUFFER;
            }
	    */
            left = event.pageX - offset.left + TOOLTIP_X_OFFSET;
            left_over = left + width - that.$element.width();
            if (left_over > -TOOLTIP_BUFFER) {
                left = event.pageX - offset.left - width - TOOLTIP_X_OFFSET -
                    TOOLTIP_BUFFER;
            }

            $tooltip.css({
                'top': top,
                'left': left
            });
        };

        return function(d) {
            var $this = $(this);
            $this.off('mouseover.tooltip').on('mouseover.tooltip', function(event) {
                var $div = $('<div></div>');
                var html = angular.isFunction(content) ? content(d) :
                               (angular.isString(content) ? content :
                                that.$scope.tooltipFormatter(that.tooltipDataGetter(d)));
                $div.append(html);
                /*FWB_CHANGE if (!content) {
                    var color = that.d3.rgb(that.color(that.tooltipColorKeyGetter(d), d));
                    color = 'rgba(' + [color.r, color.g, color.b, '0.7'].join(',') + ')';
                    $div.css('background-color', color);
                }*/
                $tooltip.empty().append($div).removeClass('hidden');
                setPosition(event);
            });

            $this.off('mousemove.tooltip').on('mousemove.tooltip', function(event) {
                setPosition(event);
            });

            $this.off('mouseleave.tooltip').on('mouseleave.tooltip', function() {
                $tooltip.addClass('hidden');
            });
        };
    };

    D3Controller.prototype.setupWatch = function() {
        var resize = $.debounce(DEBOUNCE_INTERVAL, function() {
            var newWidth = this.$element.width();
            var newHeight = this.$element.height();
            if (newWidth && newHeight && (newWidth !== this.width || newHeight !== this.height)) {
                this.width = newWidth;
                this.height = newHeight;
                this.render(this.$scope.data, true);
            }
        }.bind(this));

        // element width/height change will update width/height of this and trigger render
        this.$scope.$watch(resize);

        // data change will trigger render
        this.$scope.$watch('data', function(data) {
            this.render(data);
        }.bind(this));

        // window change will trigger resize, in turn, call render
        angular.element(this.$window).on('resize.' + this.id, resize);

        this.$scope.$on('$destroy', function() {
            angular.element(this.$window).off('resize.' + this.id);
        }.bind(this));
    };

    D3Controller.prototype.render = function(data, ignoreLoading) {
        if (!ignoreLoading) {
            this.$loading.toggleClass('hidden', false);
        }
        this.$q.when(data, function(data) {
            if (data != null && this.width && this.height) {
                this.$loading.toggleClass('hidden', true);
                this.update(data);
            }
        }.bind(this));
    };

    return D3Controller;
});

