/* global define */
/* jshint maxparams: 11 */

define(['angular', 'ng/services/injector', 'jquery', 'fweb', 'ftnt_shared', 'ng/directives/d3_base',
        'ng/services/chart/constants',
        'ng/services/chart/base_ftnt_object',
        'ng/services/chart/bubble_cluster',
        'ng/services/chart/device_type',
        'ng/services/chart/topology_tree'],
function(angular, inject, $, fweb, ftnt_shared, D3Controller, CONSTANTS, BaseFtntObject,
         BubbleCluster, DeviceType, TopologyTree) {
    'use strict';

    var LINK_STROKE_WIDTH = 2.5;
    var DEFAULT_ICON_COLOR = '#333';

    var FTNT_ICON_SIZE = 18;
    var TEXT_Y_ADJUSTMENT_FACTOR = 1 / 8;
    var CLUSTER_ZOOM_SCALE = 0.88,
        BUBBLE_ZOOM_SCALE = 0.9;
    var INITIAL_FONT_SIZE = 1;
    var MAX_FONT_SIZE = 15;
    var TRANSITION_DURATION = {
        BASE: 1000,
        ZOOM: 750,
        QUICK: 100
    };

    var INTF = {
        TEXT_MARGIN: 4
    };

    // Given a circle with radius R, let use 4 parallel horizontal circle
    // chords to cut the vertical circle chord into 5 equal segments of
    // which the length equals to (2 * R) / 5:
    //   - The 1st and the 5th area doesn't have anything
    //   - The 2nd area contains the icon
    //   - The 3rd area contains the name text
    //   - The 4th area contains the value text
    // This constant is used to calculate icon size and text position with
    // above assumption.
    var MAX_AREA = 5;

    var NAT_ROUTER_TOOLTIP_TEMPLATE = '<div class="warning-message">' +
        '  <div class="flex-row-centered">' +
        '    <f-icon class="fa-warning"></f-icon>' +
        '    <label for="messages-toggle">' +
        '    {{0}}' +
        '    </label>' +
        '  </div>' +
        '</div>';


    function valueFormatter(d) {
        /* jshint validthis:true */
        return this.$scope.shownKeyFormatter(d);
    }

    function iconSize(d) {
        return toPx(d.r * 2 / MAX_AREA);
    }

    function toPx(num) {
        return (Math.floor(num * 1000) / 1000) + 'px';
    }

    function pluckX(d) {
        return d.x;
    }

    function pluckY(d) {
        return d.y;
    }

    function pluckDeviceName(d) {
        return d.alias || d.hostname ||
            (d.device ? $.getInfo('Short::device-type.' + d.device) : '') || d.name;
    }

    function transformD(d) {
        return transformXY(d.x, d.y);
    }

    function transformXY(dx, dy) {
        return 'translate(' + dx + ',' + dy + ')';
    }

    function quickTransition(collection, attrs, styles, textFn) {
        var transition = collection.transition()
            .duration(TRANSITION_DURATION.QUICK);
        if (attrs) {
            transition.attr(attrs);
        }
        if (styles) {
            transition.style(styles);
        }
        if (textFn) {
            transition.text(textFn);
        }
    }

    function linkId(prefix) {
        return function(d) {
            return (prefix ? prefix + '-' : '') + d.source.linkId;
        };
    }

    function childrenDivisionMinX(d) {
        var minX = (d.source.divisionHMargin || {}).minX;
        return {
            x: d.source.y,
            y: minX || d.source.x
        };
    }

    function parentDivisionMaxX(d) {
        var maxX = (d.target.divisionHMargin || {}).maxX;
        return {
            x: d.target.y,
            y: maxX || d.target.x
        };
    }

    function horizontalProjection(d) {
        // The default direction of of diagonal is from top to bottom while we
        // want left to right (or right to left), so we need this projection
        return [d.y, d.x];
    }

    function D3BubbleCluster($scope, $element, $injector, injector, iconCode, $q,
                             $http, csf, fortiviewVisualization, cloudSvg) {
        $injector.invoke(D3Controller, this, {$scope: $scope, $element: $element});
        injector.injectMarked(this);
        this.$element.addClass('d3-bubble-cluster-chart');
        this.init();
        this.color = null;
        this.resetChartData();

        this.natRouterTooltipBindFn = this.createNatRouterTooltipBindFn();
        this.fgtTooltipBindFn = this.createFgtTooltipBindFn();
        this.linkTooltipBindFn = this.createLinkTooltipBindFn();
        this.interfaceTooltipBindFn = this.createInterfaceTooltipBindFn();

        this.interfaceFetch = $http.get('/api/v2/cmdb/system/interface', {
            params: {'format': 'name|macaddr|ip|vdom|role|type|managed-device'}
        }).then(function(response) {
            var intfMap = {};
            response.data.results.forEach(function(intf) {
                var ipmask = fweb.util.ip.IpMask.parse(intf.ip);
                intf.network = ipmask.full_network();
                intf.ipAddr = ipmask.addr.join('.');
                intfMap[intf.name] = intf;
            });
            this.chartData.masterFgtIntfMap = intfMap;

        }.bind(this));

        $scope.$watch('apiUrlParams()', function(newVal, oldVal) {
            if (newVal !== oldVal || (newVal && !this.csfData)) {
                if (this.shouldLoadCsf(newVal)) {
                    this.csfData = csf.fetch(newVal, fortiviewVisualization.processSource,
                                             newVal.params.vdom);
                }
            }
        }.bind(this), true);

        $scope.$on('reload', function() {
            var urlParams = $scope.apiUrlParams();
            this.csfData = csf.fetch(urlParams, fortiviewVisualization.processSource,
                                     urlParams.params.vdom);
        }.bind(this));

        this.chartData.baseFontSize = this.findBaseFontSize();

        $q.all([this.ready, iconCode.$promise, this.interfaceFetch,
                cloudSvg.$promise]).then(function() {
            this.interfaceClick = function() {
                return $scope.interfaceClick.apply(this, arguments);
            }.bind(this);
            this.cloudPath = cloudSvg.path;

            $scope.$on('$routeChangeSuccess', function() {
                this.resetChartData();
            }.bind(this));

            this.outerSvg = this.svg;
            this.defs = this.outerSvg.append('defs');
            this.createWhiteGlowFilter();

            this.zoom = this.d3.behavior.zoom()
                .translate([0, 0])
                .on('zoom', function() {
                    this.svg.attr('transform',
                                  'translate(' + this.d3.event.translate + ')' +
                                      ' scale(' + this.d3.event.scale + ')');
                }.bind(this));

            this.svg = this.outerSvg
                .append('g');

            this.outerSvg.on('mousedown', this.restrictZoomBehavior(), true)
                .call(this.zoom)
                .call(this.zoom.event);

            this.overlayRect = this.svg.append('rect')
                .attr({
                    width: this.width,
                    height: this.height,
                    'pointer-events': 'all',
                    'class': 'overlay pannable'
                })
                .style({
                    fill: 'none'
                })
                .on('click', this.overlayClicked());

            // Those layers are here to ensure:
            // - links between interface and cluster won't cross cluster bubbles
            // - links between interface and cloud svg won't cross cloud svg
            this.cloudLinksLayer = this.svg.append('g').attr('class', 'cloud-link-layer');

            this.linksLayer = this.svg.append('g').attr('class', 'link-layer');

            // This layer is for all FGT/FSW
            this.ftntLayer = this.svg.append('g')
                .attr('class', 'division-interface-layer');

            // This layer is for cloud if WAN role interface exists
            this.cloudLayer = this.svg.append('g').attr('class', 'cloud-layer');

            // This layer is for cluster wrapper
            this.clusterContainerLayer = this.svg.append('g').attr('class', 'cluster-layer');

            this.currentZoomInNodeId = null;
            this.currentZoomInClusterId = null;

            // For CSF, we divide the horizontal into multiple columns:
            //
            // | column0 | column1 | column2 | column3 | column4 |...     |
            // |         |         |         |         |         |        |
            // | cloud   | master  | master  | level1  | level2  |        |
            // |         | FGT     |  FGT    |  FGT    |  FGT    |        |
            // |         | intfs   |  bubbles|  bubbles|  bubbles|        |
            // |         |         | +       | +       | +       |        |
            // |         |         | level1  | level2  | level3  |        |
            // |         |         |  FGT    |  FGT    |  FGT    |        |
            // |         |         |  intfs  |  intfs  |  intfs  |        |
            // |         |         |         |         |         |        |
            //
            // - Cloud icon is in column0
            // - Interfaces of master FGT is in column1
            // - Bubbles of master FGT + interfaces of FGT connecting to master
            // FGT (a.k.a level1 downstream) are in column2
            // - Bubbles of level1 downstream + interfaces of FGT connecting to
            // level1 downstream (a.k.a level2 downstream) are in column3
            // and so on... Some initial workflow:
            //
            // - Connecting lines will be draws last.
            // - Bubbles in all levels will use the same radius function to
            // ensure the relativeness of bubble size and device data.
            // - Interface list will have the same scale ratio
            //
            // To satisfy so:
            // - Use base interfaces size to calculate interfaces list.
            // - Data of master device to calculate radius function.
            // - Starting from column2, total bubble cluster size will fit the
            // screen, while the interface list size will be used to track the
            // scale factor needed so that the whole picture of topology is
            // shown.
            this.columnLayers = [];

        }.bind(this));
    }

    D3BubbleCluster.prototype = Object.create(D3Controller.prototype);
    D3BubbleCluster.prototype.constructor = D3BubbleCluster;

    D3BubbleCluster.prototype.shouldLoadCsf = inject.mark(function(state) {
        return function(urlParams) {
            return (!state.vdom_mode || !!urlParams.params.vdom);
        };
    });

    D3BubbleCluster.prototype.createFgtTooltipBindFn = inject.mark(function(lang) {
        var attrs = [{
            key: 'sn',
            lang: lang('serial')
        }, {
            key: 'hostname',
            lang: lang('hostname')
        }, {
            key: 'name',
            lang: lang('name')
        }, {
            key: 'vdom',
            lang: lang('vdom')
        }, {
            key: 'version',
            lang: lang('ver_firmware')
        }, {
            key: 'build',
            lang: lang('build')
        }];

        return function() {
            return this.createTooltipBindFn(function(d) {
                var str = '<table><tbody>';
                attrs.forEach(function(attr) {
                    if (d[attr.key]) {
                        if (attr.key !== 'vdom') {
                            // always show any key differing from 'vdom'
                            // which is currently hidden until it is clear how
                            // CSF works in multi-vdom environment.
                            str += ['<tr>', '<td>', attr.lang, ':</td>',
                                    '<td>', d[attr.key], '</td>', '</tr>'].join('');
                        }
                    }
                });
                if (d.viaRouterNat) {
                    str += ['<tr>', '<td colspan="2">',
                            lang('This FortiGate is behind Router/NAT device.'),
                            '</td>','</tr>'].join('');
                }
                str += '</tbody></table>';
                return str;
            });
        };
    });

    D3BubbleCluster.prototype.createNatRouterTooltipBindFn = inject.mark(function(lang) {
        return function() {
            var msg = lang('Replace Router/NAT devices with FortiGate');
            return this.createTooltipBindFn(
                fweb.util.dom.renderTemplate(NAT_ROUTER_TOOLTIP_TEMPLATE, [msg]));
        };
    });

    D3BubbleCluster.prototype.createLinkTooltipBindFn = inject.mark(function(lang) {
        return function() {
            return this.createTooltipBindFn(function(d) {
                var attrs = [{
                    key: 'network',
                    lang: lang('network')
                }];
                var content = '';
                attrs.forEach(function(attr) {
                    var str = '';
                    if (attr.key === 'network') {
                        var srcNetwork = d.source.network;
                        var dstNetwork = d.target.network;
                        if (srcNetwork === dstNetwork) {
                            str = srcNetwork;
                        } else if (srcNetwork && dstNetwork) {
                            str = dstNetwork + ' <--> ' + srcNetwork;
                        } else if (dstNetwork) {
                            str = dstNetwork;
                        } else if (srcNetwork) {
                            str = srcNetwork;
                        }
                    } else {
                        str = d[attr.key];
                    }
                    if (str) {
                        content += ['<tr>', '<td>', attr.lang, ':</td>',
                                '<td>', str, '</td>', '</tr>'].join('');
                    }
                });
                return content ? '<table><tbody>' + content + '</tbody></table>' : '';
            });
        };
    });

    D3BubbleCluster.prototype.createInterfaceTooltipBindFn = inject.mark(function(lang) {
        return function() {
            return this.createTooltipBindFn(function(d) {
                var attrs = [{
                    key: 'name',
                    lang: lang('Name')
                }, {
                    key: 'ipAddr',
                    lang: lang('IP Address')
                }, {
                    key: 'network',
                    lang: lang('network')
                }, {
                    key: 'vlan',
                    lang: lang('vlan')
                }, {
                    key: 'role',
                    lang: lang('role')
                }];
                var html = '<table><tbody>';
                attrs.forEach(function(attr) {
                    var val = d[attr.key];
                    var str = '';
                    if (val && attr.key === 'role') {
                        val = lang('Interface::role.' + val).toString();
                        if (d.isWANRoleMissing) {
                            str += lang('{role}. Interface appears to be used for WAN traffic.', [
                                val
                            ]).toString();
                        } else if (d.role === 'undefined') {
                            str += lang('{role}. Interface requires a defined role.', [
                                val
                            ]).toString();
                        }
                        if (str) {
                            val = '<f-icon class="fa-warning"></f-icon>' +
                                '<span>' + str + '</span>';
                        }
                    } else if (val && attr.key === 'name') {
                        if (d.fortilink === 'enable' || d['fortilink-port']) {
                            val = val + ' (' + lang('fortilink_port').toString() + ')';
                        }
                    }
                    if (val) {
                        html += ['<tr>', '<td>', attr.lang, ':</td>',
                                '<td>', val, '</td>', '</tr>'].join('');
                    }
                });

                html += '</tbody></table>';
                return html;
            });
        };
    });

    D3BubbleCluster.prototype.findBaseFontSize = function() {
        var $dummyText = $('<span style="display:none"></span>')
            .appendTo($('body'));
        var fontSize = parseInt($dummyText.css('font-size')) || MAX_FONT_SIZE;
        $dummyText.remove();
        return fontSize;
    };

    D3BubbleCluster.prototype.resetChartData = function() {
        this.chartData = {
            baseFontSize: this.chartData ? this.chartData.baseFontSize : null,
            fullViewScale: 1,
            bubbleOption: this.chartData ? this.chartData.bubbleOption : '',
            accessDevice: this.chartData ? this.chartData.accessDevice : false,
            topology: this.chartData ? this.chartData.topology : ''
        };
    };

    D3BubbleCluster.prototype.update = function(data) {
        this.csfData.then(function(result) {
            this.csfUpdate(data, result);
        }.bind(this));
    };

    D3BubbleCluster.prototype.csfUpdate = function(masterData, csfData) {
        this.chartData.fullViewScale = 1;

        this.updateSize(this.width, this.height);
        angular.extend(this.chartData, {
            width: this.width,
            height: this.height,
            bubbleOption: this.$scope.bubbleOption(),
            accessDevice: this.$scope.accessDevice() === 'show',
            topology: this.$scope.topology
        });
        this.renderTopology(csfData, masterData, this.chartData);
    };

    D3BubbleCluster.prototype.isLogicalTopology = inject.mark(function(fortiviewVisualization) {
        return function(mode) {
            return mode === fortiviewVisualization.TOPOLOGY_ENUM.LOGICAL;
        };
    });

    D3BubbleCluster.prototype.renderBubbles = inject.mark(function($location, csf_chart) {
        return function(deviceNodes, clusterNodes) {

            var deviceNode = this.svg.selectAll('.node')
                .data(deviceNodes, function(d) {
                    return d.srcIntfId + d.name;
                });

            var deviceNodeAttrs = {
                class: 'node',
                transform: transformD
            };
            var enter = deviceNode.enter()
                .append('g')
                .attr(deviceNodeAttrs);
            deviceNode.exit().remove();
            deviceNode.transition()
                .duration(TRANSITION_DURATION.QUICK)
                .attr(deviceNodeAttrs)
                .call(this.transitionDone, function() {
                    if (this.currentZoomInNodeId) {
                        // If we are in 'bubble' zooming state
                        //   If the zoomed in entry still available in new data, let revise the zoom
                        //   Else, it's an orphan zoom, let reset it
                        if (BubbleCluster.bubbleMap.hasOwnProperty(this.currentZoomInNodeId)) {
                            var d = BubbleCluster.bubbleMap[this.currentZoomInNodeId];
                            this.zoomToNode(d);
                        } else {
                            this.zoomReset();
                        }
                    } else if (this.currentZoomInClusterId) {
                        // If we are in 'cluster' zooming state
                        //   If the zoomed cluster still available in new data, let revise the zoom
                        //   Else, it's an orphan zoom, let reset it
                        if (BubbleCluster.clusterMap.hasOwnProperty(this.currentZoomInClusterId)) {
                            this.zoomToCluster(this.currentZoomInClusterId);
                        } else {
                            this.zoomReset();
                        }
                    }
                }.bind(this));

            deviceNode.each(this.tooltipBindFn);

            // bubble
            var deviceCircleAttrs = {
                r: csf_chart.pluckR
            };
            var deviceCircleStyles = {
                fill: function(d) {
                    return this.color(d.srcIntfId);
                }.bind(this),
                cursor: function(d) {
                    var id = d.srcIntfId + d.name;
                    return (this.currentZoomInNodeId === id) ? 'zoom-out' : 'zoom-in';
                }.bind(this)
            };
            enter.append('circle')
                .attr(deviceCircleAttrs)
                .style(deviceCircleStyles)
                .on('click', this.bubbleClicked());
            quickTransition(deviceNode.select('circle'), deviceCircleAttrs, deviceCircleStyles);

            // 'icon'
            var svgBaseUrl = $('base').attr('href').slice(0, -1) + $location.url();
            var deviceIconAttrs = {
                class: function(d) {
                    return 'bubble-icon auto-size f-icon ' + (d.iconName || '');
                },
                'text-anchor': 'middle',
                'font-family': function(d) { return d.iconFontFamily || ''; },
                'font-size': iconSize,
                dy: function(d) { return toPx(-d.r / MAX_AREA); },
                fill: DEFAULT_ICON_COLOR,
                // Need a longer IRI instead of the short IRI like #filter
                // because it is no longer relative to the current page, but
                // to the URL indicated in the <base> tag which is '/ng'
                filter: 'url(' + svgBaseUrl + '#whiteglow)'
            };
            var deviceIconStyles = {
                'pointer-events': 'none'
            };
            var iconTextFn = function(d) { return d.iconCode || ''; };
            enter.append('text')
                .text(iconTextFn)
                .attr(deviceIconAttrs)
                .style(deviceIconStyles);
            quickTransition(deviceNode.select('text.bubble-icon'), deviceIconAttrs,
                            deviceIconStyles, iconTextFn);

            // 'name' text
            var deviceNameAttrs = {
                class: 'name auto-size',
                'text-anchor': 'middle',
                'font-size': toPx(INITIAL_FONT_SIZE)
            };
            enter.append('text')
                .text(pluckDeviceName)
                .attr(deviceNameAttrs)
                .style({
                    'pointer-events': 'none'
                });
            // transition for name node will be done later after nameFontSize is calculated

            // 'value' text
            var deviceValueAttrs = {
                class: 'value auto-size',
                'text-anchor': 'middle',
                dy: function(d) { return toPx(1.5 * d.r / MAX_AREA); },
                'font-size': toPx(INITIAL_FONT_SIZE)
            };
            enter.append('text')
                .text(valueFormatter.bind(this))
                .attr(deviceValueAttrs)
                .style({
                    'pointer-events': 'none'
                });
            // transition for value node will be done later after nameFontSize is calculated

            // calculate the font size to be used commonly by 'name' text and 'value' text
            var nameNodes = deviceNode.select('text.name')
                .text(function(d) {
                    d.initialTextLength = 0;
                    return pluckDeviceName(d);
                })
                .attr({
                    'font-size': function(d) {
                        if (d.nameFontSize && d.nameFontSize < INITIAL_FONT_SIZE) {
                            return toPx(d.nameFontSize);
                        }
                        d.nameFontSize = undefined;
                        return toPx(INITIAL_FONT_SIZE);
                    }
                });
            nameNodes.each(function(d) {
                // pre-calculate the font size to be used by both 'name' text and 'value' text
                if (!d.initialTextLength) {
                    // should padding 1 letter each side, we can't use fixed pixel
                    // because the radius and the text length varied for each bubble
                    var name = pluckDeviceName(d);
                    d.initialTextLength = (1 + (2 / name.length)) * this.getComputedTextLength();
                }
                // Here is the math:
                //  Need initialTextLength to render text in INITIAL_FONT_SIZE font
                //  So, with availableLength, the font size would be
                //                    (availableLength * INITIAL_FONT_SIZE) / initialTextLength

                // In the other hand, availableLength is one of the sides of the right triangle
                // where hypotenuse is the radius segment, the length of the other side is 1/2
                // of r/5. Applying pythagoras theorem:
                var availableLength = 2 * Math.sqrt(Math.pow(d.r, 2) - Math.pow(d.r / 10, 2));
                var initialFontSize = d.nameFontSize || INITIAL_FONT_SIZE;
                var fontSize = (availableLength * initialFontSize) /
                                   (d.initialTextLength);
                d.nameFontSize = Math.min(2 * d.r / MAX_AREA, fontSize, MAX_FONT_SIZE);
            });
            quickTransition(nameNodes, {
                'font-size': function(d) {
                    return toPx(d.nameFontSize);
                }
            });

            quickTransition(
                deviceNode.select('text.value'), {
                    dy: function(d) { return toPx(1.5 * d.r / MAX_AREA); },
                    'font-size': function(d) { return toPx(d.nameFontSize); }
                },
                null,
                valueFormatter.bind(this)
            );

            var clusterCircles = this.clusterContainerLayer.selectAll('.cluster-container')
                .data(clusterNodes, function(d) { return d.upstreamSrcIntfId; });
            var clusterEnter = clusterCircles.enter();
            var clusterCircleAttrs = {
                r: csf_chart.pluckR,
                cx: pluckX,
                cy: pluckY,
                'class': 'cluster-container'
            };
            var clusterCircleStyles = {
                fill: 'white',
                stroke: function(d) { return this.color(d.upstreamSrcIntfId); }.bind(this),
                cursor: function(d) {
                    return (this.currentZoomInClusterId === d) ? 'zoom-out' : 'zoom-in';
                }.bind(this)
            };
            clusterEnter.append('circle')
                .attr(clusterCircleAttrs)
                .style(clusterCircleStyles)
                .on('click', this.clusterCircleClicked());
            clusterCircles.exit().remove();
            quickTransition(clusterCircles, clusterCircleAttrs, clusterCircleStyles);
        };
    });

    D3BubbleCluster.prototype.renderTopology = inject.mark(
    function($injector, injector, state, csf_chart) {
        return function(csfData, masterData, settings) {
            var withAccessDevices = settings.accessDevice || false;
            var withInterfaces = this.isLogicalTopology(settings.topology);
            csfData.tree.isRoot = true;
            csfData.tree.vdom = state.current_vdom;
            csfData.tree.data.stats = {};
            csfData.tree.data.stats[state.current_vdom] = [];
            angular.copy(masterData, csfData.tree.data.stats[state.current_vdom]);

            BubbleCluster.clusterMap = {};
            BubbleCluster.bubbleMap = {};
            var topologyTree = $injector.instantiate(TopologyTree, {
                injector: injector,
                csfData: csfData,
                masterFvStats: masterData,
                withAccessDevices: withAccessDevices
            });
            var hasCloud = topologyTree.hasCloud();
            var chartNodes = topologyTree.toChart(this.d3, withInterfaces, settings.bubbleOption);

            this.color = csf_chart.createColorFn(this.d3, chartNodes.allIntfs);
            var bBox = chartNodes.bBox;
            if (bBox) {
                var dx = bBox.x2 - bBox.x1,
                    dy = bBox.y2 - bBox.y1,
                    x = (bBox.x1 + bBox.x2) / 2;

                var height = this.height - CONSTANTS.DIVISION.MARGIN.TOP;
                var width = this.width - CONSTANTS.DIVISION.MARGIN.LEFT;
                var fullViewScale = 1 / Math.max(dx / width, dy / height);
                var translate = [width / 2 - fullViewScale * x,
                                 CONSTANTS.DIVISION.MARGIN.TOP];
                // this scale factor is for overall view
                this.chartData.fullViewScale = fullViewScale < 1 ? fullViewScale : 1;
                this.chartData.translate = fullViewScale < 1 ? translate : [0, 0];
                if (fullViewScale < 1) {
                    this.updateSize(this.width / fullViewScale, this.height / fullViewScale);
                }
            }
            this.zoomReset();

            this.updateCsfMasterCloud(hasCloud, settings.baseFontSize);
            this.renderFtntDevice(this.ftntLayer, chartNodes.ftnts,
                                  settings.baseFontSize, withInterfaces);
            this.renderDeviceTypes(this.ftntLayer, chartNodes.deviceTypeGroups,
                                   settings.baseFontSize);
            this.renderBubbles(chartNodes.devices, chartNodes.clusters);
            this.updateCloudWanIntfLinks(hasCloud, chartNodes.ftnts[0], withInterfaces);
            this.renderLinks(chartNodes.links, withInterfaces);
        };
    });

    /**
     * Draw curved link from WAN-role interface and cloud icon
     **/
    D3BubbleCluster.prototype.updateCloudWanIntfLinks = inject.mark(function(cloudSvg) {
        return function(hasCloud, rootFtnt, withInterfaces) {
            var layer = this.cloudLinksLayer;
            var shownIntfs = rootFtnt.shownIntfs;
            var offset = rootFtnt.getOffset();
            var cloudX = CONSTANTS.DIVISION.MARGIN.LEFT + cloudSvg.WIDTH / 2;
            var headH = rootFtnt.getHeadHeight();
            var rowH = BaseFtntObject.BASE_ROW_HEIGHT;
            var cloudY = CONSTANTS.DIVISION.MARGIN.TOP +
                         (withInterfaces ? headH : headH / 2) +
                         (withInterfaces ? rowH / 2 : 0);

            var wanRoleLinks = [];
            if (!hasCloud) {
                wanRoleLinks = [];
            } else {
                shownIntfs.forEach(function(d, i) {
                    var targetY = withInterfaces ?
                                    offset.y + headH + i * rowH + rowH / 2 :
                                    offset.y + headH / 2;
                    if (d.isWANRole) {
                        wanRoleLinks.push({
                            source: {
                                id: d.id,
                                x: cloudX,
                                y: cloudY
                            },
                            target: {
                                id: d.id + '-' + i,
                                x: offset.x,
                                y: targetY
                            }
                        });
                    }
                });
            }

            var diagonal = this.d3.svg.diagonal()
                .source(function(d) {
                    return {
                        x: d.source.y,
                        y: d.source.x
                    };
                })
                .target(function(d) {
                    return {
                        x: d.target.y,
                        y: d.target.x
                    };
                })
                .projection(function(d) {
                    return [d.y, d.x];
                });

            var wanRolePaths = layer.selectAll('.wan-role-link')
                .data(wanRoleLinks, function(d) { return d.source.id; });

            var wanRolePathEnter = wanRolePaths.enter();
            wanRolePathEnter.append('path')
                .attr({
                    class: 'wan-role-link',
                    d: diagonal
                });
            wanRolePaths.exit().remove();

            wanRolePaths
                .style({
                    fill: 'none',
                    stroke: function(d) { return this.color(d.source.id); }.bind(this),
                    'stroke-width': LINK_STROKE_WIDTH
                })
                .attr({
                    d: diagonal
                })
                .transition()
                .duration(TRANSITION_DURATION.BASE)
                .attr({
                    d: diagonal
                })
                .style({
                    stroke: function(d) { return this.color(d.source.id); }.bind(this)
                });
        };
    });

    D3BubbleCluster.prototype.renderLinks = function(linkNodes, withInterfaces) {
        var d3 = this.d3;
        var layer = this.linksLayer;
        var linkPathIdFn = linkId('link-path');

        // We need the path to animate from right to left
        var linkPathDiagonal = d3.svg.diagonal()
            .source(childrenDivisionMinX)
            .target(parentDivisionMaxX)
            .projection(horizontalProjection);

        // To prevent curves from overlapping, we should ensure:
        // - x of all curve sources have the same value, aka the most left of
        // the objects in a division
        // - x of all curve targets have the same value, aka the most right of
        // the objects in the parent division
        // - then we draw 2 extra horizontal segments to complete the full path
        // This linkPathCustomDiagonal will help to create such a curve
        var linkPathCustomDiagonal = function(d, i) {
            var cDivisionMinX = (d.source.divisionHMargin || {}).minX;
            var pDivisionMaxX = (d.target.divisionHMargin || {}).maxX;
            var startSegment = '';
            var endSegment = '';
            if (d.source.x > cDivisionMinX) {
                startSegment = 'M' + d.source.x + ',' + d.source.y +
                               'L' + cDivisionMinX + ',' + d.source.y;
            }
            if (d.target.x < pDivisionMaxX) {
                endSegment = 'M' + pDivisionMaxX + ',' + d.target.y +
                             'L' + d.target.x + ',' + d.target.y;
            }
            var mainCurve = linkPathDiagonal(d, i);
            return startSegment + mainCurve + endSegment;
        };

        var linkPathStyle = {
            fill: 'none',
            stroke: function(d) { return this.color(d.source.colorKey); }.bind(this),
            'stroke-width': LINK_STROKE_WIDTH
        };
        var linkPathAttrs = {
            class: 'link',
            d: linkPathCustomDiagonal
        };
        var linkPaths = layer.selectAll('path.link')
            .data(linkNodes, linkPathIdFn);
        linkPaths.enter()
            .append('path')
            .style(linkPathStyle)
            .attr(linkPathAttrs);
        // animation
        linkPaths
            .style(linkPathStyle)
            .attr(linkPathAttrs)
            .transition()
            .duration(TRANSITION_DURATION.BASE)
            .style(linkPathStyle)
            .attr(linkPathAttrs)
            .attrTween('stroke-dasharray', function() {
                var len = this.getTotalLength();
                return function(t) { return (d3.interpolateString('0,' + len, len + ',0'))(t) };
            });
        if (withInterfaces) {
            linkPaths.each(this.linkTooltipBindFn);
        }
        linkPaths.exit().remove();
    };

    /**
     * FortiGate, FortiSwitch, FortiAP are now referred as FTNT devices.
     * We only have one render function but the 'ftntDeviceType' property
     * will help to differentiate FTNT devices.
     * Master FGT and others FGT/FSW are rendered on 2 different layers, and
     * layer is one of the parameter. This makes it possible to re-use the
     * function without worrying about the objects (based on different data
     * set) being wiped out.
     *
     * | Feature                     | FGT                  | FSW    | FortiAP
     * -----------------------------------------------------------------------
     * | Icon                        | yes                  | yes    |       |
     * | Display name                | hostname - vdom      | serial |       |
     * | Tooltip for hostname/serial | yes                  | no     |       |
     * | Tooltip for interfaces      | yes                  | yes    |       |
     * | Interface drilldown         | yes (for master FGT) | no     |       |
     */
    D3BubbleCluster.prototype.renderFtntDevice = inject.mark(function(csf_chart) {
        return function(layer, ftntData, baseFontSize, showInterfaces) {
            var rowHeight = BaseFtntObject.BASE_ROW_HEIGHT;
            var rowWidth = BaseFtntObject.BASE_ROW_WIDTH;
            var textMargin = INTF.TEXT_MARGIN;
            var textFontSize = baseFontSize;
            var baseIconSize = baseFontSize * 2;
            var headerTextX = (INTF.TEXT_MARGIN + 2) + baseIconSize;
            var squareSize = CONSTANTS.COLOR_SQUARE.WIDTH;
            var squarePadding = CONSTANTS.COLOR_SQUARE.PADDING;

            var ftntDeviceUniqueFn = function(d) {
                return d.sn + d.vdom + d.shownIntfs.map(csf_chart.pluckName).join('-');
            };
            var classFn = function(d) {
                var iconName = csf_chart.isFsw(d) ? 'ftnt-fortiswitch' : 'ftnt-fortigate';
                return 'f-icon ftnt-icon ' + iconName;
            };
            var headerTextFn = function(d) {
                var text = csf_chart.isFsw(d) ? d.name : d.hostname;
                return text;
            };
            var intfNameFn = function(d) {
                if (csf_chart.isFsw(d.parentFtnt)) {
                    return d.name + (!!d['fortilink-port'] ? '' : ' - ' + d.vlan);
                }
                return d.name;
            };
            var headHeightFn = function(d) {
                return d.getHeadHeight();
            };
            var headerTextIconY = function(d) {
                return d.getHeadHeight() / 2;
            };
            var iconTextFn = function(d) {
                return (d.getIcon() || {}).unicode;
            };

            var devices = layer.selectAll('g.ftnt-device').data(ftntData, ftntDeviceUniqueFn);
            var deviceAttrs = {
                transform: function(d) {
                    var offset = d.getOffset();
                    return transformXY(offset.x, offset.y);
                },
                class: 'ftnt-device'
            };
            var deviceEnter = devices.enter()
                .append('g')
                .attr(deviceAttrs);
            var headerTdAttrs = {
                width: rowWidth,
                height: headHeightFn,
                class: 'td head'
            };
            var header = deviceEnter.append('g')
                .attr({
                    class: 'th'
                });
            header.append('rect')
                .attr(headerTdAttrs);
            header.each(this.fgtTooltipBindFn);

            var headerIconAttrs = {
                x: textMargin,
                y: headerTextIconY,
                dy: toPx(baseIconSize / 2),
                'font-size': toPx(baseIconSize),
                class: classFn
            };

            header.append('text')
                .attr(headerIconAttrs)
                .text(iconTextFn);

            var headerTextAttrs = {
                x: headerTextX,
                y: headerTextIconY,
                dy: function(d) {
                    var h = d.getHeadHeight();
                    return toPx(h * TEXT_Y_ADJUSTMENT_FACTOR);
                },
                'font-size': toPx(textFontSize),
                class: 'device-name'
            };
            header.append('text')
                .attr(headerTextAttrs)
                .text(headerTextFn);

            var intfGAttrs = {
                transform: function(d, i) {
                    var h = d.parentFtnt.getHeadHeight();
                    return transformXY(0, h + i * rowHeight);
                },
                class: 'intf tr'
            };
            var intfRectAttrs = {
                width: rowWidth,
                height: rowHeight,
                class: function(d) {
                    return d.parentFtnt.isRoot ? 'td master' : 'td';
                }
            };
            var squareAttrs = {
                width: squareSize,
                height: squareSize,
                x: rowWidth - squareSize - squarePadding,
                y: squarePadding,
                class: 'color-square'
            };
            var squareStyles = {
                fill: function(d) { return this.color(d.id); }.bind(this),
                'pointer-events': 'none'
            };
            var intfTextAttrs = {
                x: textMargin,
                y: rowHeight / 2,
                dy: toPx(rowHeight / 8),
                'font-size': toPx(textFontSize),
                class: 'intf-text'
            };

            var intfs = devices.selectAll('g.intf').data(showInterfaces ? function(d) {
                return d.shownIntfs.map(function(intf) {
                    return angular.extend(intf, {parentFtnt: d});
                });
            } : [], function(d) {
                return d.id;
            });

            var intfsEnter = intfs.enter()
                .append('g')
                .attr(intfGAttrs);

            intfsEnter.append('rect')
                .attr(intfRectAttrs)
                .style({
                    cursor: 'pointer',
                    fill: 'white'
                });
            intfsEnter.append('rect')
                .attr(squareAttrs)
                .style(squareStyles);

            intfsEnter.append('text')
                .attr(intfTextAttrs)
                .style({
                    'pointer-events': 'none'
                })
                .text(intfNameFn);

            intfs.exit().remove();

            devices.exit().remove();

            // transition for header
            quickTransition(devices.selectAll('rect.td.head'), headerTdAttrs);
            quickTransition(devices.selectAll('text.ftnt-icon'), headerIconAttrs);
            quickTransition(devices.selectAll('text.device-name'), headerTextAttrs);

            // transition for interfaces rows
            quickTransition(intfs, intfGAttrs);
            quickTransition(intfs.selectAll('rect.td'), intfRectAttrs);
            quickTransition(intfs.selectAll('rect.color-square'), squareAttrs, squareStyles);
            quickTransition(intfs.selectAll('text.intf-text'), intfTextAttrs);

            // the whole FTNT Device transform to its offset
            quickTransition(devices, deviceAttrs);

            // clickable action or tooltip for interface
            intfs.selectAll('rect.td').each(this.interfaceTooltipBindFn);
            intfs.selectAll('rect.td.master').on('click', this.interfaceClick);
        };
    });

    D3BubbleCluster.prototype.renderDeviceTypes = inject.mark(function(iconCode) {
        return function(layer, deviceTypeData, baseFontSize) {
            var rowHeight = DeviceType.ROW_HEIGHT;
            var rowWidth = DeviceType.WIDTH;
            var iconX = INTF.TEXT_MARGIN;
            var iconSize = FTNT_ICON_SIZE;
            var textFontSize = baseFontSize;
            var roundedR = rowHeight / 8;
            var mWidth = DeviceType.MASK_WIDTH;
            var countTextX = iconSize + iconX * 2;

            var maxCount = 0;
            deviceTypeData.forEach(function(d) {
                d.children.forEach(function(type) {
                    if (type.count > maxCount) {
                        maxCount = type.count;
                    }
                });
            });
            var maxCountLength = (maxCount + '').length;
            var lWidth = iconSize + iconX * 2 + maxCountLength * iconSize / 2 + mWidth;
            var rWidth = rowWidth - lWidth;
            var typeTextX = lWidth;

            var devTypeGrps = layer.selectAll('g.device-type').data(deviceTypeData, function(d) {
                return d.upstreamSrcIntfId;
            });
            var devTypeGrpAttrs = {
                transform: function(d) {
                    return transformXY(d.offset.x, d.offset.y);
                },
                class: 'device-type'
            };
            devTypeGrps.enter()
                .append('g')
                .attr(devTypeGrpAttrs);

            var devTypeGAttrs = {
                transform: function(d, i) {
                    return transformXY(0, i * rowHeight);
                },
                class: 'type device-type-counter'
            };
            var devTypeLeftRectAttrs = {
                width: lWidth,
                height: rowHeight,
                rx: roundedR,
                ry: roundedR,
                class: 'left-part'
            };
            var devTypeRightRectAttrs = {
                width: rWidth,
                height: rowHeight,
                x: lWidth,
                rx: roundedR,
                ry: roundedR,
                class: 'right-part'
            };
            var devTypeMaskRectAttrs = {
                width: mWidth,
                height: rowHeight,
                x: lWidth - (mWidth / 2),
                class: 'mask'
            };
            var devTypeMaskRectStyles = {
                'stroke-dasharray': mWidth + ',' + rowHeight + ',' + (mWidth + rowHeight)
            };
            var devTypeCountAttrs = {
                x: countTextX,
                y: rowHeight / 2,
                dy: toPx(rowHeight / 6),
                'font-size': toPx(textFontSize),
                class: 'type-count'
            };

            var devTypeTextAttrs = {
                x: typeTextX,
                y: rowHeight / 2,
                dy: toPx(rowHeight / 6),
                'font-size': toPx(textFontSize),
                class: 'type-text right-part'
            };

            var deviceIconAttrs = {
                class: function(d) { return 'icon f-icon ' + d.icon.name; },
                'font-family': function(d) { return d.icon.fontFamily || ''; },
                'font-size': toPx(textFontSize),
                x: iconX,
                y: (rowHeight + iconSize) / 2,
                fill: DEFAULT_ICON_COLOR,
                dy: toPx(- iconSize * TEXT_Y_ADJUSTMENT_FACTOR)
            };
            var iconTextFn = function(d) { return d.icon.code || ''; };

            var alertIconCodeObj = iconCode.getCode('fa-warning' || '');

            var alertIconAttrs = {
                class: 'alert',
                fill: 'white',
                'font-family': alertIconCodeObj.fontFamily,
                'font-size': toPx(textFontSize),
                x: rowWidth - iconSize,
                y: (rowHeight + iconSize) / 2,
                dy: toPx(- iconSize * TEXT_Y_ADJUSTMENT_FACTOR)
            };

            var types = devTypeGrps.selectAll('g.type').data(function(d) {
                return d.children;
            });

            var typesEnter = types.enter()
                .append('g')
                .attr(devTypeGAttrs);

            typesEnter.append('rect')
                .attr(devTypeLeftRectAttrs);
            typesEnter.append('rect')
                .attr(devTypeRightRectAttrs);
            typesEnter.append('rect')
                .attr(devTypeMaskRectAttrs)
                .style(devTypeMaskRectStyles);

            typesEnter.append('text')
                .attr(deviceIconAttrs)
                .style({
                    'pointer-events': 'none'
                })
                .text(iconTextFn);

            typesEnter.append('text')
                .attr(devTypeCountAttrs)
                .style({
                    'pointer-events': 'none'
                })
                .text(function(d) { return d.count; });

            typesEnter.append('text')
                .attr(devTypeTextAttrs)
                .style({
                    'pointer-events': 'none'
                })
                .text(function(d) { return d.typeTranslated; });

            var routerDevs = types.selectAll('text.alert').data(function(d) {
                return d.type === 'router-nat-device' ? [1] : [];
            });
            routerDevs.enter()
                .append('text')
                .attr(alertIconAttrs)
                .style('cursor', 'pointer')
                .text(alertIconCodeObj.unicode || '');
            quickTransition(routerDevs, alertIconAttrs);
            routerDevs.each(this.natRouterTooltipBindFn);

            types.exit().remove();

            devTypeGrps.exit().remove();

            // transition
            quickTransition(types, devTypeGAttrs);
            quickTransition(types.selectAll('rect.left-part'), devTypeLeftRectAttrs);
            quickTransition(types.selectAll('rect.mask'),
                            devTypeMaskRectAttrs, devTypeMaskRectStyles);
            quickTransition(types.selectAll('rect.right-part'), devTypeRightRectAttrs);
            quickTransition(types.selectAll('text.type-count'), devTypeCountAttrs);
            quickTransition(types.selectAll('text.type-text'), devTypeTextAttrs);
            quickTransition(types.selectAll('text.icon'), deviceIconAttrs, null, iconTextFn);

            // the whole device types object transform to its offset
            quickTransition(devTypeGrps, devTypeGrpAttrs);
        };
    });

    /**
     * Only CSF master FGT has the cloud icon if there is WAN Role interface
     * in the traffic
     **/
    D3BubbleCluster.prototype.updateCsfMasterCloud = inject.mark(function(cloudSvg) {
        return function(visible, baseFontSize) {
            var data = visible ? [1] : [];
            var textFontSize = baseFontSize;
            var cloud = this.cloudLayer.selectAll('.cloud')
                .data(data);

            var enter = cloud.enter()
                .append('g')
                .attr({
                    class: 'cloud'
                });

            var cloudPathTransform = transformXY(CONSTANTS.DIVISION.MARGIN.LEFT,
                CONSTANTS.DIVISION.MARGIN.TOP) +  ' scale(' + cloudSvg.SCALE + ')';
            var textX = CONSTANTS.DIVISION.MARGIN.LEFT + cloudSvg.WIDTH / 2;
            var textY = CONSTANTS.DIVISION.MARGIN.TOP + cloudSvg.HEIGHT / 2;
            var cloudPath = enter.append('g').attr({
                class: 'cloud-path',
                transform: cloudPathTransform
            });
            if (cloudPath[0][0]) {
                // !== null only when enter, hence do this 1 time
                cloudPath.node().appendChild(this.cloudPath);
            }

            enter.append('text')
                .attr({
                    'text-anchor': 'middle',
                    class: 'internet-text',
                    'font-size': toPx(textFontSize),
                    x: textX,
                    y: textY
                })
                .text($.getInfo('Upstream'));

            quickTransition(this.cloudLayer.selectAll('.cloud-path'), {
                transform: cloudPathTransform
            });

            quickTransition(this.cloudLayer.selectAll('.internet-text'), {
                    'font-size': toPx(textFontSize),
                    x: textX,
                    y: textY
                });

            cloud.exit().remove();
        };
    });

    D3BubbleCluster.prototype.transitionDone = function(transition, callback) {
        if (transition.size() === 0) { callback() }
        var n = 0;
        transition
            .each(function() { ++n; })
            .each('end', function() {
                if (!--n) {
                    callback.apply(this, arguments);
                }
            });
    };

    D3BubbleCluster.prototype.updateSize = function(width, height) {
        [this.svg, this.overlayRect].forEach(function(obj) {
            obj.attr({
                width: width,
                height: height
            });
        }.bind());
    };

    D3BubbleCluster.prototype.updateCursor = function() {
        var that = this;
        that.clusterContainerLayer.selectAll('circle.cluster-container').each(function() {
            var $this = that.d3.select(this);
            var intf = $this.data()[0];
            that.d3.select(this).style({
                cursor: (that.currentZoomInClusterId === intf.upstreamSrcIntfId ||
                         that.currentZoomInNodeId) ? 'zoom-out' : 'zoom-in'
            });
        });
        that.svg.selectAll('g.node circle').each(function() {
            var $this = that.d3.select(this);
            var node = $this.data()[0];
            var id = node.srcIntfId + node.name;
            $this.style({
                cursor: (that.currentZoomInNodeId === id) ? 'zoom-out' : 'zoom-in'
            });
        });
    };

    D3BubbleCluster.prototype.zoomReset = function() {
        this.currentZoomInNodeId = null;
        this.currentZoomInClusterId = null;
        this.svg.transition()
            .duration(TRANSITION_DURATION.ZOOM)
                .call(this.zoom
                      .translate(this.chartData.translate || [0, 0])
                      .scale(this.chartData.fullViewScale || 1).event);
        this.updateCursor();
    };

    D3BubbleCluster.prototype.zoomToNode = function(d) {
        this.currentZoomInClusterId = null;
        var diameter = 2 * d.r;
        var scale = BUBBLE_ZOOM_SCALE / Math.max(diameter / this.width, diameter / this.height);
        var translate = [this.width / 2 - scale * d.x, this.height / 2 - scale * d.y];
        this.svg.transition()
            .duration(TRANSITION_DURATION.ZOOM)
                .call(this.zoom.translate(translate).scale(scale).event);
        this.updateCursor();
    };

    D3BubbleCluster.prototype.zoomToCluster = function(clusterId) {
        this.currentZoomInNodeId = null;
        var bBox = BubbleCluster.clusterMap[clusterId];
        if (bBox) {
            var dx = bBox.x2 - bBox.x1,
                dy = bBox.y2 - bBox.y1,
                x = (bBox.x1 + bBox.x2) / 2,
                y = (bBox.y1 + bBox.y2) / 2;
            var scale = CLUSTER_ZOOM_SCALE / Math.max(dx / this.width, dy / this.height),
                translate = [this.width / 2 - scale * x, this.height / 2 - scale * y];
            this.svg.transition()
                .duration(TRANSITION_DURATION.ZOOM)
                    .call(this.zoom.translate(translate).scale(scale).event);
            this.updateCursor();
        }
    };

    D3BubbleCluster.prototype.restrictZoomBehavior = function() {
        // Preventing applying zooming/panning behavior on right click
        var that = this;
        return function() {
            var stop = that.d3.event.button || that.d3.event.ctrlKey;
            if (stop) {
                that.d3.event.stopImmediatePropagation();
            }
        };
    };

    D3BubbleCluster.prototype.bubbleClicked = function() {
        var that = this;
        return function(d) {
            if (that.d3.event.defaultPrevented) {
                return;
            }
            var id = d.srcIntfId + d.name;
            if (that.currentZoomInNodeId === id) {
                // let reset (zoom out)
                that.zoomReset();
            } else {
                // let zoom in
                that.currentZoomInNodeId = id;
                that.zoomToNode(d);
            }
        };
    };

    D3BubbleCluster.prototype.overlayClicked = function() {
        var that = this;
        return function() {
            if (that.d3.event.defaultPrevented) {
                return;
            }
            if (that.currentZoomInClusterId || that.currentZoomInNodeId) {
                that.zoomReset();
            }
        };
    };

    D3BubbleCluster.prototype.clusterCircleClicked = function() {
        var that = this;
        return function(d) {
            if (that.d3.event.defaultPrevented) {
                return;
            }
            var clusterId = d.upstreamSrcIntfId;
            if (that.currentZoomInClusterId === clusterId) {
                // clicking on same cluster will reset zoom
                that.zoomReset();
            } else {
                // zoom to that new cluster
                that.currentZoomInClusterId = clusterId;
                that.zoomToCluster(clusterId);
            }
        };
    };

    D3BubbleCluster.prototype.tooltipColorKeyGetter = function(d) {
        return d.srcIntfId;
    };

    D3BubbleCluster.prototype.createWhiteGlowFilter = function() {
        var filter = this.defs.append('filter')
            .attr({
                id: 'whiteglow'
            });

        filter.append('feFlood')
            .attr({
                'flood-color': 'white'
            });

        filter.append('feComposite')
            .attr({
                in2: 'SourceAlpha',
                operator: 'in'
            });

        filter.append('feGaussianBlur')
            .attr({
                stdDeviation: 0.8
            });

        var compTransfer = filter.append('feComponentTransfer');
        compTransfer.append('feFuncA')
            .attr({
                type: 'linear',
                slope: 3,
                intercept: 0
            });

        var feMerge = filter.append('feMerge');

        feMerge.append('feMergeNode');
        feMerge.append('feMergeNode')
            .attr('in', 'SourceGraphic');
    };

    /**
     * @ngdoc directive
     * @name ng.directive:fD3BubbleClusterChart
     *
     * @description
     * This directive will render an interface list and clusters of bubble with d3.
     * Arguments (* => required):
     *   * "data": An array of objects where each object must have the properties "name" and
     *       "value".
     *     "click": An function that will be called when a circle is clicked with the respective
     *       entry.
     *     "tooltipFormatter": A function that will be called when a circle is hovered over with the
     *       respective entry. Must return tooltip content.
     *
     */
    var d3BubbleClusterChart = function() {
        return {
            restrict: 'A',
            scope: {
                data: '=',
                click: '=',
                interfaceClick: '=',
                tooltipFormatter: '=',
                shownKeyFormatter: '=',
                bubbleOption: '=',
                accessDevice: '=',
                topology: '=',
                apiUrlParams: '='
            },
            controller: D3BubbleCluster
        };
    };

    return function(providers) {
        providers.$compile.directive('fD3BubbleClusterChart', d3BubbleClusterChart);
    };
});
