define(['ml_module'], function(module) {
	'use strict';

	var S_TO_MS = 1000,
		MS_TO_S = 1 / S_TO_MS,
		TZOFFSET_TO_SECONDS = 60;

	var dateToLocalSeconds = function(date) {
		//unix epoch in local time!
		var offsetSeconds = new Date().getTimezoneOffset() * TZOFFSET_TO_SECONDS;
		return Math.round(date.getTime() * MS_TO_S - offsetSeconds);
	};

	/**
	* A facet that is being actively searched
	* @constructor
	* @param {SearchFacet|ActiveFacetModel} [facet] Search facet that is being searched
	*   or another ActiveFacetModel object to clone
	* @param {SearchFacet.ValueOption|ComplexValue} [value]  Currently selected key/value combo
	*/
	activeFacetModelFactory.$inject = ['SearchFacet', 'ComplexValue', 'lang'];
	function activeFacetModelFactory(SearchFacet, ComplexValue, lang) {
		ActiveFacetModel.prototype.assign = function(other) {
			ActiveFacetModel.call(this, other);
		};
		return ActiveFacetModel;
		function ActiveFacetModel(facet, value) {
			if (facet instanceof ActiveFacetModel) {
				facet = facet.facet;
				if (facet) {
					value = facet.normalizeOption(value);
				}
			}
			this.$new = !facet;
			this.facet = facet;
			this.name = facet && lang(facet.name);
			this.value = value;
			this.$valid = true;
			if (value &&
					!(value instanceof SearchFacet.ValueOption || value instanceof ComplexValue)) {
				throw Error('Please supply a Searchfacet.ValueOption or ' +
					'ComplexValue object as value');
			}
			this.$editing = function() { return false };
		}
	}

	/**
	* controller for fActiveFacet
	*/
	ActiveFacet.$inject = ['injector', '$scope'];
	function ActiveFacet(injector, $scope) {
		this.scope = $scope;
		this.inputs = {};
		this._undo = [];

		$scope.$editing = false;
		$scope.$preSelected = {};

		var inject = {
			updateOptions: ['$q', 'lang'],
			possibleFacets: ['SearchFacet', 'lang'],
			possibleValues: ['SearchFacet', 'ComplexValue'],
			preSelect: ['lang'],
			suggest: ['SearchFacet'],
			valueUpdated: ['$q'],
			datePickerUpdate: ['ComplexValue'],
			datePickerGetDate: ['SearchFacet', 'ComplexValue'],
			datePickerSetDate: [],
			keyPress: ['ComplexValue'],
			tab: ['$q'],
			pushUndo: ['ActiveFacetModel'],
			updateFocus: ['$timeout']
		};
		var fn;
		for (fn in inject) {
			this[fn].$inject = inject[fn];
			this[fn] = injector.bind(this[fn], this);
			this[fn].toString = genToString(fn);
		}
		//make a new select object or it will modify the prototype.select :(
		var selectInject = {
			facet: ['lang', 'SearchFacet'],
			value: ['SearchFacet', 'ComplexValue', '$q'],
			current: ['SearchFacet', 'ComplexValue']
		};
		var select = this.select;
		this.select = {};
		for (fn in selectInject) {
			//inject everything but the variable which matches the fn name.
			select[fn].$inject = selectInject[fn];
			this.select[fn] = injector.bind(select[fn], this);
		}
		this.possibleValues = this.possibleValues.bind(this);
		this.remove = this.remove.bind(this);

		$scope.$watch('$model.name', this.updateOptions);
		$scope.$watch('$model.value', this.valueUpdated);
		$scope.$watch('$editing', function(value) {
			//set focus if truthy, otherwise remove focus
			this.updateFocus(value ? this.inputs[value] : null, !!value);
		}.bind(this));
		$scope.$watch(function() {
			var d = this.datePickerGetDate();
			return d && d.getTime();
		}.bind(this), function() {
			var value = this.datePickerGetDate();
			this.datePickerSetDate(value);
		}.bind(this));
		this.datePickerUpdate();
		function genToString(name) {
			return function() {return 'function ' + name + '()'};
		}
	}

	ActiveFacet.FOCUS_DELAY = 10;


	ActiveFacet.emptyArray = [];
	ActiveFacet.value = function valueFilter(value) {
		return function(kv) {
			return kv.value === value;
		};
	};

	ActiveFacet.prototype = {
		init: function init(facetedSearch) {
			this.facetedSearch = facetedSearch;
			this.focused = null;
			var scope = this.scope;
			if (scope.$model && scope.$model.$new) {
				scope.$model.$new = false;
			}
			scope.$model.$editing = this.editing.bind(this);
		},
		addInput: function(input) {
			if (input.name in this.inputs) {
				throw new Error('FacetInput names inside an activeFacet must be unique!' +
					'Tried to add duplicate input for ' + input.name);
			}
			this.inputs[input.name] = input;
		},
		removeInput: function(input) {
			delete this.inputs[input.name];
		},
		isEmpty: function() {
			return !this.scope.$model.facet;
		},
		edit: function(name) {
			if (name in this.inputs) {
				this.scope.$editing = name;
			} else {
				throw new Error('Unable to edit ' + name + ': FacetInput does not exist');
			}
		},
		editing: function() {
			return this.scope.$editing;
		},
		/**
		* Filter and sort a list of items based on their relevance as autocomplete suggestions
		* @param {Object[]} list List of items to filter/sort
		* @param {String} [searchValue] value to autocomplete or undefined.
		* @param {Function(item):String} [pluckValue] Function that returns the searchable string
		*   value of each item.
		* @param {Function} [genRelevanceFn=SearchFacet.defaultGenRelevanceFn] Function that takes
		*   a searchValue and returns a function which returns a relevance rating for each
		*   suggestion value/item. (higher number is more relevant, null is not relevant at all).
		*   @see SearchFacet.defaultGenRelevanceFn
		* @returns {Object[]} Array of items from the list that qualify as autocomplete suggestions
		*   sorted by relevance.
		*/
		suggest: function(SearchFacet, list, searchValue, pluckValue, genRelevanceFn, sort, limit) {
			pluckValue = pluckValue || function(v) { return v };
			var getRelevance = (genRelevanceFn || SearchFacet.defaultGenRelevanceFn)(searchValue);
			// Relevancy in the following order:
			// exact match
			// starts with
			// sub-word starts with
			// contains
			var preSliced = list.map(exec).filter(notNull);
			if (preSliced.length > limit) {
				preSliced = preSliced.slice(0, limit);
			}
			return preSliced.sort(relevance).map(pluckItem);

			function notNull(details) { return details.relevance !== null }
			function pluckItem(details) { return details.item }

			function exec(item) {
				var value = pluckValue(item);
				return {
					value: value,
					item: item,
					relevance: getRelevance(value, item)
				};
			}
			function relevance(a, b) {
				//higher relevance should sort to a lower index so reverse this comparison
				var a_value = Array.isArray(a.value) ? a_value = a.value[0] : a.value;
				var b_value = Array.isArray(b.value) ? b_value = b.value[0] : b.value;
				return (b.relevance - a.relevance) ||
					sort && sort(a.item, b.item) ||
					//string compare if relevance difference == 0
					//TODO: may require optimization if strings only contain codes < 128?
					//Probably fast enough for now.
					// http://jsperf.com/operator-vs-localecompage/11
					(typeof a_value === 'string' && typeof b_value === 'string') &&
					a_value.localeCompare(b_value) ||
					//other faceted search object types support localeCompare too
					(('localeCompare' in a.item) && ('localeCompare' in b.item)) &&
					a.item.localeCompare(b.item) ||

					(typeof a_value === 'number' && typeof b_value === 'number') &&
					b_value - a_value ||

					(a_value && !b_value ? 1 : (b_value && !a_value ? -1 : 0));
			}
		},
		possibleFacets: function(SearchFacet, lang, searchName) {
			var langTable = {},
				facets = this.facetedSearch.facets();
			if (!facets || !facets.length) {
				return ActiveFacet.emptyArray;
			}
			return this.suggest(facets.filter(uniqueCheck.bind(this)),
				searchName, pluckValue);
			function uniqueCheck(facet) {
				/* jshint validthis: true */
				return (this.scope.$model.facet === facet || !facet.unique ||
					!this.facetedSearch.activeFacets.some(matches));
				function matches(otherFacet) {
					return otherFacet.facet === facet;
				}
			}
			function pluckValue(facet) {
				var facetName = langTable[facet.name] || lang(facet.name).toString();
				var selectors = Object.keys(facet.selectors).map(pluckSelector);
				langTable[facet.name] = facetName;
				return [facetName, facet.name].concat(selectors);
				function pluckSelector(s) { return facet.selectors[s] }
			}
		},
		possibleValues: function(SearchFacet, ComplexValue, searchName) {
			if (!this.scope.$valueOptions || !this.scope.$model.facet) {
				return ActiveFacet.emptyArray;
			}
			var facet = this.scope.$model.facet,
				value = this.scope.$model.value,
				key = value && (value instanceof SearchFacet.ValueOption ?
					value.key : value),
				keys = key && key instanceof ComplexValue ?
					key.values : [key],
				modelValues = this.facetedSearch.scope.model[facet.id],
				otherKeys = ComplexValue
					.flattenValues(modelValues || [])
					.filter(notMatchKeys(keys)),
				options = this.scope.$valueOptions.filter(notMatchKeys(otherKeys));
			return this.suggest(options, searchName, pluckValue,
				facet.genRelevanceFn, facet.sort, facet.suggestionLimit);

			function pluckValue(kv) {
				return [kv.value]
					.concat(kv.value !== kv.key ? [kv.key] : [])
					.concat(kv.description ? [kv.description] : []);
			}

			function notMatchKeys(keys) {
				return function(vo) {
					var hasIntersection = (vo instanceof SearchFacet.ValueOption) ||
						(vo instanceof ComplexValue);
					return hasIntersection ?
						!vo.intersection(keys) : keys.indexOf(vo) === -1;
				};
			}
		},
		cancel: function(ifEmpty, $event) {
			if ($event) {
				$event.stopPropagation();
			}
			if (ifEmpty && !this.isEmpty()) {
				return;
			}
			if (this.focused) {
				this.undo();
			}
			this.scope.$editing = null;

		},
		remove: function(ifEmpty, $event) {
			var scope = this.scope;
			if ($event) {
				$event.stopPropagation();
			}
			if (ifEmpty && !this.isEmpty()) {
				return false;
			}
			var af = this.facetedSearch.activeFacets,
				index = af.indexOf(scope.$model);
			if (index > -1) {
				af.splice(index, 1);
			}
			this.facetedSearch.syncToModel();
			this.facetedSearch.focus(true);
			return true;
		},
		pushUndo: function(ActiveFacetModel) {
			this._undo.push(new ActiveFacetModel(this.scope.$model));
		},
		undo: function() {
			if (this._undo.length) {
				this.scope.$model.assign(this._undo.pop());
			}
		},
		/**
		* Tab to the next control.
		* @param {Boolean} [shiftKey] If shiftKey is true, tab backwards
		* @param {Boolean} [next=true] If next is false, then stop editing instead
		*/
		tab: function($q, shiftKey, input) {
			var scope = this.scope;

			if (!shiftKey) {
				this.select[input.editGroup](true, true);
			}
			if (shiftKey) {
				if (input.editGroup === 'value') {
					this.scope.$edit('facet');
				} else {
					this.facetedSearch.editPrev(scope.$model);
					this.remove(true);
				}
			} else {
				if (input.editGroup === 'facet') {
					this.scope.$edit('value');
				} else {
					this.facetedSearch.editNext(scope.$model);
				}
			}
			return true;
		},
		select: {
			current: function(SearchFacet, ComplexValue) {
				// if (this.scope.$editing) {
				// 	this.select[this.scope.$editing](true);
				// }
				var value = this.scope.$model.value;
				if (this.scope.$editing) {
					if((value instanceof ComplexValue || value instanceof SearchFacet.ValueOption)
						&& value.value !== "")
						this.select[this.scope.$editing]();
					else
						this.select[this.scope.$editing](true);
				}
			},
			/**
			* Select a facet based on the current model state or provided string/facet. If facet is
			* falsy or `true` then the first suggested facet will be chosen. Accepts a boolean for
			* compatibility with select.value() below.
			* @param {DI} translateFilter Injected translateFilter service.
			* @param {DI} SearchFacet Injected SearchFacet constructor.
			* @param {String|SearchFacet|Boolean} [facet] (partial) name of facet or facet.
			* @param {Boolean} [dontEdit=false] Don't edit the value when finished.
			* @returns false if facet was not set successfully.
			*/
			facet: function(lang, SearchFacet, facet, dontEdit) {
				// allow `true` for compatibility with select.value() below.
				if (facet === true) {
					facet = null;
				}
				/* jshint validthis: true *//* this === ActiveFacet */
				if (!facet) {
					facet = this.scope.$model.name;
				}
				//allow undefined facet, just pick the first one
				if (!(facet instanceof SearchFacet)) {
					facet = this.preSelect('facet', 0) || this.possibleFacets(facet)[0];
				}
				if (!facet) {
					return false;
				}
				var scope = this.scope,
					name = lang(facet.name).toString();
				scope.$model.name = name;
				scope.$model.facet = facet;
				//allow $watch('$model.name'...) to execute
				if (!dontEdit) {
					scope.$edit('value');
				}
				return true;
			},
			/**
			* Select a value based on the current model state or provided
			* string/SeachFacet.ValueOption.
			* If there is a value typed into the input, then:
			* - if the user has pre-selected an autocomplete value it will be used
			* - if the user has not pre-selected an autocomplete value then the partial value will
			*	be used.
			* If there is no current user value then:
			* - if `value` is `true` choose the first suggested value
			* - if `value` is falsy, do nothing.
			* @param {DI} SearchFacet Injected Searchfacet constructor.
			* @param {DI} $q service.
			* @param {String|SearchFacet.ValueOption|Boolean} [value] value to select.
			* @returns true if value was set, or a promise if value is pending to be set.
			*/
			/* jshint unused: false */
			value: function(SearchFacet, ComplexValue, $q, value) {
				/* jshint validthis: true *//* this === ActiveFacet */
				var scope = this.scope,
					allowedType,
					mv = scope.$model.value,
					useSuggestion = value === true,
					preSelected = scope.$preSelected,
					forceSuggestion = !scope.$model.facet.allowUserInput &&
						!(value instanceof SearchFacet.ValueOption);
				/* jshint eqnull: true */
				/* deliberately using == null which is also true for undefined */
				if (value == null || useSuggestion) {
					value = scope.$preSelected.value;
					if (value == null) {
						value = scope.$model.value;
					}
				}
				if (!scope.$model.facet.allowUserInput) {
					useSuggestion = !(value instanceof SearchFacet.ValueOption);
				}
				if (useSuggestion || forceSuggestion) {
					//use currently preSelected option if available
					var suggestion = this.preSelect('value', 0);
					if (suggestion == null) {
						//try to get the first autocomplete value.
						suggestion = this.preSelect('value', 1);
					}
					if (suggestion != null) {
						value = suggestion;
					}
					//do nothing if there's no good default.
					if (value == null) {
						scope.$preSelected = preSelected;
						return;
					}
				}
				allowedType = value instanceof SearchFacet.ValueOption ||
					value instanceof ComplexValue;
				var done = true;
				if (allowedType) {
					var merge = true,
						values = mv instanceof ComplexValue && mv.values.length,
						modifiers = mv instanceof ComplexValue && mv.modifiers.length;
					if(modifiers || (values && values > 1)) {
						if(value instanceof SearchFacet.ValueOption)
							value = value.key;
						if(!(value instanceof ComplexValue)) {
							value = [value];
						} else {
							if(value.modifiers.length) {
								merge = false;
							}
						}

						if(merge) {
							if(mv !== value)
								scope.$model.value = mv.replaceSelectedValues(value, scope.$model.facet);
						} else {
							scope.$model.value = value;
						}
					} else {
						scope.$model.value = value;
					}
				} else {
					if (scope.$model.facet) {
						var facet = scope.$model.facet;
						var key = facet.reverseLookup(value);
						done = $q.when(key).then(function(key) {
							if (key) {
								scope.$model.value = facet.normalizeOption(key, value);
							} else if (value) {
								scope.$model.value = new ComplexValue(value);
							} else {
								scope.$model.value = undefined;
							}
						});
					}
				}
				$q.when(done).then(function() {

					scope.$preSelected.value = null;
					this.facetedSearch.syncToModel();
					this.scope.$editing = false;
				}.bind(this));
				return done;
			}
		},

		/**
		* returns true if keypress has been handled and should be prevented.
		*/
		keyPress: function(ComplexValue, which, input) {
			if (input.name === 'facet') {
				if (['=', '<', '>', '!'].indexOf(which) > -1) {
					var valueInput = this.inputs.value;
					if (which !== '=' && valueInput.modify) {
						valueInput.modify(which);
					}
					this.select.facet();
					return true;
				}
			}
			return false;
		},
		/**
		* Pre select the current auto completed facet or value
		* @param {DI} translateFilter translateFilter service provided by DI.
		* @param {String} name Name of the input that is being auto completed.
		* @param {Number} delta Adjust auto completed selected index (0 for no adjustment).
		* @param {Boolean} [accept=false] Accept preselected value and assign to model.
		* @returns {String} Newly preselected value (I18n facet name or user value).
		*/
		preSelect: function(lang, name, delta) {
			var list = null,
				scope = this.scope;
			if (name === 'facet') {
				list = this.possibleFacets(scope.$model.name);
			} else if (name === 'value') {
				list = this.possibleValues(scope.$fmtValue());
			}
			if (list && list.length) {
				var index = list.indexOf(scope.$preSelected[name]);
				if (delta) {
					if (delta < 0 && index === -1) {
						index = 0;
					}
					//add list.length and then get modulus to counter -1
					index = (index + delta + list.length) % list.length;
				}
				var value = list[index];
				if (value !== undefined) {
					if (delta) {
						scope.$preSelected[name] = value;
					}
					return value;
				}
			}
			return null;
		},
		focusPending: null,
		/**
		* Delay input.doFocus call to let other inputs in the same editGroup to gain focus first.
		*/
		delayedFocus: function($timeout, input, value) {
			var fp = (this.focusPending = this.focusPending || {});
			if (fp[input.editGroup]) {
				$timeout.cancel(fp[input.editGroup]);
			}
			fp[input.editGroup] = $timeout(function() {
				fp[input.editGroup] = null;
				input.doFocus(value);
			}, ActiveFacet.FOCUS_DELAY, true);
		},
		/*
		* @param {FacetInput|null} input if null, use currently focused input (or do nothing)
		* @param {Boolean} value true to focus, false to blur
		*/
		updateFocus: function($timeout, input, value) {
			var dontEdit = !input,
				editing;
			if (input === null) {
				input = this.focused;
				editing = this.editing();
				if (!input || value) {
					return;
				}
			}
			if (this.scope.$pseudoFocus) {
				return;
			}
			if (value) {
				if (this.focused && this.focused !== input) {
					this.focused.focus(false);
				}
				this.focused = input;
			} else {
				if (this.focused === input) {
					this.focused = null;
				}
			}
			// Retry focus until the element becomes visible. Could be better.
			var retries = 0;
			function doFocus() {
				$timeout(function() {
					if (!input.focus(value)) {
						if (retries >= 10) {
							//fweb.log.error('ActiveFacet: Unable to focus ' + input.name);
							throw new Error('ActiveFacet: Unable to focus ' + input.name);
						} else if (value) {
							// don't retry if we are un-focusing
							++retries;
							doFocus();
						}
					}
				});
			}
			doFocus();
			var focused = !!this.focused;
			if (focused === value) {
				editing = (this.editing() === input.editGroup);
				if (editing !== value) {
					this.pushUndo();
				}
				this.scope.$editing = value && input.editGroup;
			}
		},
		pseudoFocus: function(name, value) {
			this._pseudoFocus = this._pseudoFocus || {};
			this._pseudoFocus[name] = value;
			if (!value) {
				delete this._pseudoFocus[name];
			}
			this.scope.$pseudoFocus = Object.keys(this._pseudoFocus).length > 0;
		},
		datePickerGetDate: function(SearchFacet, ComplexValue) {
			var value = null,
				scope = this.scope;
			if (scope.$model.facet && scope.$model.facet.type === 'date_time') {
				value = scope.$model.value;
				if (value instanceof ComplexValue) {
					if (value.selectedValue()) {
						value = new Date(value.selectedValue() * S_TO_MS);
					} else {
						value = undefined;
					}
				} else if (value instanceof SearchFacet.ValueOption) {
					value = new Date(value.key * S_TO_MS);
				}
			}
			return value;
		},
		datePickerSetDate: function(value) {
			if (this.scope.datePickerOptions) {
				value = value || this.datePickerGetDate();
				this.scope.datePickerDate = value;
			}
		},
		datePickerUpdate: function(ComplexValue) {
			var scope = this.scope,
				oneDay = new Date(0, 0, 0, 23, 59, 59).getTime() -
					new Date(0, 0, 0, 0, 0, 0).getTime(),
				oneDaySeconds = oneDay * MS_TO_S;
			scope.datePickerOptions = null;
			if (scope.$model.facet && scope.$model.facet.type === 'date_time') {
				scope.datePickerOptions = {
					//TODO: change may work for jqui 1.11
					//TODO: move this to a separate directive?
					onSelect: function(dateStr, inst) {
						//shockingly bad api!
						var value = inst.input.datepicker('getDate');
						scope.$apply(function() {
							if (scope.$model.value instanceof ComplexValue &&
									scope.$model.value.selectedCount === 1) {
								value = startEnd(value, scope.$model.value.selected === 0);
								var cv = new ComplexValue(scope.$model.value, scope.$model.value);
								cv.setSelectedValue(value);
								value = scope.$model.value = cv;
							} else if (value) {
								value = startEnd(value).join('-');
								value = new ComplexValue(value);
								value.selected = 0; //increment to 1 later
								value.selectedCount = 1;
							} else {
								value = new Date();
							}
							this.pseudoFocus('dateSelected', false);
							var valid = true;
							if (value.splitter === '-') {
								if (value instanceof ComplexValue) {
									valid = Number(value.values[0]) <= Number(value.values[1]);
									value.setSelected((value.selected + 1) % value.values.length);
									this.datePickerSetDate();
								}
							}
							scope.$model.value = value;
							scope.$model.$valid = valid;
							//scope.datePickerOptions = null;
							this.facetedSearch.focus();
						}.bind(this));
						function startEnd(date, wantStart) {
							//return the start or end of the day (or both if wantStart is undefined)
							var start = new Date(date.getTime());
							['setHours', 'setMinutes', 'setSeconds', 'setMilliseconds']
								.forEach(function(fn) { start[fn](0) });
							start = dateToLocalSeconds(start);
							if (wantStart === undefined) {
								return [start, start + oneDaySeconds];
							}
							return start + (wantStart ? 0 : oneDaySeconds);
						}
					}.bind(this)
				};
			}
		},
		updateOptions: function($q, lang, facetName) {
			var facets = this.facetedSearch.facets();
			if (!facets || !facets.length) {
				return;
			}
			facetName = facetName && facetName.toLowerCase();
			var facet = facets.filter(facetNamed)[0];
			this.preSelect('facet', 0);
			//this.scope.$model.facet = facet;
			this.scope.$valueOptions = ActiveFacet.emptyArray;
			if (facet) {
				var p = $q.when(this.facetedSearch.getValues(facet))
					.then(assignValueOptions.bind(this));
				this.scope.$valueOptions.$promise = p;
				this.datePickerUpdate();
			}

			function facetNamed(facet) {
				return lang(facet.name).toString().toLowerCase() === facetName;
			}

			function assignValueOptions(values) {
				/* jshint validthis: true */
				//double check that we're still using the same facet.
				var scope = this.scope;
				if (this.scope.$model.facet === facet) {
					scope.$valueOptions = values.map(normalizeOption);
					if (facet && scope.$model.value && scope.$model.value.key) {
						//update selected value with correct key/value pair.
						var key = scope.$model.value.key,
							option = scope.$valueOptions.filter(matchKey(key))[0];
						if (option) {
							scope.$model.value = option;
							$q.when(option.$promise).then(function(value) {
								scope.$model.inputValue = value;
							});
						}
					}
				}
			}

			function normalizeOption(value) {
				return facet.normalizeOption(value);
			}

			function matchKey(key) { return function(option) { return option.key === key } }
		},
		valueUpdated: function($q, value) {
			if (value !== undefined) {
				var option = this.scope.$valueOptions.filter(valueNamed)[0];
				if (!option) {
					var promises = this.scope.$valueOptions.filter(pluckPromise).map(pluckPromise);
					if (promises.length) {
						$q.all.apply($q, promises).then(this.valueUpdated.bind(this, value));
						return;
					}
				}
				if (option) {
					this.scope.$model.value = option;
					this.facetedSearch.syncToModel();
				}
			}

			function valueNamed(option) { return option.value === value }
			function pluckPromise(option) { return option.$promise }
		}
	};

	/**
	 * @ngdoc directive
	 * @scope
	 * @name ng.directive:fSearchFacet
	 *
	 * @description
	 * This directive is used internally to manage a single active search facet
	 */
	fActiveFacet.$inject = ['$q', 'SearchFacet', '$window', 'ComplexValue'];
	function fActiveFacet($q, SearchFacet, $window, ComplexValue) {
		return {
			templateUrl: '/ng/directives/faceted_search_util/active_facet.html',
			require: ['^fFacetedSearch', 'fActiveFacet'],
			controller: ActiveFacet,
			replace: true,
			scope: {
				$model: '=fActiveFacet',
				$index: '=index'
			},
			link: function(scope, elem, attr, ctrls) {
				var facetedSearch = ctrls.shift(),
					activeFacet = ctrls.shift();
				activeFacet.init(facetedSearch);

				//scope functions
				scope.$fmtValue = function() {
					var v = scope.$model.value;
					return v instanceof ComplexValue || v instanceof SearchFacet.ValueOption ?
						v.formatted(scope.$model.facet) : v;
				};
				scope.$selValue = function() {
					var v = scope.$model.value;
					if (v instanceof ComplexValue) {
						return v.selectedValue();
					} else if (v instanceof SearchFacet.ValueOption) {
						return v.formatted();
					} else {
						return v;
					}
				};
				scope.$edit = activeFacet.edit.bind(activeFacet);
				scope.$on('editFacet', editFacet);
				//deal with the fact that the activefacetmodel is created before the controller.
				var editQueue = facetedSearch.popEditQueue();
				if (editQueue) {
					editFacet.apply(null, editQueue);
				}

				scope.$remove = activeFacet.remove;
				scope.$select = activeFacet.select;
				scope.$preSelect = activeFacet.preSelect;
				scope.$possibleFacets = activeFacet.possibleFacets;
				scope.$possibleValues = activeFacet.possibleValues;

				scope.$watch('$model.name', function(value) {
					scope.possibleFacets = scope.$possibleFacets(value);
				});

				scope.$watch('$selValue()', function(value) {
					scope.possibleValues = scope.$possibleValues(value);
				});

				scope.$watch('$valueOptions', function(value) {
					scope.possibleValues = scope.$possibleValues(scope.$selValue());
				});

				function editFacet(evt, index, inputName) {
					if (scope.$index === index) {
						evt.preventDefault();
						if (inputName !== activeFacet.editing()) {
							scope.$edit(inputName || 'value');
						}
					} else {
						if (index === -1) {
							if (!activeFacet.remove(true)) {
								activeFacet.select.current();
							}
						}
						activeFacet.updateFocus(null, false);
					}
				}
			}
		};
	}

	//jq-ui datepicker is such a pita
	fDatepickerSetDate.$inject = ['$timeout'];
	function fDatepickerSetDate($timeout) {
		return function(scope, element, attrs) {
			var last = null;
			scope.$watch(attrs.fDatepickerSetDate, function(value) {
				last = value;
				update(value);
			});
			scope.$watch(attrs.jqUi, scope.$evalAsync.bind(scope, update));
			function update() {
				//jquery events should happen outside the $digest
				$timeout(function() {
					if (element.is('.hasDatepicker')) {
						element.datepicker('setDate', last);
					}
				});
			}
		};
	}

	module.factory('ActiveFacetModel', activeFacetModelFactory);
	module.constant('ActiveFacet', ActiveFacet);
	module.directive('fActiveFacet', fActiveFacet);
	module.directive('fDatepickerSetDate', fDatepickerSetDate);
});
