define(['ml_module'], function(module) {
	'use strict';

	var KEYS = {
		ESC: 27,
		ENTER: 13,
		BACKSPACE: 8,
		TAB: 9,
		UPARROW: 38,
		DOWNARROW: 40
	};

	FacetInput.$inject = ['$scope', '$element', '$attrs', '$parse', '$timeout'];
	function FacetInput($scope, $element, $attrs, $parse, $timeout) {

		this.type = $element[0].type;
		this.$attr = $attrs;
		this.$scope = $scope;
		this.$elem = $element;
		this.$timeout = $timeout;

		$element.on('focus blur', function(event) {
			var focused = event.type === 'focus';
			$element.toggleClass('focused', focused);
			this.focused = focused;
		}.bind(this));
	}

	FacetInput.prototype = {
		$activeFacet: null,
		$pendingKeypress: null,
		init: function(name, activeFacet, editGroup) {
			this.name = name;
			this.editGroup = editGroup || name;
			this.$activeFacet = activeFacet;
		},
		focus: function(focus) {
			// can't focus a hidden element
			if (this.$elem.is(':hidden')) {
				return false;
			}
			focus = focus !== false;
			this.focused = focus;
			this.$elem.focus();
			return true;
		}
	};

	fFacetInput.$inject = ['$timeout', '$parse', 'KEYS'];
	function fFacetInput($timeout, $parse, KEYS) {
		return {
			scope: false,
			controller: FacetInput,
			require: ['^fFacetedSearch', '^?fActiveFacet', 'fFacetInput', '?fFacetModified'],
			link: function(scope, elem, attr, ctrls) {
				var facetedSearch = ctrls.shift(),
					activeFacet = ctrls.shift(),
					ctrl = ctrls.shift(),
					facetModified = ctrls.shift(),
					name = attr.fFacetInput;
				ctrl.init(name, activeFacet, attr.editGroup);

				if (facetModified) {
					ctrl.modify = facetModified.modify;
				}
				//button type default is 'submit' which is not very useful here
				if (elem[0].tagName.toUpperCase() === 'BUTTON') {
					if (!elem.attr('type')) {
						elem.prop('type', 'button');
					}
					ctrl.type = 'button';
				}
				if (activeFacet) {
					activeFacet.addInput(ctrl);
				}
				facetedSearch.addInput(ctrl);
				scope.$on('$destroy', function() {
					facetedSearch.removeInput(ctrl);
					if (activeFacet) {
						activeFacet.removeInput(ctrl);
					}
				});
				var preventKeypress = false,
					inputStopsEdit = true,
					keyDownVal = null,
					//default settings in ubuntu.
					KEY_PRESS_REPEAT_INTERVAL = 30,
					KEY_PRESS_REPEAT_DELAY = 500;

				elem.on('keypress', function(event) {
					if (preventKeypress) {
						event.preventDefault();
					}
					if (activeFacet) {
						scope.$apply(function() {
							if (activeFacet.keyPress(String.fromCharCode(event.which), ctrl)) {
								event.preventDefault();
							}
						});
					}
				});
				var preSelectKeys = {};
				preSelectKeys[KEYS.UPARROW] = -1;
				preSelectKeys[KEYS.DOWNARROW] = 1;
				//fake keypress for keys that don't support keypress
				function fakeKeypress(event, nextDelay) {
					$timeout.cancel(ctrl.$pendingKeypress);
					if (!preventKeypress) {
						if (activeFacet && event.which in preSelectKeys) {
							scope.$apply(function() {
								activeFacet.preSelect(name, preSelectKeys[event.which]);
								event.preventDefault();
							});
						}
						$timeout.cancel(ctrl.$pendingKeypress);
						ctrl.$pendingKeypress = $timeout(fakeKeypress.bind(null, event),
							nextDelay || KEY_PRESS_REPEAT_INTERVAL);
					}
				}
				var keydown = {};
				elem.on('focus', function() {
					keydown = {};
				});
				elem.on('keydown', function(event) {
					if (ctrl.type === 'button') {
						return;
					}
					preventKeypress = false;
					keyDownVal = elem.val();
					if (activeFacet) {
						scope.$apply(function() {
							if (event.which === KEYS.TAB) {
								activeFacet.tab(event.shiftKey, ctrl);
								event.preventDefault();
								preventKeypress = true;
							}
						});
					}
					//tab key doesn't allow a keyup (sometimes?)
					if (!preventKeypress && !event.isDefaultPrevented() &&
							event.which !== KEYS.TAB) {
						inputStopsEdit = false;
						if (event.which in preSelectKeys) {
							fakeKeypress(event, KEY_PRESS_REPEAT_DELAY);
						}
					}
					keydown[event.which] = true;
				});
				elem.on('keyup', function(event) {
					$timeout.cancel(ctrl.$pendingKeypress);
					inputStopsEdit = true;
					if (preventKeypress) {
						return;
					}
					scope.$apply(function() {
						if (event.which === KEYS.ESC) {
							if (activeFacet) {
								activeFacet.cancel();
								if (activeFacet.isEmpty()) {
									activeFacet.remove();
								}
							}
						} else if (event.which === KEYS.ENTER && keydown[event.which]) {
							//accept preselected value
							if (activeFacet && (name in activeFacet.select)) {
								activeFacet.select[name]();
							}
						}
					});
					keydown[event.which] = false;
				});
			}
		};
	}

	/**
	* fake facet_input that gets focus when you hover, loses focus on mouseout or when
	* the specified eventName is triggered. Defaults to 'click'
	*/
	function fFacetPseudoInput() {
		//focus in/out on these events
		//focusin and focusout would be nice, but don't work in chrome for <a> in this case
		var fevents = {
			mousedown: true,
			mouseup: false
		};
		return {
			scope: {
				name: '@fFacetHoverInput'
			},
			controller: 'FacetInput',
			require: '^fActiveFacet',
			link: function(scope, element, attr, activeFacet) {
				var events = Object.keys(fevents).join(' ');
				element.on(events, 'a, input, button', function(event) {
					scope.$apply(function() {
						//datepicker triggers 'mouseenter' when you click!?
						if (element.is(':visible') && event.isTrigger !== 3) {
							activeFacet.pseudoFocus(scope.name, fevents[event.type]);
						}
					});
				});
				element.on('focus', '*', function(event) {
					event.preventDefault();
				});
			}
		};
	}

	/**
	* add formatters and parsers for complexvalues based on the selected facet
	*/
	fFacetFormat.$inject = ['ComplexValue', 'SearchFacet'];
	function fFacetFormat(ComplexValue, SearchFacet) {
		return {
			scope: {
				facet: '=fFacetFormat'
			},
			require: 'ngModel',
			link: function(scope, element, attr, ngModel) {
				ngModel.$parsers.push(parse);
				//run this formatter before f-facet-modified formatter
				// (formatters are run in reverse order)
				ngModel.$formatters.push(format);
				var lastValueOption,
					errors = [];
				function parse(value) {
					if (lastValueOption && value === lastValueOption.value) {
						value = lastValueOption;
					}
					if (scope.facet) {
						errReset();
						try {
							value = scope.facet.complexValue(value, false);
							value = cvmap(value, scope.facet.parse.bind(scope.facet));
						} catch (ex) {
							doCatch(ex);
							value = undefined;
						}
					}
					return value;
				}
				function format(value) {
					lastValueOption = null;
					if (value instanceof SearchFacet.ValueOption) {
						lastValueOption = value;
						if (value.key instanceof ComplexValue) {
							value = value.key;
						} else {
							value = value.value;
						}
					}
					if (scope.facet) {
						value = cvmap(value, scope.facet.format.bind(scope.facet));
					}
					return value;
				}
				function cvmap(value, fn) {
					errReset();
					try {
						if (value instanceof ComplexValue) {
							value = value.formatted(fn, true);
						} else {
							value = fn(value);
						}
					} catch (ex) {
						doCatch(ex);
						value = undefined;
					}
					return value;
				}
				function errReset() {
					errors.forEach(function(error) {
						ngModel.$setValidity(error, true);
					});
				}
				function doCatch(ex) {

					var err = ex.constructor.name;
					if (err === 'Error') {
						err = ex.message.split(/ /).slice(0, 2).join('-');
					}
					errors.push(err);
					ngModel.$setValidity(err, false);
				}
			}
		};

	}

	module.controller('FacetInput', FacetInput);

	module.directive('fFacetInput', fFacetInput);
	module.directive('fFacetFormat', fFacetFormat);
	module.directive('fFacetPseudoInput', fFacetPseudoInput);

	module.constant('KEYS', KEYS);	
});
