define(['ml_module'], function(module) {
	'use strict';

	var MODIFIER_EXPR = /^(:?<=|>=|<|>|!)+\s*/,
		END_SPLIT_EXPR = /\s*[-,]\s*$/,
		//prevent lastIndex sharing
		SPLIT_EXPR = function() { return /\s*[-,]\s*/g; },
		//Unicode Record Separator control character
		RS = '\u001E';

	fFacetModifier.$inject = ['lang'];
	function fFacetModifier(lang) {
		return {
			scope: {
				modifier: '@fFacetModifier'
			},
			controller: 'FacetInput',
			require: ['^fFacetModifiers', 'fFacetModifier', '^?fActiveFacet', '^fFacetedSearch'],
			link: function(scope, elem, attr, ctrls) {
				var facetModifiers = ctrls.shift(),
					facetInput = ctrls.shift(),
					activeFacet = ctrls.shift(),
					facetedSearch = ctrls.shift();

				if (!facetedSearch.isComplex()) {
					elem.css('display', 'none');
				}
				facetInput.init('FacetModifier:' + attr.fFacetModifier,
					activeFacet, 'value');
				var fmBound = facetModifiers.modify.bind(facetModifiers, scope.modifier);
				elem.prop('type', 'button');
				elem.val(scope.modifier);
				var modifier = FacetModified.resolveModifier(scope.modifier);
				elem.attr('title', lang(FacetModified.modifierDefs[modifier].tip));
				elem.addClass('f-facet-modifier');
				elem.on('click', scope.$apply.bind(scope, fmBound));
				scope.$watch(
					function() {
						var model = activeFacet && activeFacet.scope.$model;
						var facet = model && model.facet;
						return facet && facet.modifiers;
					},
					function(facetModifiers) {
						var show = !facetModifiers || facetModifiers.indexOf(modifier) !== -1;
						elem.toggleClass('ng-hide', !show);
					}
				);
			}
		};
	}

	function FacetModifiers() {
	}
	FacetModifiers.prototype = {
		modified: null,
		register: function(modified) {
			this.modified = modified;
		},
		modify: function(modifier) {
			this.modified.modify(modifier);
		}
	};

	function fFacetModifiers() {
		return {
			controller: FacetModifiers,
			scope: false,
			require: 'fFacetModifiers',
			link: function(scope, elem) {
				elem.addClass('f-facet-modifiers');
			}
		};
	}

	FacetModified.$inject = ['injector'];
	function FacetModified(injector) {
		this.restoreCursor.$inject = ['$timeout'];
		this.restoreCursor = injector.bind(this.restoreCursor, this);
	}
	FacetModified.initModifiers = function() {
		var modifierDefs = FacetModified.modifierDefs,
			modifiers = FacetModified.prototype.modifiers;
		var lookup = FacetModified.prototype.modifierLookup;
		for (var m in modifierDefs) {
			var value = modifierDefs[m];
			if (typeof value === 'string') {
				//pointer to another modifier.
				lookup[m] = value;
				//initialized multiple times maybe? do we care?
				value = modifiers[m] = modifierDefs[value];
			}
			if (Array.isArray(value)) {
				//pointer to itself
				lookup[m] = m;
				//modifier data definition.
				modifierDefs[m] = {
					tip: value.shift(),
					isModifier: value.shift(),
					coexist: value.shift(),
					splitReplace: value.shift(),
					endSplitReplace: value.shift()
				};
				modifiers[m] = this.genModifier(m, modifierDefs[m]);
			}
		}
	};

	FacetModified.resolveModifier = function(modifier) {
		return FacetModified.prototype.modifierLookup[modifier];
	};

	/*
	* Generate a modifier function which adds a specific modifier to a value.
	* @param {String} modifier This modifier will be added.
	* @param {Object[]} r {
	*   {String} tip Tooltip that should be shown.
	*   {Boolean} isModifier True if modifier belongs at the start (not a separator),
	*   {String[]|Boolean} coexist array of modifiers that can coexist or true/false for all/none.
	*   {String} splitReplace RegExp replace string based on SPLIT_EXPR,
	*   {String} [endSplitReplace] RegExp replace string based on END_SPLIT_EXPR (Optional)
	* ]
	* @returns {Function(String value):String} function which returns value with modifier.
	*/
	FacetModified.genModifier = function(modifier, rules) {
		var args = [rules.endSplitReplace, rules.splitReplace];
		if (typeof coexist === 'string') {
			rules.coexist = [rules.coexist];
		}

		return function(value) {
			var expr = [END_SPLIT_EXPR, SPLIT_EXPR()];
			/* jshint validthis:true */
			for (var i = 0, len = args.length; i < len; ++i) {
				if (args[i] !== undefined) {
					value = value.replace(expr[i], args[i]);
				}
			}
			var complex = new ComplexValue(value);
			var exists = complex.modifiers.indexOf(modifier) > -1;
			complex.modifiers = complex.modifiers.filter(canCoexist);
			if (rules.isModifier && !exists) {
				complex.addModifier(modifier);
			}
			value = value.replace(MODIFIER_EXPR, '');
			if (complex.modifiers.length > 0) {
				value = complex.modifiers.join('') + ' ' + value;
			}
			if (!rules.isModifier) {
				value += rules.splitReplace;
				this.cursor.end = this.cursor.start = value.length;
			}
			return value;
		};
		function canCoexist(m) {
			if (Array.isArray(rules.coexist)) {
				return rules.coexist.indexOf(m) > -1;
			} else {
				return rules.coexist && m !== modifier;
			}
		}
	};

	//order will be used when/if modifiers co-exist
	FacetModified.modifierDefs = {
		'!': ['Negate the filter value', true, true],
		'<=': ['Less than or equal to this value', true, '!', ' '],
		'>=': ['Greater than or equal to this value', true, '!', ' '],
		'>': ['Greater than this value', true, '!', ' '],
		'<': ['Less than this value', true, '!', ' '],
		',': ['Match any of these values', false, '!', ', ', ''],
		'-': ['Between these two values (inclusive)', false, '!', ' - ', ''],
		//unicode ... (horizontal ellipsis)
		'\u2026': '-',
		'A-B': '-',
		'NOT': '!',
		'OR': ',',
		'RANGE': '-'
	};

	FacetModified.prototype = {
		//filled in by initModifiers
		modifiers: {},
		modifierLookup: {},

		modify: function(ngModel, facetInput, elem, modifier) {
			var modifiers = this.modifiers;
			modifier = FacetModified.resolveModifier(modifier);
			this.saveCursor(elem);
			var value = modifiers[modifier].call(this, ngModel.$viewValue || '');
			elem.val(value);
			elem.focus();
			ngModel.$setViewValue(value);
			this.restoreCursor(elem);
		},
		saveCursor: function(elem) {
			elem = elem[0];
			this.cursor = {
				start: elem.selectionStart,
				end: elem.selectionEnd,
				word: elem.value
			};
		},
		restoreCursor: function($timeout, $elem) {
			var value = $elem.val();
			var elem = $elem[0];
			//TODO: restore cursor to the same position in the same 'word'
			//(overall text may have changed)
			if (this.cursor) {
				if (this.cursor.start < value.length) {
					var pad = value.length - this.cursor.word.length;
					//for now (prototype), assume that the extra characters were inserted before
					this.cursor.start += pad;
					this.cursor.end += pad;
				}
				if (elem.setSelectionRange) {
					elem.setSelectionRange(this.cursor.start, this.cursor.end);
				}
			}
			$timeout(function() {
				$elem.focus();
			});
		},
		cursor: null
	};
	FacetModified.initModifiers();

	fFacetModified.$inject = ['$timeout'];
	function fFacetModified($timeout) {
		return {
			controller: FacetModified,
			scope: false,
			require: [
				'ngModel', '^fFacetModifiers', 'fFacetModified', 'fFacetInput', '?^fFacetedSearch'
			],
			link: function(scope, elem, attr, ctrls) {
				var ngModel = ctrls.shift(),
					facetModifiers = ctrls.shift(),
					facetModified = ctrls.shift(),
					facetInput = ctrls.shift(),
					facetedSearch = ctrls.shift();
				elem.addClass('f-facet-modified');
				facetModifiers.register(facetModified);
				facetModified.modify = facetModified.modify
					.bind(facetModified, ngModel, facetInput, elem);
				var $render = ngModel.$render,
					caret;
				ngModel.$render = function() {
					$render.call(this, arguments);
					if (caret) {
						if (elem[0].setSelectionRange) {
							elem[0].setSelectionRange(caret.start, caret.start + caret.length);
						}
						caret = null;
						$timeout.cancel(updateCaretPending);
					} else {
						updateCaret();
					}
					elem.attr('size', elem.val().length);
				};

				function selectedWatch() {
					return current instanceof ComplexValue &&
						[current.selected, current.selectedCount];
				}
				var current, updateCaretPending, skipUpdate, doubleClicked;
				scope.$watch(selectedWatch, function(value) {
					if (value && !skipUpdate) {
						caret = current.getCaret();
						if (caret.length === elem[0].selectionEnd - elem[0].selectionStart &&
								caret.start === elem[0].selectionStart || caret.length === 0) {
							caret = null;
						} else {
							ngModel.$render();
						}
					} else {
						skipUpdate = !value;
					}
				}, true);
				ngModel.$parsers.push(parse);
				ngModel.$formatters.push(formatter);
				function updateCaretCurrent(value) {
					current = value;
					caret =  value.getCaret();
				}
				function updateCaret() {
					if (current) {
						var ss = elem[0].selectionStart;
						var sl = elem[0].selectionEnd - ss;
						current.setCaret(ss, sl);
						var mv = ngModel.$modelValue;
						if (mv !== current && mv instanceof ComplexValue) {
							mv.setCaret(ss, sl);
						}
						skipUpdate = !doubleClicked;
						doubleClicked = false;
					}
				}
				elem.on('dblclick', function() {
					doubleClicked = true;
				});
				elem.on('input keydown keyup keypress mouseup', function() {
					caret = null;
					updateCaretPending = $timeout(updateCaret, 0, true);
				}.bind(this));
				function parse(value) {
					var complex = facetedSearch && facetedSearch.isComplex();
					if (complex) {
						value = new ComplexValue(value);
						updateCaretCurrent(value);
						updateCaretPending = $timeout(updateCaret, 0, true);
					}
					return value;
				}

				function formatter(value) {
					if (value instanceof ComplexValue) {
						updateCaretCurrent(value);
						value = value.formatted();
					}
					return value;
				}
			}
		};
	}

	/**
	* Complex value for a facet. Used to translate between text input and qlist filters worlds.
	* Be sure to implement the interface defined by {@link SearchFacet.ValueOption}
	* @constructor
	* @augments SearchFacet.ValueOption
	* @see SearchFacet.ValueOption
	* @param {String|ComplexValue} value Value may contain special grammar.
	*   - Modifiers: < > ! will be stripped from the front and placed in #modifiers array.
	*   - Splitter: , or - may be used to split multiple values. '-' trumps ',' atm.
	* @param {ComplexValue} [linkSelected] If present, setCaret() calls will also affect the
	*   selected property of the linkSelected instance
	*/
	function ComplexValue(value, linkSelected) {
		//todo: add quoting?
		if (value.__ComplexValue) {
			//might have been stored in json!
			this._restrictModifiers = value._restrictModifiers;
			value = this.formatted.call(value);
		}

		//Assigning this on the prototype won't work, json.stringify would ignore it.
		this.__ComplexValue = true;
		var olen = value.length,
			splitMatch,
			splitters = {
				'-': /\s*-\s*/g,
				',': /\s*,\s*/g
			},
			splitter,
			splitExpr = /$ ^/g;//unmatchable!
		this.modifiers = [];
		this.value = value;
		value = String(value).replace(MODIFIER_EXPR, foundModifier.bind(this));
		this._mpad = olen - value.length;
		// - precedence over , ? combinable?
		this.splitter = null;
		for (splitter in splitters) {
			if (value.indexOf(splitter) !== -1) {
				splitExpr = splitters[splitter];
				this.splitter = splitter;
				break;
			}
		}
		this.selected = 0;
		this.selectedCount = 0;
		this._spad = [];
		if (linkSelected && linkSelected.__ComplexValue) {
			this._linkSelected = linkSelected;
			this.selected = this._linkSelected.selected;
			this.selectedCount = this._linkSelected.selectedCount;
		}
		while ((splitMatch = splitExpr.exec(value)) !== null) {
			this._spad.push(splitMatch[0].length);
		}
		this.values = value.split(splitExpr);
		//if previous ComplexValue was restricted to specific modifiers we must also
		//be restricted!
		if (this._restrictModifiers) {
			var wm = this.withModifiers(this._restrictModifiers);
			['values', 'modifiers', 'splitter', '_spad', '_mpad'].forEach(function(prop) {
				this[prop] = wm[prop];
			}.bind(this));
		}

		function foundModifier(m) {
			var spaceExpr = /^\s+$/;
			/* jshint validthis: true */
			while (!spaceExpr.test(m) && MODIFIER_EXPR.test(m)) {
				for (var i = 0, len = ComplexValue.modifiers.length; i < len; ++i) {
					var mod = ComplexValue.modifiers[i];
					if (mod === m.substr(0, mod.length)) {
						m = m.substr(mod.length);
						this.modifiers.push(mod);
					}
				}
			}
			return '';
		}
	}

	/**
	* build a ComplexValue object from constituent parts
	* @param {Object[]} values
	* @param {String[]} modifiers cannonical names of modifiers
	* @param {String} splitter connonical splitter to use
	* @returns {ComplexValue}
	*/
	ComplexValue.build = function(values, modifiers, splitter) {
		return new ComplexValue({
			values: values,
			modifiers: modifiers,
			splitter: splitter,
			__ComplexValue: true
		});
	};

	ComplexValue.probablyComplex = function(value) {
		return value &&
			(value.__ComplexValue || MODIFIER_EXPR.test(value) || SPLIT_EXPR().test(value));
	};

	// flatten an array of ComplexValue objects to a single array of values
	ComplexValue.flattenValues = function(cvs) {
		return cvs.reduce(flatten, []);

		function flatten(result, cv) {
			return result.concat(ComplexValue.probablyComplex(cv) && cv.values ?
				cv.values : [cv]);
		}
	};
	ComplexValue.modifiers = Object.keys(FacetModified.prototype.modifiers);
	ComplexValue.resolveModifier = FacetModified.resolveModifier;
	ComplexValue.prototype = {
		/**
		* The index of the values array which is currently being edited by the user
		* updated by #setCaret()
		* @type Number
		*/
		selected: 0,
		/**
		* Splitter character from the input (if present)
		* @type String
		*/
		splitter: null,
		/**
		* Values parsed from the input. If there is no splitter then the array only contains one.
		* @type Object[]
		*/
		values: null,
		/**
		* Modifier parsed from the input (< > !) (there can be only one!)
		* @type String[]
		*/
		modifiers: null,
		/**
		* Original value that was passed to the constructor
		* @type String
		*/
		value: null,
		/**
		* Add a modifier to the list of modifiers.
		*/
		addModifier: function(modifier) {
			var modifierOrder = ComplexValue.modifiers;
			this.modifiers.push(modifier);
			this.modifiers.sort(modifierSort);
			function modifierSort(a, b) {
				return modifierOrder.indexOf(a) - modifierOrder.indexOf(b);
			}
		},
		/**
		* Update #selected with the #value item that is currently being edited.
		* @param {Number} selectionStart Position of the user's cursor
		*/
		setCaret: function(selectionStart, selectionLength) {
			selectionStart -= this._mpad;
			selectionLength = selectionLength || 0;
			var sel = valueIndexOf(this, selectionStart),
				selCount = selectionLength &&
					valueIndexOf(this, selectionStart + selectionLength) + 1 - sel;


			if (sel >= this.values.length) {
				//throw new Error('Selection outside value range!');
				sel = this.values.length - 1;
			}
			if (sel + selCount > this.values.length) {
				selCount = this.values.length - sel;
			}
			this.selected = sel;
			this.selectedCount = selCount;
			if (this._linkSelected instanceof ComplexValue) {
				this._linkSelected.selected = sel;
				this._linkSelected.selectedCount = selCount;
			}

			function valueIndexOf(cv, index) {
				var len = 0;
				for (var sel = 0, vlen = cv.values.length; sel < vlen; ++sel) {
					len += String(cv.values[sel]).length;
					if (len >= index) {
						break;
					}
					len += cv._spad[sel] || 0;
				}
				return sel;
			}
		},
		/**
		* @returns {Object(Number start, Number length)}
		*/
		getCaret: function() {
			var startLength = 0,
				startPad = 0,
				lvalues = this.selectedValues(),
				selPad = 0;
			if (this.selected > 0) {
				startLength = this.values.slice(0, this.selected).reduce(len, 0);
				startPad =  this._spad.slice(0, this.selected).reduce(sum, 0);
			}
			if (this.selectedCount > 1) {
				selPad = this._spad.slice(this.selected, this.selected + this.selectedCount - 1)
					.reduce(sum);
			}
			return {
				start: this._mpad + startLength + startPad,
				length: this.selectedCount && (lvalues.reduce(len, 0) + selPad)
			};

			function len(result, value) { return result + String(value).length }
			function sum(result, value) { return result + value }
		},
		/**
		* Currently selected value (based on cursor position)
		* @returns {String}
		*/
		toString: function() {
			return String(this.selectedValue());
		},
		/**
		* Currently selected value (based on cursor position)
		* @returns {Object}
		*/
		selectedValue: function() {
			return this.values[this.selected];
		},
		selectedValues: function() {
			var end = this.selected + Math.max(this.selectedCount, 1);
			return this.values.slice(this.selected, end);
		},
		/**
		* Update the value at the currently selected position
		* @param {Object} value
		*/
		setSelectedValue: function(value) {
			this.values[this.selected] = value;
		},
		setSelectedValues: function(values) {
			this.values.splice.apply(
				this.values,
				[this.selected, Math.max(this.selectedCount, 1)]
					.concat(values)
			);
			this.selectedCount = values.length;
		},
		setSelected: function(index, count) {
			this.selected = index;
			if (count !== undefined) {
				this.selectedCount = count;
			}
			if (this._linkSelected) {
				this._linkSelected.selected = index;
				if (count !== undefined) {
					this._linkSelected.selectedCount = count;
				}
			}
		},
		replaceSelectedValues: function(values, facet) {
			var result = new ComplexValue(this);
			if(values instanceof ComplexValue)
				values = values.values;

			values = values.filter(notEmpty);
			result.values.splice.apply(result.values, [this.selected, Math.max(this.selectedCount, 1)].concat(values));
			result.value = result.formatted(facet);
			result.setSelected(this.selected, values.length > 1 ? values.length : this.selectedCount);
			return new ComplexValue(result);

			function notEmpty(value) {
				return !!String(value).trim()
			}
		},
		/**
		* Proxy replace function for autocomplete which expects a string value.
		*/
		replace: function() {
			var value = this.toString();
			return value.replace.apply(value, arguments);
		},
		/**
		* check to see if the specified splitter is used, if it is, return a new ComplexValue
		* with the original value in tact. Otherwise return this
		*/
		withoutSplitter: function(splitter, noClone) {
			var result = this;
			if (this.splitter === splitter) {
				result = noClone ? this : new ComplexValue(this.value, this);
				result.splitter = null;
				result.values = result.value.replace(MODIFIER_EXPR, '')
					.replace(SPLIT_EXPR(), otherSplitter)
					.split(RS);
				result._restrictModifiers = [splitter];
			}
			return result;

			function otherSplitter(match) {
				match = match.trim();
				var current = match === result.splitter,
					noSplitter = result.splitter === null,
					oldSplitter = match === splitter;
				if (oldSplitter || !(current || noSplitter)) {
					return match;
				}
				result.splitter = match.trim();
				return RS;
			}
		},
		/**
		* Attempt to merge another value into this one, using the specified splitter.
		* @param {ComplexValue} other
		* @param {String} splitter
		* @returns {Boolean} True if successful
		*/
		merge: function(other, splitter) {
			var joinedModifiers = this.modifiers.join(RS);
			if (other.modifiers.join(RS) !== joinedModifiers ||
					['!', ''].indexOf(joinedModifiers) === -1 ||
					this.splitter != null && this.splitter !== splitter) {
				return false;
			}
			this.values.splice.apply(this.values, [0, 0].concat(other.values));
			this.splitter = splitter;
			return true;
		},
		/**
		* Get a string representation if the value isn't actually complex, otherwise return this.
		* @returns {ComplexValue|String}
		*/
		getSimple: function() {
			if (!this.splitter &&
					this.modifiers.length === 0 &&
					this.values.length === 1 &&
					!this._restrictModifiers) {
				return this.values[0];
			}
			return this;
		},
		/**
		* Only allow the specified modifiers. If others are present, discard them and return a new
		* ComplexValue containing them.
		*/
		withModifiers: function(modifiers) {
			modifiers = modifiers.map(FacetModified.resolveModifier);
			var result = this,
				currentModifiers = this.modifiers.concat(this.splitter ? [this.splitter] : []),
				removed = [];

			currentModifiers.filter(notWanted).forEach(remove.bind(this));
			if (removed.length) {
				result.values[0] = removed.filter(defined).join('') + ' ' + result.values[0];
			}
			result._restrictModifiers = modifiers;
			return result;
			function notWanted(modifier) { return modifiers.indexOf(modifier) ===  -1 }
			function defined(modifier) { return modifier }
			function remove(modifier) {
				/* jshint validthis: true */
				if (SPLIT_EXPR().test(modifier)) {
					result = result.withoutSplitter(modifier, result !== this);
				} else {
					var index = result.modifiers.indexOf(modifier);
					if (index > -1) {
						removed[index] = modifier;
						if (result === this) {
							result = new ComplexValue(this.value, this);
						}
						result.modifiers.splice(index, 1);
					}
				}
			}
		},
		/**
		* Implements SearchFacet.ValueOption#equals behavior.
		*/
		/**
		* Normalized value including modifiers and splitters.
		* Implements SearchFacet.ValueOption#formatted functionality as well.
		* @see SearchFacet.ValueOption#formatted
		* @param {Function(Object value):Object|SearchFacet} [fmt] Format function or search facet
		* @param {Boolean} [complex=false] preserve ComplexValue type and link selection
		* to be used to format values.
		* NOTE: don't call any methods here:
		*	 this function is also used to revive objects parsed from JSON
		* @returns {String}
		*/
		formatted: function(fmt, complex) {
			if (typeof fmt !== 'function') {
				fmt = fmt && fmt.format && fmt.format.bind(fmt) || function(v) { return v };
			}
			if (complex) {
				var result = new ComplexValue(this, this);
				result.values = result.values.map(fmt);
				return result;
			}
			var modifierDef = FacetModified.modifierDefs[this.splitter],
				splitReplace = modifierDef && modifierDef.splitReplace;
			return (this.modifiers.length ? this.modifiers.join('') + ' ' : '') +
				this.values.map(fmt).join(splitReplace);
		},
		equals: function(value) {
			return value instanceof ComplexValue &&
				value.formatted() === this.formatted();
		},
		localeCompare: function() {
			return this.value.localeCompare.apply(this, arguments);
		},

		/**
		* Modify the complex value so it contains only values that intersect with the supplied array
		* if it already contains any values. Otherwise just return false.
		* @see SearchFacet.ValueOption#intersection
		* @param {Object[] | ComplexValue} other Values that are allowed to persist. Any values
		*   not listed must be stripped from the object.
		* @param {Boolean} [cull=false] If true, cull the values that don't intersect when others do
		* @returns {Object[]} a list of values that were found or null if there was no match
		*/
		intersection: function(other, cull) {
			//incompatible modifiers ?
			// if (other instanceof ComplexValue) {
			// 	if (other.modifiers.join('') !== this.modifiers.join('') ||
			// 			normalizeSplitter(other.splitter) !== normalizeSplitter(this.splitter)) {
			// 		return null;
			// 	}
			// } else {
			// 	if (this.modifiers.length || normalizeSplitter(this.splitter) !== '') {
			// 		return null;
			// 	}
			// }
			// other = ComplexValue.flattenValues(other);

			// Fix bug: 0491322	
			function compatible(value) {
				if (value instanceof ComplexValue) {
					if (value.modifiers.join('') !== this.modifiers.join('') ||
						normalizeSplitter(value.splitter) !== normalizeSplitter(this.splitter)) {
						return false;
					}
				} else {
					if (this.modifiers.length || normalizeSplitter(this.splitter) !== '') {
						return false;
					}
				}
				return true;
			}
			other = ComplexValue.flattenValues(other.filter(compatible.bind(this)));

			var values = this.values.filter(intersects);
			if (values.length > 0) {
				if (cull && values.length !== this.values.length) {
					this.values.splice.apply([0, this.values.length].concat(values));
				}
				return values;
			}
			return null;

			function intersects(value) { return other.indexOf(value) > -1; }
			function normalizeSplitter(s) { return s === ',' || s === null ? '' : s }
		}
		/******** End SearchFacet.ValueOption interface ********/
	};

	module.value('ComplexValue', ComplexValue);
	module.directive('fFacetModifiers', fFacetModifiers);
	module.directive('fFacetModifier', fFacetModifier);
	module.directive('fFacetModified', fFacetModified);
});
