define([
	'ml_module',
	'./search_facet',
	'./active_facet',
	'./facet_input',
	'./facet_modifier'
	],
	function(module) {
	'use strict';

	FacetedSearch.$inject = ['injector', '$scope', '$element', '$timeout', '$document', 'KEYS'];
	function FacetedSearch(injector, $scope, $element, $timeout, $document, KEYS) {
		this.activeFacets = [];
		this.inputs = [];
		this.editQueue = null;
		this.scope = $scope;

		var inject = {
			getValues: ['ActiveFacet'],
			addFacet: ['ActiveFacetModel'],
			syncFromModel: ['ActiveFacetModel', '$q', 'ComplexValue', '$log'],
			syncToModel: ['ComplexValue', 'facetedSearchUtil'],
			mergeComplexValues: ['ComplexValue', 'SearchFacet']
		};
		var fn, locals;
		for (fn in inject) {
			locals = inject[fn];
			this[fn].$inject = locals;
			this[fn] = injector.bind(this[fn], this);
		}

		['focus', 'focused', 'focusedElement', 'editing'].forEach(function(fn) {
			this[fn] = this[fn].bind(this);
		}.bind(this));
		this.blockBackspace($scope, KEYS);
		this.monitorFocus($timeout, $document, $element);

		$scope.$watch(function() {
			var fe = this.focusedElement();
			return fe && fe.$activeFacet && !!fe.$activeFacet.focused;
		}.bind(this), function(value) {
			this.$overlay.toggleClass('ng-hide', !value);
		}.bind(this));
		$scope.$watch('[facets, options, model]', this.update.bind(this), true);
	}

	FacetedSearch.prototype = {
		editQueue: null,
		update: function update() {
			this.facetsById = null;
			this.syncFromModel();
		},
		facets: function() { return this.scope.facets },
		isComplex: function() {
			return this.scope.options && this.scope.options.complex;
		},
		getValues: function(ActiveFacet, facet) {
			var options = this.scope.options;
			if (facet && options && angular.isFunction(options.entries)) {
				var entries = options.entries();
				return entries ? facet.populate(entries, options.source) : [];
			} else {
				return ActiveFacet.emptyArray;
			}
		},
		addFacet: function(ActiveFacetModel) {
			var af = this.activeFacets;
			for (var i = af.length - 1; i >= 0; --i) {
				if (!af[i].name || !af[i].value && !af[i].editing) {
					af.splice(i, 1);
				}
			}
			var afm = new ActiveFacetModel();
			af.push(afm);
			this.editFacet(afm, 'facet');
		},
		syncFromModel: function(ActiveFacetModel, $q, ComplexValue, $log) {
			var scope = this.scope;
			if (!scope.facets || !scope.facets.length || !scope.model) {
				return;
			}
			if (!this.facetsById) {
				this.facetsById = scope.facets
				.reduce(FacetedSearch.indexById, {});
			}

			var afs = this.activeFacets,
			model = scope.model,
			facetsById = this.facetsById,
			keys,
			found;
			//clear extraneous facets
			var clearModel = angular.copy(model);
			for (var i = afs.length - 1; i >= 0; --i) {
				var af = afs[i];
				if (af.facet && af.value) {
					keys = clearModel[af.facet.id];
					found = keys && af.value && af.value.intersection(keys, true);
					if (found == null || !(af.facet.id in this.facetsById)) {
						afs.splice(i, 1);
					}
					if (found) {
						clearModel[af.facet.id] = keys.filter(notIn(found));
					}
				}
			}
			function notIn(aSet) { return function(key) { return aSet.indexOf(key) === -1} }
			//add missing facets
			for (var id in model) {
				model[id].forEach(addMissing(id));
			}

			this.mergeComplexValues();
			function addMissing(id) {
				return function(key) {
					var facet = facetsById[id];
					if (!facet) {
						$log.warn('Can\'t add missing facet from model: ' + id +
							' Available facets: ' + Object.keys(facetsById));
					} else {
						if (!afs.some(afMatches(id, key))) {
							var value = key;
							if (!(value instanceof ComplexValue)) {
								value = facet.normalizeOption(key);
							} else {
								// make a copy of model
								value = new ComplexValue(value);
							}
							var afm = new ActiveFacetModel(facet, value);
							afs.push(afm);
						}
					}
				};
			}

			function afMatches(id, key) {
				return function(af) {
					var facet = af.facet;
					return facet && facet.id === id &&
					af.value && af.value.intersection([key]);
				};
			}
		},
		syncToModel: function(ComplexValue, facetedSearchUtil) {
			this.mergeComplexValues();
			var scope = this.scope;
			if (!this.facetsById) {
				this.facetsById = scope.facets
				.reduce(FacetedSearch.indexById, {});
			}
			var newModel = this.activeFacets.reduce(toModel, {}),
			facetsById = this.facetsById;
			function toModel(model, af) {
				if (af.facet && af.value) {
					var id = af.facet.id,
					values = model[id];
					if (!values) {
						values = model[id] = [];
					}
					if (af.value instanceof ComplexValue) {
						values.push(af.value);
					} else {
						values.push(af.value.key);
					}
				}
				return model;
			}
			//preserve facets that don't apply to this segment
			for (var id in scope.model) {
				if (!(id in facetsById)) {
					newModel[id] = scope.model[id];
				}
			}
			//angular.copy creates deep copies but doesn't preserve constructor!
			angular.copy(newModel, scope.model);
			if (this.isComplex()) {
				facetedSearchUtil.jsonForModel(scope.model, true);
			}
		},
		mergeComplexValues: function(ComplexValue, SearchFacet) {
			if (!this.isComplex()) { return }
				var afmGroups = this.activeFacets.reduce(groupByFacet, {}),
			options = this.scope.options.filterMerge || {};
			Object.keys(afmGroups).forEach(function(k) {
				var group = afmGroups[k];
				if (group.length > 1) {
					if (options.useLast) {
						group.reverse();
					}
					mergeValues(this.activeFacets, group, options);
				}
			}.bind(this));


			//reduce to filterMerge.max ComplexValues if possible, mark the rest invalid
			function mergeValues(activeFacets, group, options) {
				var remove = [],
				filterCount = 1,
				value = makeCV(group[0]),
				origValue = group[0].value;
				group.slice(1).reduce(merge, value);
				if (remove.length) {

					if (origValue instanceof SearchFacet.ValueOption) {
						origValue = origValue.key;
					}
					if (origValue instanceof ComplexValue && origValue.values === value.values) {
						throw new Error('shared array fail');
					}
					group[0].value = value.getSimple();
					remove.forEach(function(afm) {
						var i = activeFacets.indexOf(afm);
						if (i > -1) {
							activeFacets.splice(i, 1);
						}
					});
				}
				function merge(result, afm) {
					//result is group[0] or null if merging failed
					var valid = result.merge(makeCV(afm), ',');
					if (valid) {
						remove.push(afm);
					} else {
						++filterCount;
					}
					afm.$valid = options.max == null || filterCount <= options.max;
					return result;
				}
			}
			function makeCV(afm) {
				var value = afm.value,
				facet = afm.facet;
				if (value instanceof SearchFacet.ValueOption) {
					value = value.key;
				}
				var ovalue = value;
				value = facet.complexValue(value, true);
				//even if it's a complexvalue already, make sure to copy it (facets may not comply)
				if (value === ovalue) {
					value = new ComplexValue(value);
				}
				if (value.values === ovalue.values) {
					throw new Error('shared array fail!');
				}
				return value;
			}
			function groupByFacet(result, afm) {
				afm.$valid = true;
				if (afm.facet && afm.value !== undefined) {
					if (!(afm.facet.id in result)) {
						result[afm.facet.id] = [];
					}
					result[afm.facet.id].push(afm);
				}
				return result;
			}
		},
		addInput: function(input) {
			this.inputs.push(input);
		},
		removeInput: function(input) {
			var index = this.inputs.indexOf(input);
			if (index > -1) {
				this.inputs.splice(index, 1);
			}
		},
		editing: function() {
			return this.activeFacets.some(isEditing);
			function isEditing(af) {
				return af.$editing();
			}
		},
		focusedElement: function() {
			var elements = this.inputs.filter(this._isFocused);
			return elements[0];
		},
		focused: function() {
			return this.inputs.some(this._isFocused);
		},
		_isFocused: function isFocused(input) { return input.focused },
		/*
		* @param {Boolean} [value=true] true to make the faceted_search bar focused
		*/
		focus: function(value) {
			if (!this.focused() && value !== false &&
				[this.overlayState, this.mouseState].indexOf('mousedown') === -1) {
				var inputs = this.inputs.slice();
			inputs.sort(priority);
			var input = inputs.filter(isVisible)[0] ||
			this.inputs.filter(isAddFilter)[0];
			if (input) {
				input.focus();
			}
		} else if (value === false) {
			this.scope.$broadcast('editFacet', -1, 'facet');
		}
		function isVisible(input) {
				//can't focus if it's not visible!
				return input.$elem.is(':visible');
			}
			function isAddFilter(input) {
				return input.name === 'addFilter';
			}
			function priority(a, b) {
				var order = [
				'addFilter', 'value', 'facet',
				'selectValue', 'selectFacet',
				'remove', 'cancelAll'
				];
				a = order.indexOf(a.name);
				b = order.indexOf(b.name);
				return (a >= 0 ? a : order.length) -
				(b >= 0 ? b : order.length);
			}
		},
		editPrev: function(model, remove) {
			var af = this.activeFacets,
			index = af.indexOf(model);
			if (index > -1 && remove) {
				af.splice(index, 1);
			}
			if (index > 0) {
				var result = this.scope.$broadcast('editFacet', index - 1);
				if (!result.defaultPrevented) {
					this._editQueued = [result, index, index];
				}
			}
		},
		editNext: function(model, addFacet) {
			var index = this.activeFacets.indexOf(model);
			if (index < this.activeFacets.length - 1) {
				var result = this.scope.$broadcast('editFacet', index + 1, 'facet');
				if (!result.defaultPrevented) {
					this._editQueued = [result, index, 'facet'];
				}
			} else if (addFacet) {
				this.addFacet();
				return;
			}
		},
		editFacet: function(model, input) {
			var index = this.activeFacets.indexOf(model);
			if (index < this.activeFacets.length) {
				var result = this.scope.$broadcast('editFacet', index, input);
				if (!result.defaultPrevented) {
					this._editQueued = [result, index, input];
				}
			}
		},
		popEditQueue: function() {
			var r = this._editQueued;
			this._editQueued = null;
			return r;
		},
		blockBackspace: function(scope, KEYS) {
			var eventName = 'keydown.facetedSearchBlockBackspace_' + scope.$id;
			angular.element(document).on(eventName, function(e) {
				var elem = this.focusedElement();
				if (e.keyCode === KEYS.BACKSPACE && elem && elem.type === 'button') {
					e.preventDefault();
				}
			}.bind(this));
			scope.$on('$destroy', function() {
				angular.element(document).off(eventName);
			});
		},
		monitorFocus: function($timeout, $document, element) {
			function ns(eventName) { return eventName + '.facetedSearchFocusMonitor_' + scope.$id }
			var scope = this.scope;
			var focusEvents = ['focusout', 'focusin'].map(ns).join(' ');
			var mouseEvents = ['mousedown', 'mouseup', 'mouseout'].map(ns).join(' ');
			var clickEvent = ns('click');

			element.on(focusEvents, function() {
				//Cause a digest after focus has settled.
				$timeout(function() {
					var nothingFocused = !$document[0].activeElement ||
					String($document[0].activeElement.tagName).toLowerCase() === 'body';
					if (nothingFocused && !this.focused()) {
						this.focus();
					}
				}.bind(this), 0, true);
			}.bind(this));
			element.on(mouseEvents, function(event) {
				this.mouseState = event.type;
			}.bind(this));

			angular.element(document).on(clickEvent, function(e) {
				if (element.find(e.target).length === 0) {
					this.focus(false);
				}
			}.bind(this));
			scope.$on('$destroy', function() {
				element.off(focusEvents + ' ' + clickEvent + ' ' + mouseEvents);
			});
		}
	};

	FacetedSearch.indexById = function(index, facet) {
		index[facet.id] = facet;
		return index;
	};

	/**
	 * @ngdoc directive
	 * @scope
	 * @name ng.directive:fFacetedSearch
	 *
	 * @description
	 * Use this directive to create a faceted search bar similar to VisualSearch
	 * Usage:
	 * f-faceted-search='{ facets: [searchFacet, searchFacet2],
	 *                     model: searchModel, options: searchOptions}'
	 * - facets is an array of SearchFacet objects
	 * - model is an Object that will be filled in to match the current search
	 * as follows: { selector: ['value'], selector2: ['value2', 'value3'] }
	 * - config is a configuration object that may be updated to reflect the
	 * current search target.
	 * - - options.source: {String} key to use from SearchFacet#selectors when
	 *  populating search values
	 * - - options.entries: {Function():String} function which returns the
	 *  array of entries to be used by facets for the populate function.
	 * (it's ok to return empty array if the results are not ready!)
	 * - - options.filterMerge: {Object} options for merging filters:
	 * - - - [max=null] {Number} Number of filters the backend supports for each facet.
	 * - - options.remainder ????
	 */
	function fFacetedSearch() {
		return {
			scope: {
				facets: '=',
				options: '=',
				model: '='
			},
			templateUrl: '/ng/directives/faceted_search_util/main.html',
			controller: FacetedSearch,
			require: 'fFacetedSearch',
			link: function(scope, elem, attr, controller) {
				elem.addClass('f-faceted-search-container');
				controller.fallback = function() {
					elem.addClass('f-fi-fallback');
				};
				scope.$activeFacets = controller.activeFacets;
				scope.$editing = controller.editing;
				scope.$focused = controller.focused;
				scope.$newFacet = newFacet;
				scope.$cancel = cancel;

				function newFacet($event) {
					controller.addFacet();
					if ($event) {
						$event.stopPropagation();
					}
				}
				function cancel() {
					scope.$activeFacets.length = 0;
					controller.syncToModel();
				}
			}
		};
	}

	function fFacetedSearchOverlay() {
		return {
			scope: false,
			require: '^fFacetedSearch',
			link: function(scope, elem, attr, facetedSearch) {
				facetedSearch.$overlay = elem;
				elem.addClass('f-faceted-search-overlay');
				elem.on('mousedown mouseup mouseout', function(event) {
					facetedSearch.overlayState = event.type;
				});
				elem.on('mousedown click contextmenu', function(event) {
					facetedSearch.editFacet(null);
					event.preventDefault();
					return false;
				});
			}
		};
	}

	FacetedSearchUtil.$inject = ['injector'];
	function FacetedSearchUtil(injector) {
		this.jsonForModel.$inject = ['ComplexValue'];
		this.jsonForModel = injector.partial(this.jsonForModel, this, ['model', 'inplace']);
	}
	FacetedSearchUtil.prototype = {
		/**
		* Decant ComplexValue objects from json storage
		* For use with fFacetedSearch#options.complex == true;
		* @param {Object} model model with ComplexValue objects replaced
		* @returns {Object} copy of fFacetedSearch model.
		*/
		jsonForModel: function(ComplexValue, model, inplace) {
			var result = inplace ? model : angular.copy(model);
			Object.keys(result).forEach(function(k) {
				var searchFacet = this.facets[k];
				result[k] = result[k].map(function(v) {
					if (ComplexValue.probablyComplex(v)) {
						return searchFacet.complexValue(v, true);
					} else {
						return v;
					}
				});
			}.bind(this));
			return result;
		},
		/**
		* For accessing facet methods
		* @see SearchFacet#complexValue
		*/
		facets: null

	};

	module.service('facetedSearchUtil', FacetedSearchUtil);

	module.directive('fFacetedSearchOverlay', fFacetedSearchOverlay);
	module.directive('fFacetedSearch', fFacetedSearch);
});
