define(['ml_module'], function(module) {
	'use strict';

	function re_escape(str) {
		return str.replace(/[\-\[\]\/\{\}\(\)\*\+\?\.\\\^\$\|]/g, '\\$&');
	};

	SearchFacetFactory.$inject = ['$injector', '$q', 'ComplexValue', 'lang'];
	function SearchFacetFactory($injector, $q, ComplexValue, lang) {
		/**
		* A Search Facet.
		* @constructor
		* @param {Object} config. Specify any property defined in the prototype
		*   to override.
		* - id, selectors and name are required.
		*/
		function SearchFacet(config) {
			/*jshint unused: false*/
			if (!(this instanceof SearchFacet)) {
				throw new Error('Please use the new keyword to create a SearchFacet');
			}

			var genRelevanceFns = {
				ip: SearchFacet.ipGenRelevanceFn
			};
			this.genRelevanceFn = genRelevanceFns[config.type] || this.genRelevanceFn;

			Object.keys(SearchFacet.prototype).forEach(function(prop) {
				if (prop in config) {
					this[prop] = config[prop];
				}
			}.bind(this));
			Object.keys(SearchFacet.prototype.tweakFilter).forEach(function(protoTF, prop) {
				if (!(prop in this.tweakFilter)) {
					this.tweakFilter[prop] = protoTF[prop];
				}
			}.bind(this, SearchFacet.prototype.tweakFilter));
		}

		SearchFacet.uniqCompact = function(values, pluckKey) {
			var result = values.reduce(makeObj, {});

			return Object.keys(result).map(getValues);
			function getValues(key) {
				return result[key];
			}
			function makeObj(result, item) {
				var key = pluckKey ? pluckKey(item) : item;
				if (key !== null && key !== '') {
					result[key] = item;
				}
				return result;
			}
		};

		/**
		* Generate a relevance function based on an array of regex parts. The searchValue will be
		* inserted between the regex parts and the resulting relevance rating funciton will return
		* the index of the leftmost matched regex subgroup. (or null for no matches)
		* Items with a property 'highlight' which is truthy or a function that returns truthy
		* will receive a +0.5 relevance bonus.
		* @param {String} searchValue (partial) value that user is searching for.
		* @param {String[]} regExParts parts of a regex that should be interleaved with the search
		*   value. Should use | to separate subgroups. Resulting expression will be wrapped
		*   automatically in a non-matching group. EG: (?:<expression>)
		* @param {(String|RegExp)[]} [wildCards] wildcard strings or regular expressions that
		*   should be replaced. (NOTE: RegExp will be run against re_escaped searchValue!)
		* @param {String[]} [wildCardExprs] Replace wildcard strings with these regex components.
		*   EG: wildCards: ['*'], wildCardExprs: '[\da-fA-F]+', searchValue: '10.*.10'
		*          will transform to '10.[\d+a-fA-F]+.10'
		*       wildCards: [/^\\\*|\\\*$/], wildCardExprs: '[\da-fA-F\.\:]+', searchValue: '10.*'
		*          will transform into '10.[\d+a-fA-F]+'
		* @returns a relevance rating function for potential autocomplete values. (higher number is
		* more relevant, null is irrelevant)
		*/
		SearchFacet.regexGenRelevanceFn = function(searchValue, regExParts, wildCards,
				wildCardExprs) {
			var escapedSearch = re_escape((searchValue || '').toString());
			if (wildCards) {
				wildCards.forEach(function(wc, i) {
					if (!(wc instanceof RegExp)) {
						wc = new RegExp(re_escape(re_escape(wc)), 'g');
					}
					escapedSearch = escapedSearch.replace(wc, wildCardExprs[i]);
				});
			}
			var expr = '(?:' + regExParts.join(escapedSearch) + ')';
			var suggestRegex = new RegExp(expr, 'i');
			var ARRAY_MATCH_RELEVANCE = 0.001;
			return suggestionRelevance;

			/**
			* Return a relevance value for the value/item. If value is an array: match any value
			* of the array, with a small boost (1/array.length)*ARRAY_MATCH_RELEVANCE for each index
			* of the array. (earlier index gets a higher boost)
			*/
			function suggestionRelevance(value, item) {
				var i, len;
				if (Array.isArray(value)) {
					var result;
					for (i = 0, len = value.length; i < len; ++i) {
						result = suggestionRelevance(value[i], item);
						if (result !== null) {
							//add a small amount to the relevance based on the array index
							//that matched. eg: [0.001, 0.0006, 0.0003] for 3 possible values
							result += ((len - i) / len) * ARRAY_MATCH_RELEVANCE;
							return result;
						}
					}
					return null;
				}
				var highlightBonus = 0;
				if (item && ('highlight' in item) &&
						typeof item.highlight === 'function' && item.highlight()) {
					highlightBonus = 0.5;
				}
				if (searchValue === undefined) {
					return 0 + highlightBonus;
				}
				var match = suggestRegex.exec(value);
				if (match) {
					//don't care about the entire regex match, just submatches
					match.shift();
					for (i = 0; i < match.length; ++i) {
						if (match[i] !== undefined) {
							return match.length - i + highlightBonus;
						}
					}
				}
				return null;
			}
		};

		/**
		* Generate a relevance rating function when given a searchValue. The returned function
		* takes a suggestion value a relevance rating. The relevance function should return lower
		* numbers for more relevant results and null for results that are not relevant at all.
		* @param {String} searchValue
		* @returns a relevance rating function for potential autocomplete values. (higher number is
		* more relevant, null is irrelevant)
		*/
		SearchFacet.defaultGenRelevanceFn = function(searchValue, wildCards, wildCardExprs) {
			var ex = ['(^', '$)|(^', ')|(\\b', ')|(', ')'];
			return SearchFacet.regexGenRelevanceFn(searchValue, ex, wildCards, wildCardExprs);
		};

		/**
		* Generate a relevance rating function for a given searchValue. The returned function
		* evaluates a single autocomplete value which should be an ipv4 or ipv6 address.
		* Relevance function assumes that if value is an array, the first index only contains an ip
		* @see SearchFacet.defaultGenRelevanceFn
		*/
		SearchFacet.ipGenRelevanceFn = function(searchValue) {
			//not sure if this is accurate but it seems to work ok
			var rel = SearchFacet.defaultGenRelevanceFn(searchValue, [/^\\\*|\\\*$/g], ['[\\da-fA-F.:]*']);
			return function(value) {
				var ip = Array.isArray(value) ? value[0] : value;
				var fields = ip.replace(/(^[|]$)/g, '').split(/[.:]/);
				var result = rel(value);
				if (fields.length) {
					result = fields.reduce(maxRel, result);
				}
				return result;
			};

			function maxRel(result, field) {
				var frel = rel(field);
				// Math.max(null, null) -> 0 ? wtfjs!
				if (result === null && frel === null) {
					return null;
				} else {
					return Math.max(result, frel);
				}
			}

		};

		SearchFacet.NO_SORT = function() {return 0};
		SearchFacet.DEFAULT_SORT = null;

		/**
		* Normalized value option key/value pair. Can have $promise if value is pending.
		* Most of this interface is duplicated in the ComplexValue type.
		* @constructor
		* @see ComplexValue
		* @param {String|Number} key Key that is used in the search model.
		* @param {String|ComplexValue} value Value displayed to the user.
		* @param {String} [description] Description value (future use)
		* //TODO: also add icon class
		*/
		SearchFacet.ValueOption = function SearchFacet_ValueOption(key, value, description) {
			this.value = value;
			this.key = key;
			this.description = description;
		};
		// $promise will be a $q promise if the value property is currently being retrieved
		SearchFacet.ValueOption.prototype = {
			//highlight this item in suggestion dropdown. Highlighted items also get a
			// +0.5 relevance bonus.
			highlight: function() { return false },
			$promise: undefined,
			formatted: function() {
				return this.value;
			},
			/**
			* Check to see if this value is equal to the supplied value.
			* @param {String|SearchFacet.ValueOption} value
			* @returns {Boolean} true if value.key || value matches this key.
			*/
			equals: function(value) {
				return value instanceof SearchFacet.ValueOption ?
					value.key === this.key :
					value === this.key;
			},
			localeCompare: function(value) {
				return value instanceof SearchFacet.ValueOption ?
					String(this.formatted()).localeCompare(value.formatted()) :
					String(this.formatted()).localeCompare(value);

			},
			/**
			* @see ComplexValue#intersection
			* @param {Object[]} values Array of values that this may intersect with
			* @returns {Object[]|null} a list of values that were found or null
			*/
			intersection: function(values) {
				if (this.key instanceof ComplexValue) {
					return this.key.intersection(values);
				}
				var result = values.reduce(intersect.bind(this), []);
				return result.length ? result : null;

				function intersect(result, value) {
					/* jshint validthis: true */
					//
					if (ComplexValue.probablyComplex(value)) {
						if (!(value instanceof ComplexValue)) {
							value = new ComplexValue(value);
						}
						return result.concat(value.intersection([this.key]) || []);
					} else {
						return result.concat(this.key === value ? [value] : []);
					}
				}
			}
		};

		SearchFacet.prototype = {
			id: null,
			name: null,
			langName: function() { return lang(this.name) },
			selectors: {},
			alias: null,
			unique: false,
			//sort compare function (a, b) { return a - b } (ascending)
			sort: null,
			//specify modifiers to restrict the allowed modifiers for this facet.
			modifiers: null,
			//placeholder to add to the value input
			valuePlaceholder: '',
			//allow user to input any value in the text box. Set to false to only allow suggestions.
			allowUserInput: true,
			// limit the number of values shown on the suggestion list
			suggestionLimit: 1000,
			type: null,
			//allow 'user' meta data for sorting etc that doesn't affect behavior of the facet.
			meta: null,
			//Hint to show in the suggestion dropdown
			hint: '',
			//highlight this facet when suggesting relevant facets.
			// Highlighted suggestions also get an automatic 0.5 relevance boost
			highlight: function() { return false },
			//compare with another filter for localized sorting
			localeCompare: function(other) {
				return lang(this.name).localeCompare(lang(other.name));
			},
			/*jshint unused: false */
			/**
			* tweak filter mapping function for different data sources. The facetDef only has to
			* specify any keys that it's changing. Otherwise the prototype value here is used.
			*/
			tweakFilter: {
				/**
				* Tweak filters right before they go to the source.
				* Some filters need to be transformed (eg: query multiple values instead of one)
				* @param {String} filter value
				* @param {String} Current report_by parameter for api monitor.
				* @param {String} 'session' or 'monitor_api' to indicate if it is realtime
				* @returns new value the back-end understands (or the old one)
				*/
				api_monitor: function api_monitor_tweakFilter(filter, reportBy, source) {
					return filter;
				},
				/**
				* Tweak filters right before they go to the history source (log filter json).
				* Mutate the passed in filter based on each field's backend oddities.
				* Optionally return a new filter to add it to the filter list.
				* @param {String} source Source type of filter
				* @param {Object} filter Original filter before tweak.
				* @returns Object Optionally return a 2nd filter to be added.
				*/
				history: function history_tweakFilter(filter) {}
			},
			allowsModifier: function(modifier) {
				return this.modifiers === null ||
					this.modifiers.indexOf(ComplexValue.resolveModifier(modifier)) > -1;
			},
			/**
			* Return valueOptions to show in context menu for the cell with value `key`
			* @param {Object} key Value for which context is requested.
			* @param {Boolean} [complex=true] Add complex value filters as well as the basic filter.
			* @returns {SearchFacet.ValueOption[][]} An array of arrays of ValueOptions. Each
			* array is treated as a 'group' of context menu items.
			*/
			contextFilters: function facetContextFilters(key, complex) {
				complex = complex !== false;
				//TODO: find a way to search for 'none'
				if (!key) {
					if (this.allowsModifier('!') && complex) {
						return [[new ComplexValue('!*')]];
					} else {
						return null;
					}
				}
				var modifiers = (complex ? ['<=', '>=', '!'] : [])
					.map(makeModifierFilter.bind(this))
					.filter(notNull);
				return [[this.normalizeOption(key)].concat(modifiers)];

				function notNull(v) { return v != null }
				function makeModifierFilter(modifier) {
					/* jshint validthis: true */
					if (this.allowsModifier(modifier)) {
						var value = ComplexValue.build([key], [modifier]);
						return new SearchFacet.ValueOption(value, value.formatted(this));
					}
					return null;
				}

			},
			getValues: function facetGetValues(entries, source) {
				var facet = this,
					MAX_OLD_ENTRIES = 200,
					oldEntries = this._entries ? this._entries.slice(0, MAX_OLD_ENTRIES) : [];
				return SearchFacet
					.uniqCompact((entries || []).map(pluckKey).concat(oldEntries), pluckUnique);
				function pluckKey(item) {
					var result = facet.getKey(item, source);
					if (result === undefined) {
						result = null;
					}
					return result;
				}
				function pluckUnique(value) {
					return !value || value.value == null ?
						value : value.value;
				}
			},
			populate: function facetPopulate(entries, source) {
				var facet = this;
				return $q.when(entries)
					.then(function(result) {
						facet._entries = facet.getValues(result, source);
						return facet._entries;
					});
			},
			getKey: function facetGetKey(entry, source) {
				/*jshint validthis: true*/
				return entry[this.selectors[source]];
			},
			/**
			* look up the key or value. Normally the key and the value are
			* the same. Override this fn to do a lookup depending on which is
			* supplied/needed.
			* @virtual
			* @param {String|Number} key Value or key to lookup.
			* @param {Boolean} [reverse=false] Whether to look up the key from
			*   the value or value from the key. If reverse is true, the result
			*   will be the key. If it is false or undefined the result is the
			*   display value.
			* @returns value or promise which returns a value.
			*/
			lookup: function facetLookup(key, reverse) {
				var facet = this,
					entries = this._entries || this.getValues();
				if (entries) {
					return $q.when(entries).then(function(entries) {
						facet._entries_cache = facet._entries_cache || {};
						facet._rentries_cache = facet._rentries_cache || {};

						var entry = reverse ? facet._rentries_cache[key] :
											  facet._entries_cache[key];
						if (entry !== undefined) {
							// found in cache
							return entry;
						}

						var ret;
						//TODO: use Array.prototype.find when ES6 lands
						entry = entries.filter(match)[0];
						if (entry instanceof SearchFacet.ValueOption) {
							ret = reverse ? entry.key : entry.value;
						} else {
							ret = entry ? entry : key;
						}
						facet._entries_cache[key] = ret;
						return ret;
					});
				}
				return key;
				function match(entry) {
					if (entry instanceof SearchFacet.ValueOption) {
						return reverse ?
							//valueOf Translated is the original string
							//valueOf most objects is the object
							entry.value !== undefined && entry.value.valueOf() === key :
							entry.key !== undefined && entry.key.valueOf() === key;
					} else {
						return entry === key;
					}
				}
			},
			/**
			* Look up the key from a given display value.
			* @param {String} value Value for which the key is desired.
			* @param {Any} ... Extra arguments (sometimes needed for context)
			*/
			reverseLookup: function facetReverseLookup(value) {
				var args = Array.prototype.slice.call(arguments, 1);
				//sometimes lookup functions have extra arguments.
				return this.lookup.apply(this, [value, true].concat(args));
			},
			clearValues: function facetClearValues() {
				this._values = null;
			},

			/**
			* convert a value into a key/value object. I think this only really
			* applyies for App/app_id but it's nice to have the option.
			* uses lookup(key) if avaliable to fill in the value from the key
			* if value isn't specified.
			* @param {String|KVD} key Lookup key in cmdb. Not shown to the user
			*   If KVD then {key:String, [value:String, description:String]}
			* @param {String} [value] value for specified key or undefined.
			* @returns {SearchFacet.ValueOption} Normalized ValueOption key/value pair.
			*/
			normalizeOption: function normalizeOption(key, value) {
				if (!(key instanceof SearchFacet.ValueOption)) {
					var result = new SearchFacet.ValueOption(key, value);
					if (key && key.description) {
						result.description = key.description;
					}
					if (value === undefined) {
						var promise = this.lookup(key);
						result.$promise = $q.when(promise, function(v) {
							delete result.$promise;
							result.value = v;
							return v;
						});
					}
					return result;
				} else {
					return key;
				}
			},
			//Generate a relevance rating function for ValueOption comparison/suggestion
			genRelevanceFn: SearchFacet.defaultGenRelevanceFn,
			//Assign a priority function to allow each facet to adjust it's relevance
			//when suggesting facets
			adjustRelevance: function(searchValue, relevance) { return relevance },

			//format and parse functions which can be used by f-facet-format to pass them to ngModel
			format: function(value) {
				return value;
			},
			parse: function(value) {
				return value;
			},
			/**
			* Output a safe complexValue for this type.
			* expected to always copy the incoming ComplexValue object
			* NOTE: When overriding, the override MUST accept null values, and return null.
			* @virtual
			* @param {ComplexValue|Object|null} value
			* @param {Boolean} [convert=false] always convert to complex
			* @returns {ComplexValue|Object} Original object or a new ComplexValue
			*/
			complexValue: function facetComplexValue(value, convert) {
				var isComplex = value instanceof ComplexValue;
				if (value != null && (isComplex || convert)) {
					value = new ComplexValue(value);
				}

				return value;
			}
		};

		$injector.annotate(SearchFacet);
		return SearchFacet;
	}

	module.factory('SearchFacet', SearchFacetFactory);
});
