/* globals define */
define(['angular', 'ng/ftnt', 'jquery', 'ng/services/loader', 'jquery.util_core',
    'ng/directives/menu/menu_items', 'ng/directives/menu/qlist_config_controller',
    'ng/directives/menu/templates'
], function(angular, module, $, loader) {
    'use strict';

    var templatePath = loader.base_path('/ng/directives/menu/partials/');

    var QLIST_SELECT_ROWS_EVENT = 'qlist_select_rows';
    var QLIST_DOUBLE_CLICK_ROW_EVENT = 'qlist_dblclick_row';
    var QLIST_CONTEXT_MENU_EVENT = 'qlist_contextmenu';
    var QLIST_CONFIGURATION_MENU_EVENT = 'qlist_config_menu';

    var namespaceEvent = function(eventName, scope) {
        return eventName + '.menu' + scope.$id;
    };

    var addQlistEventData = function(menu, data) {
        if (data.event) {
            menu.$target = angular.element(data.event.target);
        }
        menu.lastSelectedEntry = data.last_selected;
        menu.entries = data.entries || [];
        menu.rows = data.rows || [];
        menu.$sectionRow = data.$section_row;
        menu.$categoryRow = data.$category_row;
        menu.$td = data.$td;
        menu.columnSelector = data.column_selector;
        menu.columnValue = data.column_value;
    };

    /**
     * @ngdoc directive
     * @name ng.directive:fMenuBar
     *
     * @description
     * Use this directive to create a menu bar that supports different menu item directives as
     * well as plain HTML content. Optionally it can be provided a qlist jQuery selector to
     * automatically update the scope with selected row information.
     *
     * Example:
     * <div f-menu-bar qlist-selector="'#qlist-element'">
     *     Some content...
     * </div>
     */
    module.directive('fMenuBar', function() {
        return {
            restrict: 'A',
            transclude: true,
            scope: {
                qlistSelector: '=',
                qlistElement: '='
            },
            controller: function($scope, $element, $attrs, $transclude) {
                var transcludeScope;

                var setupQlistEvents = function(qlist) {
                    var selectEvent, dblclickEvent;
                    var $qlist = $(qlist);
                    if (!$qlist.length) {
                        throw new Error('Qlist selector "' + $scope.qlistSelector +
                                        '" failed to locate element');
                    }
                    selectEvent = namespaceEvent(QLIST_SELECT_ROWS_EVENT, $scope);
                    dblclickEvent = namespaceEvent(QLIST_DOUBLE_CLICK_ROW_EVENT, $scope);

                    $qlist.off(selectEvent).on(selectEvent, function(event, data) {
                        $scope.$evalAsync(function() {
                            addQlistEventData(this, data);
                        }.bind(this));
                    }.bind(this));
                    $qlist.off(dblclickEvent).on(dblclickEvent, function(event, data) {
                        $scope.$apply(function() {
                            addQlistEventData(this, data);
                        }.bind(this));
                        if (angular.isFunction(this.qlistDoubleClickHandler)) {
                            this.qlistDoubleClickHandler.apply(this, arguments);
                        }
                    }.bind(this));

                    $scope.$on('$destroy', function() {
                        $qlist.off(selectEvent).off(dblclickEvent);
                    });
                    this.$qlist = $qlist;
                }.bind(this);

                if ($attrs.qlistSelector) {
                    $scope.$watch('qlistSelector', function(qlistSelector) {
                        if (qlistSelector) {
                            setupQlistEvents(qlistSelector);
                        }
                    }.bind(this));
                } else if ($attrs.qlistElement) {
                    $scope.$watch('qlistElement.element', function(qlistElement) {
                        if (qlistElement) {
                            setupQlistEvents(qlistElement);
                        }
                    }.bind(this));
                }
                $scope.$on('clearQlistSelection', function() {
                    addQlistEventData(this, {});
                }.bind(this));

                $element.addClass('menu-bar');
                transcludeScope = $scope.$parent.$new();
                this.menuBar = true;
                transcludeScope.menu = this;

                $transclude(transcludeScope, function(clone) {
                    $element.append(clone);
                });
            }
        };
    });

    module.factory('popUpMenuUtil', function() {
        var service = {};
        var visible = {};
        var $mask;

        service.getMask = function() {
            if (!$mask) {
                $mask = angular.element('<div/>').addClass('pop-up-menu-mask').appendTo('body');
            }
            return $mask;
        };

        service.setVisible = function(id, show) {
            if (show) {
                visible[id] = true;
            } else {
                delete visible[id];
            }
            var hidden = Object.keys(visible).length === 0;
            this.getMask().toggleClass('hidden', hidden);
        };

        return service;
    });

    /**
     * The exposed pop up menu controller. Provides functions to show and hide the menu. Note that
     * any properties or methods that are prefixed with "_" are intended for internal use only.
     *
     * @constructor
     */
    /* jshint maxparams:11 */
    function PopUpMenuController($scope, $element, $attrs, $templateCache, $window, $q, $http,
                                   $compile, $timeout, loader, popUpMenuUtil) {
        // Internal state
        this._$scope = $scope;
        this._$attrs = $attrs;
        this._$templateCache = $templateCache;
        this._$window = angular.element($window);
        this._$q = $q;
        this._$http = $http;
        this._$compile = $compile;
        this._$timeout = $timeout;
        this._loader = loader;
        this._popUpMenuUtil = popUpMenuUtil;
        this._menuId = 'popUpMenu' + $scope.$id;
        this._children = [];
        this._leftAdjusted = false;
        this.toggleDefer = null;

        // Exposed state
        this.popUpMenu = true;
        this.$element = $element;
        this.visible = false;
        // Any custom info that needs to be added to the controller should be done here.
        this.metaInfo = null;
        // When a pop up is closed, the returned promise from "toggle" will be resolved with
        // this property. Reset with each new toggle. Directives can override this property if they
        // wish to communicate back information to the toggling function. This is only necessary
        // if the toggler is outside ng context and should be avoided otherwise.
        this.toggleMeta = null;
        // Qlist specific
        this.$qlist = null;
        this.entries = [];
        this.lastSelectedEntry = null;
        this.$sectionRow = null;
        this.$td = null;
        this.columnSelector = null;
        this.columnValue = null;

        if ($attrs.qlistSelector) {
            $scope.$watch('qlistSelector', function(qlistSelector) {
                if (qlistSelector) {
                    this._setupQlistEvents(qlistSelector);
                }
            }.bind(this));
        } else if ($attrs.qlistElement) {
            $scope.$watch('qlistElement.element', function(qlistElement) {
                if (qlistElement) {
                    this._setupQlistEvents(qlistElement);
                }
            }.bind(this));
        }
        if ($attrs.fPopUpMenu) {
            $scope.popUpMenu = this;
        }
        this._setupAutoPositioning();
        this._show(false);
    }

    PopUpMenuController.prototype._show = function(show) {
        this.visible = show;
        this.$element.toggleClass('hidden', !show);
        this._popUpMenuUtil.setVisible(this._menuId, show);
        if (!show) {
            this._children.forEach(function(childMenu) {
                childMenu._show(false);
            }.bind(this));
            this._leftAdjusted = false;
            if (this.toggleDefer) {
                this.toggleDefer.resolve(this.toggleMeta);
                this.toggleMeta = null;
            }
        }
    };

    PopUpMenuController.prototype._setPosition = function(position, $target) {
        var WINDOW_BUFFER = 5;

        if (position || this.visible) {
            if (!this.visible) {
                this.$element.css({
                    top: 0,
                    left: 0,
                    visibility: 'hidden'
                }).removeClass('hidden');
            }
            position = position || this.$element.offset();
            var buffer = this._$scope.windowBuffer != null ? this._$scope.windowBuffer :
                WINDOW_BUFFER;
            var top = position.top;
            var left = position.left;
            var windowWidth = this._$window.width();
            var windowHeight = this._$window.height();
            var maxWidth = windowWidth - buffer * 2;
            var maxHeight = windowHeight - buffer * 2;
            var width = Math.min(this.$element.width(), maxWidth);
            var height = Math.min(this.$element.height(), maxHeight);

            var leftOver = left + width + buffer - windowWidth;
            var topOver = top + height + buffer - windowHeight;
            if (leftOver > 0) {
                if (!this._leftAdjusted && position.leftOverAdjust) {
                    left -= position.leftOverAdjust + width + 2;
                    this._leftAdjusted = true;
                } else {
                    left -= leftOver;
                }
                if (left < buffer) {
                    left = buffer;
                }
            }
            if (topOver > 0) {
                top -= topOver;
                if (top < buffer) {
                    top = buffer;
                }
            }
            this.$element.css({
                left: left,
                top: top,
                'max-width': maxWidth,
                'max-height': maxHeight,
                'visibility': 'visible',
                // Auto will cause the scrollbar to overlap the contents of the context
                // menu so we will determine when it is necessary to show it
                'overflow-y': this.$element[0].scrollHeight > maxHeight ? 'scroll' : 'hidden'
            });

            if ($target && $target.length) {
                var isTargetMasked = this._isTargetMasked(left, top, width, height, $target);
                $target.toggleClass('is-masked-by-popup-menu', isTargetMasked);
            }

            if (position.minWidth) {
                this.$element.css('min-width', position.minWidth);
            }
        }
    };

    PopUpMenuController.prototype._setupQlistEvents = function(selector) {
        var $qlist = angular.element(selector);
        var contextEvent = namespaceEvent(QLIST_CONTEXT_MENU_EVENT, this._$scope);
        var configMenuEvent = namespaceEvent(QLIST_CONFIGURATION_MENU_EVENT, this._$scope);
        var dblclickEvent = namespaceEvent(QLIST_DOUBLE_CLICK_ROW_EVENT, this._$scope);
        var configMenuScopeKey = 'qlistConfigContextMenu';
        var configMenuTemplate = this._loader.base_path(
            '/ng/directives/menu/partials/qlist_config.html');
        var configMenuContent = this._$http.get(configMenuTemplate, {cache: this._$templateCache});
        var configMenuElement, configMenuScope;

        if (!$qlist.length) {
            throw new Error('Qlist selector "' + selector + '" failed to locate element');
        }

        $qlist.on(contextEvent, function(event, data) {
            this._$scope.$apply(function() {
                addQlistEventData(this, data);
            }.bind(this));
            // Queue positioning (and rendering) until the menu's contents have $digested and
            // rendered. This way we will know the actual dimensions.
            this._$timeout(function() {
                this._setPosition({
                    left: data.event.pageX,
                    top: data.event.pageY
                }, this.$target); // also pass in the qlist event's target (see addQlistEventData)
                this._show(true);
            }.bind(this));
        }.bind(this));

        $qlist.on(dblclickEvent, function(event, data) {
            this._$scope.$apply(function() {
                addQlistEventData(this, data);
                this._$scope.qlistDoubleClick({entry: this.lastSelectedEntry});
            }.bind(this));
        }.bind(this));

        $qlist.on(configMenuEvent, function(event, data) {
            if (configMenuScope) {
                configMenuScope.$destroy();
                configMenuElement.remove();
            }
            configMenuScope = this._$scope.$new(true);
            configMenuScope.menu = this;
            configMenuScope.options = data.options;
            configMenuContent.then(function(response) {
                configMenuElement = this._$compile('<div f-pop-up-menu="' +
                    configMenuScopeKey + '">' + response.data + '</div>')(configMenuScope);
                // Wait for the isolate scope bind
                configMenuScope.$watch(configMenuScopeKey, function(popUpMenu) {
                    if (popUpMenu) {
                        // Queue positioning (and rendering) until the menu's contents have
                        // $digested and rendered. This way we will know the actual dimensions.
                        this._$timeout(function() {
                            popUpMenu.toggle({
                                left: data.event.pageX,
                                top: data.event.pageY
                            }).then(function(meta) {
                                var action = meta.action;
                                if (action === 'cancel') {
                                    return;
                                } else if (!action) {
                                    action = 'update_columns';
                                }
                                $qlist.qlist(action, meta.chosen);
                            });
                        });
                    }
                }.bind(this));
            }.bind(this));
        }.bind(this));

        this._$scope.$on('clearQlistSelection', function() {
            addQlistEventData(this, {});
        }.bind(this));

        this._$scope.$on('$destroy', function() {
            $qlist.off(contextEvent).off(configMenuEvent).off(dblclickEvent);
        });
        this.$qlist = $qlist;
    };

    PopUpMenuController.prototype._setupAutoPositioning = function() {
        var $mask = this._popUpMenuUtil.getMask();
        var clickEvent = 'click.popUpMenu-' + this._menuId +
            ' contextmenu.popUpMenu-' + this._menuId;

        var debouncedSetPosition = $.debounce(50, function() {
            this._setPosition();
        }.bind(this));

        this._$window.on('resize', debouncedSetPosition);

        $mask.on(clickEvent, function(event) {
            event.preventDefault();
            if (this.visible) {
                this._show(false);
            }
        }.bind(this));

        this._$scope.$on('$destroy', function() {
            $mask.off(clickEvent);
        });
    };

    PopUpMenuController.prototype._addChild = function(childMenu) {
        this._children.push(childMenu);
        childMenu._parent = this;
    };

    PopUpMenuController.prototype._removeChild = function(childMenu) {
        this._children.splice(this._children.indexOf(childMenu), 1);
        childMenu._parent = null;
    };

    PopUpMenuController.prototype._getElementAndChildren = function() {
        var result = [this.$element];
        this._children.forEach(function(childMenu) {
            result = result.concat(childMenu._getElementAndChildren());
        });
        return result;
    };

    PopUpMenuController.prototype._isTargetMasked = function(left, top, width, height, $target) {
        if (!$target || !$target.length) {
            return false;
        }
        var bottom = top + height;
        var right = left + width;
        var targetOffset = $target.offset();
        targetOffset.bottom = targetOffset.top + $target.outerHeight();
        targetOffset.right = targetOffset.left + $target.outerWidth();
        if (targetOffset.right <= left || targetOffset.left >= right ||
            (targetOffset.bottom - 1) <= top || targetOffset.top >= bottom) {
            return false;
        }
        return true;
    };

    /**
     * Shows or hides the pop up menu
     *
     * @param {object|null} [position] - The position to show the pop up menu at. Must have
     *   number properties "top" and "left", or be falsy to hide.
     * @param {jQuery object} [element] - The element which initiated the toggle
     * @return {promise} - Promise that will be resolved when the pop up menu is eventually
     *   closed. If a null "position" argument is provided, the promise will be rejected
     *   immediately.
     */
    PopUpMenuController.prototype.toggle = function(position, element) {
        var show = angular.isObject(position);
        var defer = this._$q.defer();
        if (show) {
            this._setPosition(position, element);
            this.toggleDefer = defer;
        } else {
            defer.reject();
        }
        this._show(show);
        return defer.promise;
    };

    /**
     * Hides the pop up menu
     *
     * @param {boolean} parent - Hide parent as well? (and its parent and so on...)
     * @return {undefined}
     */
    PopUpMenuController.prototype.hide = function(parent) {
        this._show(false);
        if (parent && this._parent) {
            this._parent.hide(true);
        }
    };

    /**
     * @ngdoc directive
     * @name ng.directive:fPopUpMenu
     *
     * @description
     * Use this directive to create a pop up menu that supports different menu item directives as
     * well as plain HTML content. It must be provided one of the following evaluated attributes:
     *   1.) "qlist-selector": A jQuery selector for a qlist element. By subscribing to qlist
     *         events, the positioning and displaying of the pop up menu will be handled
     *         automatically.
     *   2.) "qlist-element": The actual qlist element. By subscribing to qlist events, the
     *         positioning and displaying of the pop up menu will be handled automatically.
     *   3.) "f-pop-up-menu": The scope value to bind the created pop up menu controller to.
     *         This controller contains all the necessary functionality to control the pop up
     *         menu (see above).
     *
     * Examples:
     *
     * <div f-pop-up-menu qlist-selector="'#qlist-element'">
     *     Some content...
     * </div>
     *
     * <div f-pop-up-menu qlist-element="qlistElement">
     *     Some content...
     * </div>
     * <div f-qlist qlist-element="qlistElement"></div>
     *
     * <div f-pop-up-menu="popUpMenu">
     *     Some content...
     * </div>
     *
     *
     */
    module.directive('fPopUpMenu', function() {
        return {
            restrict: 'A',
            transclude: true,
            scope: {
                popUpMenu: '=fPopUpMenu',
                qlistSelector: '=',
                qlistElement: '=',
                qlistDoubleClick: '&',
                windowBuffer: '='
            },
            controller: PopUpMenuController,
            link: function(scope, element, attrs, controller, transclude) {
                // Move to body (for absolute positioning)
                angular.element('body').append(element);
                element.addClass('pop-up-menu');

                element.on('contextmenu', function(event) {
                    event.preventDefault();
                });

                var transcludeScope = scope.$parent.$new();
                transcludeScope.menu = controller;

                transclude(transcludeScope, function(clone) {
                    element.append(clone);
                });

                scope.$on('$destroy', function() {
                    element.remove();
                });
            }
        };
    });

    /**
     * @ngdoc directive
     * @name ng.directive:fPopUpMenuToggle
     *
     * @description
     * Use this directive to automatically toggle a pop up menu below the element when clicked.
     *
     * Example:
     *
     * <button type="button" f-pop-up-menu-toggle="popUpMenu">
     *    Click me to show a context menu
     * </button>
     *
     * <div f-pop-up-menu="popUpMenu">
     *    Some context menu content...
     * </div>
     *
     */
    module.directive('fPopUpMenuToggle', function() {
        return {
            restrict: 'A',
            templateUrl: templatePath + 'pop_up_toggle.html',
            transclude: true,
            scope: {
                popUpMenu: '=fPopUpMenuToggle',
                metaInfo: '=',
                onlyIf: '&'
            },
            link: function(scope, element, attrs) {
                if (attrs.noToggleIcon == null) {
                    element.find('.flex-button-content')
                    .append('<div class="flex-spacer"></div>')
                    .append('<f-icon class="fa-caret-down toggle-indicator"></f-icon>');
                }

                element.on('click', function() {
                    if (attrs.onlyIf == null || scope.onlyIf()) {
                        scope.$apply(function() {
                            var offset;
                            if (scope.popUpMenu.visible) {
                                scope.popUpMenu.toggle(false);
                            } else {
                                scope.popUpMenu.metaInfo = scope.metaInfo;
                                offset = element.offset();
                                element.addClass('pop-up-toggled');
                                scope.popUpMenu.$element.addClass('button-toggled');
                                scope.popUpMenu.toggle({
                                    top: offset.top + element.outerHeight() - 1,
                                    left: offset.left,
                                    minWidth: element.outerWidth()
                                }, element).then(function() {
                                    element.removeClass('pop-up-toggled');
                                });
                            }
                        });
                    }
                });
            }
        };
    });
});
