/* globals define */

define(['angular', 'ng/services/loader'], function(angular) {
    'use strict';

    /**
     * @ngdoc directive
     * @name ng.directive:fTabs
     *
     * @description
     * This directive will provide a tabbed wrapper around the content specified within the
     * element.
     * @param {string[]|object[]} tabs - An array of tab string values (to be used alongside
     * "langMap") or an array of objects of the form:
     *   {
     *      value: '...',
     *      langKey: '...'
     *   }
     * @param {object} [langMap] - A mapping of tab values to language keys. Used when "tabs" is a
     * simple list of string values
     * @param {expression} [onChange] - An angular expression that will be evaluated on tab change.
     * The expression will be provided the local "tab" which be set to the selected tab
     * @param {string} [selectedTab] - The initial tab that is selected. If not provided "onChange"
     * will be evaluated with the first tab provided
     *
     * Examples:
     *
     *   <f-tabs tabs="['one', 'two', 'three']", lang-map="{one: 'One', two: 'Two', three: 'Three'}"
     *     on-change="selectedTab = tab">
     *     <div ng-switch="selectedTab">
     *       <div ng-switch-when="one">
     *          ...
     *        </div>
     *       <div ng-switch-when="two">
     *          ...
     *        </div>
     *       <div ng-switch-when="three">
     *          ...
     *        </div>
     *     </div>
     *   </f-tabs>
     *
     *   <f-tabs tabs="[{value: 'one', langKey: 'One'}, {value: 'two', langKey: 'Two'}]"
     *     on-change="selectedTab = tab" selected='two'>
     *       <div ng-switch-when="one">
     *          ...
     *        </div>
     *       <div ng-switch-when="two">
     *          ...
     *        </div>
     *   </f-tabs>
     */
    var tabs = function(loader) {
        return {
            restrict: 'E',
            scope: {
                tabs: '=',
                langMap: '=',
                hideOptions: '=',
                onChange: '&',
                selectedTab: '@selected',
		genPdf: '&'
            },
            replace: true,
            transclude: true,
            templateUrl: loader.base_path('ng/directives/tabs.html'),
            link: function(scope) {
                scope.$watch('tabs', function(tabs, previous) {
                    scope.formattedTabs = tabs.map(function(tab) {
                        if (!angular.isObject(tab)) {
                            tab = {
                                value: tab,
                                langKey: scope.langMap ? scope.langMap[tab] : tab
                            };
                        }
                        return tab;
                    });
                    // Intialize by calling "onChange" if "selected" wasn't provided
                    if (tabs === previous && !scope.selectedTab) {
                        scope.change(scope.formattedTabs[0].value);
                    }
                });

                scope.change = function(tab) {
                    scope.selectedTab = tab;
                    scope.onChange({tab: tab});
                };

		scope.gen_pdf = function() {
		    scope.genPdf();
		}
            }
        };
    };

    return function(providers) {
        providers.$compile.directive('fTabs', tabs);
    };
});
