/*globals define*/
/**
 * @ngdoc service
 * @name ng.service:adminEvent
 *
 * @description
 * This service provides the data used for the FortiView admin timeline visualization
 */
define(['fweb'], function(fweb) {
    'use strict';

    var transformLogId = (function() {
        var eventIdMap = {
            32001: 'auth-login',
            32003: 'auth-logout',
            38407: 'guest-notify',
            32131: 'guest-delete',
            32129: 'guest-add',
            32301: 'vdom-add',
            44547: 'config-add',
            44545: 'config-delete',
            32400: 'config-change',
            44546: 'config-attribute',
            32215: 'image-update',
            32202: 'image-restore',
            32138: 'system-reboot',
            32140: 'ntp-update'
        };

        return function(logId) {
            return eventIdMap[logId];
        };
    })();

    var transformLogLevel = (function() {
        var levels = [
            'information',
            'notice',
            'warning',
            'alert',
            'critical'
        ];

        return function(event) {
            var level = event.level;

            if (isConfig(event)) {
                level = 'warning';
            }

            var levelIndex = levels.indexOf(level);
            if (levelIndex >= 0) {
                return levelIndex;
            }

            return 0;
        };
    })();

    function generateHash(str) {
        /* jshint bitwise:false */

        var hash = 0;
        var i, len;

        if (str.length === 0) {
            return hash;
        }

        for (i = 0, len = str.length; i < len; ++i) {
            hash = (((hash << 5) - hash) + str.charCodeAt(i) | 0);
        }

        return hash;
    }

    function transformEvent(log) {
        var dt = fweb.util.datetime.localSecondsToDate(log.rel_time);
        var event = {
            timestamp: dt.getTime(),
            user: log.user,
            type: transformLogId(log.logid) || log.action,
            level: transformLogLevel(log),
            ui: log.ui,
            detail: {
                type: log.action,
                subtype: log.reason,
                message: log.msg
            }
        };

        event.id = generateHash(JSON.stringify(event));

        return event;
    }

    function isLogin(event) {
        return event.type === 'login';
    }

    function isLogout(event) {
        return event.type === 'logout';
    }

    function isConfig(event) {
        return ['Add', 'Edit', 'Delete'].indexOf(event.action) >= 0;
    }

    function generateSession(event) {
        return {
            starting_time: event.starting_time,
            ending_time: event.ending_time,
            related_events: [event]
        };
    }

    function joinSessionOut(event, sessions) {
        if (isLogin(event)) {
            var session = generateSession(event);
            sessions.push(session);
            return joinSessionIn;
        }

        return joinSessionOut;
    }

    function joinSessionIn(event, sessions) {
        var session = sessions[sessions.length - 1];

        if (isLogout(event)) {
            session.related_events.push(event);
            session.ending_time = event.starting_time;
            return joinSessionOut;
        }

        if (isConfig(event)) {
            session.related_events.push(event);
        }

        return joinSessionIn;
    }

    function unique(list) {
        return list.reduce(function(accumulator, value) {
            if (value != null && accumulator.indexOf(value) === -1) {
                return accumulator.concat(value);
            }
            return accumulator;
        }, []);
    }

    function groupBySession(events) {
        var sessions = {};
        var salt = 0;

        var se = events.sort(function(a, b) {
            return a.timestamp - b.timestamp;
        }).map(function(event) {
            var sessionId = null;

            event = transformEvent(event);

            if (isLogin(event) || sessions[event.ui] == null) {
                sessionId = event.id + 's' + salt;
                if (event.ui) {
                    sessions[event.ui] = sessionId;
                }
                salt += 1;
            } else if (isLogout(event)) {
                if (event.ui) {
                    sessionId = sessions[event.ui];
                    sessions[event.ui] = null;
                }
            } else {
                sessionId = sessions[event.ui];
            }

            event.session = sessionId;
            return event;
        });

        return unique(se.map(function(event) {
            return event.session;
        })).map(function(sessionId) {
            var matchSession = function(event) {
                return event.session === sessionId;
            };

            var children = se.filter(matchSession);
            return {
                id: children[0].session,
                children: children,
                starting_time: children[0].starting_time,
                ending_time: children[children.length - 1].starting_time
            };
        });
    }

    function parseEvents(rawLogs) {
        var ignoreUsers = [
            'daemon_admin', 'ntp_daemon'
        ];
        var adminEvents = rawLogs.filter(function(d) {
            return d.ui;
        });
        var users = unique(adminEvents.map(function(d) {
                return d.user;
            }))
            .sort(function(a, b) {
                return a.localeCompare(b);
            })
            .filter(function(a) {
                return ignoreUsers.indexOf(a) === -1;
            });

        var data = users.map(function(user) {
            var matchUser = function(event) {
                return event.user === user;
            };

            return {
                id: user,
                adminuser: user,
                children: groupBySession(rawLogs.filter(matchUser))
            };
        });

        return data;
    }

    var LOG_URL = '/p/logs/system_events/json/';
    var MAX_ROWS = 10000;

    function AdminEvent($q, $resource, fortiviewFacets) {
        function generateFilterParam(timeframe, filters) {
            return fortiviewFacets.facets.makeQlistFilters(filters, false, timeframe);
        }

        return {
            qlistOptions: {
                columns: [
                    {selector: 'timestamp'},
                    {selector: 'user'},
                    {selector: 'detail'}
                ],
                default_columns: [
                    'user', 'detail', 'timestamp'
                ],
                format_fn: {
                    'user': function(td, col, data) {
                        return fweb.util.dom.escapeHTML(data[col.selector]);
                    },
                    'detail': function(td, col, data) {
                        var details = data[col.selector];
                        return fweb.util.dom.escapeHTML(details.message);
                    },
                    'timestamp': function(td, col, data) {
                        return fweb.util.datetime.formatDateTime(
                            new Date(data[col.selector])
                        );
                    }
                }
            },
            get: function(timeframe, filters) {
                var get = $q.defer();
                var filter = generateFilterParam(timeframe, filters);

                $resource(LOG_URL, {filter: [filter], rows: MAX_ROWS}).get(
                    function(logData) {
                        get.resolve({
                            timeframe: timeframe,
                            events: parseEvents(logData.source)
                        });
                    }
                );

                return get.promise;
            }
        };
    }

    return function(providers) {
        providers.$provide.service('adminEvent', AdminEvent);
    };

});
