/* globals define */
define(['angular', 'ng/services/injector', 'widgets', 'fweb.util/datetime', 'fweb.util/objects',
    './util', 'ng/services/persistent_storage', 'ng/services/ftnt'
], function(angular, inject, widgets, f_datetime, f_obj, fortiviewUtil) {
    'use strict';
    function LearningReport($resource, persistentStorage, injector) {
        injector.injectMarked(this);
        this._displayMode = persistentStorage.get(this.DISPLAY_MODE_KEY) || 'summary';
    }

    LearningReport.prototype.config = {
        chart: {
            request: {
                params: {
                    'app-categories': {
                        report_by: 'app-categories'
                    },
                    'video': {
                        filter: {
                            'appcat': 'Video/Audio'
                        }
                    },
                    'social_media': {
                        filter: {
                            'appcat': 'Social.Media'
                        }
                    },
                    'p2p': {
                        filter: {
                            'appcat': 'P2P'
                        }
                    },
                    'game': {
                        filter: {
                            'appcat': 'Game'
                        }
                    }
                },
                defaultParams: {
                    report_by: 'apps-with-category',
                    sort_by: 'bytes',
                    filter: {'policymode': 'learn'},
                    no_summary: true
                }
            }
        },
        list: {
            request: {
                params: {
                    'web-category': {
                        report_by: 'top-web-categories',
                        sort_by: 'sessions'
                    },
                    'web-domain': {
                        report_by: 'top-web-domains',
                        sort_by: 'sessions'
                    },
                    'web-application': {
                        report_by: 'top-web-apps',
                        sort_by: 'sessions'
                    },
                    'high_risk_application': {
                        report_by: 'high-risk-apps'
                    },
                    'app_vuln_exploits': {
                        report_by: 'app-vuln-exploits'
                    },
                    'malware_botnets': {
                        report_by: 'malware-botnets'
                    },
                    'at_risk_device_host': {
                        report_by: 'risk-device-hosts'
                    }
                },
                defaultParams: {
                    filter: {'policymode': 'learn'},
                    no_summary: true,
                    aggregate_hosts: true,
                    count: 10
                }
            },
            config: {
                'high_risk_application': {
                    qlistOption: {
                        prefix: 'qlist_high_risk_application',
                        default_columns: [
                            'risk', 'apps', 'category', 'user_count', 'data_bytes', 'sessions'
                        ],
                        columns: [
                            { selector: 'apps',       lang_key: 'application', fixed: true },
                            { selector: 'sessions',   lang_key: 'Sessions', fixed: true },
                            { selector: 'user_count', lang_key: 'user', fixed: true },
                            { selector: 'risk',       lang_key: 'fv_risk', fixed: true },
                            { selector: 'appcat',     lang_key: 'category', fixed: true  },
                            { selector: 'data_bytes', lang_key: 'total-bytes', fixed: true }
                        ]
                    }
                },
                'app_vuln_exploits': {
                    qlistOption: {
                        prefix: 'qlist_app_vuln_exploits',
                        default_columns: [
                            'severity', 'attackname', 'victim_count', 'source_count', 'total_count'
                        ],
                        columns: [
                            { selector: 'severity',     lang_key: 'severity', fixed: true },
                            { selector: 'attackname',   lang_key: 'Threat Name', fixed: true },
                            { selector: 'victim_count', lang_key: 'Victim', fixed: true },
                            { selector: 'source_count', lang_key: 'Source', fixed: true },
                            { selector: 'total_count',  lang_key: 'count', fixed: true }
                        ]
                    }
                },
                'malware_botnets': {
                    qlistOption: {
                        prefix: 'qlist_top_malware',
                        default_columns: [
                            'virus', 'type', 'appname', 'victim_count', 'source_count',
                            'total_count'
                        ],
                        columns: [
                            { selector: 'virus',        lang_key: 'Malware Name', fixed: true },
                            { selector: 'type',         lang_key: 'type', fixed: true },
                            { selector: 'total_count',  lang_key: 'count', fixed: true },
                            { selector: 'victim_count', lang_key: 'Victim', fixed: true },
                            { selector: 'source_count', lang_key: 'Source', fixed: true },
                            { selector: 'appname',      lang_key: 'application', fixed: true }
                        ]
                    }
                },
                'at_risk_device_host': {
                    qlistOption: {
                        prefix: 'qlist_at_risk_device_host',
                        default_columns: ['srcip', 'device', 'score'],
                        columns: [
                            { selector: 'srcip',  lang_key: 'Host', fixed: true },
                            { selector: 'device', lang_key: 'device', fixed: true },
                            { selector: 'score',  lang_key: 'Threat Score (Blocked/Allowed)',
                              fixed: true }
                        ]
                    }
                },
                'web-domain': {
                    qlistOption: {
                        prefix: 'qlist_top_domains',
                        default_columns: ['domain', 'catdesc', 'visit_count'],
                        columns: [
                            { selector: 'domain',      lang_key: 'domain', fixed: true },
                            { selector: 'catdesc',     lang_key: 'category', fixed: true },
                            { selector: 'visit_count', lang_key: 'Visits', fixed: true }
                        ]
                    }
                },
                'web-category': {
                    qlistOption: {
                        prefix: 'qlist_top_categories',
                        default_columns: ['catdesc', 'user_count', 'total_count', 'data_bytes'],
                        columns: [
                            { selector: 'catdesc',     lang_key: 'URL Category', fixed: true },
                            { selector: 'user_count',  lang_key: 'User', fixed: true },
                            { selector: 'total_count', lang_key: 'count', fixed: true },
                            { selector: 'data_bytes',  lang_key: 'total-bytes', fixed: true }
                        ]
                    }
                },
                'web-application': {
                    qlistOption: {
                        prefix: 'qlist_top_applications',
                        default_columns: ['apps', 'sessions', 'data_bytes'],
                        columns: [
                            { selector: 'apps',       lang_key: 'application', fixed: true },
                            { selector: 'sessions',   lang_key: 'Sessions', fixed: true },
                            { selector: 'data_bytes', lang_key: 'total-bytes', fixed: true }
                        ]
                    }
                }
            }
        },
        summary:  {
            request: {
                params: {
                    'top_host_by_bandwidth': {
                        report_by: 'top1-host-by-bandwidth'
                    },
                    'top_host_by_session': {
                        report_by: 'top1-host-by-session',
                        sort_by: 'sessions'
                    },
                    'app_category': {
                        report_by: 'top1-app-cat'
                    },
                    'web_category': {
                        report_by: 'top1-web-cat'
                    },
                    'web_site': {
                        report_by: 'top1-web-site'
                    },
                    'attack': {
                        report_by: 'total-attack'
                    }
                },
                defaultParams: {
                    filter: {'policymode': 'learn'},
                    no_summary: true,
                    aggregate_hosts: true
                }
            },
            config: [
                {
                    name: 'security_threat',
                    summary: [
                        {
                            'key': 'attack'
                        }
                    ]
                },
                {
                    name: 'user_productivity',
                    summary: [
                        {
                            'key': 'app_category'
                        },
                        {
                            'key': 'web_category'
                        },
                        {
                            'key': 'web_site'
                        }
                    ]
                },
                {
                    name: 'network_utilization',
                    summary: [
                        {
                            'key': 'top_host_by_bandwidth'
                        },
                        {
                            'key': 'top_host_by_session'
                        }
                    ]
                }
            ]
        }
    };

    LearningReport.prototype.getData = inject.mark(
    function(fortiviewSearchState, $http, $q) {
        return function getData(configType, type) {
            var params = this.config[configType].request.params;
            var defaultParams = this.config[configType].request.defaultParams;
            if (!(type in params)) {
                return $q.when([]);
            }

            var timePeriod = fortiviewSearchState.get_timeframe_as_range();

            return $http.get('/api/v2/monitor/fortiview/statistics/', {
                'params': angular.merge({}, defaultParams, params[type], timePeriod)
            }).then(function(data) {
                var results = data.data.results;
                return results.details || [];
            });
        };
    });

    LearningReport.prototype.getAvailableTypes = function(configType) {
        return Object.keys(this.config[configType].request.params);
    };

    LearningReport.prototype.getExcutiveSummaryData = function getExcutiveSummaryData(type) {
        var valueKeys = {
            'top_host_by_bandwidth': 'srcip',
            'top_host_by_session': 'srcip',
            'app_category': 'appcat',
            'web_category': 'catdesc',
            'web_site': 'hostname',
            'attack': 'total_count'
        };

        return this.getData('summary', type).then(function(entries) {
            var result = {};
            var valueKey = valueKeys[type];
            if (entries.length) {
                var topEntry = entries[0];
                result[type] = topEntry[valueKey];
            } else {
                result[type] = null;
            }
            return result;
        });
    };

    LearningReport.prototype.getTopChartData = function getTopChart(type) {
        var MAX_ENTRIES = 10;
        var genChartPoint = function(entry) {
            return {
                'label': entry.appcat || entry.appname,
                'data': entry.bytes
            };
        };
        return this.getData('chart', type).then(function(entries) {
            return entries.map(genChartPoint).slice(0, MAX_ENTRIES);
        });
    };

    LearningReport.prototype.getTopListData =
    inject.mark(function(fortiviewUtil, $q) {
        return function getTopListData(type) {
            var reqs = [
                fortiviewUtil.get_applications().$promise,
                this.getData('list', type)
            ];
            return $q.all(reqs).then(function(data) {
                return data[1];
            });
        };
    });

    LearningReport.prototype.getTopListConfig = function() {
        return this.config.list.config;
    };

    LearningReport.prototype.getExcutiveSummaryConfig = function() {
        return this.config.summary.config;
    };

    LearningReport.prototype.getLearningPolicies = inject.mark(function($http) {
        return function getLearningPolicies() {
            return $http.get('/api/v2/cmdb/firewall/policy/', {params: {
                format:
                'policyid|name|srcaddr|dstaddr|service|status|srcintf|dstintf|devices|users|groups',
                key: 'learning-mode',
                pattern: 'enable',
                with_meta: 1,
                datasource: 1
            }}).then(function(data) {
                var results = data.data.results || [];
                var getEnabled = function(entry) {
                    return entry.status === 'enable';
                };
                return results.filter(getEnabled);
            });
        };
    });


    LearningReport.prototype.DISPLAY_MODE_KEY = 'learning_report_display_mode';

    LearningReport.prototype.displayMode = inject.mark(function(persistentStorage) {
        return function displayMode(type) {
            if (arguments.length === 1) {
                this._displayMode = type;
                persistentStorage.put(this.DISPLAY_MODE_KEY, type);
            } else {
                return this._displayMode;
            }
        };
    });

    return function(providers, loader) {
        providers.$provide.service('fortiviewLearningReportData', LearningReport);
        return loader.initModules([fortiviewUtil]);
    };
});
