/* globals define */
/* jshint maxparams: 8 */
define([
    'angular', 'jquery', 'module', 'fweb', 'ng/services/injector', 'fweb.util/datetime',
    'fweb.util/formatters', 'fweb.util/objects', 'fweb.util/dom'
], function(angular, $, module, fweb, inject, fDatetime, fFormatters, fObjects, fDom) {
    'use strict';

    var ATTACK_DETAIL_SECTIONS_CONFIG = '/ng/log/view/detail-sections.json';
    var TRAFFIC_DETAIL_SECTIONS_CONFIG = '/ng/log/view/detail-sections-traffic.json';
    var FTVW_MSG_LOG_SUB_TRAFFIC = 2;

    function fLogDetails(loader) {
        return {
            retrict: 'E',
            scope: {
                columns: '=', // <
                selectedTab: '=',
                hideClick: '&',
                entries: '=',// <
                deviceId: '=',// <
                serialNo: '<',
                logType: '=', // <
                logFormatters: '=', // <
                /**
                 * Published so that qlist can react to animation close events. $state is a local
                 * which is a unique object representing the current state. (for watching)
                 * @type {ngExpression($state)}
                 */
                animationEnd: '&'
            },
            bindToController: true,
            controllerAs: 'logDetails',
            controller: LogDetails,
            templateUrl: loader.base_path('details.html', module)
        };
    }

    function LogDetails($scope, $element, $resource, injector) {
        injector.injectMarked(this, {$scope: $scope, $element: $element});
        // separate directive?
        this.setupAnimationEnd();
        // get section json according to log type
        this._sectionConfig = $resource((this.logType && this.logType === FTVW_MSG_LOG_SUB_TRAFFIC) ? TRAFFIC_DETAIL_SECTIONS_CONFIG : ATTACK_DETAIL_SECTIONS_CONFIG).get();
        this.updateData = this.updateData.bind(this);
        this._sectionConfig.$promise.then(function() {
            $scope.$watchCollection('logDetails.entries', this.updateData);
            $scope.$watch('logDetails.columns', this.updateData);
            $scope.$watch(function() {
                    return Section.expandedState(this.sections);
                }.bind(this), this.saveExpandedSections, true);
        }.bind(this));

        // update scrollTop so archive dump tooltips can be repositioned. Useful directive?
        var $content;
        $scope.$watch('logDetails.selectedTab', function() {
            if ($content) {
                $content.off('scroll.log-details');
            }
            // have to wait until the dom has been updated
            $scope.$evalAsync(function() {
                $content = $element.find('.content');
                $content.on('scroll.log-details', function(event) {
                    this.scrollTop = event.target.scrollTop;
                }.bind(this));
                this.scrollTop = $content.scrollTop();
            }.bind(this));
        }.bind(this));
    }

    LogDetails.prototype = {
        /**
         * qlist entries that are currently selected
         * @type {Object[]}
         */
        entries: null,
        /**
         * Data extracted from selected qlist entry and columns
         * @type {Section[]}
         */
        sections: null,
        /**
         * Object containing sections as keys and arrays of details (log columns) in each section
         * @type {Resource}
         */
        _sectionConfig: null,
        /**
         * Extra threat details retrieved using the utmid in the treat column array.
         * @type {Resource[]}
         */
        threats: null,
        /**
         * Total number of threat types
         * @type {Number}
         */
        threatTypeCount: 0,
        /**
         * Get the sectionConfig for the current log type. The matching property inside the
         * `$logType` object is copied to the main config object, and `$logType` is deleted.
         * @return {Object} Object with sections as keys and arrays of field names as values.
         */
        getSectionConfig: function() {
            var result = {},
                //the "app" section has some columns that should merge with the
                //$logType.app section.
                topLevelCols = this._sectionConfig[this.logType] || [];
            for (var sectionKey in this._sectionConfig) {
                var section = this._sectionConfig[sectionKey];
                if (sectionKey[0] !== '$' && Array.isArray(section)) {
                    result[sectionKey] = section;
                }
            }
            var logTypeSection = this._sectionConfig.$logType;
            if (logTypeSection && this.logType in logTypeSection) {

                var _isNot = function(value) {
                    return function(other) {
                        return value !== other;
                    };
                };

                // Remove duplicate keys from existing sections if they exist.
                logTypeSection[this.logType].forEach(function(removeThis) {
                    for (var sectionKey in result) {
                        if (Array.isArray(result[sectionKey])) {
                            result[sectionKey] = result[sectionKey].filter(
                                _isNot(removeThis)
                            );
                        }
                    }
                });
                result[this.logType] = topLevelCols.concat(logTypeSection[this.logType]);
            }

            // Manual overrides for crazy requirements.
            if (this.logType === 'endpoint_events') {
                result.source = result.source.concat(result.destination);
                delete result.destination;
            }
            return result;
        },
        /**
         * Extract sections and data from the selected entry, and store them in #sections
         */
        updateData: function() {
            if (!this.entries || !this.columns) {
                return;
            }
            if (this.entries && this.entries.length) {
                this.entry = this.entries[0];
                this._makeSections();
                this._makeTabs();
            } else {
                this.sections = [];
            }
            if (this.tabs && this.tabs.indexOf(this.selectedTab) === -1) {
                this.selectedTab = this.tabs[0];
            }
        },
        _makeSections: function() {
            var sectionConfig = this.getSectionConfig();
            this.sections = Object.keys(sectionConfig)
                .filter(notDollar)
                .map(Section.extract(sectionConfig, this, this.logFormatters))
                .filter(notEmpty);
            this.expandSections(this.sections);

            function notDollar(key) { return key.length && key[0] !== '$' }

            function notEmpty(section) {
                return section.data && section.data.length > 0;
            }
        },
        _getThreatCountTabName: function(threatCountName) {
            return threatCountName.replace(/^count/, '');
        },
        _getThreatDetails: inject.mark(
        function($q, logViewData, logViewColumns, lang) {
            return function _getThreatDetails() {
                var result = {};
                if (this.entry && this.entry.threat) {
                    result = this.entry.threat.reduce(function(result, threat) {
                        var filter = [{'logic': {}, 'id': 'utmref',
                                       'value': [this.entry.utmref]}],
                            path = this._getThreatCountTabName(threat.key),
                            dataParams = {filter: filter, logType: path};
                        if (this.serialNo) {
                            dataParams.serial_no = this.serialNo;
                        }
                        var res = result[path] = {
                            data: logViewData.get(dataParams),
                            columns: logViewColumns.get({logType: path}),
                            sections: null,
                            expanded: true
                        };
                        $q.all({data: res.data.$promise, columns: res.columns.$promise})
                            .then(function() {
                                res.title = 'Log::threat.' + path;
                                res.columns = fObjects.makeSet(res.columns.map(pluckSelector))
                                    .difference(this.columns.map(pluckSelector));
                                res.sections = res.data.map(threatSection.bind(this));
                                res.buttons = [
                                    new SectionButton(
                                        'fa-list', 'view_log',
                                        null,
                                        logViewData.navigate.bind(null, filter, path)
                                    ),
                                    new SectionButton(
                                        'fa-download', 'downlog',
                                        null,
                                        logViewData.download.bind(null, filter, path)
                                    )
                                ];
                            }.bind(this));
                        return result;
                        function threatSection(entry) {
                            /* jshint validthis: true *//* this is ldCopy */
                            this.entry = entry;
                            var title = new lang.Translated(entry['#']),
                                section = new Section(
                                    title, res.columns, this, this.logFormatters, entry
                                );
                            section.collapsable = false;
                            return !section.empty ? section : null;
                        }

                        function pluckSelector(c) { return c.selector }

                    }.bind(this), {});
                }
                return result;
            };
        }),
        _getArchive: inject.mark(function($q, logArchiveData, lang) {
            var resources = logArchiveData;
            return function _getArchive() {
                var result = null;
                if (this.entry._is_archived) {
                    var logName = this.logType.replace(/^.*_(traffic)$/, '$1');
                    if (this.entry.is_email) {
                        // TODO: not supported atm.
                        return false;
                    }
                    //TODO: add the rest of the resource types
                    if (!(logName in resources)) { return false }
                    var resource = resources[logName],
                        params = angular.extend({
                            device_id: this.deviceId
                        }, this.entry);

                    result = resource.get(params);
                    result.$promise = result.$promise.then(function(res) {
                        var arrayData = Array.isArray(res.data);
                        if (res.error) {
                            result.error = res.err_no;
                            result.sections = [];
                        } else {
                            var sections = arrayData ? res.data : [res.data];
                            result.name = res.data.name ||
                                lang('Log::Archive::title.' + logName);
                            result.sections = sections.map(function(sectionData) {
                                var cols = Object.keys(sectionData)
                                        .filter(exclude((res.hide_columns || []).concat(['name']))),
                                    lf = this.logFormatters;
                                if ('rel_time' in sectionData) {
                                    cols = ['date', 'time']
                                        .concat(cols.filter(exclude(['rel_time'])));
                                }
                                var result = new Section('', cols, this, lf, sectionData);
                                result.dump = sectionData.dump;
                                result.name = sectionData.name;
                                return result;
                            }.bind(this));

                        }
                        var downloadData = arrayData ? this.entry : res.data,
                            downloadTitle = res.data && res.data.name ||
                                params.filename ?
                                    lang('Download {filename}', [params.filename]) : 'download';
                        result.buttons = [
                            downloadData && new SectionButton(
                                'fa-download', downloadTitle, null,
                                resource.download &&
                                    resource.download.bind(null, downloadData, params),
                                'Archived File'
                            ),
                            resource.downloadPacketCapture && new SectionButton(
                                'ftnt-packet-download', lang('Download packet capture'), null,
                                resource.downloadPacketCapture.bind(null, downloadData, params),
                                'packet_capture'
                            )
                        ].filter(function(b) { return b && b.url && b.url() });
                    }.bind(this));
                }
                return result;
                function exclude(arr) { return function(name) { return arr.indexOf(name) === -1 }}
            };
        }),
        _makeTabs: function() {
            this.tabs = [];
            this.threats = this._getThreatDetails();
            this.archive = this._getArchive();
            this.threatTypeCount = Object.keys(this.threats).length;
            if (this.threatTypeCount) {
                this.tabs.push('threats');
            }
            if (this.archive) {
                this.tabs.push('archive');
            }
            this.tabs.unshift('details');

            this.selectedTab = this.selectedTab || 'details';
        },
        setupAnimationEnd: inject.mark(function($animate, $element, $scope) {
            return function setupAnimationEnd() {
                var destroyed;
                $animate.on('leave', $element, animationEnd.bind(this, 'leave'));
                $animate.on('enter', $element, animationEnd.bind(this, 'enter'));
                $scope.$on('$destroy', function() { destroyed = true });

                function animationEnd(event, element, phase) {
                    /* jshint validthis: true *//* this = LogDetails */
                    if (phase === 'close') {
                        // have to copy because at this point we may already be disconnected
                        // from the parent scope!
                        $scope.$apply(function() {
                            var state = {event: event, phase: phase};
                            this.animationEnd({$state: state});
                        }.bind(this));
                        // clean up after ourselves!
                        if (destroyed) {
                            $animate.off('leave', $element, animationEnd);
                            $animate.off('enter', $element, animationEnd);
                        }
                    }
                }
            };
        }),
        persistKey: function(path) {
            return 'LogView.' + this.logType + '.details.' + path;
        },
        expandSections: inject.mark(function(persistentStorage) {
            return function expandSections() {
                if (this.sections) {
                    var state = persistentStorage.get(this.persistKey('expandedSections')) || {};
                    Section.expandedState(this.sections, state);
                }
            };
        }),
        saveExpandedSections: inject
            .mark(function saveExpandedSections(persistentStorage) {
                return function() {
                    var pk = this.persistKey('expandedSections');
                    var state = persistentStorage.get(pk) || {};
                    state = angular.merge(state, Section.expandedState(this.sections));
                    persistentStorage.put(pk, state);
                };
            })
    };

    function SectionButton(icon, title, action, url, text) {
        this.icon = icon;
        this.title = title;
        this.action = action;
        this.url = url;
        this.text = text;
    }

    SectionButton.prototype = {
        /**
         * Icon class for the button
         * @type {String}
         */
        icon: null,
        /**
         * Language key to assign to the button. Will be passed through the lang filter.
         * @type {String}
         */
        title: null,
        /**
         * Action to take when the button is clicked
         * @type {Function(event)}
         */
        action: null,
        /**
         * Url to go to when clicked.
         * @type {String}
         */
        url: null,
        /**
         * Text to place in the button
         * @optional
         * @type {String}
         */
        text: ''
    };

    function Section(name, config, logDetails, logFormatters, entry) {
        this.title = name;
        this.data = config
            .map(Datum.extract(logDetails, logFormatters, entry))
            .filter(function(d) { return !!d });
        this.empty = !(this.sections && this.sections.length ||
            this.data && this.data.length);
    }

    /**
     * Extract data from a log entry based on the sectionConfig passed in.
     * Returns a function for use with [].map() by binding the first 3 arguments.
     * @static
     * @param  {Object} sectionConfig Section config object from detail-sections.json
     * @param  {LogDetails[]} logDetails LogDetails controller for columns and entries.
     * @return {Function(String):Section} Creates a single for each sectionName.
     */
    Section.extract = function(sectionConfig, logDetails, logFormatters, entry) {
        return function(sectionName) {
            return new Section(
                sectionName, sectionConfig[sectionName], logDetails, logFormatters, entry
            );
        };
    };

    /**
     * Get the expanded state for an array of sections.
     * @see  #expanded
     * @param  {Section[]} sections Sections to update or get the expanded state from.
     * @param  {Object} [state] Set the expanded state.
     * @return {Object} State, same as state parameter.
     */
    Section.expandedState = function(sections, state) {
        if (sections) {
            if (state) {
                sections.forEach(function(section) {
                    if (section.title in state) {
                        section.expand = state[section.title];
                    }
                });
            } else {
                return sections.reduce(function(result, section) {
                    result[section.title] = section.expand;
                    return result;
                }, {});
            }
        }
    };

    Section.prototype = {
        /**
         * Data to show for this section
         * @type {Datum}
         */
        data: null,
        /**
         * Title of the current section
         * @type {String}
         */
        title: null,
        /**
         * Whether or not this section is expanded
         * @type {Boolean}
         */
        expand: true,
        /**
         * Set false to prevent collapse button from showing.
         * @type {Boolean}
         */
        collapsable: true,
        /**
         * Whether or not the dump has been expanded
         * @type {Boolean}
         */
        expandDump: true,
        /**
         * True if there are no data or sections inside.
         * @type {Boolean}
         */
        empty: true,
        /**
         * Data dump associated with a section
         * @type {String}
         */
        dump: null,
        dumpTipOptions: {
            style: {classes: 'dump-tip'},
            position: {
                my: 'bottom right',
                at: 'bottom right',
                //target: f-tip-target (automatic),
                adjust: { x: 9, y: 0 },
                effect: false
            },
            show: {
                //target: f-tip-target (automatic)
            },
            hide: {
                target: false,
                inactive: 5000
            }
        }
    };

    /**
     * Create a single datum (piece of data) for display.
     * @param {String} name Name of this piece of data. Should be a language table entry.
     * @param {String|Number} value Value to display.
     * @param {String|null} [icon] f-icon className to use for the icon.
     */
    function Datum(name, value, icon, valueIcon) {
        this.name = name;
        if (typeof value === 'function') {
            this.getValue = value;
        } else {
            this.value = value;
        }
        this.valueIcon = valueIcon;
        this.icon = icon;
    }

    var byteFormat = {
        format: fFormatters.metric_bytes
    };
    // sometimes we don't want the default formatter because it's dumb
    var noFormat = { format: null };
    // add device icon
    var deviceIconFormat = { icon: 'device' };

    var secondsFormat = {
        format: function(value) { return value + 's' }
    };
    /**
     * Object mapping detail-sections.json detail names to log viewer columns.
     * Some of the details are the same column formatted differently, otherwise
     * the detail-sections.json names should map directly to column names.
     * @type {Object}
     */
    Datum.columnDetailMap = {
        /*date: {
            selector: 'rel_time',
            lang: 'date',
            format: function(value) {
                return fDatetime.formatDate(fDatetime.localSecondsToDate(value));
            }
        },
        time: {
            selector: 'rel_time',
            lang: 'time',
            format: function(value) {
                return fDatetime.formatTime(fDatetime.localSecondsToDate(value));
            }
        },
        srcip: noFormat,
        srcmac: noFormat,
        dstip: noFormat,
        countapp: noFormat,
        countav: noFormat,
        countdlp: noFormat,
        countemail: noFormat,
        countips: noFormat,
        countwaf: noFormat,
        countweb: noFormat,
        lanin: byteFormat,
        lanout: byteFormat,
        wanin: byteFormat,
        wanout: byteFormat,
        rcvdbyte: byteFormat,
        sentbyte: byteFormat,
        shaperdroprecvdbyte: byteFormat,
        shaperdropsentbyte: byteFormat,
        shaperipdropbyte: byteFormat,
        osname: deviceIconFormat,
        trandisp: {
            prefix: 'Log::Details::column::trandisp.',
            skipValues: ['noop']
        },
        duration: secondsFormat*/
    };

    /**
     * Don't show these columns (key) if the other column (value) is the same
     * @type {Object}
     */
    Datum.dontDuplicate = {
        srcmac: 'mastersrcmac',
        transip: 'srcip',
        transport: 'srcport',
        tranip: 'dstip',
        tranport: 'dstport'
    };

    /**
     * Generate a function which extracts a single datum from current logDetail columns/entries.
     * @param  {LogDetails} logDetails Log Details controller instance
     * @param {LogFormatters} logFormatters Log Formatters service.
     * @param {Object} [entry] Log entry to extract datum from.
     * @return {function} Function that creates a datum object for the passed in column name.
     */
    Datum.extract = function(logDetails, logFormatters, entry) {
        var qlFormatters = $.qlist.format_fn;
        return function(name) {
            var detail = Datum.columnDetailMap[name] || {selector: name};
            // TODO: ES6 Array.prototype.find
            var column = logDetails.columns.filter(hasSelector(detail))[0],
                valueIcon = null;
            entry = entry || logDetails.entry;
            if (!entry) {
                return null;
            }
            if (!column && name in entry) {
                column = {selector: name};
            }
            if (column && !isDuplicate(column, entry) && !skipValue(column, detail, entry)) {
                var langKey = detail.lang ||
                        ('lang_key' in column ? column.lang_key : column.selector),
                    value = entry[column.selector],
                    valueAndIcon = null;
                if (detail.prefix) {
                    value = logFormatters.langPrefixFormat(detail.prefix, value);
                }
                if (detail.format !== null) {
                    if (detail.format) {
                        if (angular.isFunction(detail.format)) {
                            value = detail.format(value);
                        } else if (detail.format in logFormatters.formatters) {
                            valueAndIcon = jqFormat(detail.format, name, entry);
                        } else if (detail.format in qlFormatters) {
                            value = qlFormat(detail, name, entry);
                        }
                    } else if (column && column.selector in logFormatters.formatters) {
                        valueAndIcon = jqFormat(column.selector, name, entry);
                    } else if (column && column.selector in qlFormatters) {
                        value = qlFormat(column, name, entry);
                    } else {
                        value = fDom.escapeHTML(value);
                    }
                }
                if (valueAndIcon) {
                    value = valueAndIcon.value;
                    valueIcon = valueAndIcon.icon;
                }
                if (detail.icon) {
                    // can't sanitize f-icon! (arg)
                    valueIcon = logFormatters.getIconClass(detail.icon, entry);
                }
                return value != null && value !== '' ?
                    new Datum(langKey, value, column.icon, valueIcon) : null;
            } else {
                return null;
            }

            function skipValue(column, detail, entry) {
                var value = entry[column.selector];
                return detail && detail.skipValues && detail.skipValues.indexOf(value) !== -1;
            }

            function isDuplicate(column, entry) {
                var dd = Datum.dontDuplicate,
                    sel = column.selector;
                return (sel in dd) && entry[sel] === entry[dd[sel]];
            }
        };

        /**
         * Use a qlist formatter to format the value.
         * @param  {column|detail} columnOrDetail Column or detail object from Datum.extract().
         *   Duck typed using `format` and `selector` keys and the presence thereof.
         * @param  {String} name Column name (may be used for the selector).
         * @param  {Object} entry Qlist data row.
         * @return {Function():String} It's expected that the formatter tries some fancy
         *    after-the-fact formatting. An accessor function is returned that can monitor the
         *    cell provided.
         *    TODO: eliminate this pattern as it's always been a terrible idea.
         *    NOTE: this optimistically relies on the 'after-the-fact' formatting to occur before
         *    the end of the current digest since the Datum object has no injected services so
         *    can't cause a $digest cycle itself (with $scope.$evalAsync or $timeout etc)
         */
        function qlFormat(columnOrDetail, name, entry) {
            var format = columnOrDetail.format || columnOrDetail.selector,
                column = columnOrDetail.selector ? columnOrDetail : {selector: name};
            var $td = $('<td>'),
                value = qlFormatters[format]($td, column, entry);
            return function() {
                return $td.html() || value;
            };
        }

        function jqFormat(formatterName, name, entry) {
            // try to grab a log formatter for the column anyways
            // unfortunately all the formatters expect jquery.
            // pass 'true' as the td to generate html
            // extract the f-icon if present since angularjs will sanitize it
            // (f-icon may be fixed via https://github.com/angular/angular.js/pull/12524 and ng1.5)
            var result = {},
                formatter = logFormatters.formatters[formatterName],
                html = formatter(name, entry, true),
                $value = $(html ? ('<div>' + html + '</div>') :  null),
                $valueIcon = $value.find('f-icon');
            if ($valueIcon.length) {
                $valueIcon.remove();
                result.icon = $valueIcon[0].className;
            }
            result.value = $value.html();
            return result;
        }
        function hasSelector(detail) {
            return function(column) { return column.selector === detail.selector };
        }
    };

    Datum.prototype = {
        /**
         * Name of this piece of data. Should be a language table entry.
         * @type {String}
         */
        name: null,
        /**
         * Value to display.
         * @type {String|Number}
         */
        value: null,
        /**
         * f-icon className to use for the icon or null.
         * @type {String|null}
         */
        icon: null,
        /**
         * Value getter, in case the value is a function.
         * @function
         * @return {String} [description]
         */
        getValue: null
    };


    return function(providers, loader) {
        providers.$compile.directive({
            fLogDetails: fLogDetails
        });
        return loader.initModules([
            'f-log-detail-section', 'data', '/ng/directives/virtual_scroll'
        ], module);
    };
});
