/* globals define */
/* jshint maxparams:9 */
define([
        'jquery', 'fweb', 'fgd_common', 'byod_common',
        'fweb.util/dom', 'fweb.util/formatters', 'fweb.util/templates', 'fweb.util/datetime'
], function($, fweb, fgd_common, byod_common, fDom, fFormatters, fTemplates, fDatetime) {
    'use strict';

    function logFormattersFactory(lang) {
        function LogFormatters(logType) {
            this.logType = logType;
            this.formatters = getFormatters(this, lang, logType);
            // these functions get passed around, and should be bound methods instead
            ['formatFnAuto', 'genColumnContent'].forEach(function(fn) {
                this[fn] = this[fn].bind(this);
            }.bind(this));
        }

        LogFormatters.smartColumnMap = {
            'app': ['dstport', 'proto', 'appid'],
            'dstip': ['dstport', 'dstcountry', 'dstip_hostname'],
            'srcip': ['srcmac', 'byod_name', 'byod_device', 'user']
        };

        LogFormatters.columns = {
            'forward_traffic': {
                'utmaction': {
                    'allow': 'fa-accepted',
                    'block': 'fa-blocked',
                    'reset': 'fa-reset',
                    'traffic-shape': 'action_traffic-shape'
                },
                'action': {
                    'deny':    'fa-blocked',
                    'accept':  'fa-accepted',
                    'start':   'fa-accepted',
                    'dns':     'fa-blocked',
                    'ip-conn': 'fa-blocked',
                    'close':   'fa-accepted',
                    'timeout': 'fa-accepted'
                },
                'action_outcome': {
                    'deny':    'fa-blocked',
                    'accept':  'fa-accepted',
                    'start':   'fa-accepted',
                    'dns':     'fa-blocked',
                    'ip-conn': 'fa-blocked',
                    'close':   'fa-accepted',
                    'timeout': 'fa-accepted'
                }
            },
            'gtp': {
                // reference: http://manual.snort.org/node185.html
                'msg-type': {
                    // version 0
                    '0': {
                        1: 'echo_request',
                        2: 'echo_response',
                        3: 'version_not_supported',
                        16: 'create_pdp_context_request',
                        17: 'create_pdp_context_response',
                        18: 'update_pdp_context_request',
                        19: 'update_pdp_context_response',
                        20: 'delete_pdp_context_request',
                        21: 'delete_pdp_context_response',
                        255: 'pdu'
                    },
                    // version 1
                    '1': {
                        // no override, same as version 0
                    },
                    // version 2
                    '2': {
                        32: 'create_session_request',
                        33: 'create_session_response',
                        34: 'modify_bearer_request',
                        35: 'modify_bearer_response',
                        36: 'delete_session_request',
                        37: 'delete_session_response'
                    }
                },
                'status': {
                    'prohibited': 'fa-blocked',
                    'state-invalid': 'fa-blocked'
                }
            }
        };
        LogFormatters.columns.fortiview_traffic = LogFormatters.columns.forward_traffic;

        LogFormatters.prototype = {
            /**
             * [icon description]
             * @param  {String} iconType Get the icon of this type.
             * @param  {Object} row Log row that contains the data needed to get the icon.
             * @return {String} Icon class name (fa-* or ftnt-*)
             */
            getIconClass: function(iconType, row) {
                var iconTypes = {
                    device: function() {
                        return byod_common.gen_fw_device_class(row.byod_device);
                    }
                };
                var fn = iconTypes[iconType];
                return fn ? fn(row) : null;
            },
            /**
             * Generate a map of hover functions. Each takes an array of cells and adds
             *     tooltip or other hover functionality to each cell.
             * @param  {Boolean} [resolveApps] True to resolve app details (used by fgd_common)
             */
            hoverFns: function(resolveApps) {
                return {
                    srcip: device,
                    srcmac: device,
                    user: device,
                    threat: fgd,
                    attack: fgd,
                    countav: fgd,
                    app: fgd,
                    id: fgd
                };

                function device(cells) {
                    byod_common.setup_device_tooltips($(cells).find('.byod_details'));
                }

                function fgd(cells) {
                    fgd_common.setup_tooltips($(cells).find('.tooltip'), 'fromclass', resolveApps);
                }
            },

            autoAll: function(formatFns) {

                for (var selector in this.formatters) {
                    formatFns[selector] = this.formatFnAuto;
                }
            },
            getFormatter: function() {
                return this.formatters;
            },
            formatFnAuto: function(td, col, entry) {
                return this.formatters[col.selector](col.selector, entry, td);
            },
            formatSmartCol: function(row, obj, tag) {
                var value = '';
                tag = tag || 'f-icon';
                if (obj) {
                    var cls = obj.cls;
                    var key = obj.key;
                    var suffix = obj.suffix;
                    if (cls) {
                        value += '<span><' + tag + ' class="' + cls + '"></' + tag + '><span>';
                    }
                    if (key in this.formatters) {
                        value += this.formatters[key](key, row);
                    } else {
                        value += fDom.escapeHTML(row[key]);
                    }
                    if (suffix) {
                        value += ' ' + lang(suffix);
                    }
                    if (cls) {
                        value += '</span></span>';
                    }
                }
                return value;
            },
            getColumnDict: function(col /*, entry*/) {
                var log_dict = LogFormatters.columns[this.logType];
                return log_dict && log_dict[col];
            },
            genColumnContent: function(callback) {
                return function(name, row, td) {
                    var dict;
                    if (td && (dict = this.getColumnDict(name, row))) {
                        return callback(name, row, dict);
                    }
                    return row[name];
                }.bind(this);
            },
            langPrefixFormat: function(prefix, value) {
                return lang(lang.getPrefixedKey(prefix, value));
            }
        };

        /**
         * Formatting callback for log entries
         * @callback formattingCallback
         * @param  {String} name The selector/key of the column
         * @param  {Object} row The source data of the row
         * @param  {Boolean} [content] If result will be used in displaying the qlist
         * @return {String|jQuery}
         */

        /**
         * Map of nonstandard format functions for displaying values in the log.
         * @type {Object.<string, formattingCallback>}
         */
        function getFormatters(logFormatters, lang) {
            var formatter =  {
                '_is_archived': function(name, row, content) {
                    if (!content) {
                        return null;
                    }
                    var value = row[name];
                    if (value) {
                        return fDom.renderTemplate(
                            '<f-icon class="fa-paperclip" title="{{ title }}"></f-icon>',
                            { title: lang('archive') }
                        );
                    }
                    return '';
                },
                'pri': function(name, row, content) {
                    var value = row[name];
                    var sev_class = 'log_severity_' + value;
                    var sev = '';
                    if (!content) {
                        sev = value + ' ';
                    }
                    sev += '<span class="log_severity_level ';
                    sev += sev_class + '" title="' + value + '"></span>';
                    return sev;
                },
                'severity': function(name, row, content) {
                    var value = row[name];
                    var sev_class = 'sig_severity_' + value;
                    var sev = '';
                    if (!content) {
                        sev = value + ' ';
                    }
                    sev += '<span class="sig_severity_level ';
                    sev += sev_class + '" title="' + value + '"></span>';
                    return sev;
                },
                'apprisk': function(name, row /*, content*/) {
                    var value = row[name];
                    return fTemplates.risk_format()(value);
                },
                'app': function(name, row) {
                    var value = row[name];
                    var id = row.appid || 0;
                    var app_data = '';
                    var address = row.dstip;
                    var dport = row.dstport;
                    var proto = row.proto;
                    var service = row.service;

                    if (!value) {
                        if (service) {
                            value = service.toUpperCase();
                        } else {
                            value = lang('unknown');
                        }
                    }

                    // The destination IP can be pre-resolved by the log extension to
                    // show reverse DNS lookups. If the lookup is successful, the
                    // 'dstip' field will look like:-
                    //
                    //     "184.150.182.158 (apis.google.com)"
                    //
                    // In this case, we want only the IP portion for the FortiFlow query.
                    var address_host_idx = address ? address.indexOf(' ') : -1;
                    if (address_host_idx !== -1) {
                        address = address.substring(0, address_host_idx);
                    }

                    // Add extra data attributes to support FortiFlow queries for
                    // unknown applications.
                    if (address && dport && proto) {
                        app_data = 'data-address="' + address + '" ' +
                                   'data-dport="' + dport + '" ' +
                                   'data-protocol="' + proto + '" ';
                    }

                    value = '<span class="tooltip id_' + id + '" ' + app_data + '>' +
                            '<span class="app_icon app' + id + '"></span>' +
                            '<label class="app_label">' + value + '</label></span>';
                    return value;
                },
                'appcat': function(name, row) {
                    var value = row[name];
                    if (value) {
                        return fgd_common.gen_app_cat_html(value);
                    }
                    return '';
                },
                'attack': function(name, row) {
                    var value = row[name] || lang('unknown');
                    var id = row.attackid || 0;

                    value = '<span class="tooltip id_' + id + '">' +
                            '<label class="app_label">' + value + '</label></span>';
                    return value;
                },
                'id': function(name, row) {
                    var value = row[name];
                    if (value && row.subtype === 'waf' && row.reason === 'Signature') {
                        return '<span class="tooltip waf_sig dont-use-id" data-sig-id="' + value +
                               '" data-sig-title="' + lang('signature') +
                               '"><label class="waf_sig_label" title="' + value + '">' +
                               value + '</label></span>';
                    } else {
                        return value || '';
                    }
                },
                _refLink: function(value, href) {
                    return '<a target="_blank" href="' + (href || value) + '">' + value + '</a>';
                },
                'ref': function(name, row) {
                    var value = row[name];
                    if (value) {
                        return formatter._refLink(value);
                    } else {
                        return '';
                    }
                },
                'vulnref': function(name, row) {
                    var value = row[name];
                    return formatter._refLink(value);
                },
                'vuln': function(name, row) {
                    var value = row[name];
                    var ref = row.vulnref;
                    if (ref) {
                        return formatter._refLink(value, ref);
                    }
                    return value;
                },
                'virus': function(name, row) {
                    var value = row[name];
                    var ref = row.ref;
                    if (ref) {
                        return formatter._refLink(value, ref);
                    }
                    return value;
                },
                'rel_time': function(name, row, content) {
                    var value = row[name];
                    var dt = fDatetime.localSecondsToDate(value);
                    // TODO: see why the FortiGate is using local time instead of UTC
                    if (!dt) {
                        return '';
                    }
                    var result = fDatetime.formatDateDiff(dt);
                    if (!content) {
                        result += ' (' + value + ')';
                    }
                    return '<span title="' + dt.toLocaleString() + '">' + result + '</span>';
                },
                'timestamp': function(name, row) {
                    return fweb.util.datetime.localSecondsToDate(row[name]).toLocaleString();
                },
                'srcip': function(name, row) {
                    var values = [], value = row[name];
                    var key, keys = ['user', 'unauthuser'];
                    var mac = row.srcmac;

                    while ((key = keys.shift())) {
                        if (row[key]) { break; }
                    }
                    var id = row.byod_name;
                    if (key) {
                        values.push(formatter.user(key, row));
                    }
                    if (mac || id) {
                        // Mac/device name and icon
                        values.push(byod_common.gen_fw_device_icon(mac, row.byod_device, id));
                    } else if (value !== undefined) {
                        if (values.length) {
                            // source plus ip address
                            values.push('(' + value + ')');
                        } else {
                            // standalone ip address
                            values.push(value);
                        }
                    }
                    return values.join('&nbsp;&nbsp;&nbsp;');
                },
                'source_plain': function(name, row) {
                    row[name] = row.srcip;
                    return row.srcip || row.srcmac;
                },
                'dstip': function(name, row) {
                    var value = row[name];
                    var whois_anchor = fweb.util.formatters.create_whois_lookup_anchor(value);
                    return formatter.dstip_no_whois(name, row) + whois_anchor;
                },
                'dstip_no_whois': function(name, row) {
                    var value = row[name];
                    var country = row.dstcountry;
                    var code = row.dstcountry_code;
                    var hostname_attr = '_' + name + '_hostname';
                    var hostname = row[hostname_attr];
                    if (code) {
                        value = '<span class="country_flag country_' + code + '" title="' +
                            country + '"></span> ' + value +
                            (hostname ? ' (' + hostname + ')' : '');
                    }
                    return value || '';
                },
                'srcmac': function(name, row) {
                    var mac = row[name];
                    var device = row.byod_device;
                    name = row.byod_name;
                    if (mac || name) {
                        return byod_common.gen_fw_device_icon(mac, device, name);
                    }
                },
                'url': function(name, row) {
                    var value = row[name] || '';
                    if ('hostname' in row) {
                        var host = row.hostname;
                        if (host && host.toLowerCase() !== 'n/a') {
                            value = host + value;
                        }
                    }
                    return fFormatters.truncate_comment(value, 64);
                },
                'proto': function(name, row) {
                    if ('protocol' in row) {
                        return row.protocol;
                    }
                    return row[name];
                },
                'protocol': function(name, row) {
                    if ('proto' in row) {
                        return row.proto;
                    }
                    return row[name];
                },
                'av_details': function(name, row) {
                    var service = row.service;
                    if (service) {
                        if (service.indexOf('smtp') !== -1) {
                            return 'to: ' + fDom.escapeHTML(row.to);
                        }
                        if (service.indexOf('imap') !== -1 || service.indexOf('pop3') !== -1) {
                            return 'from: ' + fDom.escapeHTML(row.from);
                        }
                        if (service.indexOf('http') !== -1) {
                            return 'url: ' + fDom.escapeHTML(row.url);
                        }
                    }
                    return 'host: ' + row.dstip;
                },
                'sent_rcvd': function(name, row) {
                    var value = '';
                    if (row.sentbyte && row.rcvdbyte) {
                        value = fFormatters.metric_bytes(row.sentbyte) + ' / ' +
                            fFormatters.metric_bytes(row.rcvdbyte);
                    }
                    return value;
                },
                'size': function(name, row) {
                    var value = Number(row[name]);
                    return fFormatters.metric_bytes(value);
                },
                'user': function(key, row) {
                    var value = row[key] && fDom.escapeHTML(row[key]);
                    var className = (key === 'user') ?
                        'fa-user-authenticated' : 'fa-user-unauthenticated';
                    if (value) {
                        value = byod_common.gen_fw_icon(className, value);
                    }
                    return value;
                },
                'msg-type': logFormatters.genColumnContent(function(name, row, dict) {
                    var val = row[name], ver = row.version;
                    if (!dict[ver]) { return val; }
                    return dict[ver][val] || dict[0][val] || val;
                }),
                status: logFormatters.genColumnContent(function(name, row, dict) {
                    var val = row[name], cls = dict[val];
                    return cls ? '<f-icon class="' + cls + '"></f-icon><span title="' + val + '">' +
                        lang(val) + '</span>' : val;
                }),
                utmaction:
                    logFormatters.genColumnContent(function(name, row, dict) {
                    var val = row[name], cls = dict[val],
                        key = 'utmaction_' + val;
                    return cls ? '<f-icon class="' + cls + '"></f-icon><span' +
                        ' title="' + val + '">' + lang(key) + '</span>' : val;
                }),
                utmaction_outcome:
                    logFormatters.genColumnContent(function(name, row, dict) {
                    var val = row[name], cls = dict[val],
                        key = 'utmresult_' + val;
                    return cls ? '<f-icon class="' + cls + '"></f-icon><span' +
                        ' title="' + val + '">' + lang(key) + '</span>' : val;
                }),
                action: logFormatters.genColumnContent(function(name, row, dict) {
                    // TODO: Check if need to format with icon
                    // and aggregate icon for each action
                    // since each utm log has different value,
                    // better consistant with UTM Profile
                    // format for forward log, action refers to firewall action
                    // format for forward utm logs, action refers to utm action
                    var val = row[name], display = dict[val];
                    return display ? '<span title="' + val + '">' + lang('Log::Action::' + val) +
                        '</span>' : val;
                }),
                action_outcome: logFormatters.genColumnContent(function(name, row, dict) {
                    var val = row.action,
                        cls = dict[val],
                        display;
                    // 0283885, 0283869: Show UTM reset or block over policy action.
                    if (!cls || row.utmaction === 'reset' || row.utmaction === 'block') {
                        return formatter.utmaction_outcome('utmaction', row, dict);
                    } else {
                        if (cls === 'fa-accepted' && val !== 'start') {
                            display = formatter.sent_rcvd(null, row);
                        } else {
                            display = lang('Log::Action::' + val);
                        }
                        return '<f-icon class="' + cls + '"></f-icon><span title="' + val +
                            '">' + display + '</span>';
                    }
                }),
                threat: function(name, row) {
                    var value = row[name],
                        html = '';
                    for (var i = 0, len = value.length; i < len; i++) {
                        html += logFormatters.formatSmartCol(row, value[i], 'span');
                    }
                    return html;
                },
                countav: function(name, row) {
                    return '<a href="#" class="threat_details">' + row[name] + '</a>';
                },
                app_details: function(name, row) {
                    return logFormatters.formatSmartCol(row, row[name]);
                },
                checksum: function(name, row) {
                    return (row[name] || '').toString(16);
                },
                catdesc: function(name, row) {
                    var value = fDom.escapeHTML(row[name]);
                    return value;
                },
                clouddetails: function(name, row) {
                    var value = '';

                    /* Parse application "action" from application "name" field, i.e.:-
                     *
                     *   "Dropbox_File.Download" => "File Download"
                     *
                     */
                    if (row.app && row.app.indexOf('_') !== -1) {
                        var app_segments = row.app.split('_');
                        var app_name = app_segments[app_segments.length - 1];

                        value += app_name.replace('.', ' ');
                    }

                    /* Add the filename / video name / etc */
                    if (row.filename) {
                        if (value) {
                            value += ': ';
                        }
                        value += row.filename;
                    }
                    return value;
                },
                // For FortiView Threat last level Drilldown
                // back import from Widgets.js
                // TODO: Reduce redundancy
                threatlevel: function(name, row) {
                    var level = level_idmap[row[name]];
                    if (typeof level === 'undefined') {
                        return '';
                    }
                    return ['<span class=\'legend ', level, '\'>',
                            lang(level), '</span>'].join('');
                },
                // TODO: need to reexamine and back import from Widgets.js
                //       to show the icon
                threattype: function(name, row) {
                    return row[name] == null ? row[name] : lang(row[name]);
                },
                // TODO: need to reexamine and back import from Widgets.js
                //       to show the icon
                threatname: function(name, row) {
                    return row[name] == null ? row[name] : lang(row[name]);
                },
                filesize: function(name, row) {
                    return row[name] === undefined ? '' :
                        ('<div title="' + row[name] + ' ' + lang('bytes') + '">' +
                        fFormatters.metric_bytes(row[name]) + '</div>');
                },
                cfgattr: function(name, row) {
                    return fFormatters.truncate_comment(row[name], 50);
                },
                result: function(name, row) {
                    var map = {
                        'pass': 'fa-enabled',
                        'fail': 'fa-disabled'
                    };
                    var value = row[name];
                    var cls = map[value];
                    if (!cls) {
                        return value;
                    }
                    return '<span><f-icon class="' + cls + '"></f-icon></span>';
                },

                /* FWB_CHANGE */
                'http_retcode': function(name, row) {
                    var html = '';
                    if(row[name] != '0')    html = row[name];
                    return html;
                },
                'f_http_return_code': function(name, row) {
                    var html = ' ';
                    if(row[name] != '0')    html = row[name];
                    return html;
                },
                'src': function(name, row) {
                    var ip = row[name] ? row[name]:'';
                    var flag = row['country_flag'] ? row['country_flag']:'';
                    var html = '<span class="country_flag country_'+flag+'"></span>&nbsp;<span>'+ip+'</span>';
                    return html;
                },
		'original_src': function(name, row) {
		    var ip = row[name] ? row[name]:'';
		    var flag = row['ori_country_flag'] ? row['ori_country_flag']:'';
		    var html = '<span class="country_flag country_' + flag + '"></span>&nbsp;<span>' + ip + '</span>';
		    return html;
		},
                'threat_level': function(name, row) {
                    var value = row[name] ? row[name]:'Off';
                    var level = value.toLowerCase();
                    return ["<span class='severity-label wide severity-", level, "'>", $.getInfo(level), "</span>"].join("")
                },
		'flag': function(name, row) {
		    var value = row[name] ? row[name] : "None";
		    var val = value.toLowerCase().split(' ').join('_');
		    var cls = 'log_flag log_flag_' + val;
		    var html = ['<span class="', cls, '" title="', value, '"></span>' ].join('');
		    return html;
		},
		'has_comment': function(name, row, content) {
			var value = row[name] ? row[name] : "No Comment";
			var html = '';
			if(value == 'Has Comment')
				html = '<span class="log_flag log_flag_comment" title="Has Comment"></span>';
			return html;
		}
            };
            formatter.level = formatter.pri;
            formatter.start = formatter.rel_time;
            formatter.countweb = formatter.countav;
            formatter.countapp = formatter.countav;
            formatter.countips = formatter.countav;
            formatter.countdlp = formatter.countav;
            formatter.countemail = formatter.countav;
            return formatter;
        }



        var level_idmap = ['low', 'med', 'high', 'crit'];

        //transform into a map from column selector to smart column selector
        var scKeys = Object.keys(LogFormatters.smartColumnMap);
        scKeys.forEach(function(k) {
            LogFormatters.smartColumnMap[k].forEach(function(col) {
                LogFormatters.smartColumnMap[col] = k;
            });
            LogFormatters.smartColumnMap[k] = k;
        });
        return LogFormatters;
    }
    return function register(providers) {
        providers.$provide.factory('LogFormatters', logFormattersFactory);
    };
});
