/* global define */
define(['angular',
        'ng/services/injector',
        'ng/services/chart/constants',
        'ng/services/chart/base_csf_chart_object',
        'ng/services/chart/device_type',
        'ng/services/chart/device_type_group',
        'ng/services/chart/bubble_cluster'],
function(angular, inject, CONSTANTS, BaseCsfChartObject,
         DeviceType, DeviceTypeGroup, BubbleCluster) {
    'use strict';

    /**
     * The are different FTNT objects:
     * - FortiGate
     * - FortiSwitch
     * - FortiAP (future)
     * This is the base class of these objects.
     **/
    function BaseFtntObject($injector, injector, sn, vdom, shownIntfs) {
        $injector.invoke(BaseCsfChartObject, this);
        injector.injectMarked(this);

        this.sn = sn || '';
        this.vdom = vdom || '';
        this.shownIntfs = shownIntfs || [];
        this.children = [];
        this.endpointStats = [];
        this.outgoingIntf = null;
        this.icon = {};
    }
    BaseFtntObject.BASE_ROW_WIDTH = CONSTANTS.FGT.ROW.WIDTH;
    BaseFtntObject.BASE_ROW_HEIGHT = CONSTANTS.FGT.ROW.HEIGHT;
    BaseFtntObject.prototype = Object.create(BaseCsfChartObject.prototype);
    angular.extend(BaseFtntObject.prototype, {
        constructor: BaseFtntObject,
        children: [],
        endpointStats: [],
        icon: {},
        outgoingIntf: null,
        shownIntfs: [],
        sn: '',
        vdom: '',
        vdomMacMap: {},
        vdomIntfMap: {},
        vdomIntfs: [],
        viaRouterNat: false,
        addFtntChild: function(child) {
            var exists = this.children.length && this.children.some(function(d) {
                return (d.sn === child.sn && d.vdom === child.vdom);
            });
            if (!exists) {
                this.children.push(child);
            }
        },
        addBubbleCluster: inject.mark(function($injector) {
            return function(pack) {
                var bubbleCluster = $injector.instantiate(BubbleCluster, {
                    pack: pack
                });
                this.children.push(bubbleCluster);
            };
        }),
        buildDeviceType: inject.mark(function($injector, injector, iconCode, lang) {
            return function(withInterfaces) {
                var typeMap = {};
                var devices = this.endpointStats;

                devices.filter(function(dev) {
                    return dev.name && dev.srcintf !== 'root';
                }).forEach(function(dev) {
                    var device = dev.device || 'other-network-device';
                    var srcIntf = withInterfaces ? dev.srcintf : CONSTANTS.SELF;
                    typeMap[srcIntf] = typeMap[srcIntf] || {};
                    typeMap[srcIntf][device] = typeMap[srcIntf][device] || 0;
                    typeMap[srcIntf][device]++;
                });
                var that = this;
                var sn = this.sn;
                var vdom = this.vdom;
                Object.keys(typeMap).forEach(function(intfName) {
                    var map = typeMap[intfName];
                    var deviceTypes = [];
                    Object.keys(map).forEach(function(typeName) {
                        var count = map[typeName];
                        var deviceType = new DeviceType(typeName,
                                                        lang('device-type.' + typeName).toString(),
                                                        count);
                        var codeObj = iconCode.getDeviceIconCode(typeName || '');
                        if (codeObj && codeObj.code) {
                            deviceType.setIconInfo({
                                name: codeObj.name,
                                code: String.fromCharCode(parseInt(codeObj.code, 16)),
                                fontFamily: codeObj.fontFamily
                            });
                        }

                        deviceTypes.push(deviceType);
                    });
                    if (deviceTypes.length) {
                        var deviceTypeGroup = $injector.instantiate(DeviceTypeGroup, {
                            injector: injector,
                            children: deviceTypes
                        });
                        deviceTypeGroup.setUpstream({
                            sn: sn,
                            vdom: vdom,
                            intf: intfName
                        });
                        that.children.push(deviceTypeGroup);
                    }
                });
            };
        }),
        getAllIntfIds: inject.mark(function(csf_chart) {
            return function(withInterfaces) {
                if (withInterfaces) {
                    return [csf_chart.genSnVdomIntfId(this.sn, this.vdom, CONSTANTS.SELF)];
                }
                return this.shownIntfs.map(function(d) {
                    return d.id;
                });
            };
        }),
        getEndpointStats: function(trafficBased) {
            return (this.endpointStats || []).filter(function(d) {
                // if traffic based, we need to ensure value has something
                return d.name && d.srcintf && (!trafficBased || d.value);
            });
        },
        getHeadHeight: function(scale) {
            var scaleFactor = typeof scale !== 'undefined' ? scale : 1;
            return this.BASE_HEAD_HEIGHT * scaleFactor;
        },
        getHeight: function(withInterfaces, scale) {
            var scaleFactor = typeof scale !== 'undefined' ? scale : 1;
            var headH = this.getHeadHeight(scaleFactor);
            if (!withInterfaces) {
                return headH;
            }
            var rowH = BaseFtntObject.BASE_ROW_HEIGHT * scaleFactor;
            var numIntfs = this.shownIntfs.length || 0;
            var h = (headH + numIntfs * rowH);
            return h;
        },
        getInterface: function(intfName) {
            var intfs = this.shownIntfs;
            var entry;
            var index;
            intfs.some(function(intf, i) {
                if (intf.name === intfName) {
                    entry = intf;
                    index = i;
                    return true;
                }
                return false;
            });
            if (entry) {
                return {
                    entry: entry,
                    index: index
                };
            }
            return null;
        },
        getLeftHeadMidpoint: function() {
            var offset = this.offset;
            return {
                x: offset.x,
                y: offset.y + this.getHeadHeight() / 2
            };
        },
        getOutgoingIntfMidpoint: function() {
            var outgoingIntf = this.outgoingIntf;
            if (!outgoingIntf) {
                return this.getLeftHeadMidpoint();
            }
            var intfs = this.shownIntfs;
            var entry;
            var index;
            intfs.some(function(intf, i) {
                if (intf.name === outgoingIntf.name) {
                    entry = intf;
                    index = i;
                    return true;
                }
                return false;
            });
            if (!entry) {
                return this.getLeftHeadMidpoint();
            }
            var offset = this.offset;
            return {
                x: offset.x,
                y: offset.y + this.getHeadHeight() + (index + 0.5) * BaseFtntObject.BASE_ROW_HEIGHT
            };
        },
        getRightInterfaceMidpoint: function(intfName) {
            var intf = this.getInterface(intfName);
            if (!intf) {
                return this.getRightHeadMidpoint();
            }
            var index = intf.index;
            var offset = this.offset;
            return {
                x: offset.x + BaseFtntObject.BASE_ROW_WIDTH,
                y: offset.y + this.getHeadHeight() + (index + 0.5) * BaseFtntObject.BASE_ROW_HEIGHT
            };
        },
        getRightHeadMidpoint: function() {
            var offset = this.offset;
            return {
                x: offset.x + BaseFtntObject.BASE_ROW_WIDTH,
                y: offset.y + this.getHeadHeight() / 2
            };
        },
        setEndpointStats: inject.mark(function(iconCode) {
            return function(stats) {
                angular.copy(stats, this.endpointStats);
                this.endpointStats.forEach(function(device) {
                    var codeObj = iconCode.getDeviceIconCode(device.device || '');
                    if (codeObj && codeObj.code) {
                        angular.extend(device, {
                            iconName: codeObj.name,
                            iconCode: String.fromCharCode(parseInt(codeObj.code, 16)),
                            iconFontFamily: codeObj.fontFamily
                        });
                    }
                });
            };
        }),
        setOutgoingIntf: function(intf) {
            this.outgoingIntf = {};
            angular.extend(this.outgoingIntf, intf);
        },
        sortChildren: inject.mark(function(csf_chart) {
            return function(withInterfaces) {
                // Sorting should reflect the followings:
                // - starting from the same interface, FGT is shown first, then FSW
                // then bubble (or device type)
                // - children must appear in the order of its upstream interface
                // Render FGT first, then FSW, then endpoints (note that device
                // bubbles and device type don't co-exist)
                var ORDERS = {
                    FGT: 0,
                    FSW: 1,
                    ENDPOINT: 2
                };

                // within a FGT/FSW, the interface name is unique
                var intfNames = (this.shownIntfs || []).map(csf_chart.pluckName);
                var sortFn = withInterfaces ? sortByIntf : sortByType;
                this.children.sort(sortFn);

                function sortByType(a, b) {
                    var aType = getType(a);
                    var bType = getType(b);
                    var sameType = aType === bType;
                    if (sameType) {
                        if (['FGT', 'FSW'].indexOf(aType)) {
                            return 0;
                        }
                        var isBubbleCluster = csf_chart.isBubbleCluster(a);
                        if (isBubbleCluster) {
                            return csf_chart.sort.bubbleCluster(a, b);
                        }
                        return csf_chart.sort.deviceTypeGroup(a, b);
                    }
                    return ORDERS[aType] - ORDERS[bType];
                }
                function getType(obj) {
                    var type;
                    if (csf_chart.isFgt(obj)) {
                        type = 'FGT';
                    } else if (csf_chart.isFsw(obj)) {
                        type = 'FSW';
                    } else {
                        type = 'ENDPOINT';
                    }
                    return type;
                }
                function sortByIntf(a, b) {
                    var aUpstream = a.upstream;
                    var bUpstream = b.upstream;
                    var aIntf = aUpstream.intf;
                    var bIntf = bUpstream.intf;
                    var sameIntf = aIntf === bIntf;
                    if (sameIntf) {
                        return sortByType(a, b);
                    }
                    var aIntfIndex = intfNames.indexOf(aIntf);
                    var bIntfIndex = intfNames.indexOf(bIntf);
                    if (aIntfIndex === bIntfIndex) {
                        return 0;
                    }
                    // if no upstream interface, move it to the top because we
                    // can connect it to the FGT itself
                    if (aIntfIndex === -1) {
                        return -1;
                    }
                    if (bIntfIndex === -1) {
                        return 1;
                    }
                    return aIntfIndex > bIntfIndex ? 1 : -1;
                }
            };
        }),
        getIcon: function() {
            return this.icon;
        }
    });

    return BaseFtntObject;
});
