/*global define */

/**
 * Service used to render CSF chart
 */
define(['angular', 'fweb.util/interface', 'ftnt_shared', 'ng/services/chart/bubble_cluster',
       'ng/services/chart/fortigate', 'ng/services/chart/fortiswitch'],
function(angular, f_interface, ftnt_shared, BubbleCluster, FortiGate, FortiSwitch) {
    'use strict';

    var SN_VDOM_SEP = '<>';

    function CsfChartService() {
    }

    CsfChartService.prototype = {
        bound: function(fn, a, b) {
            return a === undefined ? b : fn(a, b);
        },
        /*
         * @param data The device details data
         * @param detectedWanRoleIntf The interface appears to be used for WAN
         *     traffic, we provide it just in case the WAN role isn't set, we
         *     still need to force adding it to WAN role interface list to
         *     ensure the connecting line between upstream and downstream FGT
         * @param interfaceMap The interface map to look for role information
         * @param intfResult The result reference which will be updated to have:
         * - shownIntfs: List of interfaces to be drawn under the serial number
         * - bubbleIntfs: List of interfaces to connect to the bubble clusters
         * - wanIntfs: List of WAN-role interfaces existing in the data
         **/
        collectInterfaceInfo: function(data, vdom, detectedWanRoleIntf, interfaceMap,
                                       sn, intfResult) {
            var inUseSrcInterfaces = []; // mainly for drawing bubbles
            var inUseDstInterfaces = []; // mainly for drawing WAN-cloud

            var srcIntfMap = {};
            var dstIntfMap = {};
            data.forEach(function(d) {
                if (d.srcintf && d.srcintf !== 'root') {
                    srcIntfMap[d.srcintf] = 1;
                }
                if (d.dstintf && d.dstintf !== 'root') {
                    dstIntfMap[d.dstintf] = 1;
                }
            });
            inUseSrcInterfaces = Object.keys(srcIntfMap);
            inUseDstInterfaces = Object.keys(dstIntfMap);

            // Classify WAN role interfaces and nonWANRole interfaces (bubble
            // interfaces) from in-use source interfaces
            var wanRoleInterfaces = [];
            var bubbleInterfaces = [];
            inUseSrcInterfaces.forEach(function(d) {
                if (this.isWANRole(d, interfaceMap)) {
                    if (wanRoleInterfaces.indexOf(d) === -1) {
                        wanRoleInterfaces.push(d);
                    }
                } else {
                    if (bubbleInterfaces.indexOf(d) === -1) {
                        bubbleInterfaces.push(d);
                    }
                }
            }.bind(this));

            // Recall that we have taken out the traffic from the FSW device
            // to the fortilink interface, so we need to manually add them
            // here to the bubbleInterfaces
            Object.keys(interfaceMap).forEach(function(intf) {
                if (f_interface.isFortiLinkMode(interfaceMap[intf] &&
                    interfaceMap[intf].vdom === vdom)) {
                    bubbleInterfaces.push(intf);
                }
            });

            // extract WAN role interfaces from in-use destination interfaces
            inUseDstInterfaces.forEach(function(d) {
                if (this.isWANRole(d, interfaceMap) && wanRoleInterfaces.indexOf(d) === -1) {
                    wanRoleInterfaces.push(d);
                }
            }.bind(this));

            // if the detected wan role interface isn't found in this device
            // stats, let add it to wan role list
            var missingMap = {};
            if (detectedWanRoleIntf && wanRoleInterfaces.indexOf(detectedWanRoleIntf) === -1) {
                wanRoleInterfaces.unshift(detectedWanRoleIntf);
                missingMap[detectedWanRoleIntf] = true;
            }

            // there are cases in which device stats data return two entries:
            // - srcintf: wan --> dstintf: root
            // - srcintf: wifi --> dstintf: wan
            // Or when detectedWanRoleIntf is in the bubbleInterfaces.
            // So wan will appear in both bubbleInterfaces and wanRoleInterfaces.
            // Make sure we remove that out of bubbleInterfaces
            var i = bubbleInterfaces.length;
            while (i--) {
                if (wanRoleInterfaces.indexOf(bubbleInterfaces[i]) > -1) {
                    bubbleInterfaces.splice(i, 1);
                }
            }

            // update interfaces. Note that bubbleInterfaces and wanRoleInterfaces
            // are sub sets of intfResult.shownIntfs
            // If the interface no longer exists in the new data request:
            // - Keep it if it is nonWANRole
            // - Remove it if it is WANRole
            var oldInterfaceMap = {};
            i = intfResult.shownIntfs.length;
            while (i--) {
                var intf = intfResult.shownIntfs[i];
                if (intf.isWANRole) {
                    if (wanRoleInterfaces.indexOf(intf.name) === -1) {
                        // so it used to be a WANRole but now not in the new data
                        intfResult.shownIntfs.splice(i, 1);
                    }
                }
                oldInterfaceMap[intf.name] = 1;
            }
            // add more bubble interfaces to the end
            bubbleInterfaces.forEach(function(d) {
                if (!oldInterfaceMap[d]) {
                    intfResult.shownIntfs.push({name: d});
                }
            });
            // add more WAN role interfaces to the beginning
            wanRoleInterfaces.forEach(function(d) {
                if (!oldInterfaceMap[d]) {
                    intfResult.shownIntfs.unshift({
                        name: d,
                        isWANRole: true,
                        isWANRoleMissing: missingMap[d]
                    });
                }
            });

            // now sort the bubbleInterfaces using the order as it appears in the
            // interface list. The order is to avoid path crossing.
            var interfaceNames = intfResult.shownIntfs.map(this.pluckName);
            bubbleInterfaces.sort(function(a, b) {
                return interfaceNames.indexOf(a) - interfaceNames.indexOf(b);
            });

            intfResult.bubbleIntfs = bubbleInterfaces;
            intfResult.wanIntfs = wanRoleInterfaces;
            intfResult.shownIntfs = intfResult.shownIntfs.map(function(d) {
                var intfObj = interfaceMap[d.name] || {};
                d.id = this.genSnVdomIntfId(sn, intfObj.vdom, d.name);
                d.network = intfObj.network || '';
                d.role = intfObj.role || 'undefined';
                d.fortilink = intfObj.fortilink;
                d.vdom = intfObj.vdom;
                d.ipAddr = intfObj.ipAddr || '';
                d.sn = sn;
                return d;
            }.bind(this));

            // remove devices coming from wan role interfaces
            i = data.length;
            while (i--) {
                if (wanRoleInterfaces.indexOf(data[i].srcintf) >= 0) {
                    data.splice(i, 1);
                }
            }
        },
        createColorFn: function(d3, domain) {
            var colors = [
                d3.scale.category20(),
                d3.scale.category20b(),
                d3.scale.category20c()
            ];
            var colorGroupLen = colors.length;
            var MAX_COLOR_EACH = 20;
            var maxColor = MAX_COLOR_EACH * colorGroupLen;
            var colorDomainEligibleValues = domain.slice(0, maxColor);

            var chunks = [],
                colorDomainEligibleLen = colorDomainEligibleValues.length,
                i;
            for (i = 0; i < colorDomainEligibleLen; i += MAX_COLOR_EACH) {
                chunks.push(domain.slice(i, i + MAX_COLOR_EACH));
            }
            chunks.forEach(function(chunk, i) {
                colors[i].domain(chunk);
            });
            var otherColor =  d3.scale.category10();

            return function(name, entry) {
                var idx = colorDomainEligibleValues.indexOf(name);
                if (idx >= 0) {
                    idx = Math.floor(idx / MAX_COLOR_EACH);
                    return colors[idx](name, entry);
                }
                return otherColor(name, entry);
            };
        },
        findConnectingIntfFromUpstreamStats: function(vdomStatsObject, downstreamMacMap) {
            var foundOutgoingIntf = null;
            var upstreamIntf = null;
            var upstreamVdom = null;
            Object.keys(vdomStatsObject).filter(function(d) {
                return d.indexOf('$') !== 0;
            }).some(function(vdom) {
                vdomStatsObject[vdom].some(function(dev) {
                    if (downstreamMacMap.hasOwnProperty(dev.mac)) {
                        // a mac address of the upstream stats match with a mac
                        // of downstream
                        foundOutgoingIntf = downstreamMacMap[dev.mac];
                        upstreamVdom = vdom;
                        upstreamIntf = dev.srcintf;
                    }
                    return !!foundOutgoingIntf;
                });
                return !!foundOutgoingIntf;
            });
            if (foundOutgoingIntf) {
                return {
                    upstreamIntf: upstreamIntf,
                    upstreamVdom: upstreamVdom,
                    outgoingIntf: foundOutgoingIntf
                };
            }
            return null;
        },
        fvStatsToInterfaces: function(stats, vdom, detectedWanRoleIntf, interfaceMap, sn) {
            var intfInfo = {
                shownIntfs: [],
                bubbleIntfs: [],
                wanIntfs: []
            };
            this.collectInterfaceInfo(stats, vdom, detectedWanRoleIntf,
                                      interfaceMap, sn, intfInfo);
            return intfInfo.shownIntfs;
        },
        genSnVdomIntfId: function(sn, vdom, intf) {
            // this isn't for quick lookup but is also use to color and add
            // to SVG object id, so the separator will be dash
            return [sn, vdom, intf].join('-');
        },
        genUniqueSnVdom: function(sn, vdom) {
            // this is purely for quick lookup
            return sn + SN_VDOM_SEP + vdom;
        },
        genUpstreamIntfId: function(upstream) {
            return this.genSnVdomIntfId(upstream.sn, upstream.vdom, upstream.intf);
        },
        isBubbleCluster: function(obj) {
            return (obj instanceof BubbleCluster);
        },
        isFgt: function(ftnt) {
            return (ftnt instanceof FortiGate);
        },
        isFsw: function(ftnt) {
            return (ftnt instanceof FortiSwitch);
        },
        isWANRole: function(name, interfaceMap) {
            var entry = interfaceMap[name];
            return entry && entry.role === 'wan';
        },
        /* This is derived from d3_layout_packTransform (layout/pack.js)
         * In addition to the original function purpose (to scale the circle
         * and its children using scale factor k), it will also move the
         * circle and its children along a vector d
         * (d {x: distance in x, y: distance in y})
         */
        layoutPackTransform: function(node, k, d) {
            function transform(node, cx, cy, k) {
                var children = node.children;
                node.x = cx + k * (node.x - cx) + d.x;
                node.y = cy + k * (node.y - cy) + d.y;
                node.r *= k;
                if (children) {
                    var i = -1, n = children.length;
                    while (++i < n) {
                        transform(children[i], cx, cy, k);
                    }
                }
            }
            return transform(node, node.x, node.y, k);
        },
        pluckName: function(d) {
            return d.name;
        },
        pluckR: function(d) {
            return d.r;
        },
        sort: {
            bubbleCluster: function(a, b) {
                var aPack = a.pack;
                var bPack = b.pack;
                var aR = aPack.r;
                var bR = bPack.r;
                // larger cluster will be on top
                return bR - aR;
            },
            deviceType: function(a, b) {
                var aCount = a.count;
                var bCount = b.count;
                if (aCount === bCount) {
                    return ftnt_shared.util.fastLocaleCompare(a.typeTranslated, b.typeTranslated);
                }
                // larger number of devices will be on top
                return bCount - aCount;
            },
            deviceTypeGroup: function(a, b) {
                var aChildrenCount = a.children.length;
                var bChildrenCount = b.children.length;
                // larger number of children will be on top
                return bChildrenCount - aChildrenCount;
            }
        },
        vdomHavingStats: function(vdomStatsObject) {
            // only return vdom which has stats
            return Object.keys(vdomStatsObject).filter(function(d) {
                // ignore keys like '$resolved', '$promise'
                return d.indexOf('$') !== 0 && vdomStatsObject[d].length > 0;
            });
        }
    };

    return function(providers) {
        providers.$provide.service('csf_chart', CsfChartService);
    };
});
