/* global define */
/* jshint maxparams: 8 */
define(['angular', 'ng/services/injector',
        'ng/services/chart/constants',
        'ng/services/chart/base_csf_chart_object',
        'ng/services/chart/base_ftnt_object',
        'ng/services/chart/fortigate',
        'ng/services/chart/fortiswitch',
        'ng/services/chart/device_type'],
function(angular, inject, CONSTANTS, BaseCsfChartObject, BaseFtntObject,
         FortiGate, FortiSwitch, DeviceType) {
    'use strict';


    function TopologyTree($injector, injector, csfData, masterFvStats, withAccessDevices) {
        injector.injectMarked(this);
        this.tree = null;
        this.map = {};
        this._buildTree(csfData, masterFvStats, withAccessDevices);
        return this;
    }

    TopologyTree.prototype = {
        tree: null,
        map: {},
        hasCloud: function() {
            return this.tree.shownIntfs.some(function(d) {
                return d.isWANRole;
            });
        },
        toChart: inject.mark(function(fortiviewVisualization, csf_chart, cloudSvg) {
            var bubbleOptionEnum = fortiviewVisualization.BUBBLE_OPTION_ENUM;
            return function(d3, withInterfaces, bubbleOption) {
                var hasCloud = this.hasCloud();
                if (bubbleOptionEnum.DEVICE_TYPE === bubbleOption) {
                    this._buildDeviceType(withInterfaces);
                } else if ([bubbleOptionEnum.TRAFFIC,
                            bubbleOptionEnum.CONSTANT].indexOf(bubbleOption) > -1) {
                    var trafficBased = bubbleOptionEnum.TRAFFIC === bubbleOption;
                    this._buildBubbleCluster(d3, withInterfaces, trafficBased);
                }

                // the topology object is now solid, this is the perfect time to
                // sort children of each FTNT object
                var map = this.map;
                Object.keys(map).forEach(function(key) {
                    var ftnt = map[key];
                    ftnt.sortChildren(withInterfaces);
                });
                var ftnts = [];
                var clusterNodes = [];
                var deviceNodes = [];
                var deviceTypeGroups = [];
                var linkNodes = [];

                // Now calculate the offset of each device, we keep update the
                // base line so that children are always on the same flat line
                // or under parent
                var levelHeight = {};
                var levelMaxX = {};
                var levelMinX = {};
                var allIntfs = [];
                var that = this;

                function processChartObject(ftnt, level) {
                    var offset = {x: 0, y: 0};
                    var startY;
                    var allH;
                    var maxH;
                    var height;
                    var divStartX = CONSTANTS.DIVISION.WIDTH * level +
                        (hasCloud ? (CONSTANTS.DIVISION.MARGIN.LEFT + cloudSvg.WIDTH) : 0);
                    var isFtnt = csf_chart.isFgt(ftnt) || csf_chart.isFsw(ftnt);
                    allH = Object.keys(levelHeight).filter(function(k) {
                        return isFtnt || (k < level);
                    }).map(function(k) {
                        return levelHeight[k];
                    });
                    startY = levelHeight[level] ?
                                 levelHeight[level] : CONSTANTS.DIVISION.MARGIN.TOP;
                    maxH = d3.max(allH);
                    startY = d3.max([maxH, startY]);

                    if (isFtnt) {
                        offset.y = startY + BaseCsfChartObject.V_MARGIN;
                        offset.x = divStartX +
                            (CONSTANTS.DIVISION.WIDTH - BaseFtntObject.BASE_ROW_WIDTH) / 2;
                        ftnt.setOffset(offset);
                        height = ftnt.getHeight(withInterfaces);
                        allIntfs.push.apply(allIntfs, ftnt.getAllIntfIds(withInterfaces));
                        levelHeight[level] = offset.y + height;
                        if (!levelMinX[level] || levelMinX[level] > offset.x) {
                            levelMinX[level] = offset.x;
                        }
                        if (!levelMaxX[level] ||
                            levelMaxX[level] < (offset.x + BaseFtntObject.BASE_ROW_WIDTH)) {
                            levelMaxX[level] = offset.x + BaseFtntObject.BASE_ROW_WIDTH;
                        }
                        ftnts.push(ftnt);
                        if (ftnt.upstream && ftnt.outgoingIntf) {
                            var outgoingIntfMidpoint = withInterfaces ?
                                                           ftnt.getOutgoingIntfMidpoint() :
                                                           ftnt.getLeftHeadMidpoint();
                            that._addLinks(
                                ftnt.sn,
                                linkNodes,
                                withInterfaces,
                                level,
                                outgoingIntfMidpoint,
                                withInterfaces ? ftnt.outgoingIntf.network : '',
                                ftnt.upstream
                            );
                        }
                    } else if (csf_chart.isBubbleCluster(ftnt)) {
                        offset.y = startY + BaseCsfChartObject.V_MARGIN;
                        offset.x = divStartX + BaseCsfChartObject.H_MARGIN;
                        var divisionMidX = divStartX + (CONSTANTS.DIVISION.WIDTH / 2);
                        ftnt.setOffset(offset, divisionMidX);
                        height = ftnt.getHeight();
                        levelHeight[level] = offset.y + height;
                        var clusterNode = ftnt.getClusterNode();
                        var width = height;
                        if (!levelMinX[level] || levelMinX[level] > (divisionMidX - width / 2)) {
                            levelMinX[level] = divisionMidX - width / 2;
                        }
                        if (!levelMaxX[level] ||
                            levelMaxX[level] < (divisionMidX + width / 2)) {
                            levelMaxX[level] = divisionMidX + width / 2;
                        }
                        clusterNodes.push(clusterNode);
                        deviceNodes.push.apply(deviceNodes, ftnt.getDeviceNodes());
                        if (ftnt.upstream) {
                            that._addLinks(
                                'bubbleCluster',
                                linkNodes,
                                withInterfaces,
                                level,
                                {
                                    x: divisionMidX - width / 2,
                                    y: offset.y + height / 2
                                },
                                '',
                                ftnt.upstream
                            );
                        }
                    } else {
                        // device type groups
                        offset.y = startY + BaseCsfChartObject.V_MARGIN;
                        offset.x = divStartX + (CONSTANTS.DIVISION.WIDTH - DeviceType.WIDTH) / 2;
                        ftnt.setOffset(offset);
                        height = ftnt.getHeight();
                        levelHeight[level] = offset.y + height;
                        if (!levelMinX[level] || levelMinX[level] > offset.x) {
                            levelMinX[level] = offset.x;
                        }
                        if (!levelMaxX[level] ||
                            levelMaxX[level] < (offset.x + DeviceType.WIDTH)) {
                            levelMaxX[level] = offset.x + DeviceType.WIDTH;
                        }
                        deviceTypeGroups.push(ftnt);
                        if (ftnt.upstream) {
                            that._addLinks(
                                'deviceTypeGroup',
                                linkNodes,
                                withInterfaces,
                                level,
                                {
                                    x: offset.x,
                                    y: offset.y + DeviceType.ROW_HEIGHT / 2
                                },
                                '',
                                ftnt.upstream
                            );
                        }
                    }
                }

                function treeTraverse(node, fn, level) {
                    var i;
                    fn(node, level);
                    for (i = 0; i < node.children.length; i++) {
                        if (csf_chart.isFgt(node.children[i]) ||
                              csf_chart.isFsw(node.children[i])) {
                            treeTraverse(node.children[i], fn, level + 1);
                        } else {
                            fn(node.children[i], level + 1);
                        }
                    }
                }
                treeTraverse(this.tree, processChartObject, 0);

                var allLevelH = Object.keys(levelHeight).map(function(k) {
                    return levelHeight[k];
                });
                var allLevelMaxX = Object.keys(levelMaxX).map(function(k) {
                    return levelMaxX[k];
                });
                var allLevelMinX = Object.keys(levelMinX).map(function(k) {
                    return levelMinX[k];
                });
                // revise links to add extra divisionHMargin attribute to avoid
                // link crossing issue
                this._reviseLinks(linkNodes, allLevelMinX, allLevelMaxX);

                var bBox = {
                    x1: 0,
                    y1: 0,
                    x2: d3.max(allLevelMaxX),
                    y2: d3.max(allLevelH)
                };

                return {
                    clusters: clusterNodes,
                    devices: deviceNodes,
                    deviceTypeGroups: deviceTypeGroups,
                    ftnts: ftnts,
                    links: linkNodes,
                    bBox: bBox,
                    allIntfs: allIntfs
                };
            };
        }),
        /**
         * As part of toChart process, add the link from the endpoint device or
         * downstream to upstream
         */
        _addLinks: inject.mark(function(csf_chart) {
            return function(prefix, links, withInterfaces, srcLevel, src, srcNetwork, upstream) {
                var intfId = csf_chart.genSnVdomIntfId(upstream.sn,
                                                  upstream.vdom,
                                                  withInterfaces ? upstream.intf : CONSTANTS.SELF);
                // from upstream find FGT/FSW
                var uniqueSnVdom = csf_chart.genUniqueSnVdom(upstream.sn, upstream.vdom);
                var ftnt = this.map[uniqueSnVdom];
                // from FGT/FSW, find interface index and eventually the upstream point
                var dst = withInterfaces ? ftnt.getRightInterfaceMidpoint(upstream.intf) :
                                           ftnt.getRightHeadMidpoint();
                var dstNetwork = '';
                if (withInterfaces) {
                    var intf = ftnt.getInterface(upstream.intf);
                    if (intf) {
                        dstNetwork = intf.entry.network || '';
                    }
                }

                links.push({
                    source: {
                        linkId: prefix + '_' + intfId,
                        divisionHMargin: {}, // to be updated after the whole tree is traversed
                        colorKey: intfId,
                        level: srcLevel,
                        network: srcNetwork,
                        x: src.x,
                        y: src.y
                    },
                    target: {
                        divisionHMargin: {},
                        level: srcLevel - 1,
                        network: dstNetwork,
                        x: dst.x,
                        y: dst.y
                    }
                });
            };
        }),
        _reviseLinks:  function(links, levelMinX, levelMaxX) {
            links.forEach(function(link) {
                var srcMinX = levelMinX[link.source.level];
                var srcMaxX = levelMaxX[link.source.level];
                var dstMinX = levelMinX[link.target.level];
                var dstMaxX = levelMaxX[link.target.level];
                angular.extend(link.source.divisionHMargin, {
                    minX: srcMinX,
                    maxX: srcMaxX
                });
                angular.extend(link.target.divisionHMargin, {
                    minX: dstMinX,
                    maxX: dstMaxX
                });
            });
        },
        /**
         * From endpoint stats, build device type object
         **/
        _buildDeviceType: function(withInterfaces) {
            // only build device type for FTNT object having endpoint stats
            var map = this.map;
            var ftnts = Object.keys(map).filter(function(ftntKey) {
                var ftnt = map[ftntKey];
                return ftnt.endpointStats && ftnt.endpointStats.length;
            }).map(function(ftntKey) {
                return map[ftntKey];
            });
            ftnts.forEach(function(ftnt) {
                ftnt.buildDeviceType(withInterfaces);
            });
        },
        _buildBubbleCluster: inject.mark(function(csf_chart) {
            return function(d3, withInterfaces, trafficBased) {
                // only build bubble cluster for FTNT object having endpoint stats
                // which can be included in bubble chart
                var map = this.map;
                var ftnts = Object.keys(map).filter(function(ftntKey) {
                    var ftnt = map[ftntKey];
                    var endpointStats = ftnt.getEndpointStats(trafficBased);
                    return endpointStats && endpointStats.length;
                }).map(function(ftntKey) {
                    return map[ftntKey];
                });

                if (!ftnts.length) {
                    // no FGT has endpoint stats, nothing need to be done then
                    return;
                }
                // unlike building device type which can be done on each FTNT object,
                // building bubble cluster requires getting endpoint stats from all
                // FTNT objects so that bubble size are relative to each other.
                // After that we can assign the bubble to each FTNT and each FTNT
                // object will take care of sorting.
                //
                // Work flow:
                // - Group by interface id in each FGT, get min, max value
                // - Calculate radius function if traffic based, use an r function
                // which ensures r(maxValue) / r(minValue) = MAX_MIN_BUBBLE_RATIO
                // - Push to a common root and first run the pack layout within
                // a square having side as 2 x FGT_width
                // - From created pack layout, find the min of grandchild radius (minR)
                // - if minBubbleR < MIN_BUBBLE_RADIUS, scale everything up so that min is
                // MIN_BUBBLE_RADIUS
                // - After scale find max of children radius (maxR)
                // - if maxChildR < FGT_width, scale everything up so that max isFGT_width
                var minValue;
                var maxValue;
                var intfMap = {};
                ftnts.forEach(function(ftnt) {
                    var devices = ftnt.getEndpointStats(trafficBased);
                    devices.forEach(function(d) {
                        var srcIntfId = csf_chart.genSnVdomIntfId(ftnt.sn, ftnt.vdom,
                                                     withInterfaces ? d.srcintf : CONSTANTS.SELF);
                        intfMap[srcIntfId] = intfMap[srcIntfId] || {
                            upstreamSrcIntfId: srcIntfId,
                            upstream: {
                                sn: ftnt.sn,
                                vdom: ftnt.vdom,
                                intf: withInterfaces ? d.srcintf : CONSTANTS.SELF
                            },
                            children: []
                        };
                        if (typeof maxValue === 'undefined' || d.value > maxValue) {
                            maxValue = d.value;
                        }
                        if (typeof minValue === 'undefined' || d.value < minValue) {
                            minValue = d.value;
                        }

                        var deviceCopy = {};
                        angular.copy(d, deviceCopy);
                        deviceCopy.srcIntfId = srcIntfId;
                        intfMap[srcIntfId].children.push(deviceCopy);
                    });
                });
                var radiusFn;
                if (!trafficBased) {
                    radiusFn = CONSTANTS.MIN_BUBBLE_RADIUS;
                } else {
                    var maxR1 = Math.sqrt(maxValue);
                    var minR1 = Math.sqrt(minValue);
                    var minR2 = CONSTANTS.MIN_BUBBLE_RADIUS;
                    var maxR2;
                    var radiusMaxMinRatio = maxR1 / minR1;
                    var toUseRatio = d3.min([radiusMaxMinRatio, CONSTANTS.MAX_MIN_BUBBLE_RATIO]);
                    maxR2 = minR2 * toUseRatio;

                    radiusFn = d3.scale.sqrt()
                        .domain([1, maxValue])
                        .range([minR2, maxR2]);
                }
                var root = {
                    name: 'fakeRoot',
                    children: []
                };
                Object.keys(intfMap).forEach(function(intfId) {
                    root.children.push(intfMap[intfId]);
                });
                var rectWidth = 1.5 * BaseFtntObject.BASE_ROW_WIDTH;
                var size = rectWidth;
                var pack = d3.layout.pack()
                    .sort(null)
                    .padding(CONSTANTS.PADDING.BUBBLE)
                    .size([size, size]);

                var packNodes,
                    packRoot;

                pack.radius(radiusFn);
                // pass to pack to get back the initial x, y, r
                packNodes = pack.nodes(root);
                packRoot = packNodes[0];

                var minMaxInfo = collectMinMaxRadius(d3, packRoot);
                var idealMaxR = rectWidth / 2 - CONSTANTS.PADDING.CLUSTER_VS_CLUSTER;
                if (minMaxInfo.maxCluster < idealMaxR) {
                    var scaleFactor = (idealMaxR / minMaxInfo.maxCluster);
                    csf_chart.layoutPackTransform(packRoot, scaleFactor, {x: 0, y: 0});
                }
                // now we have good packs, assign each small pack to the FTNT objects
                // Note that the pack has (x, y) as the circle center, we need to
                // update it to the new position in toChart function
                root.children.forEach(function(child) {
                    var upstream = child.upstream || {};
                    var uniqueSnVdom = csf_chart.genUniqueSnVdom(upstream.sn, upstream.vdom);
                    var ftnt = this.map[uniqueSnVdom];
                    if (!ftnt) {
                        throw new Error('FGT or FSW info is not valid', child.sn, child.vdom);
                    }
                    ftnt.addBubbleCluster(child);
                }.bind(this));

                function collectMinMaxRadius(d3, node) {
                    var minBubbleR;
                    var maxClusterR;
                    var minClusterR;
                    (node.children || []).forEach(function(d) {
                        var bubbleRs = (d.children || []).map(csf_chart.pluckR);
                        var minR = d3.min(bubbleRs);
                        if (typeof minBubbleR === 'undefined' || minR < minBubbleR) {
                            minBubbleR = minR;
                        }
                        if (typeof maxClusterR === 'undefined' || d.r > maxClusterR) {
                            maxClusterR = d.r;
                        }
                        if (typeof minClusterR === 'undefined' || d.r < minClusterR) {
                            minClusterR = d.r;
                        }
                    });
                    return {
                        minBubble: minBubbleR,
                        maxCluster: maxClusterR,
                        minCluster: minClusterR
                    };
                }
            };
        }),
        _buildTree: inject.mark(function($injector, injector, csf_chart) {
            return function(csfData, masterFvStats, withAccessDevices) {
                var csfTree = csfData.tree;
                var csfMap = csfData.map;
                var masterVdom = csfTree.vdom;
                var tree = null;
                var map = {};
                function buildFgtTree(node) {
                    var i;
                    var isRoot = node.isRoot;
                    var nodeData = node.data || {};
                    var nodeIntfData = nodeData.intfs || {};
                    var nodeFswData = nodeData.fsws || {};
                    var interfaceMap = nodeIntfData.intfs || {};
                    var macMap = nodeIntfData.macs || {};
                    var allStats = nodeData.stats || {};
                    var vdoms = csf_chart.vdomHavingStats(allStats);
                    // the upstream to which this FGT connects
                    var csfUpstreamInfo = null;
                    // the interface from which this FGT goes to upstream
                    var outgoingCsfIntf = null;

                    // collect outgoingCsfIntf and csfUpstreamInfo if available
                    if (node.intf) {
                        if (interfaceMap[node.intf]) {
                            var intfObj = interfaceMap[node.intf];
                            outgoingCsfIntf = {
                                name: node.intf,
                                vdom: intfObj.vdom,
                                fortilink: intfObj.fortilink,
                                ipAddr: intfObj.ipAddr,
                                isWANRole: intfObj.role === 'wan',
                                isWANRoleMissing: intfObj.role !== 'wan',
                                network: intfObj.network,
                                sn: node.sn,
                                id: csf_chart.genSnVdomIntfId(node.sn, intfObj.vdom, node.intf)
                            };
                        }
                    }
                    var upstreamNode = csfMap[node.upstreamSn];
                    if (upstreamNode && node.upstreamIntf) {
                        var upstreamIntfMap = (upstreamNode.data.intfs || {}).intfs;
                        var upstreamIntfObj = upstreamIntfMap[node.upstreamIntf];
                        csfUpstreamInfo = {
                            sn: node.upstreamSn,
                            vdom: upstreamIntfObj.vdom,
                            intf: node.upstreamIntf
                        };
                    }
                    function handleVdom(shownIntfs, vdom, vdomMacMap, vdomStats) {
                        var uniqueSnVdom;
                        var fgt = $injector.instantiate(FortiGate, {
                            injector: injector,
                            sn: node.sn,
                            vdom: vdom,
                            shownIntfs: shownIntfs,
                            build: node.build,
                            hostname: node.hostname,
                            vdomMode: node.vdomMode,
                            version: node.version,
                            viaRouterNat: node.viaRouterNat,
                            ipSeenByUpstream: node.ip,
                            unFilteredIntfData: nodeIntfData,
                            isRoot: isRoot
                        });
                        if (vdomStats) {
                            fgt.setEndpointStats(vdomStats);
                        }
                        uniqueSnVdom = csf_chart.genUniqueSnVdom(node.sn, vdom);
                        map[uniqueSnVdom] = fgt;

                        if (withAccessDevices) {
                            // there will be 2 rounds building up the tree with access
                            // devices: in 1st round, we add FSW as a child, in 2nd
                            // round, we revise the tree by move those behind FSW to
                            // FSW children. Here is the 1st round
                            var fsws = (nodeFswData.fsws || []).filter(function(f) {
                                return f.vdom === vdom;
                            });
                            fsws.forEach(function(f) {
                                var fswShownIntfs = [];
                                var fswToIntf = nodeIntfData.fswToIntf || {};
                                var fswIntf = fswToIntf[f.serial];
                                if (!fswIntf) {
                                    // If we don't know which interface of parent
                                    // the switch connect to, don't need to add it
                                    return;
                                }
                                var intfName = fswIntf.name;

                                var fortilinkPorts = f.ports.filter(function(p) {
                                    return !!p['fortilink-port'];
                                });
                                // by default FSW will have at least FortiLink ports,
                                // bubble ports will be added later
                                fswShownIntfs = fortilinkPorts.map(function(p) {
                                    return angular.extend(p, {
                                        vdom: f.vdom,
                                        sn: f.serial,
                                        id: csf_chart.genSnVdomIntfId(f.serial, f.vdom, p.name),
                                        network: fswIntf.network || '',
                                        ipAddr: fswIntf.ipAddr || ''
                                    });
                                });

                                var fsw = $injector.instantiate(FortiSwitch, {
                                    injector: injector,
                                    sn: f.serial,
                                    name: f.name,
                                    vdom: vdom,
                                    shownIntfs: fswShownIntfs
                                });

                                uniqueSnVdom = csf_chart.genUniqueSnVdom(f.serial, vdom);
                                map[uniqueSnVdom] = fsw;

                                fsw.setUpstream({
                                    sn: node.sn,
                                    vdom: vdom,
                                    intf: intfName
                                });
                                fsw.setOutgoingIntf(fswShownIntfs[0]);
                                uniqueSnVdom = csf_chart.genUniqueSnVdom(node.sn, vdom);
                                map[uniqueSnVdom].addFtntChild(fsw);
                                // make sure the FGT to which FSW connect must have
                                // the interface intfName
                                map[uniqueSnVdom].addIntf(fswIntf);
                            });
                        }

                        if (node.isRoot) {
                            tree = fgt;
                        } else {
                            // Now set upstream info, this FGT can go upstream via CSF or
                            // it may just route traffic to upstream
                            // First let check the upstream stats and see if there is any
                            // incoming traffic from this FGT
                            var upStats = upstreamNode.data.stats || {};
                            var conIntf = csf_chart.findConnectingIntfFromUpstreamStats(upStats,
                                                                                        vdomMacMap);
                            if (conIntf) {
                                // this is more reliable detection method as it is
                                // based on the mac address
                                uniqueSnVdom = csf_chart.genUniqueSnVdom(node.upstreamSn,
                                                                         conIntf.upstreamVdom);
                                if (map[uniqueSnVdom]) {
                                    // there exists a FGT with this sn-vdom
                                    fgt.setUpstream({
                                        sn: node.upstreamSn,
                                        vdom: conIntf.upstreamVdom,
                                        intf: conIntf.upstreamIntf
                                    });
                                    fgt.setOutgoingIntf(conIntf.outgoingIntf);
                                    map[uniqueSnVdom].addFtntChild(fgt);
                                }
                            } else if (!vdomStats ||
                                       outgoingCsfIntf && outgoingCsfIntf.vdom === vdom) {
                                // if upstream doesn't have any stats from downstream
                                // we have to use the CSF information
                                uniqueSnVdom = csf_chart.genUniqueSnVdom(node.upstreamSn,
                                                                         csfUpstreamInfo.vdom);
                                if (map[uniqueSnVdom]) {
                                    fgt.setUpstream(csfUpstreamInfo);
                                    fgt.setOutgoingIntf(outgoingCsfIntf);
                                    map[uniqueSnVdom].addFtntChild(fgt);
                                    // if there is no stats from downstream, the
                                    // shownIntfs of upstream may not have the
                                    // interface to which the downstream connect.
                                    // Make sure that interface is added to upstream.
                                    map[uniqueSnVdom].addIntfByName(csfUpstreamInfo.intf);
                                }
                            }
                        }
                    }

                    // Now build the FGT
                    // First case, if this FGT doesn't have any stats
                    if (!vdoms.length) {
                        // no stats: it is a downstream leaf or a root node alone
                        var shownIntfs = [];
                        if (outgoingCsfIntf) {
                            // it is a downstream: shownIntfs will have at least the
                            // outgoing CSF intf
                            shownIntfs.push(angular.extend({}, outgoingCsfIntf, {isWANRole: true}));
                        }
                        // note: if shownIntfs is empty, it is a root node without any
                        // interfaces
                        var vd = isRoot ? masterVdom :
                            (outgoingCsfIntf ? outgoingCsfIntf.vdom : 'root');

                        handleVdom(shownIntfs, vd, macMap, null);
                    } else {
                        // when this FGT has stats
                        vdoms.forEach(function(vd) {
                            // each sn-vdom can be presented as a FGT object

                            // root doesn't have outgoingCsfIntf while downstreams have it
                            var isCsfIntfFromThisVdom = outgoingCsfIntf &&
                                                        outgoingCsfIntf.vdom === vd;
                            // FV stats may not contain node.intf, so we may not
                            // know that node.intf is a WAN role interface, so let
                            // pass detectedWanIntf if needed
                            var detectedWanIntf = !isRoot && isCsfIntfFromThisVdom ?
                                                      node.intf : null;
                            var shownIntfs = csf_chart.fvStatsToInterfaces(allStats[vd],
                                                                  vd,
                                                                  detectedWanIntf,
                                                                  interfaceMap,
                                                                  node.sn);
                            var vdomMacMap = {};
                            Object.keys(macMap).forEach(function(mac) {
                                if (macMap[mac].vdom === vd) {
                                    vdomMacMap[mac] = macMap[mac];
                                }
                            });
                            handleVdom(shownIntfs, vd, vdomMacMap, allStats[vd]);
                        });
                    }

                    for (i = 0; i < (node.children || []).length; i++) {
                        buildFgtTree(node.children[i]);
                    }
                }
                buildFgtTree(csfTree);

                // round 2, any FGT with children need to be updated:
                //  - remove CSF FGT from endpointStats
                //  - detect if CSF FGT is behind FSW and move it if withAccessDevices flag
                var fgts = Object.keys(map).filter(function(ftntKey) {
                    var ftnt = map[ftntKey];
                    return ftnt.children && ftnt.children.length;
                }).map(function(ftntKey) {
                    return map[ftntKey];
                });
                fgts.forEach(function(fgt) {
                    fgt.refineHierarchy(withAccessDevices, map);
                });

                this.tree = tree;
                this.map = map;
            };
        })
    };

    return TopologyTree;
});
