define(['ml_module'], function(module) {
	'use strict';

	facetsFactory.$inject = ['injector', 'SearchFacet', 'lang'/*, 'datetime'*/];
	function facetsFactory(injector, SearchFacet, lang/*, datetime*/) {
		//TODO var qlist_sorting = window.qlist_sorting;
		/**
		 * Each type of facet should construct an instance and
		 * call #addFacets() to add any additional facet definitions.
		 * @see  #addFacets()
		 * @constructor
		 */
		function Facets() {
			var inject = {
				constructFacets: ['SearchFacet'],
				makeQlistFilters: ['ComplexValue'],
				fromQlistFilters: ['ComplexValue'],
				genericFacets: ['SearchFacet', 'ComplexValue', '$q', 'lang'],
				enumFacet: ['SearchFacet', 'ComplexValue', '$q', 'lang'],
				//genericConfigDateTime: ['SearchFacet', 'ComplexValue', 'lang'],
				langOptionValueGen: ['SearchFacet', 'lang']
			};
			for (var fn in inject) {
				this[fn].$inject = inject[fn];
				this[fn] = injector.bind(this[fn], this);
			}
			this.facetDefs = [];
			this.facets = this.constructFacets(this.facetDefs);
		}

		Facets.prototype = {
			_byId: null,
			byId: function(id) {
				if (id) {
					var result = this.byId()[id];
					if (!result) {
						throw new Error('Unable to find facet by id: ' + id);
					}
					return result;
				}
				if (!this._byId) {
					this._byId = this.facets.reduce(indexById, {});
				}
				return this._byId;

				function indexById(index, facet) {
					index[facet.id] = facet;
					return index;
				}
			},
			bySource: function(source) {
				return this.facets.reduce(indexBySource, {});

				function indexBySource(index, facet) {
					index[facet.selectors[source]] = facet;
					return index;
				}
			},
			addFacets: function(facets) {
				this.facets.push.apply(this.facets, facets);
				this._byId = null;
				this._bySelector = null;
			},
			_bySelector: null,
			bySelector: function(source, selector) {
				if (selector) {
					return this.bySelector(source)[selector];
				}
				this._bySelector = this._bySelector || {};
				if (!this._bySelector[source]) {
					var bs = this.facets.reduce(indexBySelector, {});
					this._bySelector[source] = bs;
				}
				return this._bySelector[source];

				function indexBySelector(index, facet) {
					var value = facet.selectors[source];
					if (value) {
						index[value] = facet;
					}
					if (facet.alias) {
						index[facet.alias] = facet;
					}
					return index;
				}
			},
			constructFacets: function(SearchFacet, facets) {
				return facets.map(function(facet) {
					if (!(facet instanceof SearchFacet)) {
						facet = new SearchFacet(facet);
					}
					return facet;
				});
			},
			makeQlistFilters: function(ComplexValue, facetsModel, extraFilters, period) {
				var facets = this.byId(),
					facetValues = Object.keys(facetsModel)
						.reduce(makeFacetValue(facetsModel), []),
					filters = facetValues.reduce(historyFilter, []);
				if (extraFilters) {
					filters.push.apply(filters, extraFilters);
					filters.sort(relTimeFirst);
				}
				if (period) {
					// filters.unshift(util_logs.timeSpanFilter(period));
					filters.unshift(period);
				}
				combine(filters);
				return filters;

				/**
				 * Sort the rel_time column so it appears first
				 * This way another time filter can override it.
				 * @param  {Object} a Filter to compare to b.
				 * @param  {Object} b Filter to compare to a.
				 * @return {Number}   Relative order
				 */
				function relTimeFirst(a, b) {
					return (a.id === 'rel_time' ? -1 : 0) - (b.id === 'rel_time' ? -1 : 0);
				}

				function makeFacetValue(valueMap) {
					return function(result, id) {
						var facet = facets[id],
							afs = valueMap[id].map(newFV);
						if (facet) {
							result.push.apply(result, afs);
						}
						return result;
						function newFV(value) {
							return {facet: facet, value: value};
						}
					};
				}
				function historyFilter(result, fv) {
					var selector = fv.facet.selectors.history,
						filter = {
							id: selector,
							logic: {is: {}, search: 'string'},
							value: [String(fv.value)]
						},
						type = fv.facet.type;
					if (ComplexValue.probablyComplex(fv.value)) {
						var value = fv.value instanceof ComplexValue ?
							fv.value : fv.facet.complexValue(fv.value);
						if (value instanceof ComplexValue) {
							filter.value = [];
							addComplexValue(value, filter);
						}
					}
					if (type) {
						filter.logic.is[type] = true;
					}
					if (selector) {
						var extra = fv.facet.tweakFilter.history(filter);
						result.push(filter);
						if (Array.isArray(extra)) {
							result = result.concat(extra);
						} else if (extra) {
							result.push(extra);
						}
					}
					return result.sort(sort);
					//move appid filter to the first one. (required for appid 0)
					//also required for threatname and type
					function sort(a, b) {
						var result = 0,
							keys = ['threattype', 'threatname', 'appid'],
							values = [0, -1, 1],
							A_LOWER = 1, B_LOWER = 2,
							IS_BOTH = A_LOWER + B_LOWER,
							indexes = {
								a: keys.indexOf(a.id),
								b: keys.indexOf(b.id)
							};
						result = (indexes.a > -1 ? A_LOWER : 0) +
								 (indexes.b > -1 ? B_LOWER : 0);
						if (result === IS_BOTH) {
							return indexes.a - indexes.b;
						}
						return values[result] || 0;
						//-1 (no keys found) return 0
						// 3 (2 keys found) whichever key is first in the keys
						// array is 'lower'
						//return 0
					}

					function addComplexValue(value, filter) {
						var modifierLogic = {
								'!': 'NOT',
								'<': 'RANGE',
								'>': 'RANGE',
								'<=': 'RANGE',
								'>=': 'RANGE'
							};
						filter.value = filter.value.concat(value.values);
						if (value.modifiers.length) {
							filter.logic.modifiers = value.modifiers.slice();
						}
						for (var i = 0, len = value.modifiers.length; i < len; ++i) {
							var m = value.modifiers[i];
							if (m in modifierLogic) {
								filter.logic[modifierLogic[m]] = 1;
							}
							var number = Number(filter.value[0]);
							var isNumber = !isNaN(number);
							var isInt = isNumber && parseInt(number, 10) === number;
							//< or > needs to be adjusted
							var adjust = 0;
							if (['<', '>'].indexOf(m) > -1) {
								adjust = isInt ? 1 : Number.MIN_VALUE;
								if (m === '<') {
									adjust *= -1;
								}
								filter.value[0] = String(number + adjust);
							}
							if (['>=', '>'].indexOf(m) > -1) {
								filter.value.push('');
							} else if (['<=', '<'].indexOf(m) > -1) {
								filter.value.unshift('');
							}
						}
						filter.logic.search = 'string';
						if (value.splitter) {
							filter.logic.splitter = value.splitter;
							if (value.splitter === '-') {
								filter.logic.RANGE = 1;
							}
						}
					}
				}
				/**
				* Logically combine filters that use the same field.
				* Works with NOT and OR logic, but not with ranges.
				* Needed when filter from a composite field is combined with a filter from
				* one of its component fields. (e.g. action_outcome and action/utmaction)
				* @param {Array} filters list to combine.
				**/
				function combine(filters) {
					var index = filters.length,
						keys = filters.map(mapIds),
						other, other_index, f;

					while (index--) {
						f = filters[index];
						// Does this filter appear more than once?
						other_index = keys.indexOf(f.id);
						if (other_index !== index) {
							other = filters[other_index];

							if (f.logic.RANGE || other.logic.RANGE) {
								continue;
							}
							if (f.logic.NOT && !other.logic.NOT) {
								// other - f
								other.logic.NOT = 0;
								other.value = other.value.filter(filterNotInCurrent);
							} else if (!f.logic.NOT && other.logic.NOT) {
								// f - other
								other.logic.NOT = 0;
								other.value = f.value.filter(filterNotInOther);
							} else if (f.logic.NOT && other.logic.NOT) {
								// intersection
								other.value = f.value.filter(filterInOther);
							} else {
								// union: append unique values from f into other
								var to_append = f.value.filter(filterNotInOther);
								other.value.push.apply(other.value, to_append);
							}
							filters.splice(index, 1);
						}
					}

					function mapIds(filter) {
						return (filter || {}).id;
					}

					function filterNotInOther(v) {
						return !(v in other.value);
					}

					function filterInOther(v) {
						return v in other.value;
					}

					function filterNotInCurrent(v) {
						return !(v in f.value);
					}
				}
			},
			fromQlistFilters: function(ComplexValue, filters, source) {
				var bySource = this.bySource(source);
				return filters.reduce(makeFacetsModel, {});

				function makeFacetsModel(model, filter) {
					var facet = bySource[filter.id];
					if (facet) {
						if (!(facet.id in model)) {
							model[facet.id] = [];
						}
						model[facet.id] = model[facet.id].concat(getComplexValues(filter));
					}
					return model;
				}

				function getComplexValues(filter) {
					if (filter.logic.NOT || filter.logic.RANGE || filter.logic.modifiers) {
						var value = new ComplexValue('');
						value.modifiers = filter.logic.NOT ? ['!'] : [];
						//extra logic that should help reverse
						if (filter.logic.modifiers) {
							var values = filter.value;
							if (!filter.splitter) {
								values = values.filter(notBlank);
							}
							value = ComplexValue
								.build(values, filter.logic.modifiers, filter.logic.splitter);
						} else {
							//try to construct a valid ComplexValue
							if (filter.logic.RANGE) {
								var blank = filter.value.indexOf('');
								if (blank === 0) {
									value.modifiers.push('>=');
								} else if (blank === 1) {
									value.modifiers.push('<=');
								} else {
									value.splitter = '-';
								}
							}
						}
						return [new ComplexValue(value)];
					} else {
						return filter.value;
					}
					function notBlank(v) { return v !== '' }
				}
			},
			langOptionValueGen: function(SearchFacet, lang, column) {
				var prefix = column.langPrefix || '';
				return function(key) {
					var value;
					if (Array.isArray(key)) {
						// Translated already by backend!
						value = new SearchFacet.ValueOption(key[0], key[1]);
					} else if (column.plainValues) {
						// Don't translate
						value = new SearchFacet.ValueOption(key, key);
					} else {
						value = new SearchFacet.ValueOption(key, lang(prefix + key));
					}
					return value;
				};
			},
			getValuesGen: function(column, values) {
				values = (values || column.values)
					.map(this.langOptionValueGen(column));
				if (column.addDescription) {
					values.forEach(column.addDescription.bind(column));
				}
				var complexValues;
				return function getValues(entries) {
					if (!complexValues) {
						complexValues = values.map(makeComplex.bind(this));
					}
					var existence;
					if (entries) {
						//assume entries is a promise if it is not an array.
						if (!Array.isArray(entries)) {
							return entries.then(getValues);
						}
						existence = entries.reduce(exists, {});
						complexValues.forEach(highlightExisting);
					}
					if (column.sortValues) {
						complexValues.sort(column.sortValues);
					}
					return complexValues;

					function exists(result, entry) {
						result[entry[column.selector]] = true;
						return result;
					}

					function highlightExisting(value) {
						value.highlight = function() { return this.key in existence };
					}

					function makeComplex(value) {
						/* jshint validthis: true */
						value.key = this.complexValue(value.key, true).getSimple();
						return value;
					}
				};
			},
			/**
			* @param {Injected} Searchfacet Service.
			* @param {Object[]} columns. Columns generated by log_filter.pyx
			* {
			*   fld_type: {Number} one of FIELD_TYPE.*
			*   is_default: {Boolean}
			*   lang_key: {String}
			*   selector: {String}
			*   (extra, attached in logging.js)
			*   type: {String} string representation of type (not sure if this is useful yet?)
			* }
			* @param {function(String searchValue, Number relevance):Number} adjustRelevance.
			*   Specify a function which will be added to each facet. Return value is an adjusted
			*   relevance based on factors that FacetedSearch
			*   may not be aware of (column visibility etc)
			*/
			genericFacets: function(SearchFacet, ComplexValue, $q, lang, columns, highlight) {
				var types = {
					ip_host: {
						//NOTE: this should be added to above Destination IP and Source IP
						// facet definitions to keep things DRY (next time it gets changed)
						type: 'ip',
						modifiers: ['!', ',', '-'],
						//TODO sort: qlist_sorting.sort_fns.ip_addr_sort_compare
					},
					//date_time: this.genericConfigDateTime(),
					enum: this.enumFacet({getValues: this.getValuesGen}),
					string: {
						modifiers: ['!', ',']
					},
					integer: {
						sort: function(a, b) {
							return parseFloat(a) - parseFloat(b);
						}
					}
				},
				overrides = {
					// eventtype: {
					// 	plainValues: true
					// },
					// action: {
					// 	sort: function(a, b) {
					// 		return $.sortCompareText(
					// 			a ? translateFilter('Log::Action::' + a.key.value) : '',
					// 			b ? translateFilter('Log::Action::' + b.key.value) : ''
					// 		);
					// 	}
					// }
				},
				facets = this;
				return columns.map(makeFacet);
				function makeFacet(column) {
					var f = {
						//generic currency symbol to avoid conflicts and denote 'generic' column
						//... questionable utility
						id: '_' + column.key,
						//will be fed to translateFilter()
						name: column.label,
						selectors: {
							history: column.key
						},
						highlight: highlight,
						type: column.type
					};
					var prop;
					if (column.type in types) {
						var type = types[column.type];
						for (prop in type) {
							f[prop] = type[prop];
						}
					}
					if (column.key in overrides) {
						column = angular.extend({}, column, overrides[column.key]);
					}
					if ('sort' in column) {
						f.sort = column.sort;
					} else if (column.type === 'enum') {
						f.sort = function(a, b) {
							a = column.values.indexOf(a.key.value);
							b = column.values.indexOf(b.key.value);
							return a - b;
						};
					}
					if ('alias' in column) {
						f.alias = column.alias;
					}
					if ('facetDef' in column) {
						for (prop in column.facetDef) {
							f[prop] = column.facetDef[prop];
						}
					}
					if (f.getValues === facets.getValuesGen) {
						f.getValues = facets.getValuesGen(column);
					}
					return new SearchFacet(f);
				}
			},
			enumFacet: function(SearchFacet, ComplexValue, $q, lang, facetDef) {
				var defaultDef = {
						modifiers: ['!', ','],
						complexValue: function(value, convert) {
							if (value) {
								var isComplex = value instanceof ComplexValue;
								if (!isComplex && convert) {
									value = new ComplexValue(value);
								}
								if (isComplex || convert) {
									value = value.withModifiers(this.modifiers);
								}
							}
							return value;
						},
						allowUserInput: false,
						lookup: function(key, reverse) {
							//enum probably doesn't have to specify a source or entries
							var values = this.getValues();
							var cv = this.complexValue.bind(this);
							var option = values.filter(findOption)[0];
							return option  ?
								(reverse ? option.key : option.value) : key;

							function findOption(option) {
								if (reverse) {
									return String(option.value) === String(key);
								} else {
									return option.key instanceof ComplexValue ?
										option.key.equals(cv(key, true)) :
										option.key === key;
								}
							}
						},
						format: function(value) {
							return this.lookup(value);
						}
					};

				return $.extend(true, defaultDef, (facetDef || {}));
			},
			genericConfigDateTime: function(SearchFacet, ComplexValue, lang) {
				return {
					valuePlaceholder: datetime.dateTimeFormat,
					parse: function(value) {
						value = datetime.parseDateTime(value);
						if (value == null) {
							throw new datetime.ParseDateTimeException();
						}
						return value && value instanceof Date &&
							datetime.dateToLocalSeconds(value);
					},
					//expects timestamp which may be a string?
					format: function(date) {
						if (date == null || date === '') { return date }
						date = Number(date);
						return datetime
							.formatDateTime(new Date(datetime.localSecondsToDate(date)));
					},
					lookup: function(key, reverse) {
						if (key == null) { return key }
						return reverse ?
							datetime.dateToLocalSeconds(datetime.parseDateTime(key)) :
							datetime.formatDateTime(datetime.localSecondsToDate(key));
					},
					tweakFilter: {
						history: function(filter) {
							filter.value = filter.value.map(fmt);
							function fmt(value, i) {
								if (value) {
									value = datetime.localSecondsToDate(value);
								} else if (filter.logic.RANGE) {
									value = i === 0 ? datetime.MIN_DATE : datetime.MAX_DATE;
								} else {
									return value;
								}
								return datetime.formatGmtDateTime(value, 'yy-mm-dd');
							}
						}
					},
					getValues: function(/*entries, source*/) {
						return [
							gtvo('Last 5 minutes', new datetime.TimeSpan(5, 0, 0)),
							gtvo('Last hour', new datetime.TimeSpan(1, 0, 0, 0)),
							gtvo('Last 24 hours', new datetime.TimeSpan(1, 0, 0, 0, 0))
						];
					},
					contextFilters: function(key) {
						return [
							[
								mvo('>=', key, this),
								mvo('<=', key, this)
							],
							this.getValues(),
							[
								gtvo('5 minutes before', new datetime.TimeSpan(5, 0, 0), key),
								gtvo('1 hour before', new datetime.TimeSpan(1, 0, 0, 0), key),
								gtvo('24 hours before',
									new datetime.TimeSpan(1, 0, 0, 0, 0), key)
							],
							[
								gtvo('5 minutes after', new datetime.TimeSpan(-5, 0, 0), key),
								gtvo('1 hour after', new datetime.TimeSpan(-1, 0, 0, 0), key),
								gtvo('24 hours after',
									new datetime.TimeSpan(-1, 0, 0, 0, 0), key)
							]
						];
					},
					genRelevanceFn: function() { return function() { return 1 } }
				};

				function mvo(modifier, key, facet) {
					var value = ComplexValue.build([key], [modifier]);
					return new SearchFacet.ValueOption(value, value.formatted(facet));
				}
				/**
				* getTimespanValueOption
				* @param {String} label to be translated
				* @param {datetime.TimeSpan} timeSpan to be subtracted from reference.
				* @param {Number} [reference=now] time in seconds to subtract from
				*/
				function gtvo(label, timeSpan, reference) {
					var before = '',
						after = '';
					if (reference === undefined) {
						before = '>= ';
						reference = datetime.dateToLocalSeconds(new Date());
					} else if (timeSpan.getTime() < 0) {
						before = reference + ' - ';
					} else if (timeSpan.getTime() > 0) {
						after = ' - ' + reference;
					}
					var rDate = datetime.localSecondsToDate(reference),
						seconds = reference - (timeSpan.getTime() * datetime.MS_TO_S);
					var cv = new ComplexValue(before + Math.round(seconds) + after);
					var vo = new SearchFacet.ValueOption(
						//Todo cv, translateFilter(label, [datetime.formatDateTime(rDate)])
						cv, lang(label)
					);
					vo.formatted = function(fn) {
						return this.key.formatted(fn);
					};
					return vo;
				}
			}
		};

		return Facets;
	}

	module.factory('Facets', facetsFactory);
});
