/**
* shared facet code between fortiview and other pages that use faceted_search
*/
/* globals define */
define([
    'angular', 'qlist.sort', 'fweb.util/logs', 'jquery', 'fweb.util/datetime',
    'directives/faceted_search'
],
function(angular, qlist_sort, util_logs, $, f_datetime) {
    'use strict';

    function Facets(injector) {
        var partialBind = {
            constructFacets: ['facets'],
            makeQlistFilters: ['facetsModel', 'extraFilters', 'realtime', 'period'],
            fromQlistFilters: ['filters', 'source'],
            genericFacets: ['columns', 'highlight'],
            genericConfigDateTime: [],
            langOptionValueGen: ['column']
        };
        for (var fn in partialBind) {
            this[fn] = injector.partial(this[fn], this, partialBind[fn]);
        }
        /*
        * List of facet definitions. Fed into #constructFacets to create proper
        * SearchFacet objects on the facet property. Properties on the 'meta' property can
        * be used to determine facet elegibility depending on current context (page/segment etc)
        *
        * See faceted_search SearchFacet for each facet's definition.
        * meta.segments is used to map to fortiview (matches
        *      `/^fortiview\.(.+)/` `\1` in
        *      `$routeSegmentProvider.when`'s second argument to an
        *       available )
        */
        this.facetDefs = [{
            id: 'source',
            name: 'Source IP',
            selectors: {
                monitor_api: 'address',
                history: 'srcip',
                session: 'saddr'
            },
            sort: qlist_sort.sort_fns.ip_addr_sort_compare,
            type: 'ip',
            modifiers: ['!', ',', '-'],
            meta: {
                //for fortiview
                segments: ['source', 'session', 'sandbox-source', 'sandbox-file']
            }
        },
        {
            id: 'destination',
            name: 'Destination IP',
            selectors: {
                monitor_api: 'address',
                history: 'dstip',
                session: 'daddr'
            },
            sort: qlist_sort.sort_fns.ip_addr_sort_compare,
            type: 'ip',
            modifiers: ['!', ',', '-'],
            meta: {
                segments: ['destination', 'session']
            }
        }];
        this.facets = this.constructFacets(this.facetDefs);
        this._extraFacets = [];
    }


    Facets.prototype = {
        _byId: null,
        byId: function(id) {
            if (id) {
                var result = this.byId()[id];
                if (!result) {
                    throw new Error('Unable to find facet by id: ' + id);
                }
                return result;
            }
            if (!this._byId) {
                this._byId =  this.facets.reduce(indexById, {});
                this._byId = this._extraFacets.reduce(indexById, this._byId);
            }
            return this._byId;

            function indexById(index, facet) {
                index[facet.id] = facet;
                return index;
            }
        },
        bySource: function(source) {
            var result = this.facetDefs.reduce(indexBySource, {});
            return this._extraFacets.reduce(indexBySource, result);

            function indexBySource(index, facet) {
                index[facet.selectors[source]] = facet;
                return index;
            }
        },
        extraFacets: function(facets) {
            this._extraFacets = facets || [];
            this._byId = null;
            this._bySelector = null;
        },
        _bySelector: null,
        bySelector: function(source, selector) {
            if (selector) {
                return this.bySelector(source)[selector];
            }
            this._bySelector = this._bySelector || {};
            if (!this._bySelector[source]) {
                var bs = this.facets.reduce(indexBySelector, {});
                bs = this.extraFacets.reduce(indexBySelector, bs);
                this._bySelector[source] = bs;
            }
            return this._bySelector[source];

            function indexBySelector(index, facet) {
                index[facet.selectors[source]] = facet;
                return index;
            }
        },
        constructFacets: function(SearchFacet, facets) {
            return facets.map(function(facet) {
                if (!(facet instanceof SearchFacet)) {
                    facet = new SearchFacet(facet);
                }
                return facet;
            });
        },
        qlistFiltersToFacetsModel: function(filters, source) {
            var facetsBySelector = this.bySelector(source);
            return filters.reduce(toFacetsModel, {});

            function toFacetsModel(model, filter) {
                var facet = facetsBySelector[filter.id];
                if (facet) {
                    model[facet.id] = filter.value;
                }
                return model;
            }
        },
        makeQlistFilters: function(ComplexValue, facetsModel, extraFilters, period) {
            // Fix bug: 0472527
            function mergeValues(model) {
                var merged = false,
                    value = new ComplexValue(model[0]),
                    origValue = model[0];
                model.slice(1).reduce(merge, value);

                if(merged)
                    model[0] = value.getSimple();

                function merge(result, value) {
                    //result is group[0] or null if merging failed
                    value = new ComplexValue(value);
                    var valid = result.merge(value, ',');
                    merged = merged || valid;
                    return result;
                }
            }

            facetsModel = angular.copy(facetsModel)

            Object.keys(facetsModel).forEach(function(k) {
                var model = facetsModel[k];
                if (model.length > 1) {
                    mergeValues(model);
                }
            }.bind(this));
            // end fix 0472527

            var facets = this.byId(),
                facetValues = Object.keys(facetsModel)
                    .reduce(makeFacetValue(facetsModel), []),
                filters = facetValues.reduce(historyFilter, []);
            if (extraFilters) {
                filters.push.apply(filters, extraFilters);
            }
            if (period) {
                //getFilters should be appendTimeSpanFilter
                filters = util_logs.getFilters(period, filters);
            }

	    // fix bug 444198
	    // added by lk, when send date/time filter, handle the time diff between device and UTC
	    // if s2utc_offset not set in the background, stay it as 0 so the convert change nothing
	    for(var i = 0; i < filters.length; i++)
	    {
		    var f = filters[i];
		    if(f.id == 'rel_time')
		    {
			    var offset = 0;
			    if(typeof s2utc_offset != 'undefined')
				    offset = s2utc_offset;

			    for(var j = 0; j < f.value.length; j++)
			    {
				    var d = new Date(f.value[j].replace(/-/g,"/"));
                    /* bug 0449912  , for ie*/
				    var t = (d.getTime() * 0.001 - offset)*1000;
				    d.setTime(t);
				    f.value[j] = f_datetime.formatDate(d, 'yy-mm-dd') + ' ' + f_datetime.formatTime(d);
			    }
		    }
	    }

            return filters;

            function makeFacetValue(valueMap) {
                return function(result, id) {
                    var facet = facets[id],
                        afs = valueMap[id].map(newFV);
                    if (facet) {
                        result.push.apply(result, afs);
                    }
                    return result;
                    function newFV(value) {
                        return {facet: facet, value: value};
                    }
                };
            }
            function historyFilter(result, fv) {
                var selector = fv.facet.selectors.history,
                    filter = {
                        id: selector,
                        logic: {is: {}, search: 'string'},
                        default_filter: fv.value.default_filter,//support deault filter
                        value: [String(fv.value)]
                    },
                    type = fv.facet.type;
                var existingFilter = result.filter(hasFilter(filter))[0];
                if (existingFilter) {
                    return result;
                }
                if (fv.value instanceof ComplexValue) {
                    filter.value = [];
                    addComplexValue(fv.value, filter);
                }
                if (type) {
                    filter.logic.is[type] = true;
                }
                if (selector) {
                    var extra =
                        fv.facet.tweakFilter.history(filter);
                    result.push(filter);
                    if (extra) {
                        result.push(extra);
                    }
                }
                return result.sort(sort);
                //move appid filter to the first one. (required for appid 0)
                //also required for threatname and type
                function sort(a, b) {
                    var result = 0,
                        keys = ['threattype', 'threatname', 'appid'],
                        values = [0, -1, 1],
                        A_LOWER = 1, B_LOWER = 2,
                        IS_BOTH = A_LOWER + B_LOWER,
                        indexes = {
                            a: keys.indexOf(a.id),
                            b: keys.indexOf(b.id)
                        };
                    result = (indexes.a > -1 ? A_LOWER : 0) +
                             (indexes.b > -1 ? B_LOWER : 0);
                    if (result === IS_BOTH) {
                        return indexes.a - indexes.b;
                    }
                    return values[result] || 0;
                    //-1 (no keys found) return 0
                    // 3 (2 keys found) whichever key is first in the keys
                    // array is 'lower'
                    //return 0
                }

                function hasFilter(filter) {
                    return function(existing) { return existing.id === filter.id };
                }

                function addComplexValue(value, filter) {
                    var modifierLogic = {
                            '!': 'NOT',
                            '<': 'RANGE',
                            '>': 'RANGE',
                            '<=': 'RANGE',
                            '>=': 'RANGE'
                        };
                    filter.value = filter.value.concat(value.values);
                    if (value.modifiers.length) {
                        filter.logic.modifiers = value.modifiers.slice();
                    }
                    for (var i = 0, len = value.modifiers.length; i < len; ++i) {
                        var m = value.modifiers[i];
                        if (m in modifierLogic) {
                            filter.logic[modifierLogic[m]] = 1;
                        }
                        var number = Number(filter.value[0]);
                        var isNumber = !isNaN(number);
                        var isInt = isNumber && parseInt(number, 10) === number;
                        //< or > needs to be adjusted
                        var adjust = 0;
                        if (['<', '>'].indexOf(m) > -1) {
                            adjust = isInt ? 1 : Number.MIN_VALUE;
                            if (m === '<') {
                                adjust *= -1;
                            }
                            filter.value[0] = String(number + adjust);
                        }
                        if (['>=', '>'].indexOf(m) > -1) {
                            filter.value.push('');
                        } else if (['<=', '<'].indexOf(m) > -1) {
                            filter.value.unshift('');
                        }
                    }
                    filter.logic.search = 'string';
                    if (value.splitter) {
                        filter.logic.splitter = value.splitter;
                        if (value.splitter === '-') {
                            filter.logic.RANGE = 1;
                        }
                    }
                }
            }
        },
        fromQlistFilters: function(ComplexValue, filters, source) {
            var bySource = this.bySource(source);
            return filters.reduce(makeFacetsModel, {});

            function makeFacetsModel(model, filter) {
                var facet = bySource[filter.id];
                if (facet) {
                    if (!(facet.id in model)) {
                        model[facet.id] = [];
                    }
                    model[facet.id] = model[facet.id].concat(getComplexValues(filter));
                }
                return model;
            }

            function getComplexValues(filter) {
                if (filter.logic.NOT || filter.logic.RANGE || filter.logic.modifiers) {
                    var value = new ComplexValue('');
                    value.modifiers = filter.logic.NOT ? ['!'] : [];
                    //extra logic that should help reverse
                    if (filter.logic.modifiers) {
                        var values = filter.value;
                        if (!filter.splitter) {
                            values = values.filter(notBlank);
                        }
                        value = ComplexValue
                            .build(values, filter.logic.modifiers, filter.logic.splitter);
                    } else {
                        //try to construct a valid ComplexValue
                        if (filter.logic.RANGE) {
                            var blank = filter.value.indexOf('');
                            if (blank === 0) {
                                value.modifiers.push('>=');
                            } else if (blank === 1) {
                                value.modifiers.push('<=');
                            } else {
                                value.splitter = '-';
                            }
                        }
                    }
                    return [new ComplexValue(value)];
                } else {
                    return filter.value;
                }
                function notBlank(v) { return v !== '' }
            }
        },


        langOptionValueGen: function(SearchFacet, lang, column) {
            var prefix = column.langPrefix || '';
            return function(key) {
                var value;
                if (Array.isArray(key)) {
                    // Translated already by backend!
                    value = new SearchFacet.ValueOption(key[0], key[1]);
                } else if (column.plainValues) {
                    // Don't translate
                    value = new SearchFacet.ValueOption(key, key);
                } else {
                    value = new SearchFacet.ValueOption(key, lang(prefix + key));
                }
                return value;
            };
        },
        valuesGetterGen: function(column, values) {
            values = (values || column.values)
                .map(this.langOptionValueGen(column));
            if (column.addDescription) {
                values.forEach(column.addDescription.bind(column));
            }
            var complexValues;
            return function getValues(entries) {
                if (!complexValues) {
                    complexValues = values.map(makeComplex.bind(this));
                }
                var existence;
                if (entries) {
                    //assume entries is a promise if it is not an array.
                    if (!Array.isArray(entries)) {
                        return entries.then(getValues);
                    }
                    existence = entries.reduce(exists, {});
                    complexValues.forEach(highlightExisting);
                }
                if (column.sortValues) {
                    complexValues.sort(column.sortValues);
                }
                return complexValues;

                function exists(result, entry) {
                    result[entry[column.selector]] = true;
                    return result;
                }

                function highlightExisting(value) {
                    value.highlight = function() { return this.key in existence };
                }

                function makeComplex(value) {
                    /* jshint validthis: true */
                    value.key = this.complexValue(value.key, true).getSimple();
                    return value;
                }
            };
        },
        /**
        * @param {Injected} Searchfacet Service.
        * @param {Object[]} columns. Columns generated by log_filter.pyx
        * {
        *   fld_type: {Number} one of FIELD_TYPE.*
        *   is_default: {Boolean}
        *   lang_key: {String}
        *   selector: {String}
        *   (extra, attached in logging.js)
        *   type: {String} string representation of type (not sure if this is useful yet?)
        * }
        * @param {function(String searchValue, Number relevance):Number} adjustRelevance. Specify
        *   a function which will be added to each facet. Return value is an adjusted relevance
        *   based on factors that FacetedSearch may not be aware of (column visibility etc)
        */
        genericFacets: function(SearchFacet, ComplexValue, $q, lang, columns, highlight) {
            var types = {
                ip_host: {
                    //NOTE: this should be added to above Destination IP and Source IP
                    // facet definitions to keep things DRY (next time it gets changed)
                    type: 'ip',
                    modifiers: ['!', ',', '-'],
                    sort: qlist_sort.sort_fns.ip_addr_sort_compare
                },
                date_time: this.genericConfigDateTime(),
                "enum": {
                    modifiers: ['!', ','],
                    complexValue: function(value, convert) {
                        var isComplex = value instanceof ComplexValue;
                        if (!isComplex && convert) {
                            value = new ComplexValue(value);
                        }
                        if (isComplex || convert) {
                            value = value.withModifiers(this.modifiers);
                        }
                        return value;
                    },
                    allowUserInput: false,
                    lookup: function(key, reverse) {
                        //enum probably doesn't have to specify a source or entries
                        var values = this.getValues();
                        var cv = this.complexValue.bind(this);
                        var option = values.filter(findOption)[0];
                        return option  ?
                            (reverse ? option.key : option.value) : key;

                        function findOption(option) {
                            return reverse ?
                                String(option.value) === String(key) :
                                option.key.equals(cv(key, true));
                        }
                    },
                    format: function(value) {
                        return this.lookup(value);
                    },
                    getValues: this.valuesGetterGen
                },
                string: {
                    modifiers: ['!', ',']
                },
                integer: {
                    sort: function(a, b) {
                        return parseFloat(a) - parseFloat(b);
                    }
                }
            },
            overrides = {
                eventtype: {
                    plainValues: true
                }
            },
            facets = this;
            return columns.map(makeFacet);
            function makeFacet(column) {
                var f = {
                    //generic currency symbol to avoid conflicts and denote 'generic' column
                    //... questionable utility
                    id: '_' + column.selector,
                    //will be fed to lang()
                    name: column.lang_key,
                    nameHtml: column.title_html,
                    selectors: {
                        'history': column.selector
                    },
                    highlight: highlight,
                    type: column.type
                };
                var prop;
                if (column.type in types) {
                    var type = types[column.type];
                    for (prop in type) {
                        f[prop] = type[prop];
                    }
                }
                if (column.selector in overrides) {
                    column = angular.extend({}, column, overrides[column.selector]);
                }
                if (column.type === 'enum') {
                    f.sort = function(a, b) {
                        a = column.values.indexOf(a.key.value);
                        b = column.values.indexOf(b.key.value);
                        return a - b;
                    };
                }
                if ('facetDef' in column) {
                    for (prop in column.facetDef) {
                        f[prop] = column.facetDef[prop];
                    }
                }
                if (f.getValues === facets.valuesGetterGen) {
                    f.getValues = facets.valuesGetterGen(column);
                }
                return new SearchFacet(f);
            }
        },
        genericConfigDateTime: function(SearchFacet, ComplexValue, lang) {
            return {
                valuePlaceholder: f_datetime.dateTimeFormat,
                parse: function(value) {
                    value = f_datetime.parseDateTime(value);
                    if (value == null) {
                        throw new f_datetime.ParseDateTimeException();
                    }
                    return value && value instanceof Date && f_datetime.dateToLocalSeconds(value);
                },
                //expects timestamp which may be a string?
                format: function(date) {
                    if (date == null || date === '') { return date }
                    date = Number(date);
                    return f_datetime.formatDateTime(new Date(f_datetime.localSecondsToDate(date)));
                },
                lookup: function(key, reverse) {
                    if (key == null) { return key }
                    return reverse ?
                        f_datetime.dateToLocalSeconds(f_datetime.parseDateTime(key)) :
                        f_datetime.formatDateTime(f_datetime.localSecondsToDate(key));
                },
                tweakFilter: {
                    history: function(filter) {
                        filter.value = filter.value.map(fmt);
                        function fmt(value, i) {
                            if (value) {
                                value = f_datetime.localSecondsToDate(value);
                            } else if (filter.logic.RANGE) {
                                value = i === 0 ? f_datetime.MIN_DATE : f_datetime.MAX_DATE;
                            } else {
                                return value;
                            }
                            return f_datetime.formatDateTime(value, 'yy-mm-dd');
                        }
                    }
                },
                populate: function(/*entries, source*/) {
                    return [
                        gtvo('Last 5 minutes', new f_datetime.TimeSpan(5, 0, 0)),
                        gtvo('Last hour', new f_datetime.TimeSpan(1, 0, 0, 0)),
                        gtvo('Last 24 hours', new f_datetime.TimeSpan(1, 0, 0, 0, 0))
                    ];
                },
                contextFilters: function(key) {
                    return [
                        [
                            mvo('>=', key, this),
                            mvo('<=', key, this)
                        ],
                        this.populate(),
                        [
                            gtvo('5 minutes before', new f_datetime.TimeSpan(5, 0, 0), key),
                            gtvo('1 hour before', new f_datetime.TimeSpan(1, 0, 0, 0), key),
                            gtvo('24 hours before',
                                new f_datetime.TimeSpan(1, 0, 0, 0, 0), key)
                        ],
                        [
                            gtvo('5 minutes after', new f_datetime.TimeSpan(-5, 0, 0), key),
                            gtvo('1 hour after', new f_datetime.TimeSpan(-1, 0, 0, 0), key),
                            gtvo('24 hours after',
                                new f_datetime.TimeSpan(-1, 0, 0, 0, 0), key)
                        ]
                    ];
                },
                genRelevanceFn: function() { return function() { return 1 } }
            };

            function mvo(modifier, key, facet) {
                var value = ComplexValue.build([key], [modifier]);
                return new SearchFacet.ValueOption(value, value.formatted(facet));
            }
            /**
            * getTimespanValueOption
            * @param {String} label to be translated
            * @param {f_datetime.TimeSpan} timeSpan to be subtracted from reference.
            * @param {Number} [reference=now] time in seconds to subtract from
            */
            function gtvo(label, timeSpan, reference) {
                var before = '',
                    after = '';
                if (reference === undefined) {
                    before = '>= ';
                    reference = f_datetime.dateToLocalSeconds(new Date());
		    var d = new Date();
		    reference += d.getTimezoneOffset()*60;
		    if(typeof origin_offset != 'undefined')
			reference += origin_offset;
		    if(typeof s2utc_offset != 'undefined')
			reference += s2utc_offset;
                } else if (timeSpan.getTime() < 0) {
                    before = reference + ' - ';
                } else if (timeSpan.getTime() > 0) {
                    after = ' - ' + reference;
                }
                var rDate = f_datetime.localSecondsToDate(reference),
                    seconds = reference - (timeSpan.getTime() * f_datetime.MS_TO_S);
                var cv = new ComplexValue(before + Math.round(seconds) + after, undefined, true);
                var vo = new SearchFacet.ValueOption(
                    cv, lang(label, [f_datetime.formatDateTime(rDate)])
                );
                vo.formatted = function(fn) {
                    return this.key.formatted(fn);
                };
                return vo;
            }
        }
    };
    return function(providers) {
        providers.$provide.service('facets', Facets);
    };
});
