/* global define */

define(['angular', 'notify', 'ng/ftnt', 'fweb'], function(angular, Notify, module, fweb) {
    'use strict';

    var PASSWORD_CHANGE_URI = '/api/v2/monitor/system/change-password';
    var PASSWORD_CHANGE_ID =  'passwordChange';

    function PasswordService($q, $http, $window, lang, slide, state, passwordPolicyService) {
        this.$q = $q;
        this.$http = $http;
        this.lang = lang;
        this.slide = slide;
        this.state = state;
        this.passwordPolicyService = passwordPolicyService;
        angular.element($window).on('message', function(event) {
            var e = event.originalEvent || event,
                data = e.data,
                origin = fweb.util.structure.getOrigin();

            if (e.origin === origin && data.messageType === this.OPEN_DIALOG_EVENT) {
                this.openPasswordDialog();
            }
        }.bind(this));
    }

    PasswordService.prototype._savePassword = function(model) {
        var dialogDfd = this.$q.defer();
        var data =  {
            mkey: model.admin.name,
            new_password: model.newPassword
        };
        if (model.oldPassword) {
            data.old_password = model.oldPassword.value;
        }
        this.$http({
            method: 'POST',
            url: PASSWORD_CHANGE_URI,
            data: data
        }).then(function(res) {
            var message;
            var results = res.data.results;
            if (results.change_status) {
                dialogDfd.resolve();
            } else {
                if (!results.valid) {
                    message = '-31';
                } else if (results.password_reused) {
                    message = 'Cannot reuse old passwords';
                }
                dialogDfd.reject(message);
            }
        }, function() {
            dialogDfd.reject();
        });
        return dialogDfd.promise;
    };

    // Require old password if any of the following conditions are met:
    //  1. fips_cc_enabled => security admin can change others password without confirmation
    //  2. Admin changing their own password
    //  3. Super admin trying to change another super admin's password
    PasswordService.prototype._requiresOldPassword = function(targetAdmin, isGuestAdmin) {
        var state = this.state,
            currentAdmin = state.admin,
            changeCurrentUserPwd = !targetAdmin ||
                                   (currentAdmin.name === targetAdmin.name),
            changeSuperAdminPwd = currentAdmin.super_admin &&
                                  (!!targetAdmin && targetAdmin['is-admin']);
        var policies = this.passwordPolicyService.policies;
        var policy = isGuestAdmin ? policies['guest-admin'] : policies.admin;
        if (policy.status === 'enable' && policy['change-4-characters'] === 'enable') {
            return true;
        }
        if (state.fips_cc_enabled && !changeCurrentUserPwd) {
            return false;
        }
        return changeCurrentUserPwd || changeSuperAdminPwd;
    };

    PasswordService.prototype.OPEN_DIALOG_EVENT = 'openPasswordDialog';

    PasswordService.prototype._openDialog = function(model) {
        this.slide.open(PASSWORD_CHANGE_ID, {
            template: '<password-changer></password-changer>',
            data: model,
            options: {
                title: this.lang('Edit Password').toString(),
                okButtonLabel: this.lang('confirm').toString(),
                fullHeight: true
            }
        });
    };

    PasswordService.prototype.openPasswordDialog = function(admin) {
        var model = {
            admin: admin || this.state.admin,
            newPassword: '',
            oldPassword: {
                value: '',
                valid: true
            },
            confirmPassword: ''
        };
        model.isGuestAdmin = !!admin ?
            admin['guest-auth'] === 'enable' : this.state.admin.guest_admin;

        this.passwordPolicyService.getAllPolicies().then(function() {
            if (this._requiresOldPassword(admin, model.isGuestAdmin)) {
                model.oldPasswordRequired = true;
            }
            this._openDialog(model);
        }.bind(this));
    };

    function PasswordPolicyService(CMDB, $q) {
        var ADMIN_POLICY = new CMDB('system', 'password-policy'),
            GUEST_ADMIN_POLICY = new CMDB('system', 'password-policy-guest-admin');

        this.getAllPolicies = function() {
            this.policies = {
                'admin': ADMIN_POLICY.fetch(),
                'guest-admin': GUEST_ADMIN_POLICY.fetch()
            };

            return $q.all([
                this.policies.admin.$promise,
                this.policies['guest-admin'].$promise
            ]);
        };
    }


    module.service('passwordService', PasswordService);
    module.service('passwordPolicyService', PasswordPolicyService);

    module.controller('PasswordChanger', function($scope, slide, lang, passwordService) {
        var current = slide.get(PASSWORD_CHANGE_ID);
        var controller = this;
        this.model = current.data;

        this.passwordPolicy = {
            status: 'disable',
            'apply-to': []
        };
        this.passwordPolicyIsActive = function(applyTo) {
            return this.passwordPolicy.status === 'enable' &&
                this.passwordPolicy['apply-to'].indexOf(applyTo) !== -1;
        };

        this.submit = function() {
            if (!this.form.$valid) {
                $scope.$broadcast('validation-msg_show');
                return;
            }
            return passwordService._savePassword(controller.model).then(function() {
                Notify.post(lang('Administrator password changed').toString(), 'success');
                current.slide.done();
            }, function(errorMessage) {
                controller.errorMessage = errorMessage || 'sslvpn_login_change_password_failed';
            });
        };
    });

    module.directive('passwordChanger', function(loader) {
        return {
            restrict: 'E',
            controller: 'PasswordChanger',
            controllerAs: 'ctrl',
            templateUrl: loader.base_path('/ng/admin/password/main.html')
        };
    });

});
