/*
 * This is a script to compute your browser's fingerprint
*/

(function(scope) {
    "use strict";

    var fontDetector;
    var uaData;

    var fingerprint = function() {
        fontDetector = new Detector();

        var parser = new(window.UAParser || exports.UAParser);
        uaData     = parser.getResult();

        return this;
    };
	
	function isStable(unstable_key){
		if (unstable_key == "Cookie" ||
			unstable_key == "MIME Types" ||
			unstable_key == "Java" ||
			unstable_key == "Flash" ||
			unstable_key == "Plugins" ||
			unstable_key == "Fonts" ||
			unstable_key == "User Agent" ||
			unstable_key == "Browser" ||
			unstable_key == "Engine") {
				return 0;
			}
		else {
			return 1;
		}
	};

    fingerprint.prototype = {
        // methods for language
        getLanguage: function() {
            return navigator.language || "N/A";
        },

        // methods for time
        getTimeZone: function() {
            var date   = new Date();
            var offset = date.getTimezoneOffset() / -60;
            if (offset < 0) {
                return "UTC" + offset;
            } else {
                return "UTC+" + offset;
            }
        },


        // methods for storage
        enabledLocalStorage: function() {
            try {
                return (!!scope.localStorage) ? "Enabled" : "Disabled";
            } catch (e) {
                // SecurityError means it exists
                return "Enabled";
            }
        },

        enabledSessionStorage: function() {
            try {
                return (!!scope.sessionStorage) ? "Enabled" : "Disabled";
            } catch (e) {
                // SecurityError means it exists
                return "Enabled";
            }
        },

        enabledIndexedDB: function () {
        	try {  //fix 如果firefox禁止cookie，无法运行该js
		   		 return (!!window.indexedDB) ? "Enabled" : "Disabled";
			} catch (e) {
				 return "Enabled";
			}
        },

        enabledOpenDatabase: function () {
			try {
	            return (!!window.openDatabase) ? "Enabled" : "Disabled";
    		} catch (e) {
				return "Enabled";
			}
	    },

        enabledCookie: function() {
            return navigator.cookieEnabled ? "Enabled" : "Disabled";
        },


        // methods for MIME
        getMimeTypes: function() {
            if (typeof navigator.mimeTypes === "undefined"
                || navigator.mimeTypes.length === 0) {
                return "N/A";
            }

            var types = "";
            for (var i = 0; i < navigator.mimeTypes.length; i++) {
                if (i == navigator.mimeTypes.length - 1) {
                    types += navigator.mimeTypes[i].description;
                } else {
                    types += navigator.mimeTypes[i].description + ", ";
                }
            }
            return types;
        },


        // methods for screen
        getCurrentResolution: function() {
            return screen.width + "x" + screen.height;
        },

        getAvailableResolution: function() {
            return screen.availWidth + "x" + screen.availHeight;
        },

        getPixelRatio: function() {
            try {
				return window.devicePixelRatio.toString(); }
			catch(e) { 
				return "N/A";
			}
        },

        getColorDepth: function() {
            return screen.colorDepth.toString();
        },

        // NOTE: device DPIs are ONLY supported by IE
        getDeviceXDPI: function() {
            return (typeof screen.deviceXDPI === "undefined") ?
                   "N/A" : screen.deviceXDPI.toString();
        },

        getDeviceYDPI: function() {
            return (typeof screen.deviceYDPI === "undefined") ?
                   "N/A" : screen.deviceYDPI.toString();
        },


        // methods for plugins
        enabledJava: function() {
            return navigator.javaEnabled() ? "Enabled" : "Disabled";
        },

        getJavaVersion: function() {
            var version = deployJava.getJREs();

            if (this.enabledJava() === "Enabled" && version.length > 0) {
                return version.toString();
            }

            return "N/A";
        },

        enabledFlash: function() {
            return (navigator.plugins["Shockwave Flash"]) ?
                   "Enabled" : "Disabled";
        },

        getFlashVersion: function() {
            if (this.enabledFlash() === "Enabled") {
                var version = swfobject.getFlashPlayerVersion();
                return version.major + "." +
                       version.minor + "." +
                       version.release;
            }

            return "N/A";
        },

        getPlugins: function() {
            if (typeof navigator.plugins === "undefined"
                || navigator.plugins.length === 0) {
                return "N/A";
            }

            var list = "";
            for (var i = 0; i < navigator.plugins.length; i++) {
                if (i == navigator.plugins.length - 1) {
                    list += navigator.plugins[i].name;
                } else {
                    list += navigator.plugins[i].name + ", ";
                }
            }
            return list;
        },


        // methods for extensions
        enabledADBlock: function() {
            var ads = document.createElement("div");
            ads.innerHTML = "&nbsp;";
            ads.className = "adsbox";
            var result = false;

            // body may not exist, that's why we need try/catch
            try {
                document.body.appendChild(ads);
                result = document.getElementsByClassName("adsbox")[0].offsetHeight === 0;
                document.body.removeChild(ads);
            } catch (e) {
                result = false;
            }

            return result ? "Enabled" : "Disabled";
        },


        // methods for canvas & webgl
        // XXX: how about the complicated method to compute canvas (fpjs2)?
        enabledCanvas: function() {
            var canvas = document.createElement("canvas");

            // try / catch for older browsers that don't support canvas
            try {
                return (!!(canvas.getContext && canvas.getContext("2d"))) ?
                       "Enabled" : "Disabled";
            } catch (e) {
                return "Disabled";
            }
        },

        getCanvasImage: function() {
            if (this.enabledCanvas() === "Enabled") {
                var canvas = document.createElement("canvas");
                var ctx    = canvas.getContext("2d");

                // https://www.browserleaks.com/canvas#how-does-it-work
                // text with lowercase/uppercase/punctuation symbols
                var txt = "~ FWB Fingerprint: !@#$%^&*() <canvas> 1.0 =";

                ctx.font         = "14px 'Arial'";
                ctx.textBaseline = "alphabetic";

                ctx.fillStyle = "#f60";
                ctx.fillRect(125, 1, 62, 20);

                ctx.fillStyle = "#069";
                ctx.fillText(txt, 2, 15);

                ctx.fillStyle = "rgba(102, 204, 0, 0.7)";
                ctx.fillText(txt, 4, 17);

                return canvas.toDataURL();
            }

            return "N/A";
        },

        enabledWebGL: function() {
            // code taken from Modernizr
            if (this.enabledCanvas() === "Disabled") {
                return "Disabled";
            }

            var canvas = document.createElement("canvas");
            var glContext;

            try {
                glContext = canvas.getContext &&
                            (canvas.getContext("webgl") ||
                             canvas.getContext("experimental-webgl"));
            } catch (e) {
                glContext = false;
            }

            return (!!window.WebGLRenderingContext && !!glContext) ?
                   "Enabled" : "Disabled";
        },

        getWebGLFP: function() {
            if (this.enabledWebGL() === "Disabled") {
                return "N/A";
            }

            var canvas = document.createElement("canvas");
            var gl     = canvas.getContext("webgl") ||
                         canvas.getContext("experimental-webgl");
            if (!gl) {
                return "N/A";
            }

            var fa2s = function(fa) {
                gl.clearColor(0.0, 0.0, 0.0, 1.0);
                gl.enable(gl.DEPTH_TEST);
                gl.depthFunc(gl.LEQUAL);
                gl.clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT);
                return "[" + fa[0] + ", " + fa[1] + "]";
            };

            var maxAnisotropy = function(gl) {
              var ext =
                gl.getExtension("EXT_texture_filter_anisotropic")        ||
                gl.getExtension("WEBKIT_EXT_texture_filter_anisotropic") ||
                gl.getExtension("MOZ_EXT_texture_filter_anisotropic");

              var anisotropy;
              return ext ?
                  (anisotropy =
                      gl.getParameter(ext.MAX_TEXTURE_MAX_ANISOTROPY_EXT),
                      0 === anisotropy && (anisotropy = 2),
                      anisotropy) : null;
            };

            // WebGL fingerprinting is a combination of techniques
            //     found in MaxMind antifraud script & Augur fingerprinting
            // First it draws a gradient object with shaders and convers
            //     the image to the Base64 string
            // Then it enumerates all WebGL extensions & capabilities
            //     and appends them to the Base64 string
            //     resulting in a huge WebGL string
            //     potentially very unique on each device
            // Since iOS supports webgl starting from version 8.1 and 8.1
            //     runs on several graphics chips
            //     the results may be different across ios devices
            //     but we need to verify it
            var result = [];
            var vShaderTemplate = "attribute vec2 attrVertex;"        +
                                  "varying vec2 varyinTexCoordinate;" +
                                  "uniform vec2 uniformOffset;"       +
                                  "void main(){varyinTexCoordinate="  +
                                  "attrVertex+uniformOffset;"         +
                                  "gl_Position=vec4(attrVertex,0,1);}";
            var fShaderTemplate = "precision mediump float;"          +
                                  "varying vec2 varyinTexCoordinate;" +
                                  "void main() {gl_FragColor="        +
                                  "vec4(varyinTexCoordinate,0,1);}";
            var vertexPosBuffer = gl.createBuffer();
            gl.bindBuffer(gl.ARRAY_BUFFER, vertexPosBuffer);
            var vertices = new Float32Array([-.2, -.9, 0, .4, -.26, 0, 0,
                                             .732134444, 0]);
            gl.bufferData(gl.ARRAY_BUFFER, vertices, gl.STATIC_DRAW);
            vertexPosBuffer.itemSize = 3;
            vertexPosBuffer.numItems = 3;
            var program = gl.createProgram();
            var vshader = gl.createShader(gl.VERTEX_SHADER);
            gl.shaderSource(vshader, vShaderTemplate);
            gl.compileShader(vshader);
            var fshader = gl.createShader(gl.FRAGMENT_SHADER);
            gl.shaderSource(fshader, fShaderTemplate);
            gl.compileShader(fshader);
            gl.attachShader(program, vshader);
            gl.attachShader(program, fshader);
            gl.linkProgram(program);
            gl.useProgram(program);
            program.vertexPosAttrib =
                gl.getAttribLocation(program, "attrVertex");
            program.offsetUniform =
                gl.getUniformLocation(program, "uniformOffset");
            gl.enableVertexAttribArray(program.vertexPosArray);
            gl.vertexAttribPointer(program.vertexPosAttrib,
                                   vertexPosBuffer.itemSize, gl.FLOAT,
                                   !1, 0, 0);
            gl.uniform2f(program.offsetUniform, 1, 1);
            gl.drawArrays(gl.TRIANGLE_STRIP, 0, vertexPosBuffer.numItems);
            if (gl.canvas != null) {
                result.push(gl.canvas.toDataURL());
            }

            result.push("extensions:"                             + gl.getSupportedExtensions().join(";"));
            result.push("webgl aliased line width range:"         + fa2s(gl.getParameter(gl.ALIASED_LINE_WIDTH_RANGE)));
            result.push("webgl aliased point size range:"         + fa2s(gl.getParameter(gl.ALIASED_POINT_SIZE_RANGE)));
            result.push("webgl alpha bits:"                       + gl.getParameter(gl.ALPHA_BITS));
            result.push("webgl antialiasing:"                     + (gl.getContextAttributes().antialias ? "yes" : "no"));
            result.push("webgl blue bits:"                        + gl.getParameter(gl.BLUE_BITS));
            result.push("webgl depth bits:"                       + gl.getParameter(gl.DEPTH_BITS));
            result.push("webgl green bits:"                       + gl.getParameter(gl.GREEN_BITS));
            result.push("webgl max anisotropy:"                   + maxAnisotropy(gl));
            result.push("webgl max combined texture image units:" + gl.getParameter(gl.MAX_COMBINED_TEXTURE_IMAGE_UNITS));
            result.push("webgl max cube map texture size:"        + gl.getParameter(gl.MAX_CUBE_MAP_TEXTURE_SIZE));
            result.push("webgl max fragment uniform vectors:"     + gl.getParameter(gl.MAX_FRAGMENT_UNIFORM_VECTORS));
            result.push("webgl max render buffer size:"           + gl.getParameter(gl.MAX_RENDERBUFFER_SIZE));
            result.push("webgl max texture image units:"          + gl.getParameter(gl.MAX_TEXTURE_IMAGE_UNITS));
            result.push("webgl max texture size:"                 + gl.getParameter(gl.MAX_TEXTURE_SIZE));
            result.push("webgl max varying vectors:"              + gl.getParameter(gl.MAX_VARYING_VECTORS));
            result.push("webgl max vertex attribs:"               + gl.getParameter(gl.MAX_VERTEX_ATTRIBS));
            result.push("webgl max vertex texture image units:"   + gl.getParameter(gl.MAX_VERTEX_TEXTURE_IMAGE_UNITS));
            result.push("webgl max vertex uniform vectors:"       + gl.getParameter(gl.MAX_VERTEX_UNIFORM_VECTORS));
            result.push("webgl max viewport dims:"                + fa2s(gl.getParameter(gl.MAX_VIEWPORT_DIMS)));
            result.push("webgl red bits:"                         + gl.getParameter(gl.RED_BITS));
            result.push("webgl renderer:"                         + gl.getParameter(gl.RENDERER));
            result.push("webgl shading language version:"         + gl.getParameter(gl.SHADING_LANGUAGE_VERSION));
            result.push("webgl stencil bits:"                     + gl.getParameter(gl.STENCIL_BITS));
            result.push("webgl vendor:"                           + gl.getParameter(gl.VENDOR));
            result.push("webgl version:"                          + gl.getParameter(gl.VERSION));
            if (!gl.getShaderPrecisionFormat) {
 //               alert("incomplete WebGL: the browser does not support getShaderPrecisionFormat");
                return result.join("~");
            }

            result.push("webgl vertex shader high float precision:"              + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.HIGH_FLOAT).precision);
            result.push("webgl vertex shader high float precision rangeMin:"     + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.HIGH_FLOAT).rangeMin);
            result.push("webgl vertex shader high float precision rangeMax:"     + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.HIGH_FLOAT).rangeMax);
            result.push("webgl vertex shader medium float precision:"            + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.MEDIUM_FLOAT).precision);
            result.push("webgl vertex shader medium float precision rangeMin:"   + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.MEDIUM_FLOAT).rangeMin);
            result.push("webgl vertex shader medium float precision rangeMax:"   + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.MEDIUM_FLOAT).rangeMax);
            result.push("webgl vertex shader low float precision:"               + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.LOW_FLOAT).precision);
            result.push("webgl vertex shader low float precision rangeMin:"      + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.LOW_FLOAT).rangeMin);
            result.push("webgl vertex shader low float precision rangeMax:"      + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.LOW_FLOAT).rangeMax);
            result.push("webgl fragment shader high float precision:"            + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.HIGH_FLOAT).precision);
            result.push("webgl fragment shader high float precision rangeMin:"   + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.HIGH_FLOAT).rangeMin);
            result.push("webgl fragment shader high float precision rangeMax:"   + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.HIGH_FLOAT).rangeMax);
            result.push("webgl fragment shader medium float precision:"          + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.MEDIUM_FLOAT).precision);
            result.push("webgl fragment shader medium float precision rangeMin:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.MEDIUM_FLOAT).rangeMin);
            result.push("webgl fragment shader medium float precision rangeMax:" + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.MEDIUM_FLOAT).rangeMax);
            result.push("webgl fragment shader low float precision:"             + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.LOW_FLOAT).precision);
            result.push("webgl fragment shader low float precision rangeMin:"    + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.LOW_FLOAT).rangeMin);
            result.push("webgl fragment shader low float precision rangeMax:"    + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.LOW_FLOAT).rangeMax);
            result.push("webgl vertex shader high int precision:"                + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.HIGH_INT).precision);
            result.push("webgl vertex shader high int precision rangeMin:"       + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.HIGH_INT).rangeMin);
            result.push("webgl vertex shader high int precision rangeMax:"       + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.HIGH_INT).rangeMax);
            result.push("webgl vertex shader medium int precision:"              + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.MEDIUM_INT).precision);
            result.push("webgl vertex shader medium int precision rangeMin:"     + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.MEDIUM_INT).rangeMin);
            result.push("webgl vertex shader medium int precision rangeMax:"     + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.MEDIUM_INT).rangeMax);
            result.push("webgl vertex shader low int precision:"                 + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.LOW_INT).precision);
            result.push("webgl vertex shader low int precision rangeMin:"        + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.LOW_INT).rangeMin);
            result.push("webgl vertex shader low int precision rangeMax:"        + gl.getShaderPrecisionFormat(gl.VERTEX_SHADER, gl.LOW_INT).rangeMax);
            result.push("webgl fragment shader high int precision:"              + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.HIGH_INT).precision);
            result.push("webgl fragment shader high int precision rangeMin:"     + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.HIGH_INT).rangeMin);
            result.push("webgl fragment shader high int precision rangeMax:"     + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.HIGH_INT).rangeMax);
            result.push("webgl fragment shader medium int precision:"            + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.MEDIUM_INT).precision);
            result.push("webgl fragment shader medium int precision rangeMin:"   + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.MEDIUM_INT).rangeMin);
            result.push("webgl fragment shader medium int precision rangeMax:"   + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.MEDIUM_INT).rangeMax);
            result.push("webgl fragment shader low int precision:"               + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.LOW_INT).precision);
            result.push("webgl fragment shader low int precision rangeMin:"      + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.LOW_INT).rangeMin);
            result.push("webgl fragment shader low int precision rangeMax:"      + gl.getShaderPrecisionFormat(gl.FRAGMENT_SHADER, gl.LOW_INT).rangeMax);
            return result.join("~");
        },


        // methods for fonts
        // XXX: useless?
        enabledFont: function(font) {
            return (fontDetector.detect(font)) ? "Enabled" : "Disabled";
        },

        // XXX: it's too slow!
        getFonts: function() {
            var fontArray = [
                "Abadi MT Condensed Light", "Adobe Fangsong Std",
                "Adobe Hebrew", "Adobe Ming Std", "Agency FB", "Aharoni",
                "Andalus", "Angsana New", "AngsanaUPC", "Aparajita", "Arab",
                "Arabic Transparent", "Arabic Typesetting", "Arial Baltic",
                "Arial Black", "Arial CE", "Arial CYR", "Arial Greek",
                "Arial TUR", "Arial", "Batang", "BatangChe", "Bauhaus 93",
                "Bell MT", "Bitstream Vera Serif", "Bodoni MT",
                "Bookman Old Style", "Braggadocio", "Broadway",
                "Browallia New", "BrowalliaUPC", "Calibri Light", "Calibri",
                "Californian FB", "Cambria Math", "Cambria", "Candara",
                "Castellar", "Casual", "Centaur", "Century Gothic",
                "Chalkduster", "Colonna MT", "Comic Sans MS", "Consolas",
                "Constantia", "Copperplate Gothic Light", "Corbel",
                "Cordia New", "CordiaUPC", "Courier New Baltic",
                "Courier New CE", "Courier New CYR", "Courier New Greek",
                "Courier New TUR", "Courier New", "DFKai-SB", "DaunPenh",
                "David", "DejaVu LGC Sans Mono", "Desdemona", "DilleniaUPC",
                "DokChampa", "Dotum", "DotumChe", "Ebrima", "Engravers MT",
                "Eras Bold ITC", "Estrangelo Edessa", "EucrosiaUPC",
                "Euphemia", "Eurostile", "FangSong", "Forte", "FrankRuehl",
                "Franklin Gothic Heavy", "Franklin Gothic Medium",
                "FreesiaUPC", "French Script MT", "Gabriola", "Gautami",
                "Georgia", "Gigi", "Gisha", "Goudy Old Style", "Gulim",
                "GulimChe", "GungSeo", "Gungsuh", "GungsuhChe",
                "Haettenschweiler", "Harrington", "Hei S", "HeiT",
                "Heisei Kaku Gothic", "Hiragino Sans GB", "Impact",
                "Informal Roman", "IrisUPC", "Iskoola Pota", "JasmineUPC",
                "KacstOne", "KaiTi", "Kalinga", "Kartika", "Khmer UI",
                "Kino MT", "KodchiangUPC", "Kokila", "Kozuka Gothic Pr6N",
                "Lao UI", "Latha", "Leelawadee", "Levenim MT", "LilyUPC",
                "Lohit Gujarati", "Loma", "Lucida Bright", "Lucida Console",
                "Lucida Fax", "Lucida Sans Unicode", "MS Gothic", "MS Mincho",
                "MS PGothic", "MS PMincho", "MS Reference Sans Serif",
                "MS UI Gothic", "MV Boli", "Magneto", "Malgun Gothic",
                "Mangal", "Marlett", "Matura MT Script Capitals", "Meiryo UI",
                "Meiryo", "Menlo", "Microsoft Himalaya", "Microsoft JhengHei",
                "Microsoft New Tai Lue", "Microsoft PhagsPa",
                "Microsoft Sans Serif", "Microsoft Tai Le",
                "Microsoft Uighur", "Microsoft YaHei", "Microsoft Yi Baiti",
                "MingLiU", "MingLiU-ExtB", "MingLiU_HKSCS",
                "MingLiU_HKSCS-ExtB", "Miriam Fixed", "Miriam",
                "Mongolian Baiti", "MoolBoran", "NSimSun", "Narkisim",
                "News Gothic MT", "Niagara Solid", "Nyala", "PMingLiU",
                "PMingLiU-ExtB", "Palace Script MT", "Palatino Linotype",
                "Papyrus", "Perpetua", "Plantagenet Cherokee", "Playbill",
                "Prelude Bold", "Prelude Condensed Bold",
                "Prelude Condensed Medium", "Prelude Medium",
                "PreludeCompressedWGL Black", "PreludeCompressedWGL Bold",
                "PreludeCompressedWGL Light", "PreludeCompressedWGL Medium",
                "PreludeCondensedWGL Black", "PreludeCondensedWGL Bold",
                "PreludeCondensedWGL Light", "PreludeCondensedWGL Medium",
                "PreludeWGL Black", "PreludeWGL Bold", "PreludeWGL Light",
                "PreludeWGL Medium", "Raavi", "Rachana", "Rockwell", "Rod",
                "Sakkal Majalla", "Sawasdee", "Script MT Bold", "Segoe Print",
                "Segoe Script", "Segoe UI Light", "Segoe UI Semibold",
                "Segoe UI Symbol", "Segoe UI", "Shonar Bangla",
                "Showcard Gothic", "Shruti", "SimHei", "SimSun",
                "SimSun-ExtB", "Simplified Arabic Fixed", "Simplified Arabic",
                "Snap ITC", "Sylfaen", "Symbol", "Tahoma",
                "Times New Roman Baltic", "Times New Roman CE",
                "Times New Roman CYR", "Times New Roman Greek",
                "Times New Roman TUR", "Times New Roman", "TlwgMono",
                "Traditional Arabic", "Trebuchet MS", "Tunga",
                "Tw Cen MT Condensed Extra Bold", "Ubuntu", "Umpush",
                "Univers", "Utopia", "Utsaah", "Vani", "Verdana", "Vijaya",
                "Vladimir Script", "Vrinda", "Webdings", "Wide Latin",
                "Wingdings"];

            var fonts = "";
            for (var i = 0; i < fontArray.length; i++) {
                if (fontDetector.detect(fontArray[i])) {
                    if (i == fontArray.length - 1) {
                        fonts += fontArray[i];
                    } else {
                        fonts += fontArray[i] + ", ";
                    }
                }
            }

            return fonts;
        },


        // methods for OS
        getCPUClass: function() {
            return (navigator.cpuClass) ? navigator.cpuClass : "N/A";
        },

        getPlatform: function() {
            return (navigator.platform) ? navigator.platform : "N/A";
        },


        // methods for others
        enabledDoNotTrack: function() {
            return navigator.doNotTrack ? "Enabled" : "Disabled";
        },

        enabledAddBehavior: function() {
            // body might not exist
            return (document.body && document.body.addBehavior) ?
                   "Enabled" : "Disabled";
        },


        // NOTE: methods for those parsed from User Agent have a "UA" mark

        getUAData: function() {
            // an array that includes ua, browser, engine, os, device, cpu
            return uaData;
        },

        // ONLY User Agent itself
        getUAUserAgent: function() {
            return uaData.ua;
        },

        getUAUserAgentLowerCase: function() {
            return this.getUAUserAgent().toLowerCase();
        },

        // ONLY browser
        getUABrowser: function() {
            return uaData.browser.name;
        },

        getUABrowserVersion: function() {
            return uaData.browser.version;
        },

        getUABrowserMajorVersion: function() {
            return uaData.browser.major;
        },

        isUAIE: function() {
            return (/IE/i.test(this.getUABrowser()));
        },

        isUAChrome: function() {
            return (/Chrome/i.test(this.getUABrowser()));
        },

        isUAFirefox: function() {
            return (/Firefox/i.test(this.getUABrowser()));
        },

        isUASafari: function() {
            return (/Safari/i.test(this.getUABrowser()));
        },

        isUAMobileSafari: function() {
            return (/Mobile\sSafari/i.test(this.getUABrowser()));
        },

        isUAOpera: function() {
            return (/Opera/i.test(this.getUABrowser()));
        },

        // ONLY engine
        getUAEngine: function() {
            return uaData.engine.name;
        },

        getUAEngineVersion: function() {
            return uaData.engine.version;
        },

        // ONLY OS
        getUAOS: function() {
            return uaData.os.name;
        },

        getUAOSVersion: function() {
            return uaData.os.version;
        },

        isUAWindows: function() {
            return (/Windows/i.test(this.getUAOS()));
        },

        isUAMac: function() {
            return (/Mac/i.test(this.getUAOS()));
        },

        isUALinux: function() {
            return (/Linux/i.test(this.getUAOS()));
        },

        isUAUbuntu: function() {
            return (/Ubuntu/i.test(this.getUAOS()));
        },

        isUASolaris: function() {
            return (/Solaris/i.test(this.getUAOS()));
        },

        // ONLY CPU
        getUACPU: function() {
            return (uaData.cpu.architecture) ?
                   uaData.cpu.architecture : "N/A";
        },

        // mobile check
        isUAMobile: function() {
          // detectmobilebrowsers.com JavaScript Mobile Detection Script
          var dataString =
              this.getUAUserAgent() || navigator.vendor || window.opera;
          return (/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|mobile.+firefox|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows (ce|phone)|xda|xiino/i.test(dataString) || /1207|6310|6590|3gso|4thp|50[1-6]i|770s|802s|a wa|abac|ac(er|oo|s\-)|ai(ko|rn)|al(av|ca|co)|amoi|an(ex|ny|yw)|aptu|ar(ch|go)|as(te|us)|attw|au(di|\-m|r |s )|avan|be(ck|ll|nq)|bi(lb|rd)|bl(ac|az)|br(e|v)w|bumb|bw\-(n|u)|c55\/|capi|ccwa|cdm\-|cell|chtm|cldc|cmd\-|co(mp|nd)|craw|da(it|ll|ng)|dbte|dc\-s|devi|dica|dmob|do(c|p)o|ds(12|\-d)|el(49|ai)|em(l2|ul)|er(ic|k0)|esl8|ez([4-7]0|os|wa|ze)|fetc|fly(\-|_)|g1 u|g560|gene|gf\-5|g\-mo|go(\.w|od)|gr(ad|un)|haie|hcit|hd\-(m|p|t)|hei\-|hi(pt|ta)|hp( i|ip)|hs\-c|ht(c(\-| |_|a|g|p|s|t)|tp)|hu(aw|tc)|i\-(20|go|ma)|i230|iac( |\-|\/)|ibro|idea|ig01|ikom|im1k|inno|ipaq|iris|ja(t|v)a|jbro|jemu|jigs|kddi|keji|kgt( |\/)|klon|kpt |kwc\-|kyo(c|k)|le(no|xi)|lg( g|\/(k|l|u)|50|54|\-[a-w])|libw|lynx|m1\-w|m3ga|m50\/|ma(te|ui|xo)|mc(01|21|ca)|m\-cr|me(rc|ri)|mi(o8|oa|ts)|mmef|mo(01|02|bi|de|do|t(\-| |o|v)|zz)|mt(50|p1|v )|mwbp|mywa|n10[0-2]|n20[2-3]|n30(0|2)|n50(0|2|5)|n7(0(0|1)|10)|ne((c|m)\-|on|tf|wf|wg|wt)|nok(6|i)|nzph|o2im|op(ti|wv)|oran|owg1|p800|pan(a|d|t)|pdxg|pg(13|\-([1-8]|c))|phil|pire|pl(ay|uc)|pn\-2|po(ck|rt|se)|prox|psio|pt\-g|qa\-a|qc(07|12|21|32|60|\-[2-7]|i\-)|qtek|r380|r600|raks|rim9|ro(ve|zo)|s55\/|sa(ge|ma|mm|ms|ny|va)|sc(01|h\-|oo|p\-)|sdk\/|se(c(\-|0|1)|47|mc|nd|ri)|sgh\-|shar|sie(\-|m)|sk\-0|sl(45|id)|sm(al|ar|b3|it|t5)|so(ft|ny)|sp(01|h\-|v\-|v )|sy(01|mb)|t2(18|50)|t6(00|10|18)|ta(gt|lk)|tcl\-|tdg\-|tel(i|m)|tim\-|t\-mo|to(pl|sh)|ts(70|m\-|m3|m5)|tx\-9|up(\.b|g1|si)|utst|v400|v750|veri|vi(rg|te)|vk(40|5[0-3]|\-v)|vm40|voda|vulc|vx(52|53|60|61|70|80|81|83|85|98)|w3c(\-| )|webc|whit|wi(g |nc|nw)|wmlb|wonu|x700|yas\-|your|zeto|zte\-/i.test(dataString.substr(0, 4)));
        },

        isUAMobileAndroid: function() {
            if (this.getUAUserAgent().match(/Android/i)) {
                return true;
            }
            return false;
        },

        isUAMobileOpera: function() {
            if (this.getUAUserAgent().match(/Opera Mini/i)) {
                return true;
            }
            return false;
        },

        isUAMobileWindows: function() {
            if (this.getUAUserAgent().match(/IEMobile/i)) {
                return true;
            }
            return false;
        },

        isUAMobileBlackBerry: function() {
            if (this.getUAUserAgent().match(/BlackBerry/i)) {
                return true;
            }
            return false;
        },

        // mobile Apple devices
        isUAMobileIOS: function() {
            if (this.getUAUserAgent().match(/iPhone|iPad|iPod/i)) {
                return true;
            }
            return false;
        },

        isUAIphone: function() {
            if (this.getUAUserAgent().match(/iPhone/i)) {
                return true;
            }
            return false;
        },

        isUAIpad: function() {
            if (this.getUAUserAgent().match(/iPad/i)) {
                return true;
            }
            return false;
        },

        isUAIpod: function() {
            if (this.getUAUserAgent().match(/iPod/i)) {
                return true;
            }
            return false;
        },

        getUAMobileInfo: function() {
            if (this.isUAMobileAndroid()) {
                return "Mobile Android";
            }

            if (this.isUAMobileOpera()) {
                return "Mobile Opera";
            }

            if (this.isUAMobileWindows()) {
                return "Mobile Windows";
            }

            if (this.isUAMobileBlackBerry()) {
                return "Mobile BlackBerry";
            }

            if (this.isUAMobileIOS()) {
                var type = "IOS";

                if (this.isUAIphone()) {
                    type += " (Iphone)";
                } else if (this.isUAIpad()) {
                    type += " (Ipad)";
                } else if (this.isUAIpod()) {
                    type += " (Ipod)";
                }

                return type;
            }

            return "N/A";
        },


        // methods for validation
        hasFakeLanguages: function() {
            // check if navigator.language is equal to
            //     the first language of navigator.languages
            if (navigator.languages) {
                try {
                    var first = navigator.languages[0].substr(0, 2);
                    if (first !== navigator.language.substr(0, 2)) {
                      return "Yes";
                    }
                } catch (e) {
                    return "Yes";
                }
            }

            return "No";
        },

        hasFakeResolution: function() {
            return (screen.width < screen.availWidth ||
                    screen.height < screen.availHeight) ? "Yes" : "No";
        },

        hasFakeOS: function() {
            var ua       = (navigator.userAgent) ?
                           navigator.userAgent.toLowerCase() : "";
            var platform = (navigator.platform) ?
                           navigator.platform.toLowerCase() : "";
            var oscpu    = (navigator.oscpu) ?
                           navigator.oscpu.toLowerCase() : "";
            var os;

            // extract the OS from UA (respect the order)
            if (ua.indexOf("windows phone") >= 0) {
                os = "Windows Phone";
            } else if (ua.indexOf("win") >= 0) {
                os = "Windows";
            } else if (ua.indexOf("android") >= 0) {
                os = "Android";
            } else if (ua.indexOf("linux") >= 0) {
                os = "Linux";
            } else if (ua.indexOf("iphone") >= 0 ||
                       ua.indexOf("ipad") >= 0   ||
                       ua.indexOf("ipod") >= 0) {
                os = "iOS";
            } else if (ua.indexOf("mac") >= 0) {
                os = "Mac";
            } else {
                os = "Other";
            }

            // whether the person uses a mobile device
            var mobileDevice;
            if (("ontouchstart" in window)       ||
                (navigator.maxTouchPoints   > 0) ||
                (navigator.msMaxTouchPoints > 0)) {
                mobileDevice = true;
            } else {
                mobileDevice = false;
            }

            if (mobileDevice && (os !== "Windows Phone"
                                 && os !== "Android"
                                 && os !== "iOS"
                                 && os !== "Other")) {
                return "Yes";
            }

            // compare oscpu with the OS extracted from UA
            if (oscpu) {
                if (oscpu.indexOf("win") >= 0
                    && (os !== "Windows" && os !== "Windows Phone")) {
                    return "Yes";
                } else if (oscpu.indexOf("linux") >= 0
                           && (os !== "Linux" && os !== "Android")) {
                    return "Yes";
                } else if (oscpu.indexOf("mac") >= 0
                           && (os !== "Mac" && os !== "iOS")) {
                    return "Yes";
                } else if (oscpu.indexOf("win") === 0
                           && oscpu.indexOf("linux") === 0
                           && oscpu.indexOf("mac") >= 0 && os !== "other") {
                    return "Yes";
                }
            }

            // compare platform with the OS extracted from the UA
            if (platform) {
                if (platform.indexOf("win") >= 0
                    && (os !== "Windows" && os !== "Windows Phone")) {
                    return "Yes";
                } else if ((platform.indexOf("linux") >= 0
                            || platform.indexOf("android") >= 0
                            || platform.indexOf("pike") >= 0)
                           && os !== "Linux" && os !== "Android") {
                    return "Yes";
                } else if((platform.indexOf("mac") >= 0
                           || platform.indexOf("ipad") >= 0
                           || platform.indexOf("ipod") >= 0
                           || platform.indexOf("iphone") >= 0)
                          && os !== "Mac" && os !== "iOS") {
                    return "Yes";
                } else if (platform.indexOf("win") === 0
                           && platform.indexOf("linux") === 0
                           && platform.indexOf("mac") >= 0
                           && os !== "other") {
                    return "Yes";
                }
            }

            if (typeof navigator.plugins === "undefined"
                && (os !== "Windows" && os !== "Windows Phone")) {
                  // in the case where IE is used
                  // therefore we can infer that it's Windows
                return "Yes";
            }

            return "No";
        },

        hasFakeBrowser: function() {
            var ua         = (navigator.userAgent) ?
                             navigator.userAgent.toLowerCase() : "";
            var productSub = navigator.productSub;

            // extract the browser from UA (respect the order)
            var browser;
            if (ua.indexOf("firefox") >= 0) {
                browser = "Firefox";
            } else if (ua.indexOf("opera") >= 0 || ua.indexOf("opr") >= 0) {
                browser = "Opera";
            } else if (ua.indexOf("chrome") >= 0) {
                browser = "Chrome";
            } else if (ua.indexOf("safari") >= 0) {
                browser = "Safari";
            } else if (ua.indexOf("trident") >= 0) {
                browser = "Internet Explorer";
            } else {
                browser = "Other";
            }

            if ((browser === "Chrome"
                 || browser === "Safari"
                 || browser === "Opera")
                && productSub !== "20030107") {
                return "Yes";
            }

            var code_block = eval.toString().length;
            if(code_block === 37 && (browser !== "Safari"
                                 && browser !== "Firefox"
                                 && browser !== "Other")) {
                return "Yes";
            } else if (code_block === 39 && (browser !== "Internet Explorer"
                                         && browser !== "Other")) {
                return "Yes";
            } else if (code_block === 33 && (browser !== "Chrome"
                                         && browser !== "Opera"
                                         && browser !== "Other")) {
                return "Yes";
            }

            // create an error to see how it works
            var errFirefox;
            try {
                throw "err";
            } catch (e) {
                try {
                    err.toSource();
                    errFirefox = true;
                } catch (e4e) {
                    errFirefox = false;
                }
            }

            if (errFirefox && (browser !== "Firefox"
                               && browser !== "Other")) {
                return "Yes";
            }

            return "No";
        },


        // collect all items
        getAllItems: function() {
            var items = new Object();

            // language
            items["Language"]         = this.getLanguage();

            // time
            items["Time Zone"] = this.getTimeZone();

            // storage
            items["Local Storage"]   = this.enabledLocalStorage();
            items["Session Storage"] = this.enabledSessionStorage();
            items["Indexed DB"]      = this.enabledIndexedDB();
            items["Open Database"]   = this.enabledOpenDatabase();
            items["Cookie"]          = this.enabledCookie();

            // MIME
            items["MIME Types"] = this.getMimeTypes();

            // screen
            items["Current Resolution"]   = this.getCurrentResolution();
            items["Available Resolution"] = this.getAvailableResolution();
            items["Pixel Ratio"]          = this.getPixelRatio();
            items["Color Depth"]          = this.getColorDepth();
            items["Device XDPI"]          = this.getDeviceXDPI();
            items["Device YDPI"]          = this.getDeviceYDPI();

            // plugins
            items["Java"]        = this.enabledJava() + "(" +
                                   this.getJavaVersion() + ")";
            items["Flash"]       = this.enabledFlash() + "(" +
                                   this.getFlashVersion() + ")";
            items["Plugins"]     = this.getPlugins();

            // extensions
            items["AD Block"] = this.enabledADBlock();

            // canvas
            items["Canvas"]       = this.enabledCanvas();
            items["Canvas Image"] = this.getCanvasImage();

            // WebGL
            items["WebGL"]    = this.enabledWebGL();
            items["WebGL FP"] = this.getWebGLFP();

            // fonts
            items["Fonts"] = this.getFonts();

            // OS
            items["CPU Class"]    = this.getCPUClass();
            items["Platform"]     = this.getPlatform();

            // others
            items["Do Not Track"] = this.enabledDoNotTrack();
            items["Add Behavior"] = this.enabledAddBehavior();

            // UA data
            // XXX: how about getUAUserAgentLowerCase
            items["User Agent"] = this.getUAUserAgent();
            items["Browser"]    = this.getUABrowser() + "(" +
                                  this.getUABrowserVersion() + ")";
            items["Engine"]     = this.getUAEngine() + "(" +
                                  this.getUAEngineVersion() + ")";
            items["OS"]         = this.getUAOS() + "(" +
                                  this.getUAOSVersion() + ")";
            items["CPU"]        = this.getUACPU();
            items["Mobile"]     = this.getUAMobileInfo();

            // validation
            items["Fake Languages"]  = this.hasFakeLanguages();
            items["Fake Resolution"] = this.hasFakeResolution();
            items["Fake OS"]         = this.hasFakeOS();
            items["Fake Browser"]    = this.hasFakeBrowser();

            return items;
        },

        // collect POST items
        getPostItems: function() {
            var items = new Object();
			
			var digestItems = new msgDigest();

            // language
            items["Language"]         = this.getLanguage();

            // time
            items["Time Zone"] = this.getTimeZone();

            // screen
            items["Current Resolution"]   = this.getCurrentResolution();
            items["Color Depth"]          = this.getColorDepth();


            // canvas
            items["Canvas"]       = digestItems.computeMD5(this.getCanvasImage());

            // WebGL
            items["WebGL"]    = digestItems.computeMD5(this.getWebGLFP());

            // UA data
            // XXX: how about getUAUserAgentLowerCase
            items["Browser"]    = this.getUABrowser();
            items["OS"]         = this.getUAOS() + "(" +
                                  this.getUAOSVersion() + ")";
            items["CPU"]        = this.getUACPU();

            return items;
        },
		
        // compute the signature
        getFingerprint: function() {
            var items = this.getAllItems();

            var src = "";
			var stable_src = "";
            var sep = "\x11";
            var indicator = ": ";
            var first = 1;
            for (var key in items) {
				if (items["Fake Languages"] == "Yes" && key == "Language") {
					continue;
				}
				if (items["Fake Resolution"] == "Yes" && key == "Current Resolution") {
					continue;
				}
				if (items["Fake Resolution"] == "Yes" && key == "Available Resolution") {
					continue;
				}
				if (items["Fake OS"] == "Yes" && key == "OS") {
					continue;
				}
				if (items["Fake Browser"] == "Yes" && key == "Browser") {
					continue;
				}
				if (key == "Fake Languages" || key == "Fake Resolution" 
					|| key == "Fake OS" || key == "Fake Browser") {
					continue;
				}
                if (first) {
                    src  += key + indicator + items[key];
					if(isStable(key)){
						stable_src += key + indicator + items[key];
					}
                    first = 0;
                } else {
                    src += sep + key + indicator + items[key];
					if(isStable(key)){
						stable_src += key + indicator + items[key];
					}
                }
            }
//            alert("raw msg before digesting: " + src);

            var digestTool = new msgDigest();

            var digests = {
//                "MurmurHash3-32"  : digestTool.computeMurmurHash3(src, 32),
//              "MurmurHash3-128" : digestTool.computeMurmurHash3(src, 128),
                "MD5"             : digestTool.computeMD5(src),
				"Stable_MD5"      : digestTool.computeMD5(stable_src)
//                "SHA1"            : digestTool.computeSHA1(src),
//               "SHA256"          : digestTool.computeSHA256(src),
 //               "SHA256"          : digestTool.computeSHA256("duancanran"),
            };
            return digests;
        }
    };

    // export it
    if (typeof module === "object" && typeof exports !== "undefined") {
        module.exports = fingerprint;
    }
    scope.fingerprint = fingerprint;
})(window);

