/*
 * This is a script to compute your browser's fingerprint
 * including:
 *      Murmur Hash 3 (x86-32bit)
 *      Murmur Hash 3 (x64-128bit)
 *      MD5
 *      SHA1
 *      SHA256
*/

(function(scope) {
    "use strict";

    var msgDigest = function() {
        return this;
    };

    msgDigest.prototype = {
        /****************************************************************
        * Murmur Hash 3 (32bit): https://github.com/jackspirou/clientjs *
        ****************************************************************/

        // 32-bit positive integer hash will be returned
        _innerMurmurHash3_32: function(src) {
            var seed = 256; // fixed

            var remainder, bytes, h1, h1b, c1, c1b, c2, c2b, k1, i;
            remainder = src.length & 3; // src.length % 4
            bytes     = src.length - remainder;
            h1        = seed;
            c1        = 0xCC9E2D51;
            c2        = 0x1B873593;
            i         = 0;

            while (i < bytes) {
                k1 =   (src.charCodeAt(i)    & 0xFF)
                     | ((src.charCodeAt(++i) & 0xFF) << 8)
                     | ((src.charCodeAt(++i) & 0xFF) << 16)
                     | ((src.charCodeAt(++i) & 0xFF) << 24);
                ++i;

                k1 = (((k1 & 0xFFFF) * c1)
                      + ((((k1 >>> 16) * c1) & 0xFFFF) << 16)) & 0xFFFFFFFF;
                k1 = (k1 << 15) | (k1 >>> 17);
                k1 = (((k1 & 0xFFFF) * c2)
                      + ((((k1 >>> 16) * c2) & 0xFFFF) << 16)) & 0xFFFFFFFF;

                h1 ^= k1;
                h1 = (h1 << 13) | (h1 >>> 19);
                h1b = (((h1 & 0xFFFF) * 5)
                       + ((((h1 >>> 16) * 5) & 0xFFFF) << 16)) & 0xFFFFFFFF;
                h1 = (((h1b & 0xFFFF) + 0x6B64)
                      + ((((h1b >>> 16) + 0xE654) & 0xFFFF) << 16));
            }

            k1 = 0;

            switch (remainder) {
                case 3: k1 ^= (src.charCodeAt(i + 2) & 0xFF) << 16;
                case 2: k1 ^= (src.charCodeAt(i + 1) & 0xFF) << 8;
                case 1: k1 ^= (src.charCodeAt(i)     & 0xFF);

                k1 = (((k1 & 0xFFFF) * c1)
                      + ((((k1 >>> 16) * c1) & 0xFFFF) << 16)) & 0xFFFFFFFF;
                k1 = (k1 << 15) | (k1 >>> 17);
                k1 = (((k1 & 0xFFFF) * c2)
                      + ((((k1 >>> 16) * c2) & 0xFFFF) << 16)) & 0xFFFFFFFF;
                h1 ^= k1;
            }

            h1 ^= src.length;
            h1 ^= h1 >>> 16;
            h1  = (((h1 & 0xFFFF) * 0x85EBCA6B) +
                   ((((h1 >>> 16) * 0x85EBCA6B) & 0xFFFF) << 16))
                  & 0xFFFFFFFF;
            h1 ^= h1 >>> 13;
            h1  = (((h1 & 0xFFFF) * 0xC2B2AE35) +
                   ((((h1 >>> 16) * 0xC2B2AE35) & 0xFFFF) << 16))
                  & 0xFFFFFFFF;
            h1 ^= h1 >>> 16;

            return h1 >>> 0;
        },


        /**************************************************
        * Murmur Hash 3 (128bit):                         *
        *     https://github.com/Valve/fingerprintjs2.git *
        **************************************************/

        // basic interfaces
        _x64Add: function(m, n) {
            m = [m[0] >>> 16, m[0] & 0xFFFF, m[1] >>> 16, m[1] & 0xFFFF];
            n = [n[0] >>> 16, n[0] & 0xFFFF, n[1] >>> 16, n[1] & 0xFFFF];

            var o = [0, 0, 0, 0];
            o[3] += m[3] + n[3];
            o[2] += o[3] >>> 16;
            o[3] &= 0xFFFF;
            o[2] += m[2] + n[2];
            o[1] += o[2] >>> 16;
            o[2] &= 0xFFFF;
            o[1] += m[1] + n[1];
            o[0] += o[1] >>> 16;
            o[1] &= 0xFFFF;
            o[0] += m[0] + n[0];
            o[0] &= 0xFFFF;

            return [(o[0] << 16) | o[1], (o[2] << 16) | o[3]];
        },

        _x64Multiply: function(m, n) {
            m = [m[0] >>> 16, m[0] & 0xFFFF, m[1] >>> 16, m[1] & 0xFFFF];
            n = [n[0] >>> 16, n[0] & 0xFFFF, n[1] >>> 16, n[1] & 0xFFFF];

            var o = [0, 0, 0, 0];
            o[3] += m[3] * n[3];
            o[2] += o[3] >>> 16;
            o[3] &= 0xFFFF;
            o[2] += m[2] * n[3];
            o[1] += o[2] >>> 16;
            o[2] &= 0xFFFF;
            o[2] += m[3] * n[2];
            o[1] += o[2] >>> 16;
            o[2] &= 0xFFFF;
            o[1] += m[1] * n[3];
            o[0] += o[1] >>> 16;
            o[1] &= 0xFFFF;
            o[1] += m[2] * n[2];
            o[0] += o[1] >>> 16;
            o[1] &= 0xFFFF;
            o[1] += m[3] * n[1];
            o[0] += o[1] >>> 16;
            o[1] &= 0xFFFF;
            o[0] += (m[0] * n[3]) + (m[1] * n[2]) +
                    (m[2] * n[1]) + (m[3] * n[0]);
            o[0] &= 0xFFFF;

            return [(o[0] << 16) | o[1], (o[2] << 16) | o[3]];
        },

        _x64Rotl: function(m, n) {
            n %= 64;

            if (n === 32) {
                return [m[1], m[0]];
            } else if (n < 32) {
                return [(m[0] << n) | (m[1] >>> (32 - n)),
                        (m[1] << n) | (m[0] >>> (32 - n))];
            } else {
                n -= 32;
                return [(m[1] << n) | (m[0] >>> (32 - n)),
                        (m[0] << n) | (m[1] >>> (32 - n))];
            }
        },

        _x64LeftShift: function(m, n) {
            n %= 64;

            if (n === 0) {
                return m;
            } else if (n < 32) {
                return [(m[0] << n) | (m[1] >>> (32 - n)), m[1] << n];
            } else {
                return [m[1] << (n - 32), 0];
            }
        },

        _x64Xor: function(m, n) {
            return [m[0] ^ n[0], m[1] ^ n[1]];
        },

        _x64Fmix: function(h) {
            h = this._x64Xor(h, [0, h[0] >>> 1]);
            h = this._x64Multiply(h, [0xFF51AFD7, 0xED558CCD]);
            h = this._x64Xor(h, [0, h[0] >>> 1]);
            h = this._x64Multiply(h, [0xC4CEB9FE, 0x1A85EC53]);
            h = this._x64Xor(h, [0, h[0] >>> 1]);

            return h;
        },

        // 128-bit positive integer hash will be returned
        _innerMurmurHash3_128: function(src) {
            var seed = 31; // fixed

            var remainder = src.length % 16;
            var bytes = src.length - remainder;
            var h1 = [0, seed];
            var h2 = [0, seed];
            var k1 = [0, 0];
            var k2 = [0, 0];
            var c1 = [0x87C37B91, 0x114253D5];
            var c2 = [0x4CF5AD43, 0x2745937F];

            for (var i = 0; i < bytes; i = i + 16) {
                k1 = [ (src.charCodeAt(i + 4) & 0xFF)        |
                      ((src.charCodeAt(i + 5) & 0xFF) << 8)  |
                      ((src.charCodeAt(i + 6) & 0xFF) << 16) |
                      ((src.charCodeAt(i + 7) & 0xFF) << 24),
                       (src.charCodeAt(i)     & 0xFF)        |
                      ((src.charCodeAt(i + 1) & 0xFF) << 8)  |
                      ((src.charCodeAt(i + 2) & 0xFF) << 16) |
                      ((src.charCodeAt(i + 3) & 0xFF) << 24)];

                k2 = [ (src.charCodeAt(i + 12) & 0xFF)        |
                      ((src.charCodeAt(i + 13) & 0xFF) << 8)  |
                      ((src.charCodeAt(i + 14) & 0xFF) << 16) |
                      ((src.charCodeAt(i + 15) & 0xFF) << 24),
                       (src.charCodeAt(i + 8)  & 0xFF)        |
                      ((src.charCodeAt(i + 9)  & 0xFF) << 8)  |
                      ((src.charCodeAt(i + 10) & 0xFF) << 16) |
                      ((src.charCodeAt(i + 11) & 0xFF) << 24)];

                k1 = this._x64Multiply(k1, c1);
                k1 = this._x64Rotl(k1, 31);
                k1 = this._x64Multiply(k1, c2);
                h1 = this._x64Xor(h1, k1);
                h1 = this._x64Rotl(h1, 27);
                h1 = this._x64Add(h1, h2);
                h1 = this._x64Add(this._x64Multiply(h1, [0, 5]),
                                  [0, 0x52dce729]);
                k2 = this._x64Multiply(k2, c2);
                k2 = this._x64Rotl(k2, 33);
                k2 = this._x64Multiply(k2, c1);
                h2 = this._x64Xor(h2, k2);
                h2 = this._x64Rotl(h2, 31);
                h2 = this._x64Add(h2, h1);
                h2 = this._x64Add(this._x64Multiply(h2, [0, 5]),
                                  [0, 0x38495ab5]);
            }

            k1 = [0, 0];
            k2 = [0, 0];

            switch (remainder) {
                case 15:
                    k2 = this._x64Xor(k2,
                             this._x64LeftShift([0, src.charCodeAt(i + 14)],
                                                48));
                case 14:
                    k2 = this._x64Xor(k2,
                             this._x64LeftShift([0, src.charCodeAt(i + 13)],
                                                40));
                case 13:
                    k2 = this._x64Xor(k2,
                             this._x64LeftShift([0, src.charCodeAt(i + 12)],
                                                32));
                case 12:
                    k2 = this._x64Xor(k2,
                             this._x64LeftShift([0, src.charCodeAt(i + 11)],
                                                24));
                case 11:
                    k2 = this._x64Xor(k2,
                             this._x64LeftShift([0, src.charCodeAt(i + 10)],
                                                16));
                case 10:
                    k2 = this._x64Xor(k2,
                             this._x64LeftShift([0, src.charCodeAt(i + 9)],
                                                8));
                case 9:
                    k2 = this._x64Xor(k2, [0, src.charCodeAt(i + 8)]);
                    k2 = this._x64Multiply(k2, c2);
                    k2 = this._x64Rotl(k2, 33);
                    k2 = this._x64Multiply(k2, c1);
                    h2 = this._x64Xor(h2, k2);
                case 8:
                    k1 = this._x64Xor(k1,
                             this._x64LeftShift([0, src.charCodeAt(i + 7)],
                                                56));
                case 7:
                    k1 = this._x64Xor(k1,
                             this._x64LeftShift([0, src.charCodeAt(i + 6)],
                                                48));
                case 6:
                    k1 = this._x64Xor(k1,
                             this._x64LeftShift([0, src.charCodeAt(i + 5)],
                                                40));
                case 5:
                    k1 = this._x64Xor(k1,
                             this._x64LeftShift([0, src.charCodeAt(i + 4)],
                                                32));
                case 4:
                    k1 = this._x64Xor(k1,
                             this._x64LeftShift([0, src.charCodeAt(i + 3)],
                                                24));
                case 3:
                    k1 = this._x64Xor(k1,
                             this._x64LeftShift([0, src.charCodeAt(i + 2)],
                                                16));
                case 2:
                    k1 = this._x64Xor(k1,
                             this._x64LeftShift([0, src.charCodeAt(i + 1)],
                                                8));
                case 1:
                    k1 = this._x64Xor(k1, [0, src.charCodeAt(i)]);
                    k1 = this._x64Multiply(k1, c1);
                    k1 = this._x64Rotl(k1, 31);
                    k1 = this._x64Multiply(k1, c2);
                    h1 = this._x64Xor(h1, k1);
            }

            h1 = this._x64Xor(h1, [0, src.length]);
            h2 = this._x64Xor(h2, [0, src.length]);
            h1 = this._x64Add(h1, h2);
            h2 = this._x64Add(h2, h1);
            h1 = this._x64Fmix(h1);
            h2 = this._x64Fmix(h2);
            h1 = this._x64Add(h1, h2);
            h2 = this._x64Add(h2, h1);

            return ("00000000" + (h1[0] >>> 0).toString(16)).slice(-8) +
                   ("00000000" + (h1[1] >>> 0).toString(16)).slice(-8) +
                   ("00000000" + (h2[0] >>> 0).toString(16)).slice(-8) +
                   ("00000000" + (h2[1] >>> 0).toString(16)).slice(-8);
        },


        /*******************************************************
        *  MD5: https://github.com/blueimp/JavaScript-MD5.git  *
        *******************************************************/

        // basic interfaces
        _safeAdd: function(x, y) {
            var lsw = (x & 0xFFFF) + (y & 0xFFFF);
            var msw = (x >> 16) + (y >> 16) + (lsw >> 16);
            return (msw << 16) | (lsw & 0xFFFF);
        },

        _basicMD5: function(q, a, b, x, s, t) {
          return this._safeAdd(this._bitRotateLeft(
                                   this._safeAdd(this._safeAdd(a, q),
                                                 this._safeAdd(x, t)), s), b);
        },

        _opFF: function(a, b, c, d, x, s, t) {
          return this._basicMD5((b & c) | ((~b) & d), a, b, x, s, t);
        },

        _opGG: function(a, b, c, d, x, s, t) {
          return this._basicMD5((b & d) | (c & (~d)), a, b, x, s, t);
        },

        _opHH: function(a, b, c, d, x, s, t) {
          return this._basicMD5(b ^ c ^ d, a, b, x, s, t);
        },

        _opII: function(a, b, c, d, x, s, t) {
          return this._basicMD5(c ^ (b | (~d)), a, b, x, s, t);
        },

        // calculate MD5 of an array of little-endian words & a bit length
        _innerMD5: function(src, len) {
            // append padding
            src[len >> 5]                      |= 0x80 << (len % 32);
            src[(((len + 64) >>> 9) << 4) + 14] = len;

            var i, olda, oldb, oldc, oldd;
            var a = 1732584193;
            var b = -271733879;
            var c = -1732584194;
            var d = 271733878;

            for (i = 0; i < src.length; i += 16) {
                olda = a;
                oldb = b;
                oldc = c;
                oldd = d;

                a = this._opFF(a, b, c, d, src[i],       7, -680876936);
                d = this._opFF(d, a, b, c, src[i + 1],  12, -389564586);
                c = this._opFF(c, d, a, b, src[i + 2],  17, 606105819);
                b = this._opFF(b, c, d, a, src[i + 3],  22, -1044525330);
                a = this._opFF(a, b, c, d, src[i + 4],   7, -176418897);
                d = this._opFF(d, a, b, c, src[i + 5],  12, 1200080426);
                c = this._opFF(c, d, a, b, src[i + 6],  17, -1473231341);
                b = this._opFF(b, c, d, a, src[i + 7],  22, -45705983);
                a = this._opFF(a, b, c, d, src[i + 8],   7, 1770035416);
                d = this._opFF(d, a, b, c, src[i + 9],  12, -1958414417);
                c = this._opFF(c, d, a, b, src[i + 10], 17, -42063);
                b = this._opFF(b, c, d, a, src[i + 11], 22, -1990404162);
                a = this._opFF(a, b, c, d, src[i + 12],  7, 1804603682);
                d = this._opFF(d, a, b, c, src[i + 13], 12, -40341101);
                c = this._opFF(c, d, a, b, src[i + 14], 17, -1502002290);
                b = this._opFF(b, c, d, a, src[i + 15], 22, 1236535329);

                a = this._opGG(a, b, c, d, src[i + 1],   5, -165796510);
                d = this._opGG(d, a, b, c, src[i + 6],   9, -1069501632);
                c = this._opGG(c, d, a, b, src[i + 11], 14, 643717713);
                b = this._opGG(b, c, d, a, src[i],      20, -373897302);
                a = this._opGG(a, b, c, d, src[i + 5],   5, -701558691);
                d = this._opGG(d, a, b, c, src[i + 10],  9, 38016083);
                c = this._opGG(c, d, a, b, src[i + 15], 14, -660478335);
                b = this._opGG(b, c, d, a, src[i + 4],  20, -405537848);
                a = this._opGG(a, b, c, d, src[i + 9],   5, 568446438);
                d = this._opGG(d, a, b, c, src[i + 14],  9, -1019803690);
                c = this._opGG(c, d, a, b, src[i + 3],  14, -187363961);
                b = this._opGG(b, c, d, a, src[i + 8],  20, 1163531501);
                a = this._opGG(a, b, c, d, src[i + 13],  5, -1444681467);
                d = this._opGG(d, a, b, c, src[i + 2],   9, -51403784);
                c = this._opGG(c, d, a, b, src[i + 7],  14, 1735328473);
                b = this._opGG(b, c, d, a, src[i + 12], 20, -1926607734);

                a = this._opHH(a, b, c, d, src[i + 5],   4, -378558);
                d = this._opHH(d, a, b, c, src[i + 8],  11, -2022574463);
                c = this._opHH(c, d, a, b, src[i + 11], 16, 1839030562);
                b = this._opHH(b, c, d, a, src[i + 14], 23, -35309556);
                a = this._opHH(a, b, c, d, src[i + 1],   4, -1530992060);
                d = this._opHH(d, a, b, c, src[i + 4],  11, 1272893353);
                c = this._opHH(c, d, a, b, src[i + 7],  16, -155497632);
                b = this._opHH(b, c, d, a, src[i + 10], 23, -1094730640);
                a = this._opHH(a, b, c, d, src[i + 13],  4, 681279174);
                d = this._opHH(d, a, b, c, src[i],      11, -358537222);
                c = this._opHH(c, d, a, b, src[i + 3],  16, -722521979);
                b = this._opHH(b, c, d, a, src[i + 6],  23, 76029189);
                a = this._opHH(a, b, c, d, src[i + 9],   4, -640364487);
                d = this._opHH(d, a, b, c, src[i + 12], 11, -421815835);
                c = this._opHH(c, d, a, b, src[i + 15], 16, 530742520);
                b = this._opHH(b, c, d, a, src[i + 2],  23, -995338651);

                a = this._opII(a, b, c, d, src[i],       6, -198630844);
                d = this._opII(d, a, b, c, src[i + 7],  10, 1126891415);
                c = this._opII(c, d, a, b, src[i + 14], 15, -1416354905);
                b = this._opII(b, c, d, a, src[i + 5],  21, -57434055);
                a = this._opII(a, b, c, d, src[i + 12],  6, 1700485571);
                d = this._opII(d, a, b, c, src[i + 3],  10, -1894986606);
                c = this._opII(c, d, a, b, src[i + 10], 15, -1051523);
                b = this._opII(b, c, d, a, src[i + 1],  21, -2054922799);
                a = this._opII(a, b, c, d, src[i + 8],   6, 1873313359);
                d = this._opII(d, a, b, c, src[i + 15], 10, -30611744);
                c = this._opII(c, d, a, b, src[i + 6],  15, -1560198380);
                b = this._opII(b, c, d, a, src[i + 13], 21, 1309151649);
                a = this._opII(a, b, c, d, src[i + 4],   6, -145523070);
                d = this._opII(d, a, b, c, src[i + 11], 10, -1120210379);
                c = this._opII(c, d, a, b, src[i + 2],  15, 718787259);
                b = this._opII(b, c, d, a, src[i + 9],  21, -343485551);

                a = this._safeAdd(a, olda);
                b = this._safeAdd(b, oldb);
                c = this._safeAdd(c, oldc);
                d = this._safeAdd(d, oldd)
            }

            return [a, b, c, d];
        },

        // convert a raw string to an array of little-endian words
        //      characters (> 255) have their high-byte silently ignored
        _raw2LEArray: function(src) {
            var i;
            var dst = [];
            dst[(src.length >> 2) - 1] = undefined;
            for (i = 0; i < dst.length; i += 1) {
                dst[i] = 0;
            }
            for (i = 0; i < src.length * 8; i += 8) {
                dst[i >> 5] |= (src.charCodeAt(i / 8) & 0xFF) << (i % 32);
            }
            return dst;
        },

        // convert an array of little-endian words to a string
        _LEArray2raw: function(src) {
            var i;
            var dst = "";
            for (i = 0; i < src.length * 32; i += 8) {
                dst += String.fromCharCode((src[i >> 5] >>> (i % 32)) & 0xFF);
            }
            return dst;
        },


        /********************************************************
        * SHA1: http://www.movable-type.co.uk/scripts/sha1.html *
        ********************************************************/

        _f4SHA1: function(s, x, y, z)  {
            switch (s) {
                case 0: return (x & y) ^ (~x & z);           // Ch()
                case 1: return  x ^ y  ^  z;                 // Parity()
                case 2: return (x & y) ^ (x & z) ^ (y & z);  // Maj()
                case 3: return  x ^ y  ^  z;                 // Parity()
            }
        },

        _innerSHA1: function(src) {
            // constants
            var K = [0x5A827999, 0x6ED9EBA1, 0x8F1BBCDC, 0xCA62C1D6];

            // add trailing '1' bit (and 0's padding)
            src += String.fromCharCode(0x80);

            // convert it into 512-bit / 16-integer block arrays
            var l = src.length / 4 + 2; // length + '1' + appended length
            var N = Math.ceil(l / 16); // num of the blocks required to hold l
            var M = new Array(N);

            for (var i = 0; i < N; i++) {
                M[i] = new Array(16);
                for (var j = 0; j < 16; j++) { // 4 chars per int (big-endian)
                    M[i][j] = (src.charCodeAt(i * 64 + j * 4) << 24)
                              | (src.charCodeAt(i * 64 + j * 4 + 1) << 16)
                              | (src.charCodeAt(i * 64 + j * 4 + 2) << 8)
                              | (src.charCodeAt(i * 64 + j * 4 + 3));
                }
            }

            // add length into final pair of 32-bit integers (BE)
            // NOTE: most significant word would be (len - 1) * 8 >>> 32,
            //       but since JS converts bitwise-op args to 32 bits,
            //       we need to simulate this by arithmetic operators
            M[N - 1][14] = ((src.length - 1) * 8) / Math.pow(2, 32);
            M[N - 1][14] = Math.floor(M[N - 1][14]);
            M[N - 1][15] = ((src.length - 1) * 8) & 0xFFFFFFFF;

            // set initial hash value
            var H0 = 0x67452301;
            var H1 = 0xEFCDAB89;
            var H2 = 0x98BADCFE;
            var H3 = 0x10325476;
            var H4 = 0xC3D2E1F0;

            // hash computation
            var W = new Array(80);
            var a, b, c, d, e;
            for (var i = 0; i < N; i++) {
                // prepare message schedule W
                for (var t = 0; t < 16; t++) {
                    W[t] = M[i][t];
                }

                for (var t = 16; t < 80; t++) {
                    W[t] = this._bitRotateLeft(W[t - 3] ^ W[t - 8] ^
                                               W[t - 14] ^ W[t - 16], 1);
                }

                // initialize 5 working variables: a, b, c, d, e
                a = H0;
                b = H1;
                c = H2;
                d = H3;
                e = H4;

                // main loop
                for (var t = 0; t < 80; t++) {
                    // seq for blocks of _f4SHA1 and K
                    var s = Math.floor(t / 20);

                    var T = (this._bitRotateLeft(a, 5)
                             + this._f4SHA1(s, b, c, d)
                             + e + K[s] + W[t]) & 0xFFFFFFFF;
                    var s = Math.floor(t / 20);
                    e = d;
                    d = c;
                    c = this._bitRotateLeft(b, 30);
                    b = a;
                    a = T;
                }

                // compute the new intermediate hash value
                H0 = (H0 + a) & 0xFFFFFFFF;
                H1 = (H1 + b) & 0xFFFFFFFF;
                H2 = (H2 + c) & 0xFFFFFFFF;
                H3 = (H3 + d) & 0xFFFFFFFF;
                H4 = (H4 + e) & 0xFFFFFFFF;
            }

            return [H0, H1, H2, H3, H4];
        },


        /************************************************************
        * SHA256: http://www.movable-type.co.uk/scripts/sha256.html *
        ************************************************************/

        _upperSigma0: function(x) {
            return this._bitRotateRight(2, x) ^ this._bitRotateRight(13, x)
                   ^ this._bitRotateRight(22, x);
        },

        _upperSigma1: function(x) {
            return this._bitRotateRight(6, x) ^ this._bitRotateRight(11, x)
                   ^ this._bitRotateRight(25, x);
        },

        _lowerSigma0: function(x) {
            return this._bitRotateRight(7, x) ^ this._bitRotateRight(18, x)
                   ^ (x >>> 3);
        },

        _lowerSigma1: function(x) {
            return this._bitRotateRight(17, x) ^ this._bitRotateRight(19, x)
                   ^ (x >>> 10);
        },

        _Ch: function(x, y, z) {
            return (x & y) ^ (~x & z);
        },

        _Maj: function(x, y, z) {
            return (x & y) ^ (x & z) ^ (y & z);
        },

        _innerSHA256: function(src) {
            var K = [0x428A2F98, 0x71374491, 0xB5C0FBCF, 0xE9B5DBA5,
                     0x3956C25B, 0x59F111F1, 0x923F82A4, 0xAB1C5ED5,
                     0xD807AA98, 0x12835B01, 0x243185BE, 0x550C7DC3,
                     0x72BE5D74, 0x80DEB1FE, 0x9BDC06A7, 0xC19BF174,
                     0xE49B69C1, 0xEFBE4786, 0x0FC19DC6, 0x240CA1CC,
                     0x2DE92C6F, 0x4A7484AA, 0x5CB0A9DC, 0x76F988DA,
                     0x983E5152, 0xA831C66D, 0xB00327C8, 0xBF597FC7,
                     0xC6E00BF3, 0xD5A79147, 0x06CA6351, 0x14292967,
                     0x27B70A85, 0x2E1B2138, 0x4D2C6DFC, 0x53380D13,
                     0x650A7354, 0x766A0ABB, 0x81C2C92E, 0x92722C85,
                     0xA2BFE8A1, 0xA81A664B, 0xC24B8B70, 0xC76C51A3,
                     0xD192E819, 0xD6990624, 0xF40E3585, 0x106AA070,
                     0x19A4C116, 0x1E376C08, 0x2748774C, 0x34B0BCB5,
                     0x391C0CB3, 0x4ED8AA4A, 0x5B9CCA4F, 0x682E6FF3,
                     0x748F82EE, 0x78A5636F, 0x84C87814, 0x8CC70208,
                     0x90BEFFFA, 0xA4506CEB, 0xBEF9A3F7, 0xC67178F2];

            var H = [0x6A09E667, 0xBB67AE85, 0x3C6EF372, 0xA54FF53A,
                     0x510E527F, 0x9B05688C, 0x1F83D9AB, 0x5BE0CD19];

            src += String.fromCharCode(0x80);

            var l = src.length / 4 + 2;
            var N = Math.ceil(l / 16);
            var M = new Array(N);

            for (var i = 0; i < N; i++) {
                M[i] = new Array(16);
                for (var j = 0; j < 16; j++) {
                    M[i][j] = (src.charCodeAt(i * 64 + j * 4) << 24)
                              | (src.charCodeAt(i * 64 + j * 4 + 1) << 16)
                              | (src.charCodeAt(i * 64 + j * 4 + 2) << 8)
                              | (src.charCodeAt(i * 64 + j * 4 + 3));
                }
            }

            M[N - 1][14] = ((src.length - 1) * 8) / Math.pow(2, 32);
            M[N - 1][14] = Math.floor(M[N - 1][14]);
            M[N - 1][15] = ((src.length - 1) * 8) & 0xFFFFFFFF;

            var W = new Array(64);
            var a, b, c, d, e, f, g, h;

            for (var i = 0; i < N; i++) {
                for (var t = 0; t < 16; t++) {
                    W[t] = M[i][t];
                }

                for (var t = 16; t < 64; t++) {
                    W[t] = (this._lowerSigma1(W[t - 2]) + W[t - 7]
                            + this._lowerSigma0(W[t - 15]) + W[t - 16])
                           & 0xFFFFFFFF;
                }

                a = H[0];
                b = H[1];
                c = H[2];
                d = H[3];
                e = H[4];
                f = H[5];
                g = H[6];
                h = H[7];

                for (var t = 0; t < 64; t++) {
                    var T1 = h + this._upperSigma1(e) + this._Ch(e, f, g)
                               + K[t] + W[t];
                    var T2 =     this._upperSigma0(a) + this._Maj(a, b, c);
                    h = g;
                    g = f;
                    f = e;
                    e = (d + T1) & 0xFFFFFFFF;
                    d = c;
                    c = b;
                    b = a;
                    a = (T1 + T2) & 0xFFFFFFFF;
                }

                H[0] = (H[0] + a) & 0xFFFFFFFF;
                H[1] = (H[1] + b) & 0xFFFFFFFF;
                H[2] = (H[2] + c) & 0xFFFFFFFF;
                H[3] = (H[3] + d) & 0xFFFFFFFF;
                H[4] = (H[4] + e) & 0xFFFFFFFF;
                H[5] = (H[5] + f) & 0xFFFFFFFF;
                H[6] = (H[6] + g) & 0xFFFFFFFF;
                H[7] = (H[7] + h) & 0xFFFFFFFF;
            }

            return [H[0], H[1], H[2], H[3], H[4], H[5], H[6], H[7]];
        },


        // public interfaces
        _bitRotateLeft: function(x, n) {
            return (x << n) | (x >>> (32 - n));
        },

        _bitRotateRight: function(n, x) {
            return (x >>> n) | (x << (32 - n));
        },

        _string2Hex: function(str) {
            var map = "0123456789abcdef";
            var dst = "";
            var c, i;
            for (i = 0; i < str.length; i += 1) {
                c = str.charCodeAt(i);
                dst += map.charAt((c >>> 4) & 0x0F) + map.charAt(c & 0x0F);
            }
            return dst;
        },

        _num2Hex: function(num) {
            // NOTE: can't use toString(16) as it is implementation-dependant,
            //       and in IE returns signed numbers when used on full words
            var s = "";
            var v;
            for (var i = 7; i >= 0; i--) {
                v  = (num >>> (i * 4)) & 0xF;
                s += v.toString(16);
            }
            return s;
        },


        // external interfaces
        computeMurmurHash3: function(src, bitNum) {
            if (bitNum === 128) {
                return this._innerMurmurHash3_128(src);
            } else {
                // default is 32-bit
                return this._innerMurmurHash3_32(src);
            }
        },

        computeMD5: function(src) {
            var raw = this._LEArray2raw(this._innerMD5(this._raw2LEArray(src),
                                                       src.length * 8));
            return this._string2Hex(raw);
        },

        computeSHA1: function(src) {
            var obj = this._innerSHA1(src);

            var dst = "";
            for (var i in obj) {
                dst += this._num2Hex(obj[i]);
            }

            return dst;
        },

        computeSHA256: function(src) {
            var obj = this._innerSHA256(src);

            var dst = "";
            for (var i in obj) {
                dst += this._num2Hex(obj[i]);
            }

            return dst;
        }
    };

    // export it
    if (typeof module === "object" && typeof exports !== "undefined") {
        module.exports = msgDigest;
    }
    scope.msgDigest = msgDigest;
})(window);

