#!/bin/sh

#####################################################
#
# feature list:
# 1. generate Metadata for an SP
# 2. add / modify an SP
# 3. delete an SP
#
#####################################################

# internal variables
_MAIN_DIR="/data/etc/saml/shibboleth"
_TEMP_META="$_MAIN_DIR/template_metadata.xml"
_TEMP_SP="$_MAIN_DIR/template_service_provider.xml"
_TEMP_CFG="$_MAIN_DIR/template_cfg.xml"
_SP_DIR="$_MAIN_DIR/service_providers"
_CERT_GENERATOR="$_MAIN_DIR/keygen.sh"
_MAIN_CFG="shibboleth2.xml"
_APP_CFG="app.xml"
_CERT_NAME="sp-cert.pem"
_KEY_NAME="sp-key.pem"
_TMPFILE="/tmp/saml.`date +%s%N`"

# external parameters
ACTION="x"
APP_ID="x"
SP_ENTITY_ID="x"
LIFETIME="x"
TIMEOUT="x"
CERT="x"
SERVICE_PATH="x"
SIGNING_OPTION='' # XXX: just null
SSO_BIND="x"
SSO_PATH="x"
SLO_BIND="x"
SLO_PATH="x"

usage()
{
    echo "usage: $0 -a action args..." >& 2
    echo "    action list:" >& 2
    echo "        meta - generate Metadata for an SP" >& 2
    echo "           -i: application ID" >& 2
    echo "           -e: SP entity ID" >& 2
    echo "           -c: certificate for signing & encryption" >& 2
    echo "           -s: service path" >& 2
    echo "        -b/-p: SSO bind type / path" >& 2
    echo "        -B/-P: SLO bind type / path" >& 2
    echo >& 2
    echo "        addsp - add / modify an SP" >& 2
    echo "           -i: application ID" >& 2
    echo "           -e: SP entity ID" >& 2
    echo "           -l: session lifetime (seconds)" >& 2
    echo "           -L: session timeout (seconds)" >& 2
    echo "           -s: service path" >& 2
    echo "           -S: sign by force" >& 2
    echo "        -b/-p: SSO bind type / path" >& 2
    echo "        -B/-P: SLO bind type / path" >& 2
    echo >& 2
    echo "        delsp - delete an SP" >& 2
    echo "           -i: application ID" >& 2
    echo >& 2
    echo "    examples:" >& 2
    echo "        $0 -a meta -i testshib -e 'https://domain.com/saml' \\" >& 2
    echo "           -c $_CERT_NAME -s '/saml' -b post \\" >& 2
    echo "           -p '/SSO/POST' -B redirect -P '/SLO/Redirect'" >& 2
    echo >& 2
    echo "        $0 -a addsp -i adfs -e 'https://domain.com/saml' \\" >& 2
    echo "           -l 28800 -L 3600 -s '/saml' [-S] -b post \\" >& 2
    echo "            -p '/SSO/POST' -B redirect -P '/SLO/Redirect'" >& 2
    echo >& 2
    echo "        $0 -a delsp -i adfs" >& 2
}

while getopts a:i:e:l:L:c:s:Sb:p:B:P: arg
    do
        case $arg in
            a) ACTION=$OPTARG;;

            i) APP_ID=$OPTARG;;

            e) SP_ENTITY_ID=$OPTARG;;

            l) LIFETIME=$OPTARG;;
            L) TIMEOUT=$OPTARG;;

            c) CERT=$OPTARG;;

            s) SERVICE_PATH=$OPTARG;;

            S) SIGNING_OPTION='signing="true"';;

            b) SSO_BIND=$OPTARG;;
            p) SSO_PATH=$OPTARG;;

            B) SLO_BIND=$OPTARG;;
            P) SLO_PATH=$OPTARG;;

            *) usage
               exit 1;;
        esac
    done

err_exit()
{
    echo "$@" >& 2
    exit 1
}

generate_metadata()
{
    if [ $APP_ID = "x" ] || [ $SP_ENTITY_ID = "x" ] || \
        [ ! -f $CERT ] || [ $SERVICE_PATH = "x" ] || \
        [ $SSO_BIND = "x" ] || [ $SSO_PATH = "x" ] || \
        [ $SLO_BIND = "x" ] || [ $SLO_PATH = "x" ]; then
        usage
        exit 1
    fi

    echo "$SP_ENTITY_ID" | grep '^https\?://[^/]' &> /dev/null \
        || err_exit "invalid SP entity ID"

    if [ ! -f $_TEMP_META ]; then
        err_exit "can't find Metadata template: $_TEMP_META"
    fi

    IGNORE_SSO='AssertionConsumerService.*'
    IGNORE_SLO='SingleLogoutService.*'

    if [ $SSO_BIND = "redirect" ]; then
        IGNORE_SSO=$IGNORE_SSO"POST"
    else
        IGNORE_SSO=$IGNORE_SSO"Redirect"
    fi

    if [ $SLO_BIND = "redirect" ]; then
        IGNORE_SLO=$IGNORE_SLO"POST"
    else
        IGNORE_SLO=$IGNORE_SLO"Redirect"
    fi

    SAML_SP_ENTITY_ID=`echo "$SP_ENTITY_ID" | sed 's#\/#\\\/#g'`

    cat $CERT | sed -ne '/-BEGIN CERTIFICATE-/,/-END CERTIFICATE-/p' | \
        grep -v '^-' > $_TMPFILE || err_exit "invalid certificate"

    PATH_PREFIX=`echo "$SP_ENTITY_ID" | \
                 grep -o '^.*://[^/]*'`"/$APP_ID$SERVICE_PATH"

    SAML_SSO_PATH=`echo "$PATH_PREFIX" | sed 's/\/$//g'`"$SSO_PATH"
    SAML_SSO_PATH=`echo "$SAML_SSO_PATH" | sed 's#\/#\\\/#g'`

    SAML_SLO_PATH=`echo "$PATH_PREFIX" | sed 's/\/$//g'`"$SLO_PATH"
    SAML_SLO_PATH=`echo "$SAML_SLO_PATH" | sed 's#\/#\\\/#g'`

    cat $_TEMP_META | grep -v "$IGNORE_SSO" | grep -v "$IGNORE_SLO" | \
        sed "s/SAML_SP_ENTITY_ID/$SAML_SP_ENTITY_ID/g" | \
        sed "/SAML_CERT/r $_TMPFILE" | grep -v 'SAML_CERT' | \
        sed "s/SAML_SSO_PATH/$SAML_SSO_PATH/g" | \
        sed "s/SAML_SLO_PATH/$SAML_SLO_PATH/g" \
        || err_exit "failed to generate Metadata"
}

generate_certificate()
{
    if [ $SP_ENTITY_ID = "x" ] || [ $APP_ID = "x" ]; then
        usage
        exit 1
    fi

    APP_DIR="$_SP_DIR/$APP_ID"
    if [ ! -d $APP_DIR ]; then
        err_exit "application directory $APP_DIR doesn't exist"
    fi

    # skip if existing
    if  [ -s $APP_DIR/$_CERT_NAME -a -s $APP_DIR/$_KEY_NAME ] ; then
        return
    fi

    echo "$SP_ENTITY_ID" | grep '^https\?://[^/]' &> /dev/null \
        || err_exit "invalid SP entity ID"

    if [ ! -f $_CERT_GENERATOR ]; then
        err_exit "certificate generator $_CERT_GENERATOR doesn't exist"
    fi

    HOSTNAME=`echo "$SP_ENTITY_ID" | grep -o '://[^/]*' \
              | busybox cut -d '/' -f 3`

    $_CERT_GENERATOR -f -h "$HOSTNAME" -o "$APP_DIR" &> /dev/null \
        || err_exit "failed to generate certificate for $APP_ID"
}

create_sp()
{
    if [ $APP_ID = "x" ] || [ $SP_ENTITY_ID = "x" ] || \
        [ $LIFETIME = "x" ] || [ $TIMEOUT = "x" ] || \
        [ $SERVICE_PATH = "x" ] || \
        [ $SSO_BIND = "x" ] || [ $SSO_PATH = "x" ] || \
        [ $SLO_BIND = "x" ] || [ $SLO_PATH = "x" ]; then
        usage
        exit 1
    fi

    echo "$SP_ENTITY_ID" | grep '^https\?://[^/]' &> /dev/null \
        || err_exit "invalid SP entity ID"


    if [ ! -f $_TEMP_SP ]; then
        err_exit "can't find SP template: $_TEMP_SP"
    fi


    APP_DIR="$_SP_DIR/$APP_ID"

    CERT="$APP_DIR/$_CERT_NAME"
    if [ ! -f $CERT ]; then
        err_exit "certificate $CERT doesn't exist"
    fi

    KEY="$APP_DIR/$_KEY_NAME"
    if [ ! -f $KEY ]; then
        err_exit "private key $KEY doesn't exist"
    fi

    IDP_META="$APP_DIR/Metadata.xml"
    if [ ! -f $IDP_META ]; then
        err_exit "IDP Metadata $IDP_META doesn't exist"
    fi

    IDP_ENTITY_ID=`grep -o 'entityID=".*"' "$IDP_META" \
                   | busybox head -n 1 | busybox cut -d '"' -f 2`
    if [ -z $IDP_ENTITY_ID ]; then
        err_exit "invalid IDP Metadata: failed to find IDP entity ID"
    fi
     

    IGNORE_SSO='AssertionConsumerService.*'
    IGNORE_SLO='SingleLogoutService.*'

    if [ $SSO_BIND = "redirect" ]; then
        IGNORE_SSO=$IGNORE_SSO"POST"
    else
        IGNORE_SSO=$IGNORE_SSO"Redirect"
    fi

    if [ $SLO_BIND = "redirect" ]; then
        IGNORE_SLO=$IGNORE_SLO"POST"
    else
        IGNORE_SLO=$IGNORE_SLO"Redirect"
    fi

    SAML_SIGNING_OPTION="$SIGNING_OPTION"

    SAML_SP_ENTITY_ID=`echo "$SP_ENTITY_ID" | sed 's#\/#\\\/#g'`
    SAML_SERVICE_PATH=`echo "/$APP_ID$SERVICE_PATH" | sed 's#\/#\\\/#g'`
    SAML_IDP_ENTITY_ID=`echo "$IDP_ENTITY_ID" | sed 's#\/#\\\/#g'`

    SAML_SSO_PATH=`echo "$SSO_PATH" | sed 's#\/#\\\/#g'`
    SAML_SLO_PATH=`echo "$SLO_PATH" | sed 's#\/#\\\/#g'`

    SAML_IDP_METADATA=`echo "$IDP_META" | sed 's#\/#\\\/#g'`

    SAML_CERT=`echo "$CERT" | sed 's#\/#\\\/#g'`
    SAML_KEY=`echo "$KEY" | sed 's#\/#\\\/#g'`

    cat $_TEMP_SP | grep -v "$IGNORE_SSO" | grep -v "$IGNORE_SLO" | \
        sed "s/SAML_APP_ID/$APP_ID/g" | \
        sed "s/SAML_SP_ENTITY_ID/$SAML_SP_ENTITY_ID/g" | \
        sed "s/SAML_SIGNING_OPTION/$SAML_SIGNING_OPTION/g" | \
        sed "s/SAML_LIFETIME/$LIFETIME/g" | \
        sed "s/SAML_TIMEOUT/$TIMEOUT/g" | \
        sed "s/SAML_SERVICE_PATH/$SAML_SERVICE_PATH/g" | \
        sed "s/SAML_IDP_ENTITY_ID/$SAML_IDP_ENTITY_ID/g" | \
        sed "s/SAML_SSO_PATH/$SAML_SSO_PATH/g" | \
        sed "s/SAML_SLO_PATH/$SAML_SLO_PATH/g" | \
        sed "s/SAML_IDP_METADATA/$SAML_IDP_METADATA/g" | \
        sed "s/SAML_CERT/$SAML_CERT/g" | sed "s/SAML_KEY/$SAML_KEY/g" \
        > "$APP_DIR/$_APP_CFG" \
        || err_exit "failed to create application configurations for an SP"
}

remove_sp()
{
    if [ $APP_ID = "x" ]; then
        usage
        exit 1
    fi

    APP_DIR="$_SP_DIR/$APP_ID"
    if [ ! -d $APP_DIR ]; then
        err_exit "application directory $APP_DIR doesn't exist"
    fi

    rm -rf $APP_DIR || err_exit "failed to remove $APP_DIR"
}

merge_sp()
{
    if [ ! -d $_SP_DIR ]; then
        err_exit "SP directory $_SP_DIR doesn't exist"
    fi

    if [ ! -f $_TEMP_CFG ]; then
        err_exit "can't find configuration template: $_TEMP_CFG"
    fi

    find "$_SP_DIR" -type f -name "$_APP_CFG" | xargs cat >> $_TMPFILE

    cat $_TEMP_CFG | sed "/<\!--APPLICATION-OVERRIDE-->/r $_TMPFILE" \
        > "$_MAIN_DIR/$_MAIN_CFG"

    # XXX: check if it's valid
}

case $ACTION in
    meta)
        generate_metadata
        ;;

    addsp)
        generate_certificate
        create_sp
        merge_sp
        ;;

    delsp)
        remove_sp
        merge_sp
        ;;

    *)
        usage
        exit 1
        ;;
esac

exit 0

