﻿IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[GetMachineType]') AND type in (N'P', N'FN'))
DROP FUNCTION [dbo].[GetMachineType]
GO
CREATE FUNCTION [dbo].[GetMachineType] (@machineType tinyint)
RETURNS nvarchar(10) AS
BEGIN
   return case when @machineType = 1 then 'Physical' when @machineType = 2 then 'Cloud' else 'Virtual' end  
END
GO
 
----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[MachineTypes]'))
DROP VIEW [dbo].[MachineTypes]

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[MachineTypes]
AS
    SELECT 'Cloud' [TypeName], 2 [TypeValue]
    UNION ALL
    SELECT 'Physical', 1
    UNION ALL
    SELECT 'Virtual', 0
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[HostTypes]'))
DROP VIEW [dbo].[HostTypes]

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[HostTypes]
AS
    SELECT 'Cloud' [TypeName], 2 [TypeValue]
    UNION ALL
    SELECT 'Virtual', 0
GO

----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[ChargeBackMachineView]'))
DROP VIEW [dbo].[ChargeBackMachineView]

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[ChargeBackMachineView]
AS
SELECT *
FROM
	BaseMachineCostView

GO
----------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ChargebackByGroup]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ChargebackByGroup]
GO
CREATE PROCEDURE [dbo].[usp_ChargebackByGroup]
	@group uniqueidentifier = NULL,
	@startDate DateTime = NULL,
	@endDate DateTime = NULL,
	@machineType tinyint = NULL

AS
BEGIN

IF @endDate IS NULL SET @endDate = GETDATE()
IF @startDate IS NULL SET @startDate = DATEADD(DAY, -30, @endDate)
IF @startDate > @endDate SET @startDate = @endDate

SELECT GroupName, VirtualMachineName, dbo.GetMachineType(MachineType) [MachineType], UserName, VirtualMachineTemplateName
,CPUs, Memory, Storage, BlueprintCost, CPUCost, MemoryCost, StorageCost, [Days], [Days] * TotalCost [TotalCost]
FROM (
	SELECT *
	,CASE WHEN VMCreationDate > @startDate AND VMDeleteDate < @endDate THEN DATEDIFF(day, VMCreationDate, VMDeleteDate)
	 WHEN VMCreationDate > @startDate THEN DATEDIFF(day, VMCreationDate, @endDate) 
	 WHEN VMDeleteDate < @endDate THEN DATEDIFF(day, @startDate, VMDeleteDate) 
	 ELSE DATEDIFF(day, @startDate, @endDate) END [Days]
	FROM ChargeBackMachineView
	WHERE VMCreationDate < @endDate AND (VMDeleteDate IS NULL OR VMDeleteDate >= @startDate)
	AND (@group IS NULL OR @group = GroupID) AND (@machineType IS NULL OR MachineType = @machineType)
	AND MachineType != 3
) cb
ORDER BY GroupName, VirtualMachineName

END
GO

----------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ChargebackByOwner]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ChargebackByOwner]
GO
CREATE PROCEDURE [dbo].[usp_ChargebackByOwner]
	@userName NVarChar(1024) = NULL,
	@startDate DateTime = NULL,
	@endDate DateTime = NULL,
	@machineType tinyint = NULL

AS
BEGIN

IF @endDate IS NULL SET @endDate = GETDATE()
IF @startDate IS NULL SET @startDate = DATEADD(DAY, -30, @endDate)
IF @startDate > @endDate SET @startDate = @endDate

SELECT UserName, VirtualMachineName, dbo.GetMachineType(MachineType) [MachineType], GroupName, VirtualMachineTemplateName
,CPUs, Memory, Storage, BlueprintCost, CPUCost, MemoryCost, StorageCost, [Days], [Days] * TotalCost [TotalCost]
FROM (
	SELECT *
	,CASE WHEN VMCreationDate > @startDate AND VMDeleteDate < @endDate THEN DATEDIFF(day, VMCreationDate, VMDeleteDate)
	 WHEN VMCreationDate > @startDate THEN DATEDIFF(day, VMCreationDate, @endDate) 
	 WHEN VMDeleteDate < @endDate THEN DATEDIFF(day, @startDate, VMDeleteDate) 
	 ELSE DATEDIFF(day, @startDate, @endDate) END [Days]
	FROM ChargeBackMachineView
	WHERE VMCreationDate < @endDate AND (VMDeleteDate IS NULL OR VMDeleteDate >= @startDate)
	AND (@userName IS NULL OR @userName = UserName) AND (@machineType IS NULL OR MachineType = @machineType)
	AND MachineType != 3
) cb
ORDER BY UserName, VirtualMachineName

END
GO

----------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ChargebackByGroupReservation]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ChargebackByGroupReservation]
GO
CREATE PROCEDURE [dbo].[usp_ChargebackByGroupReservation]
	@group uniqueidentifier = NULL,
	@startDate DateTime = NULL,
	@endDate DateTime = NULL,
	@machineType tinyint = NULL

AS
BEGIN

IF @endDate IS NULL SET @endDate = GETDATE()
IF @startDate IS NULL SET @startDate = DATEADD(DAY, -30, @endDate)
IF @startDate > @endDate SET @startDate = @endDate

SELECT GroupName, HostReservationName, min(dbo.GetMachineType(MachineType)) [MachineType]
,sum(CPUs) [CPUs], dbo.DivideAsDecimal(sum(Memory), 1024) [Memory], sum(Storage) [Storage]
,sum([Days] * BlueprintCost) [BlueprintCost], sum([Days] * CPUCost) [CPUCost], sum([Days] * MemoryCost) [MemoryCost], sum([Days] * StorageCost) [StorageCost], sum([Days] * TotalCost) [TotalCost]
FROM (
	SELECT *
	,CASE WHEN VMCreationDate > @startDate AND VMDeleteDate < @endDate THEN DATEDIFF(day, VMCreationDate, VMDeleteDate)
	 WHEN VMCreationDate > @startDate THEN DATEDIFF(day, VMCreationDate, @endDate) 
	 WHEN VMDeleteDate < @endDate THEN DATEDIFF(day, @startDate, VMDeleteDate) 
	 ELSE DATEDIFF(day, @startDate, @endDate) END [Days]
	FROM ChargeBackMachineView
	WHERE VMCreationDate < @endDate AND (VMDeleteDate IS NULL OR VMDeleteDate >= @startDate)
	AND (@group IS NULL OR @group = GroupID) AND (@machineType IS NULL OR MachineType = @machineType)
	AND MachineType != 3
) cb
GROUP BY HostReservationName, GroupName
ORDER BY GroupName, HostReservationName

END
GO

----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[DeletedMachinesCostView]'))
DROP VIEW [dbo].[DeletedMachinesCostView]

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[DeletedMachinesCostView]
AS
	SELECT	vm.VirtualMachineID
		,	vm.MachineType 
		,	vm.VMDeleteDate
		,	vm.[Owner]
		,	hr.GroupID
		,	cpus.CPUs
		,	memory.Memory
		,	storage.Storage
		,	blueprintcost.BlueprintCost 
			+ cputotalcost.CPUCost 
			+ memorytotalcost.MemoryCost
			+ storagetotalcost.StorageCost AS Cost
	FROM VirtualMachineHistory vm
	INNER JOIN		Users					u on u.UserID = vm.[Owner]
	LEFT OUTER JOIN Host					h on h.HostID = vm.HostID
	LEFT OUTER JOIN CostProfile				cp on cp.CostProfileID = h.CostProfileID
	INNER JOIN		HostReservation			hr on hr.HostReservationID = vm.HostReservationID
	INNER JOIN		VirtualMachineTemplate	vmt	on vmt.VirtualMachineTemplateID = vm.VirtualMachineTemplateID
	LEFT OUTER JOIN PhysicalMachines		pm on pm.PhysicalMachineID = vm.PhysicalMachineID
	LEFT OUTER JOIN CostProfile				cpp on cpp.CostProfileID = pm.CostProfileID
	LEFT OUTER JOIN
	(
		-- calculate storage cost by machine id
		SELECT	vm.VirtualMachineID,
				SUM(COALESCE(scp.DiskCost, cp.DiskCost, 0) * COALESCE(vdh.Capacity, vm.VMTotalStorageGB, 0))	AS StorageCost
		FROM	VirtualMachineHistory vm
		LEFT OUTER JOIN Host h						ON h.HostID = vm.HostID 
		LEFT OUTER JOIN CostProfile cp				ON cp.CostProfileID = h.CostProfileID 
		LEFT OUTER JOIN VMDiskHardwareHistory vdh	on vdh.VirtualMachineID = vm.VirtualMachineID
		LEFT OUTER JOIN HostToStorage hs			ON hs.HostToStorageID = vdh.HostToStorageID 
		LEFT OUTER JOIN StorageCostProfiles scp		ON scp.Id = hs.StorageCostProfileId
		GROUP BY vm.VirtualMachineID
	)  AS storagetotalcost  ON vm.VirtualMachineID = storagetotalcost.VirtualMachineID		
	-- calculated fields
	CROSS APPLY (SELECT CASE WHEN vm.MachineType != 1 THEN ISNULL(vm.VMCPUs, 0) ELSE pm.ProcessorCount END) AS cpus(CPUs)
	CROSS APPLY (SELECT CASE WHEN vm.MachineType != 1 THEN ISNULL(vm.VMTotalMemoryMB, 0) ELSE pm.MemoryInMB END) AS memory(Memory)
	CROSS APPLY (SELECT CASE WHEN vm.MachineType != 1 THEN ISNULL(vm.VMTotalStorageGB, 0) ELSE NULL END) AS storage(Storage)
	CROSS APPLY (SELECT ISNULL(vmt.Cost, 0)) AS blueprintcost(BlueprintCost)
	CROSS APPLY (
		SELECT 
			CASE 
				WHEN vm.MachineType = 0 THEN ISNULL(cp.CpuCost, 0) 
				WHEN vm.MachineType = 1 THEN ISNULL(cpp.CpuCost, 0) 
			ELSE NULL 
		END) AS cpucost(CPUCost)
	CROSS APPLY (
		SELECT 
			CASE 
				WHEN vm.MachineType = 0 THEN ISNULL(cp.MemoryCost, 0) 
				WHEN vm.MachineType = 1 THEN ISNULL(cpp.MemoryCost, 0) 
			ELSE NULL 
		END) AS memorycost(MemoryCost)
	CROSS APPLY (SELECT cpus.CPUs * cpucost.CPUCost) AS cputotalcost(CPUCost)
	CROSS APPLY (SELECT dbo.Divideasdecimal(memory.Memory, 1024) * memorycost.MemoryCost) AS memorytotalcost(MemoryCost)
GO

----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReclamationDashboard]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReclamationDashboard]
GO
CREATE PROCEDURE [dbo].[usp_ReclamationDashboard]
	@machineType tinyint = NULL
AS
BEGIN
	SET NOCOUNT ON;

	DECLARE @lastWeek int
	DECLARE @lastMonth int
	DECLARE @lastQuarter int
	DECLARE @lastYear int

	DECLARE @dtToday datetime = GETDATE()
	DECLARE @dtLastWeek datetime = datediff(day,7, @dtToday)
	DECLARE @dtLastMonth datetime = datediff(day,30, @dtToday)
	DECLARE @dtLastQuarter datetime = datediff(day,90, @dtToday)
	DECLARE @dtLastYear datetime = datediff(day,365, @dtToday)

	SELECT @lastWeek = COUNT(*) FROM VirtualMachineHistory 
		where VMDeleteDate >= @dtLastWeek and (@machineType IS NULL or MachineType = @machineType)
	SELECT @lastMonth = COUNT(*) FROM VirtualMachineHistory
		where VMDeleteDate >= @dtLastMonth and (@machineType IS NULL or MachineType = @machineType)
	SELECT @lastQuarter = COUNT(*) FROM VirtualMachineHistory
		where VMDeleteDate >= @dtLastQuarter and (@machineType IS NULL or MachineType = @machineType)
	SELECT @lastYear = COUNT(*) FROM VirtualMachineHistory
		where VMDeleteDate >= @dtLastYear and (@machineType IS NULL or MachineType = @machineType)

	DECLARE @SUMMARY TABLE(Period nvarchar(128), VMCount int)
	INSERT @SUMMARY VALUES('Last 7 Days', @lastWeek)
	INSERT @SUMMARY VALUES('Last 30 Days', @lastMonth)
	INSERT @SUMMARY VALUES('Last 90 Days', @lastQuarter)
	INSERT @SUMMARY VALUES('Last 365 Days', @lastYear)
	
	SELECT * FROM @SUMMARY
END
GO

----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReclamationByGroup]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReclamationByGroup]
GO
CREATE PROCEDURE [dbo].[usp_ReclamationByGroup]
	@startDate datetime = NULL,
	@machineType tinyint = NULL,
	@reclaimedOnly bit = 0,
	@costOnly bit = 0
AS
BEGIN
	SET NOCOUNT ON;

	declare @currentDate datetime = GETDATE()

	select pg.GroupName
	,pgi.MachinesTotal + pgi.MachinesReclaimed [MachinesTotal]
	,pgi.MachinesReclaimed
	,dbo.Percentage(pgi.MachinesReclaimed, pgi.MachinesTotal + pgi.MachinesReclaimed) [MachinesReclaimedPercent]
	,pgi.CPUs, pgi.Memory, pgi.Storage, pgi.Cost
	from ProvisioningGroup pg
	join (
		select pg.GroupId
		,(select count(vm.VirtualMachineID) from VirtualMachine vm join HostReservation hr on hr.HostReservationID = vm.HostReservationID 
		  where hr.GroupID = pg.GroupID and vm.IsManaged = 1 and (@machineType IS NULL or vm.MachineType = @machineType)) [MachinesTotal]
		,COUNT(dm.VirtualMachineID) [MachinesReclaimed]
		,ISNULL(SUM(dm.CPUs), 0) [CPUs]
		,dbo.DivideAsDecimal(ISNULL(SUM(dm.Memory), 0), 1024) [Memory]
		,ISNULL(SUM(dm.Storage), 0) [Storage]
		,ISNULL(SUM(DATEDIFF(day, dm.VMDeleteDate, @currentDate) * dm.Cost), 0) [Cost]
		from ProvisioningGroup pg
		join (
			select * from DeletedMachinesCostView
			where (@startDate IS NULL or @startDate <= VMDeleteDate)
			and (@machineType IS NULL or MachineType = @machineType)
		) dm on pg.GroupID = dm.GroupID
		where pg.GroupType = 0
		group by pg.GroupID
	) pgi on pgi.GroupID = pg.GroupID
	where (@reclaimedOnly = 0 or pgi.MachinesReclaimed > 0)
	and (@costOnly = 0 or pgi.Cost > 0)
	order by pg.GroupName
END
GO

----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReclamationByOwner]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReclamationByOwner]
GO
CREATE PROCEDURE [dbo].[usp_ReclamationByOwner]
	@startDate datetime = NULL,
	@machineType tinyint = NULL
AS
BEGIN
	SET NOCOUNT ON;

	declare @currentDate datetime = GETDATE()

	select u.UserName
	,ui.MachinesTotal + ui.MachinesReclaimed [MachinesTotal]
	,ui.MachinesReclaimed
	,dbo.Percentage(ui.MachinesReclaimed, ui.MachinesTotal + ui.MachinesReclaimed) [MachinesReclaimedPercent]
	,ui.CPUs, ui.Memory, ui.Storage, ui.Cost
	from Users u
	join (
		select u.UserID
		,(select count(vm.VirtualMachineID) from VirtualMachine vm 
		  where vm.[Owner] = u.UserID and vm.IsManaged = 1 and (@machineType IS NULL or vm.MachineType = @machineType)) [MachinesTotal]
		,COUNT(dm.VirtualMachineID) [MachinesReclaimed]
		,ISNULL(SUM(dm.CPUs), 0) [CPUs]
		,dbo.DivideAsDecimal(ISNULL(SUM(dm.Memory), 0), 1024) [Memory]
		,ISNULL(SUM(dm.Storage), 0) [Storage]
		,ISNULL(SUM(DATEDIFF(day, dm.VMDeleteDate, @currentDate) * dm.Cost), 0) [Cost]
		from Users u
		join (
			select * from DeletedMachinesCostView
			where (@startDate IS NULL or @startDate <= VMDeleteDate)
			and (@machineType IS NULL or MachineType = @machineType)
		) dm on u.UserID = dm.[Owner]
		group by u.UserID
	) ui on ui.UserID= u.UserID
	order by u.UserName
END
GO

----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[ManagedHostsView]'))
DROP VIEW [dbo].[ManagedHostsView]

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[ManagedHostsView]
AS
	SELECT *
	FROM Host 
	WHERE ClusterHostID is null and MachineType != 1 
	AND HostID IN (SELECT HostID from AdminGroupToHost)
GO

----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[ManagedHostsCapacityView]'))
DROP VIEW [dbo].[ManagedHostsCapacityView]

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[ManagedHostsCapacityView]
AS
	select hf.HostName
	,hf.HostProcessors [CPUs]
	,hf.HostTotalMemoryMB [MemoryPhysical]
	,hf.HostTotalStorageGB [StoragePhysical]
	,hf.MachineType
	,hi.*
	from host hf 
	join(
		select h.hostId
		,ISNULL(SUM(hrx.MachinesTotal), 0) [MachinesTotal]
		,ISNULL(SUM(hrx.MachinesAllocated), 0) [MachinesAllocated]
		,ISNULL(SUM(hr.ReservationMemorySizeMB), 0) [MemoryReserved]
		,ISNULL(SUM(hrx.MemoryAllocated), 0) [MemoryAllocated]
		,ISNULL(SUM(dbo.DiffNoNegBigInt(hr.ReservationMemorySizeMB, hrx.MemoryAllocated)), 0) [MemoryFree]
		,ISNULL(SUM(hr.ReservationStorageSizeGB), 0) [StorageReserved]
		,ISNULL(SUM(hrx.StorageAllocated), 0) [StorageAllocated]
		,ISNULL(SUM(dbo.DiffNoNegBigInt(hr.ReservationStorageSizeGB, hrx.StorageAllocated)), 0) [StorageFree]
		,ISNULL(SUM(hrx.StorageUsed), 0) [StorageUsed]
		from ManagedHostsView h
		left outer join HostReservation hr on h.HostID = hr.HostID
		left outer join HostReservationStats hrx on hr.HostReservationID = hrx.HostReservationID
		group by h.HostID
	) hi on hf.HostID = hi.HostID
GO

----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[ManagedGroupsCapacityView]'))
DROP VIEW [dbo].[ManagedGroupsCapacityView]

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[ManagedGroupsCapacityView]
AS
	select pg.GroupName, pg.GroupID, hr.MachineType
	,ISNULL(SUM(hrx.MachinesTotal), 0) MachinesTotal
	,ISNULL(SUM(hrx.MachinesAllocated), 0) MachinesAllocated
	,ISNULL(SUM(hr.ReservationMemorySizeMB), 0) [MemoryReserved]
	,ISNULL(SUM(hrx.MemoryAllocated), 0) [MemoryAllocated]
	,ISNULL(SUM(dbo.DiffNoNegBigInt(hr.ReservationMemorySizeMB, hrx.MemoryAllocated)), 0) [MemoryFree]
	,ISNULL(SUM(hr.ReservationStorageSizeGB), 0) [StorageReserved]
	,ISNULL(SUM(hrx.StorageAllocated), 0) [StorageAllocated]
	,ISNULL(SUM(dbo.DiffNoNegBigInt(hr.ReservationStorageSizeGB, hrx.StorageAllocated)), 0) [StorageFree]
	,ISNULL(SUM(hrx.StorageUsed), 0) [StorageUsed]
	from ProvisioningGroup pg
	join HostReservation hr on pg.GroupID = hr.GroupID
	left outer join HostReservationStats hrx on hr.HostReservationID = hrx.HostReservationID
	left outer join ManagedHostsView h on hr.HostID = h.HostID
	where pg.GroupType = 0
	and (hr.MachineType IS NULL or hr.MachineType = 1 or h.HostID IS NOT NULL)
	group by pg.GroupID, pg.GroupName, hr.MachineType
GO

----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_BlueprintCapacityChart]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_BlueprintCapacityChart]
GO
CREATE PROCEDURE [dbo].[usp_BlueprintCapacityChart]
	@blueprint uniqueidentifier = NULL,
	@machineType tinyint = null
AS
BEGIN
	SET NOCOUNT ON;

	select 
		vmt.VirtualMachineTemplateName [Name]
		,dbo.GetMachineType(vmt.MachineType) MachineType
		,dbo.GetGroupNamesForTemplate(vmt.VirtualMachineTemplateID) [Group]
		,COUNT(vm.VirtualMachineTemplateID) [MachinesTotal]
		,SUM(CASE WHEN vm.MachineType = 1 or vm.IsRunning = 1 THEN 1 ELSE 0 END) [MachinesAllocated]
		,case when vmt.MachineType != 1 then dbo.DivideAsDecimal(ISNULL(SUM(CAST(vm.IsRunning AS INT) * vm.VMTotalMemoryMB), 0), 1024) 
		 else dbo.DivideAsDecimal(ISNULL(SUM(pm.MemoryInMB), 0), 1024)  end [Memory]
		,case when vmt.MachineType != 1 then dbo.DivideAsDecimal(ISNULL(SUM(vm.VMTotalStorageGB), 0), 1024) else null end [Storage]
		,case when vmt.MachineType = 0 then dbo.DivideAsDecimal(ISNULL(SUM(vm.VMUsedStorageGB), 0), 1024) else null end [Used]
	from VirtualMachineTemplate vmt
	left outer join VirtualMachine vm on vm.VirtualMachineTemplateID = vmt.VirtualMachineTemplateID
	left outer join PhysicalMachines pm on pm.PhysicalMachineID = vm.PhysicalMachineID
	where (@blueprint IS NULL OR vmt.VirtualMachineTemplateID = @blueprint)
	AND (@machineType IS NULL OR vmt.MachineType = @machineType)
	group by vmt.VirtualMachineTemplateID, vmt.VirtualMachineTemplateName, vmt.MachineType
	order by vmt.VirtualMachineTemplateName
END
GO

----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_OwnerCapacityChart]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_OwnerCapacityChart]
GO
CREATE PROCEDURE [dbo].[usp_OwnerCapacityChart]
	@pOwner VARCHAR(128) = NULL,
	@machineType tinyint = NULL
AS
BEGIN
	SET NOCOUNT ON;

	SELECT u.UserName [Owner]
		,COUNT(vm.VirtualMachineID) [MachinesTotal]
		,SUM(CASE WHEN vm.MachineType = 1 or vm.IsRunning = 1 THEN 1 ELSE 0 END) [MachinesAllocated]
		,dbo.DivideAsDecimal(SUM(CASE WHEN vm.MachineType != 1 AND vm.IsRunning = 1 THEN VMTotalMemoryMB WHEN vm.MachineType = 1 THEN pm.MemoryInMB ELSE 0 END), 1024) [Memory]
		,CASE WHEN @machineType != 1 THEN dbo.DivideAsDecimal(SUM(VMTotalStorageGB), 1024) ELSE NULL END [Storage]
		,CASE WHEN @machineType = 0 THEN dbo.DivideAsDecimal(SUM(VMUsedStorageGB), 1024) ELSE NULL END [Used]
	FROM VirtualMachine vm
	left outer join PhysicalMachines pm on pm.PhysicalMachineID = vm.PhysicalMachineID
    JOIN Users u ON u.UserID = vm.[Owner]
	WHERE (@pOwner IS NULL OR u.UserName = @pOwner)
	AND (@machineType IS NULL OR vm.MachineType = @machineType)
	AND vm.IsManaged = 1
	GROUP BY u.UserName
	ORDER BY u.UserName
END
GO

----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_HostCapacityChart]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_HostCapacityChart]
GO
CREATE PROCEDURE [dbo].[usp_HostCapacityChart]
	@host uniqueidentifier = NULL,
	@machineType tinyint = NULL
AS
BEGIN
	SET NOCOUNT ON;

	select HostName
	,MachinesTotal
	,MachinesAllocated
	,CASE WHEN MachineType = 0 THEN CPUs ELSE NULL END [CPUs]
	,CASE WHEN MachineType = 0 THEN dbo.DivideAsDecimal(MemoryPhysical, 1024) ELSE NULL END [MemoryPhysical]
	,CASE WHEN MachineType = 0 THEN dbo.DivideAsDecimal(MemoryReserved, 1024) ELSE NULL END [MemoryReserved]
	,CASE WHEN MachineType = 0 THEN dbo.Percentage(MemoryReserved, MemoryPhysical) ELSE NULL END [MemoryReservedPercent]
	,dbo.DivideAsDecimal(MemoryAllocated, 1024) [MemoryAllocated]
	,CASE WHEN MachineType = 0 THEN dbo.Percentage(MemoryAllocated, MemoryReserved) ELSE NULL END [MemoryAllocatedPercent]
	,CASE WHEN MachineType = 0 THEN dbo.DivideAsDecimal(StoragePhysical, 1024) ELSE NULL END [StoragePhysical]
	,CASE WHEN MachineType = 0 THEN dbo.DivideAsDecimal(StorageReserved, 1024) ELSE NULL END [StorageReserved]
	,CASE WHEN MachineType = 0 THEN dbo.Percentage(StorageReserved, StoragePhysical) ELSE NULL END [StorageReservedPercent]
	,dbo.DivideAsDecimal(StorageAllocated, 1024) [StorageAllocated]
	,CASE WHEN MachineType = 0 THEN dbo.Percentage(StorageAllocated, StorageReserved) ELSE NULL END [StorageAllocatedPercent]
	,CASE WHEN MachineType = 0 THEN dbo.DivideAsDecimal(StorageUsed, 1024) ELSE NULL END [StorageUsed]
	from ManagedHostsCapacityView
	WHERE (@host IS NULL OR HostID = @host) AND (@machineType IS NULL OR MachineType = @machineType)
	order by HostName
END
GO

----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_GroupCapacityChart]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_GroupCapacityChart]
GO
CREATE PROCEDURE [dbo].[usp_GroupCapacityChart]
	@group uniqueidentifier = NULL,
	@machineType tinyint = NULL
AS
BEGIN
	SET NOCOUNT ON;

	select GroupName
	,MachinesTotal
	,MachinesAllocated
	,dbo.Percentage(MachinesAllocated, MachinesTotal) [MachinesAllocatedPercent]
	,CASE WHEN MachineType = 0 THEN dbo.DivideAsDecimal(MemoryReserved, 1024) ELSE NULL END [MemoryReserved]
	,dbo.DivideAsDecimal(MemoryAllocated, 1024) [MemoryAllocated]
	,CASE WHEN MachineType = 0 THEN dbo.Percentage(MemoryAllocated, MemoryReserved) ELSE NULL END [MemoryAllocatedPercent]
	,CASE WHEN MachineType = 0 THEN dbo.DivideAsDecimal(StorageReserved, 1024) ELSE NULL END [StorageReserved]
	,CASE WHEN MachineType != 1 THEN dbo.DivideAsDecimal(StorageAllocated, 1024) ELSE NULL END [StorageAllocated]
	,CASE WHEN MachineType = 0 THEN dbo.Percentage(StorageAllocated, StorageReserved) ELSE NULL END [StorageAllocatedPercent]
	,CASE WHEN MachineType = 0 THEN dbo.DivideAsDecimal(StorageUsed, 1024) ELSE NULL END [StorageUsed]
	from ManagedGroupsCapacityView
	where (@group IS NULL OR GroupID = @group) AND MachineType = @machineType
	order by GroupName
END
GO

----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_HostCapacityMemoryStorage]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_HostCapacityMemoryStorage]
GO
CREATE PROCEDURE [dbo].[usp_HostCapacityMemoryStorage]
	@host uniqueidentifier = NULL,
	@machineType tinyint = NULL
AS
BEGIN
	SET NOCOUNT ON;

	SELECT dbo.DivideAsDecimal(SUM(CASE WHEN MachineType = 0 THEN MemoryReserved ELSE 0 END), 1024) [MemoryReserved],
		   dbo.DivideAsDecimal(SUM(CASE WHEN MachineType = 0 THEN MemoryAllocated ELSE 0 END), 1024) [MemoryAllocated],
		   dbo.DivideAsDecimal(SUM(CASE WHEN MachineType = 0 THEN MemoryFree ELSE 0 END), 1024) [MemoryFree],
		   dbo.DivideAsDecimal(SUM(CASE WHEN MachineType = 0 THEN StorageReserved ELSE 0 END), 1024) [StorageReserved],
		   dbo.DivideAsDecimal(SUM(CASE WHEN MachineType = 0 THEN StorageAllocated ELSE 0 END), 1024) [StorageAllocated],
		   dbo.DivideAsDecimal(SUM(CASE WHEN MachineType = 0 THEN StorageFree ELSE 0 END), 1024) [StorageFree],
		   CASE WHEN @host IS NULL THEN 'All Hosts' ELSE MAX(HostName) END [HostName]
	FROM ManagedHostsCapacityView h
	WHERE (@host IS NULL OR HostID = @host) AND (@machineType IS NULL OR MachineType = @machineType)
END
GO

----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_GroupCapacityMemoryStorage]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_GroupCapacityMemoryStorage]
GO
CREATE PROCEDURE [dbo].[usp_GroupCapacityMemoryStorage]
	@group uniqueidentifier = NULL,
	@machineType tinyint = NULL
AS
BEGIN
	SET NOCOUNT ON;

	SELECT dbo.DivideAsDecimal(SUM(CASE WHEN MachineType = 0 THEN MemoryReserved ELSE 0 END), 1024) [MemoryReserved],
		   dbo.DivideAsDecimal(SUM(CASE WHEN MachineType = 0 THEN MemoryAllocated ELSE 0 END), 1024) [MemoryAllocated],
		   dbo.DivideAsDecimal(SUM(CASE WHEN MachineType = 0 THEN MemoryFree ELSE 0 END), 1024) [MemoryFree],
		   dbo.DivideAsDecimal(SUM(CASE WHEN MachineType = 0 THEN StorageReserved ELSE 0 END), 1024) [StorageReserved],
		   dbo.DivideAsDecimal(SUM(CASE WHEN MachineType = 0 THEN StorageAllocated ELSE 0 END), 1024) [StorageAllocated],
		   dbo.DivideAsDecimal(SUM(CASE WHEN MachineType = 0 THEN StorageFree ELSE 0 END), 1024) [StorageFree],
		   CASE WHEN @group IS NULL THEN 'All Groups' ELSE MAX(GroupName) END [GroupName]
	FROM ManagedGroupsCapacityView
	WHERE (@group IS NULL OR GroupID = @group) AND MachineType = @machineType
END
GO

----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[getNextDayBegining]') AND type in (N'P', N'FN'))
DROP FUNCTION [dbo].[getNextDayBegining]
GO
CREATE FUNCTION [dbo].[getNextDayBegining] (@date datetime)
RETURNS datetime AS
BEGIN
   return DATEADD(day, DATEDIFF(day, 0, @date), 1)
END
GO


IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[get_week_start]') AND type in (N'P', N'FN'))
DROP FUNCTION [dbo].[get_week_start]
GO

CREATE FUNCTION [dbo].[get_week_start] (@date datetime)

RETURNS datetime AS

BEGIN

   return dateadd(yyyy, datepart(yyyy,

      dateadd(weekday,1-datepart(weekday, @date),@date))-1900, 0)

    + dateadd(dy, datepart(dy,

      dateadd(weekday,1-datepart(weekday, @date),@date))-1,0)

END
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[VirtualMachineView]'))
DROP VIEW [dbo].[VirtualMachineView]

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[VirtualMachineView]
AS
SELECT     
	VM.*, 
	ISNULL(CONVERT(VARCHAR(20), VM.Expires, 106), 'Never') ExpireDate
FROM uv_VirtualMachineView AS VM where IsTemplate = 0 and MachineType = 0
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[PhysicalMachineView]'))
DROP VIEW [dbo].[PhysicalMachineView]

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[PhysicalMachineView]
AS
SELECT     
	VM.*
FROM uv_VirtualMachineView AS VM where IsTemplate = 0 and MachineType = 1
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[CloudMachineView]'))
DROP VIEW [dbo].[CloudMachineView]

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[CloudMachineView]
AS
SELECT     
	VM.*
FROM uv_VirtualMachineView AS VM where IsTemplate = 0 and MachineType = 2
GO
------------------------------------------------------------------------------------------------------------------------------------------
-- Host view with costs
IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[HostPerDayCostView]'))
DROP VIEW [dbo].[HostPerDayCostView]
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[HostPerDayCostView]
AS
SELECT     
	H.*, 
	ISNULL(CP.MemoryCost,0) MemoryPerDay,
	ISNULL(CP.DiskCost,0) DiskPerDay,
	ISNULL(CP.CpuCost, 0) AS CPUPerDay,
    (SELECT HostName FROM Host c WHERE c.HostID = H.ClusterHostID) ClusterHostName
FROM dbo.HostCapacityView AS H LEFT OUTER JOIN 
	dbo.CostProfile CP ON H.CostProfileID = CP.CostProfileID
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[VirtualMachinePerDayCostView]'))
DROP VIEW [dbo].[VirtualMachinePerDayCostView]
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[VirtualMachinePerDayCostView]
AS
SELECT     
	VM.*, 
	ISNULL(CP.MemoryCost,0) MemoryPerDay,
	ISNULL(CP.DiskCost,0) DiskPerDay,
	ISNULL(VMT.Cost, 0) TemplatePerDay
FROM VirtualMachineView VM LEFT OUTER JOIN 
	dbo.VirtualMachineTemplate AS VMT ON VMT.VirtualMachineTemplateID = VM.VirtualMachineTemplateID LEFT OUTER JOIN
    dbo.Host AS H ON H.HostID = VM.HostID LEFT OUTER JOIN 
	dbo.CostProfile CP ON H.CostProfileID = CP.CostProfileID
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[VirtualMachineCostView]'))
DROP VIEW [dbo].[VirtualMachineCostView]
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[VirtualMachineCostView]
AS
SELECT     
	VMC.*,
	(VMC.MemoryPerDay * VMC.VMTotalMemoryMB/1024) MemoryCost,
	(VMC.DiskPerDay * VMC.VMTotalStorageGB) DiskCost,
	(VMC.DiskPerDay * VMC.VMUsedStorageGB) DiskCostByUsed,
	TemplatePerDay TemplateCost,
	((VMC.MemoryPerDay * VMC.VMTotalMemoryMB/1024) +  (VMC.DiskPerDay * VMC.VMTotalStorageGB) + TemplatePerDay) VMCost,
	((VMC.MemoryPerDay * VMC.VMTotalMemoryMB/1024) +  (VMC.DiskPerDay * VMC.VMUsedStorageGB) + TemplatePerDay) VMCostByUsed
FROM VirtualMachinePerDayCostView VMC
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[HostReservationCompatView]'))
DROP VIEW [dbo].[HostReservationCompatView]
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[HostReservationCompatView]
AS
SELECT     
	HR.*,
	HR.ReservationMemorySizeMB ReservationMemorySize, 
	HR.ReservationStorageSizeGB ReservationDiskSize,
	HR.MaxVMsPowerOn VirtualMachineQuota
FROM dbo.HostReservation AS HR
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportGroupCapacityChart]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportGroupCapacityChart]
GO
CREATE PROCEDURE [dbo].[usp_ReportGroupCapacityChart]
	@pGroupID VARCHAR(128) = NULL
AS
BEGIN
	SET NOCOUNT ON;
	IF @pGroupID IS NULL 
		SELECT SUM(NumOfVMsCreated) NumOfVMsCreated, 
				SUM(GroupReservedStorageGB) GroupReservedStorageGB, 
				SUM(GroupUsedStorageGB) GroupUsedStorageGB, 
				SUM(GroupFreeStorageGB) GroupFreeStorageGB, 
				SUM(GroupReservedStorageTB) GroupReservedStorageTB, 
				SUM(GroupUsedStorageTB) GroupUsedStorageTB, 
				SUM(GroupFreeStorageTB) GroupFreeStorageTB, 
				SUM(GroupUsedStorageGB)*100/NULLIF(SUM(GroupReservedStorageTB),0) PercentStorageAllocated,
				SUM(GroupReservedMemoryMB) GroupReservedMemoryMB, 
				SUM(GroupReservedMemoryGB) GroupReservedMemoryGB,
				SUM(GroupUsedMemoryGB) GroupUsedMemoryGB, 
				SUM(GroupFreeMemoryGB) GroupFreeMemoryGB, 
				SUM(GroupUsedMemoryGB)*100/NULLIF(SUM(GroupReservedMemoryMB), 0) PercentMemoryAllocated,
				'allgroupid' GroupID,
				'All Groups' GroupName
		FROM ProvisioningGroupCapacityView
	ELSE 
		SELECT * FROM ProvisioningGroupCapacityView WHERE GroupID = @pGroupID
END
GO


IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[UserCapacitySummaryView]'))
DROP VIEW [dbo].[UserCapacitySummaryView]
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[UserCapacitySummaryView]
AS
	select Owner, count(VirtualMachineID) NumberOfVMs
		,dbo.DivideAsDecimal(SUM(VMTotalMemoryMB), 1024) MemoryGB
		,dbo.DivideAsDecimal(SUM(VMTotalStorageGB), 1024) DiskTB
		,dbo.DivideAsDecimal(SUM(VMUsedStorageGB), 1024) UsedDiskTB
        ,dbo.Percentage(SUM(VMTotalStorageGB)-SUM(VMUsedStorageGB), SUM(VMTotalStorageGB)) PercentSpaceEfficiency   
	
	from 
		VirtualMachineView
	GROUP BY Owner 
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[TemplateCapacitySummaryView]'))
DROP VIEW [dbo].[TemplateCapacitySummaryView]
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[TemplateCapacitySummaryView]
AS
SELECT
	(SELECT count(*) FROM VirtualMachineView VMT WHERE VMT.VirtualMachineTemplateName = VM.VirtualMachineTemplateName) NumberOfVMs,     
	(SELECT CAST(CAST(Sum(VMTotalMemoryMB) as decimal(28,3))/1024 as decimal(28,2)) FROM VirtualMachineView VMT WHERE VMT.IsRunning=1 AND VMT.VirtualMachineTemplateName = VM.VirtualMachineTemplateName) Memory, 
	Sum(VMTotalStorageGB) Disk,
	CAST(CAST(Sum(VMTotalStorageGB) as decimal(28,3))/1024 as decimal(28,2)) DiskTB,
	CAST(CAST(Sum(VMUsedStorageGB) as decimal(28,3))/1024 as decimal(28,2)) UsedDiskTB,
	dbo.Percentage(SUM(VMTotalStorageGB)-SUM(VMUsedStorageGB), SUM(VMTotalStorageGB)) PercentSpaceEfficiency,
	VirtualMachineTemplateID,
	VirtualMachineTemplateName
FROM VirtualMachineView VM
	GROUP BY VirtualMachineTemplateID, VirtualMachineTemplateName 
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[VirtualMachineTemplateView]'))
DROP VIEW [dbo].[VirtualMachineTemplateView]
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[VirtualMachineTemplateView]
AS
SELECT
	VMT.*,
	MOD.WorkflowName,
	MOD.AssemblyFQN,
	MOD.WorkflowDescription,
	GT.GroupID
FROM VirtualMachineTemplate VMT LEFT OUTER JOIN 
	WorkflowInfo MOD ON MOD.WorkflowID = VMT.WorkflowID  LEFT OUTER JOIN 
	GroupToTemplate GT ON GT.VirtualMachineTemplateID = VMT.VirtualMachineTemplateID
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportVMsCreatedDateRange]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportVMsCreatedDateRange]
GO
CREATE PROCEDURE [dbo].[usp_ReportVMsCreatedDateRange]
	@pGroupID VARCHAR(128) = NULL,
	@pHostID VARCHAR(128) = NULL,
	@pStartDate DateTime = NULL,
	@pEndDate DateTime = NULL

AS
BEGIN
	SET NOCOUNT ON;
	SELECT * FROM (
		select
			VM.*
		from 
			VirtualMachineView VM
	) AS TT WHERE
		GroupID IS NOT NULL AND			
		(GroupID = @pGroupID OR @pGroupID IS NULL) AND
		(HostID = @pHostID OR @pHostID IS NULL) and
		(VMCreationDate >= @pStartDate OR @pStartDate IS NULL) AND
		(VMCreationDate <= [dbo].[getNextDayBegining](@pEndDate) OR @pEndDate IS NULL)		
	ORDER BY 
		VMCreationDate, Owner, VirtualMachineName DESC
	END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportVMsExpireDateRange]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportVMsExpireDateRange]
GO
CREATE PROCEDURE [dbo].[usp_ReportVMsExpireDateRange]
	@pGroupID VARCHAR(128) = NULL,
	@pOwner VARCHAR(128) = NULL,
	@pEndDate DateTime = NULL

AS
BEGIN
	SET NOCOUNT ON;
	select
		VM.* 
	from 
		VirtualMachineView VM 
	where
		VM.Expires IS NOT NULL AND
		(VM.GroupID = @pGroupID OR @pGroupID IS NULL) AND
		(VM.Owner = @pOwner OR @pOwner IS NULL) AND
		(VM.Expires <= [dbo].[getNextDayBegining](@pEndDate) OR @pEndDate IS NULL)		
	ORDER BY 
		VM.Expires, VM.Owner, VM.VirtualMachineName DESC
	END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportVMsAlreadyExpired]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportVMsAlreadyExpired]
GO
CREATE PROCEDURE [dbo].[usp_ReportVMsAlreadyExpired]
AS
BEGIN
	SET NOCOUNT ON;
	select *
	from 
		VirtualMachineView 
	where 
		Expires IS NOT NULL AND
		VMDeleteDate IS NULL AND 
		(Expires <= DATEDIFF(D, 1, GETDATE()))		

	ORDER BY 
		Expires, Owner, VirtualMachineName DESC
	END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportVMsDeletedDateRange]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportVMsDeletedDateRange]
GO
CREATE PROCEDURE [dbo].[usp_ReportVMsDeletedDateRange]
	@pStartDate DateTime = NULL,
	@pEndDate DateTime = NULL

AS
BEGIN
	SET NOCOUNT ON;
	SELECT * FROM (
		select *
		from 
			VirtualMachineHistory VM
	) AS TT 
	where 
		VMDeleteDate IS NOT NULL AND
		(VMDeleteDate >= @pStartDate OR @pStartDate IS NULL) AND
		(VMDeleteDate <= [dbo].[getNextDayBegining](@pEndDate) OR @pEndDate IS NULL)		
	ORDER BY 
		VMDeleteDate, Owner, VirtualMachineName DESC
	END
GO


IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportHostCapacityChart]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportHostCapacityChart]
GO
CREATE PROCEDURE [dbo].[usp_ReportHostCapacityChart]
	@pHostID VARCHAR(128) = NULL
AS
BEGIN
	SET NOCOUNT ON;
	IF @pHostID IS NULL 
	BEGIN 
		SELECT SUM(NumOfVMsCreated), 
				SUM(HostReservedStorageGB) HostReservedStorageGB, 
				SUM(HostUsedStorageGB) HostUsedStorageGB, 
				SUM(HostFreeStorageGB) HostFreeStorageGB, 
				CAST(SUM(HostReservedStorageTB) as decimal(28,3)) HostReservedStorageTB, 
				CAST(SUM(HostAllocatedStorageTB) as decimal(28,3)) HostAllocatedStorageTB, 
				CAST(SUM(HostFreeStorageTB) as decimal(28,3)) HostFreeStorageTB, 
				SUM(HostAllocatedStorageTB)*100/NULLIF(SUM(HostReservedStorageTB),0) PercentStorageAllocated,
				SUM(HostReservedMemoryGB) HostReservedMemoryGB, 
				SUM(HostAllocatedMemoryGB) HostAllocatedMemoryGB, 
				SUM(HostFreeMemoryGB) HostFreeMemoryGB, 
				SUM(HostAllocatedMemoryGB)*100/NULLIF(SUM(HostReservedMemoryGB),0) PercentMemoryAllocated,
				'allhostid' HostID,
				'All Hosts' HostName
		FROM HostCapacityView
	END
	ELSE 
		SELECT * FROM HostCapacityView WHERE HostID = @pHostID
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportHostVirtualMachines]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportHostVirtualMachines]
GO
CREATE PROCEDURE [dbo].[usp_ReportHostVirtualMachines]
	@pHostID VARCHAR(128) = NULL
AS
BEGIN
	SET NOCOUNT ON;
	select *
	from 
		VirtualMachineView
	WHERE 
		(HostID = @pHostID OR @pHostID IS NULL)
	ORDER BY HostName, VirtualMachineName
	END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportTemplateCapacity]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportTemplateCapacity]
GO
CREATE PROCEDURE [dbo].[usp_ReportTemplateCapacity]
	@pTemplateID VARCHAR(128) = NULL
AS
BEGIN
	SET NOCOUNT ON;
	select 
		c.*,
		(SELECT dbo.GetGroupNamesForTemplate(c.VirtualMachineTemplateID)) GroupNames

	from TemplateCapacitySummaryView as c
	WHERE 
		c.VirtualMachineTemplateID IS NOT NULL AND
		(c.VirtualMachineTemplateID = @pTemplateID OR @pTemplateID IS NULL)
	ORDER BY c.VirtualMachineTemplateName
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportOwnerVirtualMachines]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportOwnerVirtualMachines]
GO
CREATE PROCEDURE [dbo].[usp_ReportOwnerVirtualMachines]
	@pOwner VARCHAR(128) = NULL
AS
BEGIN
	SET NOCOUNT ON;
	select *	
	from 
		VirtualMachineView
	WHERE 
		(Owner = @pOwner OR @pOwner IS NULL) AND 
		Owner is not null
	ORDER BY Owner, VirtualMachineName
	END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportOwnerCapacity]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportOwnerCapacity]
GO
CREATE PROCEDURE [dbo].[usp_ReportOwnerCapacity]
	@pOwner VARCHAR(128) = NULL
AS
BEGIN
	SET NOCOUNT ON;
	select Owner, count(VirtualMachineID) NumOfVMsCreated
		,dbo.DivideAsDecimal(SUM(VMTotalMemoryMB), 1024) MemoryGB
		,dbo.DivideAsDecimal(SUM(VMTotalStorageGB), 1024) DiskTB
		,dbo.DivideAsDecimal(SUM(VMUsedStorageGB), 1024) UsedDiskTB
        ,dbo.Percentage(SUM(VMTotalStorageGB)-SUM(VMUsedStorageGB), SUM(VMTotalStorageGB)) PercentSpaceEfficiency   
	
	from 
		VirtualMachineView
	WHERE 
		(Owner = @pOwner OR @pOwner IS NULL)
	GROUP BY Owner
	ORDER BY Owner
	END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_VirtualMachineStates]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_VirtualMachineStates]
GO
CREATE PROCEDURE [dbo].[usp_VirtualMachineStates]
AS
BEGIN
	SET NOCOUNT ON;
	select distinct VirtualMachineState ID, VirtualMachineState Name from VirtualMachine where VirtualMachineState is not null
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportPowerStateVirtualMachines]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportPowerStateVirtualMachines]
GO
CREATE PROCEDURE [dbo].[usp_ReportPowerStateVirtualMachines]
	@pGroupId VARCHAR(128) = NULL,
	@pStateId VARCHAR(128) = NULL
AS
BEGIN
	SET NOCOUNT ON;
	select *
	from VirtualMachineView
	WHERE 
		(GroupID = @pGroupId OR @pGroupId IS NULL) AND 
		(VirtualMachineState = @pStateId OR @pStateId IS NULL) AND 
		VirtualMachineState is not null
	ORDER BY VirtualMachineName, HostName	
	END
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[RogueVirtualMachineView]'))
DROP VIEW [dbo].[RogueVirtualMachineView]
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[RogueVirtualMachineView]
AS
	select *
	from VirtualMachineView
	WHERE IsRogue = 1

GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportUpdateGetRougeMachines]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportUpdateGetRougeMachines]
GO
CREATE PROCEDURE [dbo].[usp_ReportUpdateGetRougeMachines]
AS
BEGIN
	SELECT * FROM RogueVirtualMachineView
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportUpdateGetMissingMachines]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportUpdateGetMissingMachines]
GO
CREATE PROCEDURE [dbo].[usp_ReportUpdateGetMissingMachines]
AS
BEGIN
	SELECT * FROM VirtualMachineView WHERE IsMissing = 1 and IsTemplate = 0 and HostReservationID is not null
END
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[TemplatesView]'))
DROP VIEW [dbo].[TemplatesView]
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[TemplatesView]
AS
SELECT     
	VMT.*
	,CASE WHEN MemoryMaxMB is not null and  MemoryMaxMB > 0 THEN convert(varchar(5), MemoryMB) + '-' + convert(varchar(5), MemoryMaxMB) ELSE convert(varchar(5), MemoryMB) END TemplateMemory
	,CASE WHEN DiskSizeMaxGB is not null and  DiskSizeMaxGB > 0 THEN convert(varchar(5), DiskSize0GB) + '-' + convert(varchar(5), DiskSizeMaxGB) ELSE convert(varchar(5), DiskSize0GB) END TemplateStorage
	,(SELECT dbo.GetGroupNamesForTemplate(VMT.VirtualMachineTemplateID)) GroupNames
FROM VMTemplateView AS VMT 
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[VirtualMachineRequestView]'))
DROP VIEW [dbo].[VirtualMachineRequestView]

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[VirtualMachineRequestView]
AS
SELECT     
	VMT.*, 
	REQ.RequestID, 
	U.UserName AS RequestUser,
	REQ.RequestDate
FROM Request REQ 
LEFT OUTER JOIN
	TemplatesView VMT ON VMT.VirtualMachineTemplateID = REQ.VirtualMachineTemplateID
LEFT OUTER JOIN
	Users U ON U.UserID = REQ.RequestUser
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[LeaseExtensionView]'))
DROP VIEW [dbo].[LeaseExtensionView]

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[LeaseExtensionView]
AS
SELECT     
	VM.*, 
	R.RequestID, 
	U.UserName AS RequestUser,
	R.RequestDate
FROM Request R 
LEFT OUTER JOIN
    VirtualMachineView VM ON VM.VirtualMachineID = R.VirtualMachineID
LEFT OUTER JOIN 
	Users U ON U.UserID = R.RequestUser
WHERE R.RequestTypeID = 1
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[HostReservationsView]'))
DROP VIEW [dbo].[HostReservationsView]
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[HostReservationsView]
AS
SELECT     
	HR.HostReservationID, 
	HR.HostReservationName,
	HR.Enabled,
	HR.HostID,
	HN.HostNicID,
--	HR.ReservationPath,
	HR.GroupID,
	HR.ReservationMemorySize/1024 ReservationMemorySize,
	HR.ReservationDiskSize ReservationDiskSize,
	HR.VirtualMachineQuota,
--	HR.ReservationPriority,
	HN.NetworkName,
	RP.ResourcePoolName,
	HOST.HostName, 
	GRP.GroupName
FROM dbo.HostReservationCompatView AS HR LEFT OUTER JOIN
     dbo.Host AS HOST ON HOST.HostID = HR.HostID LEFT OUTER JOIN 
	 dbo.ProvisioningGroup as GRP ON GRP.GroupID = HR.GroupID LEFT OUTER JOIN
	dbo.HostNicToReservation AS HNC ON HNC.HostReservationID = HR.HostReservationID LEFT OUTER JOIN
	dbo.HostNic AS HN ON HN.HostNicID = HNC.HostNicID LEFT OUTER JOIN
	dbo.ResourcePool AS RP ON RP.ResourcePoolID = HR.ResourcePoolID
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[HostReservationCostView]'))
DROP VIEW [dbo].[HostReservationCostView]
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[HostReservationCostView]
AS
SELECT *, 
	ReservationMemorySize * MemoryPerDay MemoryCost, 
	ReservationDiskSize * DiskPerDay DiskCost	 
FROM (
	SELECT     
		HR.*, 
		ISNULL(CP.MemoryCost,0) MemoryPerDay,
		ISNULL(CP.DiskCost,0) DiskPerDay
	FROM dbo.HostReservationsView AS HR LEFT OUTER JOIN
		dbo.Host AS H ON H.HostID = HR.HostID LEFT OUTER JOIN 
		dbo.ProvisioningGroup AS GRP ON GRP.GroupID = HR.GroupID LEFT OUTER JOIN 
		dbo.CostProfile CP ON H.CostProfileID = CP.CostProfileID
) TT
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardVMsStat]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardVMsStat]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardVMsStat]
AS
BEGIN
	DECLARE @vmCount int
	DECLARE @hostCount int
	DECLARE @groupCount int
	DECLARE @templateCount int
	DECLARE @reservationCount int
	DECLARE @userCount int
	DECLARE @totalCPUs int
	DECLARE @totalStorage  decimal(28,2)
	DECLARE @totalMemory  decimal(28,2)
	
	DECLARE @VRMvmCount int
	DECLARE @VRMhostCount int
	DECLARE @VRMtotalCPUs int
	DECLARE @VRMtotalStorage  decimal(28,2)
	DECLARE @VRMtotalMemory  decimal(28,2)
	
	CREATE TABLE #VMS (ID [uniqueidentifier], Name varchar(128), Value varchar(128), VRMManagedValue varchar(128)  )

	SELECT @vmCount = count(*) from VirtualMachineView WHERE IsMissing = 0
	SELECT @VRMvmCount = count(*) from VirtualMachineView WHERE IsMissing = 0 AND IsManaged = 1
	SELECT @hostCount = Count(*) from Host Where ClusterHostID is null and MachineType = 0
	SELECT @VRMhostCount = Count(*) from Host Where ClusterHostID is null and MachineType = 0 AND HostID in (SELECT HostID from AdminGroupToHost)

	SELECT @groupCount = count(*) from dbo.ProvisioningGroupView
	SELECT @templateCount = Count(*) from VirtualMachineTemplate where MachineType = 0
	SELECT @reservationCount = Count(*) from HostReservation where MachineType = 0
	SELECT @userCount = count(*) from Users where UserName in (select [Owner] from VirtualMachineView WHERE IsManaged = 1)

	select @totalCPUs = SUM(HostProcessors), @totalMemory = dbo.DivideAsDecimal(SUM(HostTotalMemoryMB), 1024) from Host where ClusterHostID is null and MachineType = 0
	select @VRMtotalCPUs = SUM(HostProcessors), @VRMtotalMemory = dbo.DivideAsDecimal(SUM(HostTotalMemoryMB), 1024) from Host WHERE ClusterHostID is null and MachineType = 0 and HostID IN (SELECT HostID from AdminGroupToHost)
	select @totalStorage = dbo.DivideAsDecimal(SUM(StorageTotalCapacityGB), 1024) from Storage where StorageID in (Select StorageID from HostToStorage where HostID in (SELECT HostID from Host Where ClusterHostID is null and MachineType = 0 ))
	select @VRMtotalStorage = dbo.DivideAsDecimal(SUM(StorageTotalCapacityGB), 1024) from Storage where StorageID in (Select StorageID from HostToStorage where HostID in (SELECT HostID from Host Where ClusterHostID is null and MachineType = 0 AND HostID in (SELECT HostID from AdminGroupToHost)))

	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Machines', @vmCount, @VRMvmCount)
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Compute resources', @hostCount, @VRMhostCount)
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES ('Total sockets', ISNULL(@totalCPUs,0), ISNULL(@VRMtotalCPUs,0))
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES ('Total memory (GB)',ISNULL(@totalMemory,0), ISNULL(@VRMtotalMemory,0))
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES ('Total storage (TB)', ISNULL(@totalStorage,0), ISNULL(@VRMtotalStorage,0))

	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Provisioning groups', null, @groupCount)
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Blueprints', null, @templateCount)
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Reservations', null, @reservationCount)
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Users', null, @userCount)

	SELECT Name, Value , VRMManagedValue from #VMS
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardCMsStat]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardCMsStat]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardCMsStat]
AS
BEGIN
	DECLARE @vmCount int
	DECLARE @hostCount int
	DECLARE @groupCount int
	DECLARE @templateCount int
	DECLARE @reservationCount int
	DECLARE @userCount int

	DECLARE @VRMvmCount int
	DECLARE @VRMhostCount int

	CREATE TABLE #VMS (ID [uniqueidentifier], Name varchar(128), Value varchar(128), VRMManagedValue varchar(128)  )

	SELECT @vmCount = count(*) from CloudMachineView WHERE IsMissing = 0
	SELECT @VRMvmCount = count(*) from CloudMachineView WHERE IsMissing = 0 AND IsManaged = 1
	SELECT @hostCount = Count(*) from Host Where MachineType = 2
	SELECT @VRMhostCount = Count(*) from Host Where MachineType = 2 AND HostID in (SELECT HostID from AdminGroupToHost)

	SELECT @groupCount = count(*) from dbo.ProvisioningGroupView
	SELECT @templateCount = Count(*) from VirtualMachineTemplate where MachineType = 2
	SELECT @reservationCount = Count(*) from HostReservation where MachineType = 2
	SELECT @userCount = count(*) from Users where UserName in (select [Owner] from CloudMachineView WHERE IsManaged = 1)

	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Machines', @vmCount, @VRMvmCount)
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Compute resources', @hostCount, @VRMhostCount)

	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Provisioning groups', null, @groupCount)
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Blueprints', null, @templateCount)
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Reservations', null, @reservationCount)
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Users', null, @userCount)

	SELECT Name, Value , VRMManagedValue from #VMS
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardPMsStat]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardPMsStat]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardPMsStat]
AS
BEGIN
	DECLARE @pmCount int
	DECLARE @userCount int
	DECLARE @templateCount int
	DECLARE @reservationCount int
	DECLARE @groupCount int
	DECLARE @totalCPUs int
	DECLARE @totalMemory  decimal(28,2)

	DECLARE @VRMpmCount int
	DECLARE @VRMtotalCPUs int
	DECLARE @VRMtotalMemory  decimal(28,2)

	CREATE TABLE #VMS (ID [uniqueidentifier], Name varchar(128), Value varchar(128), VRMManagedValue varchar(128)  )

	SELECT @pmCount = count(*) from PhysicalMachineView
	SELECT @VRMpmCount = count(*) from PhysicalMachineView WHERE IsManaged = 1
	SELECT @totalCPUs = SUM(ProcessorCount), @totalMemory = dbo.DivideAsDecimal(SUM(MemoryInMB), 1024) from PhysicalMachines where PhysicalMachineID in (select PhysicalMachineID from PhysicalMachineView)
	SELECT @VRMtotalCPUs = SUM(ProcessorCount), @VRMtotalMemory = dbo.DivideAsDecimal(SUM(MemoryInMB), 1024) from PhysicalMachines where PhysicalMachineID in (select PhysicalMachineID from PhysicalMachineView where IsManaged = 1)

	SELECT @groupCount = count(*) from dbo.ProvisioningGroupView
	SELECT @templateCount = Count(*) from VirtualMachineTemplate where MachineType = 1
	SELECT @reservationCount = Count(*) from HostReservation where MachineType = 1
	SELECT @userCount = count(*) from Users where UserName in (select [Owner] from PhysicalMachineView WHERE IsManaged = 1)

	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Machines', @pmCount, @VRMpmCount)
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Total sockets', ISNULL(@totalCPUs,0), ISNULL(@VRMtotalCPUs,0))
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Total memory (GB)',ISNULL(@totalMemory,0), ISNULL(@VRMtotalMemory,0))
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Provisioning Groups', null, @groupCount)
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Blueprints', null, @templateCount)
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Reservations', null, @reservationCount)
	INSERT INTO #VMS(Name, Value, VRMManagedValue) VALUES('Users', null, @userCount)

	SELECT Name, Value , VRMManagedValue from #VMS
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardVMsByState]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardVMsByState]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardVMsByState]
AS
BEGIN
	DECLARE @leased int
	DECLARE @permanant int
	DECLARE @tombstoned int
	DECLARE @rouge int
	DECLARE @missing int

	CREATE TABLE #VMS (ID [uniqueidentifier], Name varchar(128), Value int  )

	SELECT @leased = count(*) from VirtualMachineView where Expires >= GETDATE() and IsMissing = 0 AND IsManaged = 1
	SELECT @tombstoned = count(*) from VirtualMachineView where Expires < GETDATE() and IsMissing = 0 AND IsManaged = 1
	SELECT @permanant = count(*) from VirtualMachineView where Expires is null and IsMissing = 0 AND IsManaged = 1
	SELECT @rouge = count(*) from VirtualMachineView where IsMissing = 0 AND IsManaged = 0
	SELECT @missing = count(*) from VirtualMachineView where IsMissing = 1 AND IsManaged = 1
	
	INSERT INTO #VMS(Name, Value) VALUES('Missing', @missing)
	INSERT INTO #VMS(Name, Value) VALUES('Permanent', @permanant)
	INSERT INTO #VMS(Name, Value) VALUES('Leased', @leased)
	INSERT INTO #VMS(Name, Value) VALUES('Archived', @tombstoned)
	INSERT INTO #VMS(Name, Value) VALUES('Unmanaged', @rouge)

	SELECT Name, Value from #VMS
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardCMsByState]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardCMsByState]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardCMsByState]
AS
BEGIN
	DECLARE @leased int
	DECLARE @permanant int
	DECLARE @missing int
	DECLARE @rouge int

	CREATE TABLE #VMS (ID [uniqueidentifier], Name varchar(128), Value int  )

	SELECT @leased = count(*) from CloudMachineView where Expires is not null and IsMissing = 0 AND IsManaged = 1
	SELECT @permanant = count(*) from CloudMachineView where Expires is null and IsMissing = 0 AND IsManaged = 1
	SELECT @missing = count(*) from CloudMachineView where IsMissing = 1 AND IsManaged = 1
	SELECT @rouge = count(*) from CloudMachineView where IsMissing = 0 AND IsManaged = 0
	
	INSERT INTO #VMS(Name, Value) VALUES('Missing', @missing)
	INSERT INTO #VMS(Name, Value) VALUES('Permanent', @permanant)
	INSERT INTO #VMS(Name, Value) VALUES('Leased', @leased)
	INSERT INTO #VMS(Name, Value) VALUES('Unmanaged', @rouge)

	SELECT Name, Value from #VMS
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardVMCountByTemplate]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardVMCountByTemplate]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardVMCountByTemplate]
	@machineType int = 0
AS
BEGIN
	DECLARE @templates TABLE
	(
		Id uniqueidentifier,
		Name VARCHAR(256),
		NumberOfVMs int
	)
	DECLARE @top5 TABLE
	(
		[Order] int IDENTITY(1,1),
		Id uniqueidentifier,
		Name VARCHAR(256),
		NumberOfVMs int
	)
	DECLARE @rowCount int

	INSERT INTO @templates
	SELECT VirtualMachineTemplateID as Id, VirtualMachineTemplateName as Name, COUNT(*) as NumberOfVMs
	FROM uv_VirtualMachineView 
	WHERE IsManaged = 1 AND IsTemplate = 0 AND MachineType = @machineType
	GROUP BY VirtualMachineTemplateID, VirtualMachineTemplateName

	INSERT INTO @top5 
	SELECT top 5 Id, Name, NumberOfVMs FROM @templates ORDER BY NumberOfVMs DESC

	SELECT @rowCount = count(*) FROM @templates
	IF @rowCount > 5
	BEGIN 
		INSERT INTO @top5 
		SELECT NULL, 'All other blueprints' AS Name, SUM(NumberOfVMs) AS NumberOfVMs FROM @templates 
		WHERE Id not in (SELECT Id FROM @top5)
	END

	SELECT Name, NumberOfVMs FROM @top5 ORDER BY [Order] DESC
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardVMCountByGroup]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardVMCountByGroup]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardVMCountByGroup]
	@machineType int = 0
AS
BEGIN
	DECLARE @groups TABLE
	(
		Id uniqueidentifier,
		Name VARCHAR(256),
		NumberOfVMs int
	)
	DECLARE @top5 TABLE
	(
		[Order] int IDENTITY(1,1),
		Id uniqueidentifier,
		Name VARCHAR(256),
		NumberOfVMs int
	)
	DECLARE @rowCount int

	INSERT INTO @groups
	SELECT GroupID as Id, GroupName as Name, COUNT(*) as NumberOfVMs
	FROM uv_VirtualMachineView 
	WHERE GroupId is not null AND IsTemplate = 0 AND MachineType = @machineType
	GROUP BY GroupID, GroupName

	INSERT INTO @top5 
	SELECT top 5 Id, Name, NumberOfVMs FROM @groups ORDER BY NumberOfVMs DESC

	SELECT @rowCount = count(*) FROM @groups
	IF @rowCount > 5
	BEGIN 
		INSERT INTO @top5 
		SELECT NULL, 'All other groups' AS Name, SUM(NumberOfVMs) AS NumberOfVMs FROM @groups 
		WHERE Id not in (SELECT Id FROM @top5)
	END

	SELECT Name, NumberOfVMs FROM @top5 ORDER BY [Order] DESC
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardVMCountByReservation]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardVMCountByReservation]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardVMCountByReservation]
	@machineType int = 0
AS
BEGIN
	DECLARE @reservations TABLE
	(
		Id uniqueidentifier,
		Name VARCHAR(256),
		NumberOfVMs int
	)
	DECLARE @top5 TABLE
	(
		[Order] int IDENTITY(1,1),
		Id uniqueidentifier,
		Name VARCHAR(256),
		NumberOfVMs int
	)
	DECLARE @rowCount int

	INSERT INTO @reservations
	SELECT HostReservationID as Id, HostReservationName as Name, COUNT(*) as NumberOfVMs
	FROM uv_VirtualMachineView 
	WHERE HostReservationId is not null  AND IsTemplate = 0 AND MachineType = @machineType
	GROUP BY HostReservationID, HostReservationName

	INSERT INTO @top5 
	SELECT top 5 Id, Name, NumberOfVMs FROM @reservations ORDER BY NumberOfVMs DESC

	SELECT @rowCount = count(*) FROM @reservations
	IF @rowCount > 5
	BEGIN 
		INSERT INTO @top5 
		SELECT NULL, 'All other reservations' AS Name, SUM(NumberOfVMs) AS NumberOfVMs FROM @reservations 
		WHERE Id not in (SELECT Id FROM @top5)
	END

	SELECT Name, NumberOfVMs FROM @top5 ORDER BY [Order] DESC
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardPMsByState]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardPMsByState]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardPMsByState]
AS
BEGIN
      DECLARE @leased int
      DECLARE @permanant int
      DECLARE @reserved int
      DECLARE @unmanaged int

      CREATE TABLE #VMS (ID [uniqueidentifier], Name varchar(128), Value int  )

      SELECT @leased = count(*) from PhysicalMachineView where Expires is not null and IsManaged = 1
      SELECT @permanant = count(*) from PhysicalMachineView where Expires is null and IsManaged = 1
      SELECT @reserved = count(*) from PhysicalMachineView where IsManaged = 0 and PhysicalMachineID in (select PhysicalMachineID from PhysicalMachines where HostReservationID is not null)
      SELECT @unmanaged = count(*) from PhysicalMachineView where IsManaged = 0 and PhysicalMachineID in (select PhysicalMachineID from PhysicalMachines where HostReservationID is null)
      
      INSERT INTO #VMS(Name, Value) VALUES('Permanent', @permanant)
      INSERT INTO #VMS(Name, Value) VALUES('Leased', @leased)
      INSERT INTO #VMS(Name, Value) VALUES('Reserved', @reserved)
      INSERT INTO #VMS(Name, Value) VALUES('Unmanaged', @unmanaged)

      SELECT Name, Value from #VMS
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardVMCountByUser]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardVMCountByUser]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardVMCountByUser]
AS
BEGIN
	DECLARE @rowCount int
	CREATE TABLE #VMS1 (UserName varchar(128), NumberOfVMs int  )
	CREATE TABLE #VMS2 (UserName varchar(128), NumberOfVMs int  )

	INSERT INTO #VMS1(UserName, NumberOfVMs) 
		SELECT Owner, count(*) from VirtualMachine GROUP BY Owner
	INSERT INTO #VMS2(UserName, NumberOfVMs) 
		SELECT top 5 * from #VMS1 order by NumberOfVMs desc
	SELECT @rowCount = count(*) FROM #VMS1
	IF @rowCount > 5
	BEGIN
		INSERT INTO #VMS2(UserName, NumberOfVMs) 
			(SELECT 'All other Users', SUM(NumberOfVMs) NumberOfVMs from #VMS1 
				WHERE UserName not in (SELECT UserName FROM #VMS2))
	END
	SELECT UserName, NumberOfVMs FROM #VMS2
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardVMUsageByGroup]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardVMUsageByGroup]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardVMUsageByGroup]
AS
BEGIN
	DECLARE @rowCount int
	CREATE TABLE #VMS (ID [uniqueidentifier], Name varchar(128), Used int, Free int  )
	INSERT INTO #VMS(ID, Name, Used, Free) 
		SELECT top 5 GroupID, GroupName, VirtualMachineQuota-FreeVirtualMachineQuota, FreeVirtualMachineQuota from ProvisioningGroupCapacityView order by VirtualMachineQuota desc
	SELECT @rowCount = count(*) FROM ProvisioningGroupCapacityView
	IF @rowCount > 5 
	BEGIN
		INSERT INTO #VMS(ID, Name, Used, Free) 
			(SELECT null, 'All other Groups', SUM(VirtualMachineQuota-FreeVirtualMachineQuota), SUM(FreeVirtualMachineQuota) from ProvisioningGroupCapacityView 
				WHERE GroupID not in (SELECT ID FROM #VMS))
	END
	SELECT Name as GroupName, Used, Free FROM #VMS
END
GO


IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardMemoryByGroup]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardMemoryByGroup]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardMemoryByGroup]
AS
BEGIN
	DECLARE @rowCount int
	CREATE TABLE #VMS (ID [uniqueidentifier], Name varchar(128), Used int, Free int  )

	INSERT INTO #VMS(ID, Name, Used, Free) 
		SELECT top 5 GroupID, GroupName, GroupUsedMemoryGB, GroupFreeMemoryGB from ProvisioningGroupCapacityView order by GroupUsedMemoryGB desc
	SELECT @rowCount = count(*) FROM ProvisioningGroupCapacityView
	IF @rowCount > 5
	BEGIN
		INSERT INTO #VMS(ID, Name, Used, Free) 
			(SELECT null, 'All other Groups', SUM(GroupUsedMemoryGB), SUM(GroupFreeMemoryGB) from ProvisioningGroupCapacityView 
				WHERE GroupID not in (SELECT ID FROM #VMS))
	END
	SELECT Name as GroupName, Used UsedMemory, Free FreeMemory FROM #VMS
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardDiskByGroup]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardDiskByGroup]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardDiskByGroup]
AS
BEGIN
	DECLARE @rowCount int
	CREATE TABLE #VMS (ID [uniqueidentifier], Name varchar(128), Allocated decimal(28,3), Free decimal(28,3), Used decimal(28,3), Savings decimal(28,3)  )

	INSERT INTO #VMS(ID, Name, Allocated, Free, Used, Savings) 
		SELECT top 5 GroupID, GroupName, GroupUsedStorageTB, GroupFreeStorageTB, GroupUsedFlCloneStorageTB, GroupFlCloneStorageSavingsTB from ProvisioningGroupCapacityView order by GroupUsedStorageTB desc
	SELECT @rowCount = count(*) FROM ProvisioningGroupCapacityView
	IF @rowCount > 5 
	BEGIN
		INSERT INTO #VMS(ID, Name, Allocated, Free, Used, Savings) 
			(SELECT null, 'All other Groups', SUM(GroupUsedStorageTB), SUM(GroupFreeStorageTB), SUM(GroupUsedFlCloneStorageTB), SUM(GroupFlCloneStorageSavingsTB) from ProvisioningGroupCapacityView 
				WHERE GroupID not in (SELECT ID FROM #VMS))
	END
	SELECT Name as GroupName, Allocated AllocatedDisk, Free FreeDisk, Used UsedDisk, Savings FROM #VMS
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardMemoryByTemplate]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardMemoryByTemplate]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardMemoryByTemplate]
AS
BEGIN
	DECLARE @rowCount int
	CREATE TABLE #VMS (ID [uniqueidentifier], Name varchar(128), Used int  )

	INSERT INTO #VMS(ID, Name, Used) 
		SELECT top 5 VirtualMachineTemplateID, VirtualMachineTemplateName, Memory from TemplateCapacitySummaryView order by Memory desc
	SELECT @rowCount = count(*) FROM TemplateCapacitySummaryView
	IF @rowCount > 5 
	BEGIN
		INSERT INTO #VMS(ID, Name, Used) 
			(SELECT null, 'All other Blueprints', SUM(Memory) from TemplateCapacitySummaryView 
				WHERE VirtualMachineTemplateID not in (SELECT ID FROM #VMS))
	END
	SELECT Name as VirtualMachineTemplateName, Used UsedMemory FROM #VMS
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardDiskByTemplate]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardDiskByTemplate]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardDiskByTemplate]
AS
BEGIN
	DECLARE @rowCount int
	CREATE TABLE #VMS (ID [uniqueidentifier], Name varchar(128), Allocated decimal(28,3), Used decimal(28,3)  )

	INSERT INTO #VMS(ID, Name, Allocated, Used) 
		SELECT top 5 VirtualMachineTemplateID, VirtualMachineTemplateName, DiskTB, UsedDiskTB from TemplateCapacitySummaryView order by Disk desc
	SELECT @rowCount = count(*) FROM TemplateCapacitySummaryView 
	IF @rowCount > 5 
	BEGIN
		INSERT INTO #VMS(ID, Name, Allocated, Used) 
			(SELECT null, 'All other Blueprints', SUM(DiskTB), SUM(UsedDiskTB) from TemplateCapacitySummaryView 
				WHERE VirtualMachineTemplateID not in (SELECT ID FROM #VMS))
	END
	SELECT Name as VirtualMachineTemplateName, Allocated AllocatedDisk, Used UsedDisk, (Allocated - Used) Savings FROM #VMS
	END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardMemoryUsage]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardMemoryUsage]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardMemoryUsage]
AS
BEGIN
	DECLARE @total int
	DECLARE @used int
	DECLARE @free int
	DECLARE @vmCount int
	DECLARE @hostCount int

	CREATE TABLE #VMS (ID [uniqueidentifier], Name varchar(128), Value int  )

	SELECT @total = SUM(ReservationMemorySize) FROM HostReservationsView
	SELECT @used = SUM(VMTotalMemoryMB) FROM VirtualMachineView
	SELECT @free = @total - @used

	INSERT INTO #VMS(Name, Value) VALUES('Total Reserved Memory', @total)
	INSERT INTO #VMS(Name, Value) VALUES('Total Free Memory', @free)
	INSERT INTO #VMS(Name, Value) VALUES('% Free Memory', @free/NULLIF(@total,0))

	SELECT @vmCount = count(*) from VirtualMachine
	SELECT @hostCount = Count(*) from Host
	INSERT INTO #VMS(Name, Value) VALUES('Avg. Memory/Host', @total / NULLIF(@hostCount,0))
	INSERT INTO #VMS(Name, Value) VALUES('Avg. Memory/VM', @total / NULLIF(@vmCount,0))

	SELECT Name, Value FROM #VMS
	END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardDiskUsage]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardDiskUsage]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardDiskUsage]
AS
BEGIN
	DECLARE @total int
	DECLARE @used int
	DECLARE @free int
	DECLARE @vmCount int
	DECLARE @hostCount int

	CREATE TABLE #VMS (ID [uniqueidentifier], Name varchar(128), Value int  )

	SELECT @total = SUM(ReservationDiskSize) FROM HostReservationsView
	SELECT @used = SUM(VMTotalStorageGB) FROM VirtualMachineView
	SELECT @free = @total - @used

	INSERT INTO #VMS(Name, Value) VALUES('Total Reserved Disk', @total)
	INSERT INTO #VMS(Name, Value) VALUES('Total Free ', @free)
	INSERT INTO #VMS(Name, Value) VALUES('% Free Disk', @free/NULLIF(@total,0))

	SELECT @vmCount = count(*) from VirtualMachine
	SELECT @hostCount = Count(*) from Host
	INSERT INTO #VMS(Name, Value) VALUES('Avg. Disk/Host', @total / NULLIF(@hostCount,0))
	INSERT INTO #VMS(Name, Value) VALUES('Avg. Disk/VM', @total / NULLIF(@vmCount,0))

	SELECT Name, Value FROM #VMS
	END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardVMsProvisioned]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardVMsProvisioned]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardVMsProvisioned]
AS
BEGIN
	DECLARE @created int
	DECLARE @deleted int
	DECLARE @expired int

	DECLARE @dtTo datetime
	DECLARE @dtFrom datetime

	CREATE TABLE #VMS (Name varchar(128), Created int, Deleted int, Expired int, starttime datetime, endtime datetime)

	SELECT @dtTo = DATEADD(day, DATEDIFF(day, 0, GETDATE()), 0)
	SELECT @created = COUNT(*) from VirtualMachine where VMCreationDate >= @dtTo
	SELECT @deleted = COUNT(*) from VirtualMachine where VMDeleteDate >= @dtTo
	SELECT @expired = COUNT(*) from VirtualMachine where Expires >= @dtTo AND Expires <= GETDATE() -- takes care of what is expired today until now
	INSERT INTO #VMS VALUES ('Today', @created, @deleted, @expired, @dtTo, null)

	SELECT @dtFrom = DATEADD(day, DATEDIFF(day, 1, GETDATE()), 0)
	SELECT @dtTo = DATEADD(day, DATEDIFF(day, 0, GETDATE()), 0)
	SELECT @created = COUNT(*) from VirtualMachine where VMCreationDate >= @dtFrom AND VMCreationDate < @dtTo
	SELECT @deleted = COUNT(*) from VirtualMachine where VMDeleteDate >= @dtFrom AND VMDeleteDate < @dtTo
	SELECT @expired = COUNT(*) from VirtualMachine where Expires >= @dtFrom AND Expires < @dtTo
	INSERT INTO #VMS VALUES ('Yesterday', @created, @deleted, @expired, @dtFrom, @dtTo)

	SELECT @dtFrom = [dbo].get_week_start(GETDATE())
	SELECT @dtTo = DATEADD(day, DATEDIFF(day, 0, GETDATE()), 0)
	SELECT @created = COUNT(*) from VirtualMachine where VMCreationDate >= @dtFrom AND VMCreationDate < @dtTo
	SELECT @deleted = COUNT(*) from VirtualMachine where VMDeleteDate >= @dtFrom AND VMDeleteDate < @dtTo
	SELECT @expired = COUNT(*) from VirtualMachine where Expires >= @dtFrom AND Expires < @dtTo
	INSERT INTO #VMS VALUES ('This Week', @created, @deleted, @expired, @dtFrom, @dtTo)

	SELECT @dtTo = @dtFrom
	SELECT @dtFrom = [dbo].get_week_start(DATEADD(day, DATEDIFF(day, 1, @dtTo), 0))
	SELECT @created = COUNT(*) from VirtualMachine where VMCreationDate >= @dtFrom AND VMCreationDate < @dtTo 
	SELECT @deleted = COUNT(*) from VirtualMachine where VMDeleteDate >= @dtFrom AND VMDeleteDate < @dtTo
	SELECT @expired = COUNT(*) from VirtualMachine where Expires >= @dtFrom AND Expires < @dtTo
	SELECT @expired = COUNT(*) from VirtualMachine where Expires >= @dtFrom AND Expires < @dtTo
	INSERT INTO #VMS VALUES ('Last Week', @created, @deleted, @expired, @dtFrom, @dtTo)

	SELECT @dtTo = GETDATE()
	SELECT @dtFrom = dateadd(m,datediff(m,0, @dtTo),0)
	SELECT @created = COUNT(*) from VirtualMachine where VMCreationDate >= @dtFrom AND VMCreationDate < @dtTo 
	SELECT @deleted = COUNT(*) from VirtualMachine where VMDeleteDate >= @dtFrom AND VMDeleteDate < @dtTo
	SELECT @expired = COUNT(*) from VirtualMachine where Expires >= @dtFrom AND Expires < @dtTo
	INSERT INTO #VMS VALUES ('This Month', @created, @deleted, @expired, @dtFrom, @dtTo)

	SELECT @dtTo = @dtFrom
	SELECT @dtFrom = dateadd(m,datediff(m,0, DATEADD(day, DATEDIFF(day, 1, @dtFrom), 0)),0)
	SELECT @created = COUNT(*) from VirtualMachine where VMCreationDate >= @dtFrom AND VMCreationDate < @dtTo 
	SELECT @deleted = COUNT(*) from VirtualMachine where VMDeleteDate >= @dtFrom AND VMDeleteDate < @dtTo
	SELECT @expired = COUNT(*) from VirtualMachine where Expires >= @dtFrom AND Expires < @dtTo
	INSERT INTO #VMS VALUES ('Last Month', @created, @deleted, @expired, @dtFrom, @dtTo)

	SELECT Name, Created, Deleted, Expired, starttime, endtime  FROM #VMS
	END
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[UserLogViewInterim]'))
DROP VIEW [dbo].[UserLogViewInterim]
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[UserLogViewInterim]
AS
SELECT     
	UserLogId, [UserName], [Timestamp], [Type], [Message], GroupID,
	substring([Message], charIndex('Machine ', [Message])+8, lEN([Message])) STRAFTERVM, 
	charIndex(':', [Message])- charIndex('Machine ', [Message])-8 COLONID,
	Status = CASE [Type] WHEN 0 THEN 'Info'
						 WHEN 1 THEN 'Warning'
						 WHEN 2 THEN 'Error'
						 ELSE 'Unknown'
	END
FROM UserLog
GO

IF  EXISTS (SELECT * FROM sys.views WHERE object_id = OBJECT_ID(N'[dbo].[UserLogView]'))
DROP VIEW [dbo].[UserLogView]
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
CREATE VIEW [dbo].[UserLogView]
AS
SELECT     
	UserLogId, [UserName], [Timestamp], [Type], [Message], GroupID,
	VirtualMachineName = CASE WHEN COLONID > 0 THEN rtrim(ltrim(substring(STRAFTERVM, 0, COLONID+1))) ELSE NULL END,
	[Status]
FROM [UserLogViewInterim]
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportUserLogDateRange]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportUserLogDateRange]
GO
CREATE PROCEDURE [dbo].[usp_ReportUserLogDateRange]
	@pUser VARCHAR(128) = NULL,
	@pGroupID VARCHAR(128) = NULL,
	@pVMId VARCHAR(128) = NULL,
	@pStartDate DateTime = NULL,
	@pEndDate DateTime = NULL

AS
BEGIN
	SET NOCOUNT ON;
	select 
		LOG.*,
		GroupName
	from 
		[UserLogView] LOG LEFT OUTER JOIN ProvisioningGroup GRP ON GRP.GroupID = LOG.GroupID
	where
		(LOG.[UserName] = @pUser OR @pUser IS NULL) AND
		(LOG.GroupID = @pGroupID OR @pGroupID IS NULL) AND
		(LOG.Message LIKE '%' + @pVMId + '%' OR @pVMId IS NULL) AND
		(Timestamp >= @pStartDate OR @pStartDate IS NULL) AND
		(Timestamp < [dbo].[getNextDayBegining](@pEndDate) OR @pEndDate IS NULL)		
	ORDER BY 
		Timestamp DESC
	END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportUserLogByHostDateRange]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportUserLogByHostDateRange]
GO
CREATE PROCEDURE [dbo].[usp_ReportUserLogByHostDateRange]
	@pUser VARCHAR(128) = NULL,
	@pHostID VARCHAR(128) = NULL,
	@pStartDate DateTime = NULL,
	@pEndDate DateTime = NULL

AS
BEGIN
	SELECT LOG.*, Host.HostName, GRP.GroupName
	FROM [UserLogView] LOG INNER JOIN 
		( select VirtualMachineName, HostReservationID from VirtualMachine 
		  union all
		  select VirtualMachineName, HostReservationID from VirtualMachineHistory ) VM
		ON VM.VirtualMachineName = LOG.VirtualMachineName INNER JOIN 
		HostReservation HR ON HR.HostReservationID = VM.HostReservationID INNER JOIN 
		Host ON Host.HostID = HR.HostID LEFT OUTER JOIN
		ProvisioningGroup GRP ON GRP.GroupID = LOG.GroupID
	WHERE (LOG.[UserName] = @pUser OR @pUser IS NULL) AND
		(Timestamp >= @pStartDate OR @pStartDate IS NULL) AND
		(Timestamp <= [dbo].[getNextDayBegining](@pEndDate) OR @pEndDate IS NULL) AND
		(Host.HostID = @pHostID OR @pHostID IS NULL) AND HOST.MachineType != 1
	ORDER BY 
		Timestamp DESC
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportUserLogByTemplateDateRange]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportUserLogByTemplateDateRange]
GO
CREATE PROCEDURE [dbo].[usp_ReportUserLogByTemplateDateRange]
	@pUser VARCHAR(128) = NULL,
	@pTemplateID VARCHAR(128) = NULL,
	@pStartDate DateTime = NULL,
	@pEndDate DateTime = NULL
AS
BEGIN
	SELECT LOG.*, VMT.VirtualMachineTemplateName, GRP.GroupName 
	FROM [UserLogView] LOG INNER JOIN
		( select VirtualMachineName, VirtualMachineTemplateID from VirtualMachine 
		  union all
		  select VirtualMachineName, VirtualMachineTemplateID from VirtualMachineHistory ) VM
		ON VM.VirtualMachineName = LOG.VirtualMachineName INNER JOIN
		VirtualMachineTemplate VMT ON VMT.VirtualMachineTemplateID = VM.VirtualMachineTemplateID LEFT OUTER JOIN
		ProvisioningGroup GRP ON GRP.GroupID = LOG.GroupID
	WHERE (LOG.[UserName] = @pUser OR @pUser IS NULL) AND
		(Timestamp >= @pStartDate OR @pStartDate IS NULL) AND
		(Timestamp <= [dbo].[getNextDayBegining](@pEndDate) OR @pEndDate IS NULL) AND
		(VMT.VirtualMachineTemplateID = @pTemplateID OR @pTemplateID IS NULL)
	ORDER BY 
		Timestamp DESC
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportDashboardCapacitySummary]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportDashboardCapacitySummary]
GO
CREATE PROCEDURE [dbo].[usp_ReportDashboardCapacitySummary]
AS
BEGIN
	DECLARE @physicalMemory decimal(28,2)
	DECLARE @reservedMemory decimal(28,2)
	DECLARE @usedMemory decimal(28,2)
	DECLARE @allocMemory decimal(28,2)
	DECLARE @physicalDisk decimal(28,2)
	DECLARE @reservedDisk decimal(28,2)
	DECLARE @allocatedDisk decimal(28,2)
	DECLARE @usedDisk decimal(28,2)

	CREATE TABLE #VMS (Resource nvarchar(128), Physical decimal(28,2), Reserved decimal(28,2), PerCentReserved decimal(28,2), Allocated decimal(28,2), PercentAllocated decimal(28,2), InUse decimal(28,2))

	select @physicalDisk = SUM(StorageTotalCapacityGB) from Storage where StorageID in (Select StorageID from HostToStorage where HostID in (SELECT HostID from ManagedHostsView where MachineType = 0))

	SELECT @physicalMemory = SUM(MemoryPhysical),
		   @reservedMemory = SUM(MemoryReserved),
		   @reservedDisk = SUM(StorageReserved),
		   @allocMemory = SUM(MemoryAllocated),
		   @allocatedDisk = SUM(StorageAllocated),
	       @usedDisk = SUM(StorageUsed)
	FROM ManagedHostsCapacityView 
	WHERE MachineType = 0

	INSERT INTO #VMS VALUES('Memory (GB)', 
			dbo.DivideAsDecimal(@physicalMemory, 1024),
			dbo.DivideAsDecimal(@reservedMemory, 1024),
			dbo.Percentage(@reservedMemory, @physicalMemory),
			dbo.DivideAsDecimal(@allocMemory, 1024),
			dbo.Percentage(@allocMemory, @reservedMemory),
			null)
	
	INSERT INTO #VMS VALUES('Storage (TB)', 
			dbo.DivideAsDecimal(@physicalDisk, 1024),
			dbo.DivideAsDecimal(@reservedDisk, 1024),
			dbo.Percentage(@reservedDisk, @physicalDisk),
			dbo.DivideAsDecimal(@allocatedDisk, 1024),
			dbo.Percentage(@allocatedDisk, @reservedDisk),
			dbo.DivideAsDecimal(@usedDisk, 1024)
			)
	SELECT * FROM #VMS
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportChargebackByGroup]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportChargebackByGroup]
GO
CREATE PROCEDURE [dbo].[usp_ReportChargebackByGroup]
	@pGroupID VARCHAR(128) = NULL,
	@pStartDate DateTime,
	@pEndDate DateTime

AS
BEGIN
	DECLARE @endDate datetime;
	SET NOCOUNT ON;
	SELECT @endDate = [dbo].[getNextDayBegining](@pEndDate);
	CREATE TABLE #ActiveDays (ID [uniqueidentifier], ChargeStart DateTime, ChargeEnd DateTime, NumOfDays int)
	INSERT INTO #ActiveDays
	SELECT	VirtualMachineID, 
			(CASE WHEN VMCreationDate < @pStartDate THEN @pStartDate ELSE VMCreationDate END),  
			(CASE WHEN ISNULL(VMDeleteDate, @endDate) < @endDate THEN VMDeleteDate ELSE @endDate END),
			0
	FROM VirtualMachinePerDayCostView 
	WHERE VMCreationDate < @endDate AND 
		(VMDeleteDate IS NULL OR VMDeleteDate > @pStartDate) AND 
		(GroupID = @pGroupID OR @pGroupID IS NULL);	
	UPDATE #ActiveDays SET NumOfDays = DATEDIFF(day, ChargeStart, ChargeEnd)
	
	CREATE TABLE #VMCosts (GroupID [uniqueidentifier], GroupName varchar(128), 
					VirtualMachineID [uniqueidentifier], VirtualMachineName varchar(128), Owner varchar(128),
					MemoryCost decimal(28,3), DiskCost decimal(28,3), DiskCostByUsed decimal(28,3), TemplateCost decimal(28,3),
					TotalCost decimal(28,3), TotalCostByUsed decimal(28,3))
	INSERT INTO #VMCosts
	SELECT 
		VMC.GroupID, VMC.GroupName,
		VMC.VirtualMachineID, VMC.VirtualMachineName,
		VMC.Owner,
		VMC.MemoryCost * AD.NumOfDays,
		VMC.DiskCost * AD.NumOfDays,
		VMC.DiskCostByUsed * AD.NumOfDays,
		VMC.TemplateCost * AD.NumOfDays,
		0, 0
	FROM [VirtualMachineCostView] VMC, #ActiveDays AD WHERE VMC.VirtualMachineID = AD.ID;
	UPDATE #VMCosts SET TotalCost = MemoryCost + DiskCost + TemplateCost, TotalCostByUsed = MemoryCost + DiskCostByUsed + TemplateCost

	CREATE TABLE #GroupCosts (GroupID [uniqueidentifier], GroupName varchar(128),
					GroupMemoryCost decimal(28,3), GroupDiskCost decimal(28,3), GroupDiskCostByUsed decimal(28,3), GroupTemplateCost decimal(28,3),
					GroupTotalCost decimal(28,3), GroupTotalCostByUsed decimal(28,3))
	INSERT INTO #GroupCosts
	SELECT 
		VMC.GroupID, VMC.GroupName,
		SUM(VMC.MemoryCost),
		SUM(VMC.DiskCost),
		SUM(VMC.DiskCostByUsed),
		SUM(VMC.TemplateCost),
		SUM(VMC.TotalCost),
		SUM(VMC.TotalCostByUsed)
	FROM #VMCosts VMC GROUP BY VMC.GroupID, VMC.GroupName;

	SELECT	VMC.*, 
			GRP.GroupMemoryCost, GRP.GroupDiskCost, GRP.GroupDiskCostByUsed, GRP.GroupTemplateCost, GRP.GroupTotalCost, GRP.GroupTotalCostByUsed,
			VMCV.MemoryCost MemoryPrice,
			VMCV.DiskCost DiskPrice,
			VMCV.TemplateCost TemplatePrice,
			VMCV.VMTotalMemoryMB,
			VMCV.VMTotalStorageGB,
			VMCV.VMUsedStorageGB,
			VMCV.VirtualMachineTemplateName, 
			VMCV.HostName,
			AD.NumOfDays
	FROM VirtualMachineCostView VMCV, #VMCosts VMC, #GroupCosts GRP, #ActiveDays AD
	WHERE VMC.GroupID = GRP.GroupID AND 
		  VMCV.VirtualMachineID = VMC.VirtualMachineID AND
		  AD.ID = VMCV.VirtualMachineID
	ORDER BY GroupName, VirtualMachineName DESC
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportChargebackByGroupReservations]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportChargebackByGroupReservations]
GO
CREATE PROCEDURE [dbo].[usp_ReportChargebackByGroupReservations]
	@pGroupID VARCHAR(128) = NULL,
	@pStartDate DateTime,
	@pEndDate DateTime
AS
BEGIN
	DECLARE @activeDays int
	DECLARE @endDate datetime;
	SET NOCOUNT ON;

	SELECT @endDate = [dbo].[getNextDayBegining](@pEndDate);
	SELECT @activeDays = DATEDIFF(day, @pStartDate, @endDate)
	
	CREATE TABLE #ReservationCosts (GroupID [uniqueidentifier], GroupName varchar(128), 
					HostReservationID [uniqueidentifier], HostReservationName varchar(128),
					MemoryCost decimal(28,3), DiskCost decimal(28,3),
					TotalCost decimal(28,3))
	INSERT INTO #ReservationCosts
	SELECT 
		HR.GroupID, HR.GroupName,
		HR.HostReservationID, HR.HostReservationName,
		HR.MemoryCost * @activeDays,
		HR.DiskCost * @activeDays,
		0
	FROM HostReservationCostView HR;
	UPDATE #ReservationCosts SET TotalCost = MemoryCost + DiskCost

	CREATE TABLE #GroupCosts (GroupID [uniqueidentifier], GroupName varchar(128),
					GroupMemoryCost decimal(28,3), GroupDiskCost decimal(28,3),
					GroupTotalCost decimal(28,3))
	INSERT INTO #GroupCosts
	SELECT 
		HR.GroupID, HR.GroupName,
		SUM(HR.MemoryCost),
		SUM(HR.DiskCost),
		SUM(HR.TotalCost)
	FROM #ReservationCosts HR GROUP BY HR.GroupID, HR.GroupName;

	SELECT	HR.*, 
			GRP.GroupMemoryCost, GRP.GroupDiskCost, GRP.GroupTotalCost,
			HRCV.MemoryCost MemoryPrice,
			HRCV.DiskCost DiskPrice,
			HRCV.ReservationMemorySize,
			HRCV.ReservationDiskSize,
			HRCV.HostName,
			@activeDays NumOfDays
	FROM HostReservationCostView HRCV, #ReservationCosts HR, #GroupCosts GRP
	WHERE HR.GroupID = GRP.GroupID AND 
		  HRCV.HostReservationID = HR.HostReservationID
	ORDER BY GroupName, HostReservationName DESC
END
GO



IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportChargebackByUser]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportChargebackByUser]
GO
CREATE PROCEDURE [dbo].[usp_ReportChargebackByUser]
	@pOwner VARCHAR(50) = NULL,
	@pStartDate DateTime,
	@pEndDate DateTime

AS
BEGIN
	DECLARE @endDate datetime;
	SET NOCOUNT ON;

	SELECT @endDate = [dbo].[getNextDayBegining](@pEndDate);
	CREATE TABLE #ActiveDays (ID [uniqueidentifier], ChargeStart DateTime, ChargeEnd DateTime, NumOfDays int)
	INSERT INTO #ActiveDays
	SELECT	VirtualMachineID, 
			(CASE WHEN VMCreationDate < @pStartDate THEN @pStartDate ELSE VMCreationDate END),  
			(CASE WHEN ISNULL(VMDeleteDate, @endDate) < @endDate THEN VMDeleteDate ELSE @endDate END),
			0
	FROM VirtualMachine 
	WHERE VMCreationDate < @endDate AND 
		(VMDeleteDate IS NULL OR VMDeleteDate > @pStartDate) AND 
		(@pOwner IS NULL OR Owner = @pOwner);
	
	UPDATE #ActiveDays SET NumOfDays = DATEDIFF(day, ChargeStart, ChargeEnd)
	
	CREATE TABLE #VMCosts (GroupID [uniqueidentifier], GroupName varchar(128), 
					VirtualMachineID [uniqueidentifier], VirtualMachineName varchar(128), Owner varchar(128),
					MemoryCost decimal(28,3), DiskCost decimal(28,3), DiskCostByUsed decimal(28,3), TemplateCost decimal(28,3),
					TotalCost decimal(28,3), TotalCostByUsed decimal(28,3))
	INSERT INTO #VMCosts
	SELECT 
		VMC.GroupID, VMC.GroupName,
		VMC.VirtualMachineID, VMC.VirtualMachineName,
		VMC.Owner,
		VMC.MemoryCost * AD.NumOfDays,
		VMC.DiskCost * AD.NumOfDays,
		VMC.DiskCostByUsed * AD.NumOfDays,
		VMC.TemplateCost * AD.NumOfDays,
		0, 0
	FROM [VirtualMachineCostView] VMC, #ActiveDays AD WHERE VMC.VirtualMachineID = AD.ID;
	UPDATE #VMCosts SET TotalCost = MemoryCost + DiskCost + TemplateCost, TotalCostByUsed = MemoryCost + DiskCostByUsed + TemplateCost

	CREATE TABLE #OwnerCosts (Owner varchar(128),
					OwnerMemoryCost decimal(28,3), OwnerDiskCost decimal(28,3), OwnerUsedDiskCost decimal(28,3), OwnerTemplateCost decimal(28,3),
					OwnerTotalCost decimal(28,3), OwnerTotalCostByUsed decimal(28,3))
	INSERT INTO #OwnerCosts
	SELECT 
		VMC.Owner,
		SUM(VMC.MemoryCost),
		SUM(VMC.DiskCost),
		SUM(VMC.DiskCostByUsed),
		SUM(VMC.TemplateCost),
		SUM(VMC.TotalCost),
		SUM(VMC.TotalCostByUsed)
	FROM #VMCosts VMC GROUP BY VMC.Owner;

	SELECT	VMC.*, 
			OW.OwnerMemoryCost, OW.OwnerDiskCost, OW.OwnerUsedDiskCost, OW.OwnerTemplateCost, OW.OwnerTotalCost, OW.OwnerTotalCostByUsed, 
			VMCV.MemoryCost MemoryPrice,
			VMCV.DiskCost DiskPrice,
			VMCV.TemplateCost TemplatePrice,
			VMCV.VMTotalMemoryMB,
			VMCV.VMTotalStorageGB,
			VMCV.VMUsedStorageGB,
			VMCV.VirtualMachineTemplateName, 
			VMCV.HostName,
			AD.NumOfDays
	FROM VirtualMachineCostView VMCV, #VMCosts VMC, #OwnerCosts OW, #ActiveDays AD
	WHERE VMC.Owner = OW.Owner AND 
		  VMCV.VirtualMachineID = VMC.VirtualMachineID AND
		  AD.ID = VMCV.VirtualMachineID
	ORDER BY Owner, VirtualMachineName DESC
END
GO
----------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportReclamationDashboardSummary]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportReclamationDashboardSummary]
GO
CREATE PROCEDURE [dbo].[usp_ReportReclamationDashboardSummary]
AS
BEGIN
	DECLARE @lastWeek int
	DECLARE @lastMonth int
	DECLARE @lastQuarter int
	DECLARE @lastYear int

	DECLARE @dtToday datetime
	DECLARE @dtLastWeek datetime
	DECLARE @dtLastMonth datetime
	DECLARE @dtLastQuarter datetime
	DECLARE @dtLastYear datetime

	SELECT @dtToday = DATEADD(day, DATEDIFF(day, 0, GETDATE()), 0)
	SELECT @dtLastWeek = datediff(day,7, @dtToday)
	SELECT @dtLastMonth = datediff(day,30, @dtToday)
	SELECT @dtLastQuarter = datediff(day,90, @dtToday)
	SELECT @dtLastYear = datediff(day,365, @dtToday)

	SELECT @lastWeek = COUNT(*) FROM ReclaimedVirtualMachineView 
		where VMDeleteDate > @dtLastWeek OR (Expires <= GETDATE() AND Expires > @dtLastWeek)
	SELECT @lastMonth = COUNT(*) FROM ReclaimedVirtualMachineView 
		where VMDeleteDate > @dtLastMonth OR (Expires <= GETDATE() AND Expires > @dtLastMonth)
	SELECT @lastQuarter = COUNT(*) FROM ReclaimedVirtualMachineView 
		where VMDeleteDate > @dtLastQuarter OR (Expires <= GETDATE() AND Expires > @dtLastQuarter)
	SELECT @lastYear = COUNT(*) FROM ReclaimedVirtualMachineView 
		where VMDeleteDate > @dtLastYear OR (Expires <= GETDATE() AND Expires > @dtLastYear)	
	--SELECT 'VirtualMachinesReclaimed' VirtualMachinesReclaimed, @lastWeek LastWeek, @lastMonth LastMonth, @lastQuarter LastQuarter, @lastYear LastYear
	CREATE TABLE #SUMMARY (Period nvarchar(128), VMCount int)
	INSERT #SUMMARY VALUES('Last 7 Days', @lastWeek)
	INSERT #SUMMARY VALUES('Last 30 Days', @lastMonth)
	INSERT #SUMMARY VALUES('Last 90 Days', @lastQuarter)
	INSERT #SUMMARY VALUES('Last 365 Days', @lastYear)
	
	SELECT * FROM #SUMMARY
	END
GO
----------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportReclamedResourcesStorageByGroup]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportReclamedResourcesStorageByGroup]
GO
CREATE PROCEDURE [dbo].[usp_ReportReclamedResourcesStorageByGroup]
	@pPeriod1 VARCHAR(50) = NULL
AS
BEGIN
	DECLARE @dtToday datetime
	DECLARE @dtPeriod datetime
	DECLARE @activeDays int
	
	SELECT @dtToday = DATEADD(day, DATEDIFF(day, 0, GETDATE()), 0)
	SELECT @dtPeriod = CASE WHEN @pPeriod1 = '7d' THEN datediff(day,7, @dtToday)
						WHEN @pPeriod1 = '30d' THEN datediff(day,30, @dtToday)
						WHEN @pPeriod1 = '90d' THEN datediff(day,90, @dtToday)
						WHEN @pPeriod1 = 'ytd' THEN DATEADD(year, DATEPART(yyyy, GETDATE())-1900, 0)
						ELSE datediff(day,365, @dtToday) END;
	SELECT @activeDays = CASE WHEN @pPeriod1 = '7d' THEN 7
						WHEN @pPeriod1 = '30d' THEN 30
						WHEN @pPeriod1 = '90d' THEN 90
						WHEN @pPeriod1 = 'ytd' THEN DATEPART(Dy, @dtToday)
						ELSE 365 END;	
	SELECT GroupID, COUNT(*) NumOfReclaimedVMs, SUM(VMCPUs), SUM(VMTotalMemoryMB), SUM(VMTotalStorageGB), SUM(VMCost) * @activeDays
		FROM ReclaimedVirtualMachineCostView
		where VMDeleteDate > @dtPeriod OR (Expires <= GETDATE() AND Expires > @dtPeriod)
	GROUP BY GroupID
	
END
GO
----------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportReclamationMemoryStorageByGroup]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportReclamationMemoryStorageByGroup]
GO
CREATE PROCEDURE [dbo].[usp_ReportReclamationMemoryStorageByGroup]
	@pPeriod VARCHAR(50) = NULL
AS
BEGIN

	CREATE TABLE #GRS (GroupID [uniqueidentifier], 
					NumOfReclaimedVMs bigint, NumOfReclaimedCPUs bigint, GroupReclaimedMemoryMB bigint, 
					GroupReclaimedStorageGB bigint, Savings decimal(28, 3))

	INSERT INTO #GRS EXEC usp_ReportReclamedResourcesStorageByGroup @pPeriod1 = @pPeriod
	
	SELECT g.*,
		ISNULL(NumOfReclaimedVMs, 0) NumOfReclaimedVMs,
		ISNULL(NumOfReclaimedCPUs, 0) NumOfReclaimedCPUs,
		ISNULL(GroupReclaimedMemoryMB, 0) GroupReclaimedMemoryMB,
		ISNULL(GroupReclaimedStorageGB, 0) GroupReclaimedStorageGB,
		dbo.DivideAsDecimal(GroupReclaimedMemoryMB, 1024) GroupReclaimedMemoryGB,
		dbo.DivideAsDecimal(GroupReclaimedStorageGB, 1024) GroupReclaimedStorageTB,
        dbo.Percentage(NumOfReclaimedVMs, NumOfReclaimedVMs + NumberOfVMs) PercentRecliamedVMs,
        ISNULL(Savings, 0) Savings
	FROM ProvisioningGroupView g LEFT OUTER JOIN #GRS r ON g.GroupID = r.GroupID
	END
GO
----------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportReclamationSavingsByGroupChart]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportReclamationSavingsByGroupChart]
GO
CREATE PROCEDURE [dbo].[usp_ReportReclamationSavingsByGroupChart]
	@pPeriod VARCHAR(50) = NULL
AS
BEGIN
	DECLARE @numOfGroups int
	CREATE TABLE #GRS (GroupID [uniqueidentifier], GroupName nvarchar(256),
					NumOfReclaimedVMs bigint, NumOfReclaimedCPUs bigint, GroupReclaimedMemoryMB bigint, 
					GroupReclaimedStorageGB bigint, Savings decimal(28, 3))
	CREATE TABLE #GRS1 (GroupID [uniqueidentifier], GroupName nvarchar(256),
					NumOfReclaimedVMs bigint, NumOfReclaimedCPUs bigint, GroupReclaimedMemoryMB bigint, 
					GroupReclaimedStorageGB bigint, Savings decimal(28, 3))

	INSERT INTO #GRS (GroupID, NumOfReclaimedVMs, NumOfReclaimedCPUs, GroupReclaimedMemoryMB, GroupReclaimedStorageGB, Savings)
		EXEC usp_ReportReclamedResourcesStorageByGroup @pPeriod1 = @pPeriod
	update #GRS set GroupName = g.GroupName FROM #GRS t, ProvisioningGroup g WHERE t.GroupID = g.GroupID
	INSERT INTO #GRS 
	SELECT GroupID, GroupName, 0, 0, 0, 0, 0 FROM ProvisioningGroupView WHERE GroupID NOT IN (SELECT GroupID FROM #GRS)	
	INSERT INTO #GRS1 SELECT top 5 * from #GRS ORDER BY NumOfReclaimedVMs desc
	SELECT @numOfGroups = COUNT(GroupID) FROM #GRS
	IF @numOfGroups > 5 
	BEGIN
		INSERT INTO #GRS1
			SELECT null, 'All other Groups', SUM(NumOfReclaimedVMs), SUM(NumOfReclaimedCPUs),
				SUM(GroupReclaimedMemoryMB), SUM(GroupReclaimedStorageGB), SUM(Savings) from #GRS 
				WHERE GroupID not in (SELECT GroupID FROM #GRS1)		
	END 	
	select * from #GRS1
END
GO

----------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportInventoryVirtualMachines]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportInventoryVirtualMachines]
GO
CREATE PROCEDURE [dbo].[usp_ReportInventoryVirtualMachines]
	@pGroupID VARCHAR(128) = NULL,
	@pHostID VARCHAR(128) = NULL,
	@pRegistered VARCHAR(50) = NULL
AS
BEGIN
	select * from VirtualMachineView
	WHERE 
		GroupID IS NOT NULL AND
		(GroupID = @pGroupID OR @pGroupID IS NULL) AND 
		(HostID = @pHostID OR @pHostID IS NULL) AND 
		((@pRegistered = 'Unregistered' AND HostReservationID is null) OR (@pRegistered = 'Registered' AND HostReservationID is not null) OR @pRegistered IS NULL) 
	ORDER BY VirtualMachineName, HostName
END
GO
----------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportReclamationSavingsByUser]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportReclamationSavingsByUser]
GO
CREATE PROCEDURE [dbo].[usp_ReportReclamationSavingsByUser]
	@pPeriod VARCHAR(50) = NULL
AS
BEGIN

	DECLARE @dtToday datetime
	DECLARE @dtPeriod datetime
	DECLARE @activeDays int
	
	SELECT @dtToday = DATEADD(day, DATEDIFF(day, 0, GETDATE()), 0)
	SELECT @dtPeriod = CASE WHEN @pPeriod = '7d' THEN datediff(day,7, @dtToday)
						WHEN @pPeriod = '30d' THEN datediff(day,30, @dtToday)
						WHEN @pPeriod = '90d' THEN datediff(day,90, @dtToday)
						WHEN @pPeriod = 'ytd' THEN DATEADD(year, DATEPART(yyyy, GETDATE())-1900, 0)
						ELSE datediff(day,365, @dtToday) END;
	SELECT @activeDays = CASE WHEN @pPeriod = '7d' THEN 7
						WHEN @pPeriod = '30d' THEN 30
						WHEN @pPeriod = '90d' THEN 90
						WHEN @pPeriod = 'ytd' THEN DATEPART(Dy, @dtToday)
						ELSE 365 END;	
	SELECT *, 
		dbo.Percentage(NumOfReclaimedVMs, NumOfReclaimedVMs + NumberOfVMs) PercentRecliamedVMs,
		dbo.DivideAsDecimal(ReclaimedMemoryMB, 1024) ReclaimedMemoryGB,
		dbo.DivideAsDecimal(ReclaimedStorageGB, 1024) ReclaimedStorageTB
	FROM (
		SELECT Owner, COUNT(*) NumOfReclaimedVMs, 
			(SELECT COUNT(*) from VirtualMachine v where v.Owner = Owner) NumberOfVMs,
			SUM(VMCPUs) NumOfReclaimedCPUs, SUM(VMTotalMemoryMB) ReclaimedMemoryMB, SUM(VMTotalStorageGB) ReclaimedStorageGB, SUM(VMCost) * @activeDays Savings
		FROM ReclaimedVirtualMachineCostView
		where VMDeleteDate > @dtPeriod OR (Expires <= GETDATE() AND Expires > @dtPeriod)
		GROUP BY Owner
	) ttt
	ORDER BY Owner
END
GO
----------------------------------------------------------------------------------------------------------------------
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[usp_ReportNetAppSavingsByGroup]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[usp_ReportNetAppSavingsByGroup]
GO
CREATE PROCEDURE [dbo].[usp_ReportNetAppSavingsByGroup]
	@pGroupID VARCHAR(128) = NULL
AS
BEGIN
	SELECT	pg.GroupName,
			COUNT(vm.VirtualMachineID) as NumberOfVMs, 
			SUM(vm.VMTotalStorageGB) as TotalStorage, 
			SUM(ISNULL(vm.VMEstimatedUsedSpace,vm.VMTotalStorageGB)) as EstimatedStorageUsed,
			SUM(vm.VMTotalStorageGB - ISNULL(vm.VMEstimatedUsedSpace,vm.VMTotalStorageGB)) as SavedStorage,
			(SUM(vm.VMTotalStorageGB - ISNULL(vm.VMEstimatedUsedSpace,vm.VMTotalStorageGB)) / SUM(vm.VMTotalStorageGB))*100 as PercentSaved,
			SUM(ISNULL(cp.DiskCost,0) * vm.VMTotalStorageGB) - SUM(ISNULL(cp.DiskCost,0) * ISNULL(vm.VMEstimatedUsedSpace,vm.VMTotalStorageGB)) as TotalSavings
	FROM VirtualMachine vm
	JOIN
		HostReservation hr on hr.HostReservationID = vm.HostReservationID
	JOIN
		ProvisioningGroup pg on hr.GroupID = pg.GroupID
	JOIN
		Host h on h.HostID = vm.HostID
	LEFT JOIN
		CostProfile cp on h.CostProfileID = cp.CostProfileID
	WHERE
		pg.GroupID = @pGroupID OR @pGroupID IS NULL
	GROUP BY
		pg.GroupName
END
GO