/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.zookeeper.server.quorum;

import java.io.DataInputStream;
import java.io.BufferedOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Enumeration;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.Date;
import org.apache.zookeeper.server.quorum.QuorumPeer.QuorumServer;
import java.util.Map.Entry;
import java.util.NoSuchElementException;

import org.apache.log4j.Logger;

/**
 * This class implements a connection manager for leader election using TCP. It
 * maintains one connection for every pair of servers. The tricky part is to
 * guarantee that there is exactly one connection for every pair of servers that
 * are operating correctly and that can communicate over the network.
 * If two servers try to start a connection concurrently, then the connection
 * manager uses a very simple tie-breaking mechanism to decide which connection
 * to drop based on the server ID of the two parties. The connection
 * initiated by the peer with higher ID is retained whereas, the connection
 * initiated by a peer with lower ID is rejected.
 *
 * Two worker threads are created for each peer, one for sending messages and
 * another for receiving messages. The {@link SendWorkerThread} writes messages
 * queued by {@link FastLeaderElection}. The {@link RecvWorkerThread}
 * reads messages and inserts them into a read queue maintained by
 * {@link FastLeaderElection}.
 */

public class QuorumCnxManager {
    private static final Logger LOG = Logger.getLogger(QuorumCnxManager.class);

    /*
     * Maximum capacity of thread queues
     */

    static final int RECV_CAPACITY = 100;
    static final int SEND_CAPACITY = 1;
    static final int PACKETMAXSIZE = 1024 * 1024;
    /*
     * Maximum number of attempts to connect to a peer
     */

    static final int MAX_CONNECTION_ATTEMPTS = 2;

    /*
     * Negative counter for observer server ids.
     */

    private long observerCounter = -1;

    /*
     * Connection time out value in milliseconds
     */

    private int cnxTO = 5000;

    /*
     * Local IP address
     */
    final QuorumPeer self;

    /*
     * Mapping from Peer to Thread number
     */
    final ConcurrentHashMap<Long, SendWorker> senderWorkerMap;
    final ConcurrentHashMap<Long, ArrayBlockingQueue<ByteBuffer>> queueSendMap;
    final ConcurrentHashMap<Long, ByteBuffer> lastMessageSent;

    /*
     * Reception queue
     */
    public final ArrayBlockingQueue<Message> recvQueue;

    /*
     * Shutdown flag
     */

    boolean shutdown = false;

    /*
     * Socket timeout for connection with peers
     */

    private int peerSockTimeout = 0;

    /*
     * Listener thread
     */
    public final Listener listener;

    /*
     * Counter to count worker threads
     */
    private AtomicInteger threadCounter = new AtomicInteger(0);

    static public class Message {
        Message(ByteBuffer buffer, long sid) {
            this.buffer = buffer;
            this.sid = sid;
        }

        ByteBuffer buffer;
        long sid;
    }

    public QuorumCnxManager(QuorumPeer self) {
        this.recvQueue = new ArrayBlockingQueue<Message>(RECV_CAPACITY);
        this.queueSendMap = new ConcurrentHashMap<Long, ArrayBlockingQueue<ByteBuffer>>();
        this.senderWorkerMap = new ConcurrentHashMap<Long, SendWorker>();
        this.lastMessageSent = new ConcurrentHashMap<Long, ByteBuffer>();
        for (QuorumServer server : self.getView().values()) {
            initializeMessageQueue(server.id);
        }
        String cnxToValue = System.getProperty("zookeeper.cnxTimeout");
        if(cnxToValue != null){
            this.cnxTO = new Integer(cnxToValue);
        }
        this.self = self;

        // Starts listener thread that waits for connection requests
        listener = new Listener();
        peerSockTimeout = self.tickTime * self.syncLimit;
    }

    /**
     * Processes invoke this message to queue a message to send. Currently, only
     * leader election uses it.
     */
    public void toSend(Long sid, ByteBuffer b) {
        /*
         * If sending message to myself, then simply enqueue it (loopback).
         *  Otherwise send to the corresponding thread to send.
         */
        if (self.getId() == sid) {
            b.position(0);
            addToRecvQueue(new Message(b.duplicate(), sid));
        } else {
            ArrayBlockingQueue<ByteBuffer> bq = queueSendMap.get(sid);
            if (bq != null) {
                addToSendQueue(bq, b);
            } else {
                LOG.fatal("No queue for server " + sid);
                dumpQueueSendMap();
                return;
            }
            startWorkerifAbsent(null, sid);
        }
    }

    /**
     * Try to establish a connection with each server if one doesn't exist.
     */

    public void connectAll() {
        long sid;
        for(Enumeration<Long> en = queueSendMap.keys();
            en.hasMoreElements();){
            sid = en.nextElement();
            startWorkerifAbsent(null, sid);
        }
    }


    /**
     * Check if all queues are empty, indicating that all messages have been delivered.
     */
    boolean haveDelivered() {
        for (ArrayBlockingQueue<ByteBuffer> queue : queueSendMap.values()) {
            LOG.debug("Queue size: " + queue.size());
            if (queue.size() == 0) {
                return true;
            }
        }

        return false;
    }

    /**
     * Flag that it is time to wrap up all activities and interrupt the listener.
     */
    public void halt() {
        shutdown = true;
        LOG.debug("Halting listener");
        listener.halt();

        softHalt();
    }

    /**
     * A soft halt simply finishes workers.
     */
    public void softHalt() {
        for (SendWorker sw : senderWorkerMap.values()) {
            LOG.debug("Halting sender: " + sw);
            sw.shutdown();
        }
    }

    public ByteBuffer getLastMessageSent(Long sid) {
       return lastMessageSent.get(sid);
    }

    private void recordLastMessageSent(Long sid, ByteBuffer b) {
        lastMessageSent.put(sid, b);
    }

    /**
     * Helper method to set socket options.
     * @param sock
     *            Reference to socket
     * @param timeout
     *             value for SO_TIMEOUT
     */
    private void setSockOpts(Socket sock, int timeout) throws SocketException {
        sock.setTcpNoDelay(true);
        sock.setSoTimeout(timeout);
        sock.setKeepAlive(true);
    }

    /**
     * Helper method to close a socket.
     * @param sock
     *            Reference to socket
     */
    private void closeSocket(Socket sock) {
        if (sock != null) {
            try {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Closing socket");
                }
                sock.close();
            } catch (IOException ie) {
                LOG.error("Ignoring Exception while closing ", ie);
            }
        }
    }

    /**
     * Return number of worker threads
     */
    public long getThreadCount() {
        return threadCounter.get();
    }

    /**
     * Return reference to QuorumPeer
     */
    public QuorumPeer getQuorumPeer() {
        return self;
    }

    /**
     * Check if the server belongs to the cluster.
     */
    public boolean isIdValid(long sid) {
      if (sid != QuorumPeer.OBSERVER_ID && !self.viewContains(sid)) {
         return false;
      }
      return true;
    }

    /**
     * Thread to listen on some port
     */
    public class Listener extends Thread {

        volatile ServerSocket ss = null;

        /**
         * Sleeps on accept().
         */
        @Override
        public void run() {
            int numRetries = 0;
            while((!shutdown) && (numRetries < 3)){
                try {
                    ss = new ServerSocket();
                    ss.setReuseAddress(true);
                    int port = self.quorumPeers.get(self.getId()).electionAddr
                            .getPort();
                    InetSocketAddress addr = new InetSocketAddress(port);
                    LOG.info("My election bind port: " + addr.toString());
                    setName("myid=" + self.getId() + ":addr=" +
                          self.quorumPeers.get(self.getId()).electionAddr.toString());
                    ss.bind(addr);
                    while (!shutdown) {
                        Socket client = ss.accept();
                        setSockOpts(client, peerSockTimeout);
                        LOG.info("Accepted connection request from "
                                + client.getRemoteSocketAddress());
                        startWorkerifAbsent(client, null);
                        numRetries = 0;
                    }
                } catch (IOException e) {
                    LOG.error("Exception while listening", e);
                    numRetries++;
                    try {
                        if (ss != null) {
                            ss.close();
                        }
                        Thread.sleep(1000);
                    } catch (IOException ie) {
                        LOG.error("Error closing server socket", ie);
                    } catch (InterruptedException ie) {
                        LOG.error("Interrupted while sleeping. " +
                                  "Ignoring exception", ie);
                    }
                }
            }
            LOG.info("Leaving listener");
            if (!shutdown) {
                LOG.fatal("As I'm leaving the listener thread, "
                        + "I won't be able to participate in leader "
                        + "election any longer: "
                        + self.quorumPeers.get(self.getId()).electionAddr);
            }
        }

        /**
         * Halts this listener thread.
         */
        void halt(){
            try{
                LOG.debug("Trying to close listener: " + ss);
                if(ss != null) {
                    LOG.debug("Closing listener: " + self.getId());
                    ss.close();
                }
            } catch (IOException e){
                LOG.warn("Exception when shutting down listener: " + e);
            }
        }
    }

    /**
     * Thread to send messages. Instance waits on a queue, and send a message as
     * soon as there is one available.
     */
    class SendWorker extends Thread {
        Long sid;
        Socket sock;
        RecvWorker recvWorker;
        volatile boolean running = true;
        DataOutputStream dout;

        /**
         * An instance of this thread receives messages to send
         * through a queue and sends them to the server sid.
         *
         * @param sock
         *            Socket to remote peer. If we are supposed to initiate
         *            a connection to the peer, then this socket must be null.
         * @param sid
         *            Server identifier must be null only if we have received
         *            connection from a peer (and hence don't know the sid until
         *            we read it).
         */
        SendWorker(Socket sock, Long sid) {
            super("myid=" + self.getId() + ":");
            this.sid = sid;
            this.sock = sock;
            recvWorker = null;
        }

        /**
         * cleanup socket and threads.
         */
        private void finish() {
            running = false;
            if (sock != null) {
                closeSocket(sock);
            }
            if (recvWorker != null) {
                recvWorker.shutdown();
            }
            if (sid != null) {
                senderWorkerMap.remove(sid, this);
            }
            String id = ((sid != null) ? sid.toString() : "unknown");
            LOG.info("SendWorker exiting sid=" + id);
        }

        /**
         * Write a buffer to the socket.
         */
        private void send(ByteBuffer b) throws IOException {
            byte[] msgBytes = new byte[b.capacity()];
            try {
                b.position(0);
                b.get(msgBytes, 0, b.capacity());
            } catch (BufferUnderflowException be) {
                LOG.fatal("BufferUnderflowException ", be);
                return;
            }
            dout.writeInt(b.capacity());
            dout.write(msgBytes);
            dout.flush();
        }

        /**
         * Keep sending messages until the thread is killed or a network error
         * is encoutered.
         */
        private void processMessages() throws Exception {
             ArrayBlockingQueue<ByteBuffer> bq = queueSendMap.get(sid);
             if (bq == null) {
                 dumpQueueSendMap();
                 throw new Exception("No queue for incoming messages for " +
                         "sid=" + sid);
             }
            try {
                /**
                 * If there is nothing in the queue to send, then we
                 * send the lastMessage to ensure that the last message
                 * was received by the peer. The message could be dropped
                 * in case self or the peer shutdown their connection
                 * (and exit the thread) prior to reading/processing
                 * the last message. Duplicate messages are handled correctly
                 * by the peer.
                 *
                 * If the send queue is non-empty, then we have a recent
                 * message than that stored in lastMessage. To avoid sending
                 * stale message, we should send the message in the send queue.
                 */
                if (isSendQueueEmpty(bq)) {
                   ByteBuffer b = getLastMessageSent(sid);
                   if (b != null) {
                       LOG.debug("Attempting to send lastMessage to sid=" + sid);
                       send(b);
                   }
                }
            } catch (IOException e) {
                LOG.error("Failed to send last message to " + sid, e);
                throw e;
            }
            try {
                while (running && !shutdown && sock != null) {
                    ByteBuffer b = null;
                    try {
                        b = pollSendQueue(bq, 1000, TimeUnit.MILLISECONDS);
                        if(b != null){
                            recordLastMessageSent(sid, b);
                            send(b);
                        }
                    } catch (InterruptedException e) {
                        LOG.warn("Interrupted while waiting for message on " +
                                 "queue", e);
                    }
                }
            } catch (Exception e) {
                LOG.warn("Exception when using channel: for id " + sid
                         + " my id = " + self.getId() + " error = ", e);
                throw e;
            }
        }

        /**
         * Shutdown this thread
         */
        public void shutdown() {
            running = false;
            this.interrupt();
        }

        /**
         * Before this thread begins processing messages, it needs to complete
         * some initialization activities.
         *
         * If we are supposed to initialize a connection to the remote peer,
         * this tread connects to the peer and sends self ID. In this case,
         * it disconnects and exits if self id is less than peer server ID.
         * Otherwise, it starts sending messages.
         *
         * If we have received a connection from a peer, then this thread
         * reads peer server ID. If peer server ID is less than self ID it
         * disconnects and reconnects with the peer. For received connections,
         * this thread will kill old worker threads, if found.  We need to
         * disconnect because:
         *  1. Peer 0 connects to Peer 1. Then peer 1 will disconnect peer 0's
         *  connection and connect back to peer 0. But peer 0 may not have
         *  stopped its thread.
         *
         *  2. Peer 1 has worker to peer 0. It is waiting in poll(). Some time
         *  later, peer 0 restarts and reconnects to peer 1. To wakeup the old
         *  worker, we can either queue the lastMessageSent() and
         *  that will make poll() to return, or we can kill this thread and
         *  restart a new thread. The second approach is preferred because
         *  it eliminates the risk of queuing lastMessageSent in wrong order.
         */

        @Override
        public void run() {
            try {
                LOG.info("SendWorker started sid=" + sid);
                threadCounter.incrementAndGet();
                if (sock == null) {
                    // I am supposed to connect to remote peer
                    SendWorker osw = senderWorkerMap.putIfAbsent(sid, this);
                    if (osw != null && osw != this) {
                        /* Do nothing if a SendWorker is already present for
                         * this thread. There could be a race condition while
                         * starting these threads due to call to
                         * startWorkerifAbsent from toSend and Listener.run().
                         * SendWorker that is able to insert itself first wins.
                         */
                        throw new Exception("During connect senderWorker " +
                                            "already present for sid=" + sid);
                    }
                    sock = handleInitiatedConnection(sid);
                } else {
                    // I have received a connection from a peer
                    sid = readRemoteServerID(sock);
                    SendWorker osw = senderWorkerMap.put(sid, this);
                    if (osw != null && osw != this) {
                        String addr = sock.getInetAddress().toString();
                        // If received connection, then kill existing thread.
                        LOG.info("Shutting down older worker thread " + osw
                                + " for <"
                                + sid + "," + addr + ">");
                        osw.shutdown();
                    }
                    sock = handleReceivedConnection(sock, sid);
                }
                if (sock == null) {
                    throw new Exception("Unable to establish channel with "
                            + sid);
                }
                dout = new DataOutputStream(new BufferedOutputStream(
                                                sock.getOutputStream()));
                recvWorker = new RecvWorker(sock, sid, this);
                recvWorker.start();
                // start sending messages
                this.processMessages();
            } catch (Exception e) {
                LOG.error("Exiting SendWorker. myid=" +self.getId(), e);
            } finally {
                this.finish();
                threadCounter.decrementAndGet();
            }
        }
    }

    /**
     * Thread to receive messages. Instance waits on a socket read. If the
     * channel breaks, then removes itself from the pool of receivers.
     */
    class RecvWorker extends Thread {
        Long sid;
        Socket sock;
        volatile boolean running = true;
        SendWorker sw;
        DataInputStream din;
        RecvWorker(Socket sock, Long sid, SendWorker sw) {
            super("myid=" + self.getId() + ":");
            this.sid = sid;
            this.sock = sock;
            this.sw = sw;
        }

        /**
         * Shuts down this worker
         */
        public void shutdown() {
            running = false;
            this.interrupt();
        }

        /**
         * Placeholder cleanup method
         */
        private void finish() {
            LOG.info("RecvWorker exiting sid=" + sid);
        }

        @Override
        public void run() {
            try {
                LOG.info("RecvWorker started sid=" + sid);
                threadCounter.incrementAndGet();
                // OK to wait until socket disconnects while reading.
                setSockOpts(sock, 0);
                din = new DataInputStream(sock.getInputStream());
                while (running && !shutdown) {
                    /**
                     * Reads the first int to determine the length of the
                     * message
                     */
                    int length = din.readInt();
                    if (length <= 0 || length > PACKETMAXSIZE) {
                        throw new IOException(
                                "Received packet with invalid packet: "
                                        + length);
                    }
                    /**
                     * Allocates a new ByteBuffer to receive the message
                     */
                    if (length > PACKETMAXSIZE) {
                        throw new IOException("Invalid packet of length "
                                + length);
                    }
                    byte[] msgArray = new byte[length];
                    din.readFully(msgArray, 0, length);
                    ByteBuffer message = ByteBuffer.wrap(msgArray);
                    addToRecvQueue(new Message(message.duplicate(), sid));
                }
            } catch (Exception e) {
                LOG.warn("Connection broken for id " + sid + ", my id = " +
                        self.getId(), e);
            } finally {
               LOG.warn("Interrupting SendWorker");
               sw.shutdown();
               finish();
               threadCounter.decrementAndGet();
            }
        }
    }

    /**
     * This method is expected to be called immediately after a connection
     * request from a remote peer is accepted. It returns the first 8 bytes sent
     * by the peer.
     * @param sock
     *            Socket for remote peer
     * @return Remote peer's ID
     * @throws Exception
     *              If error is encountered while using the socket.
     */
    public Long readRemoteServerID(Socket sock) throws Exception {
        DataInputStream din = new DataInputStream(sock.getInputStream());
        Long remoteSid = null;
        try {
           remoteSid = din.readLong();
           if (!isIdValid(remoteSid)) {
              throw new Exception("Ignoring connection from a non " +
                    "cluster member sid=" + remoteSid + " IP=" +
                    sock.getInetAddress().getHostAddress());
           }
           if (remoteSid == QuorumPeer.OBSERVER_ID) {
               /*
                * Choose identifier at random. We need a value to identify
                * the connection.
                */

               remoteSid = observerCounter--;
               initializeMessageQueue(remoteSid);
               LOG.info("Setting arbitrary identifier to observer: " + remoteSid);
           }
           if (remoteSid == self.getId()) {
               LOG.error("Remote sid " + remoteSid + " from " +
                       sock.getInetAddress().getHostAddress() + " is same as my id "
                       + self.getId());
               throw new Exception("Invalid sid " + remoteSid);
           }
        } catch (Exception e) {
           LOG.error("Error while reading remote server ID ", e);
           throw e;
        }
        return remoteSid;
    }

    /**
     * Creates a message for for the provided sid.
     *
     * @param sid
     *      Server ID
     */
    private void initializeMessageQueue(Long sid) {
        ArrayBlockingQueue<ByteBuffer> bq = new ArrayBlockingQueue<ByteBuffer>(
                SEND_CAPACITY);
        queueSendMap.put(sid, bq);
    }

    /**
     * Prints {@link queueSendMap}.
     */
    private void dumpQueueSendMap() {
        LOG.info("queueSendMap size=" + queueSendMap.size());
        for(Enumeration<Long> en = queueSendMap.keys();
            en.hasMoreElements();){
            Long sid = en.nextElement();
            LOG.info("Found sid=" + sid + " in queueSendMap");
        }
    }

    /**
     * This method is used to verify if a connection between two peers is valid
     * or not. To prevent two peers from establishing two connections with each
     * other, we enforce the policy where the connection initiated by a peer
     * with higher ID is valid. Connection initiated by server with lower sid is
     * considered invalid, and therefore, should be closed by the caller.
     *
     * @param sourceSid
     *            The server ID of the peer that initiated the connect request.
     * @param destSid
     *            The server ID of the peer that accepted the connect request.
     * @param sc
     *            Socket corresponding to the <sourceSid, destSid> pair.
     * @return True if the connection is valid. Otherwise, returns false.
     */
    public boolean keepConnection(Long sourceSid, Long destSid) {
        if (sourceSid < destSid) {
            LOG.info("The sid if connection initiator " + sourceSid
                    + " is smaller than destination " + destSid);
            return false;
        }
        return true;
    }

    /**
     * Helper method to start {@link SendWorker}
     *
     * @param client
     *            Reference to Socket to remoteSid. Must be null only if
     *            we are supposed to initiate connection to the peer.
     * @param remoteSid
     *            Server Id of remote peer. Must be null only if we have
     *            received a connection from remote peer.
     */
    private void startWorkerifAbsent(Socket client, Long remoteSid) {
        if (remoteSid != null && senderWorkerMap.containsKey(remoteSid)) {
                LOG.info("Worker threads already present for " + remoteSid);
        } else {
            // Start SendWorker for this peer.
            // RecvWorker is started by SendWorker
            SendWorker sw = new SendWorker(client, remoteSid);
            sw.start();
        }
    }

    /**
     * This method initiates connection to the remote peer. It uses a blocking
     * connect call. The timeout used for this call is specified in
     * zookeeper.cnxTimeout property.
     *
     * @param remoteSid
     *            Server Id of the remote peer.
     * @return On success returns reference to Socket to remote peer.
     *         Otherwise, returns null.
     */
    public Socket doConnect(Long remoteSid) throws Exception {
        InetSocketAddress electionAddr;
        Socket sock = null;
        if (self.quorumPeers.containsKey(remoteSid)) {
            electionAddr = self.quorumPeers.get(remoteSid).electionAddr;
        } else {
            throw new Exception("Invalid server id: " + remoteSid);
        }
        try {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Opening channel to server " + remoteSid);
            }

            sock = new Socket();
            setSockOpts(sock, peerSockTimeout);
            sock.connect(self.getView().get(remoteSid).electionAddr, cnxTO);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Connected to server " + remoteSid + " " + self.getView().get(remoteSid).electionAddr);
            }
        } catch (Exception e) {
            LOG.warn("Cannot open channel to " + remoteSid
                    + " at election address " + electionAddr,
                    e);
            throw e;
        }
        return sock;
    }

    /*
     * Writes self ID to the socket.
     * @param sock
     *              Reference to the socket.
     */
    public void writeMyID(Socket sock) throws IOException {
        DataOutputStream dout = new DataOutputStream(sock.getOutputStream());
        dout.writeLong(self.getId());
        dout.flush();
    }

    /**
     * This method should be invoked after we have initiated a connection to a
     * remote peer. After connection is established self ID is sent to the
     * remote peer. The channel is closed if the peer's ID > than self, since we
     * expect the peer with higher IDs to initiate connection to us.
     *
     * @param sc
     *            Channel to remote peer
     * @param remoteSid
     *            ID of the peer
     * @return Socket
     *          null if the remote peer's ID > self ID. Otherwise,
     *          reference to the created Socket.
     */
    public Socket handleInitiatedConnection(Long remoteSid) throws Exception {
        Socket sock = null;
        try {
            // Sending id and challenge
            sock = doConnect(remoteSid);
            writeMyID(sock);
        } catch (IOException e) {
            LOG.warn("Exception writing challenge to " + remoteSid);
            if (sock != null) {
                closeSocket(sock);
            }
            throw e;
        }
        if (!keepConnection(self.getId(), remoteSid)) {
            closeSocket(sock);
            throw new Exception("Have smaller ID " + self.getId() + " than "
                    + remoteSid);
        }
        return sock;
    }

    /**
     * This method should be invoked to handle a connection request from
     * a remote peer. Based on the peer's server ID, it determines whether to
     * keep the connection or to open a new one. If peer's ID is less than self
     * ID, the accepted connection is closed, new connection request is sent
     * to the peer, and self ID is sent to the peer.
     * @param sock
     *              Socket of the accepted connection
     * @param remoteSid
     *              Remote peer server ID
     * @return
     *          Socket that should be used to communicate with the peer.
     */
    public Socket handleReceivedConnection(Socket sock, Long remoteSid) {
        try {
            if (!keepConnection(remoteSid, self.getId())) {
                closeSocket(sock);
                sock = null;
                sock = doConnect(remoteSid);
                writeMyID(sock);
            }
        } catch (Exception e) {
            LOG.warn("Exception while using socket to " + remoteSid);
            if (sock != null) {
                closeSocket(sock);
            }
            return null;
        }
        return sock;
    }

    /**
     * Inserts an element in the specified queue. If the Queue is full, this
     * method removes an element from the head of the Queue and then inserts
     * the element at the tail. It can happen that the an element is removed
     * by another thread in {@link SendWorker#processMessage() processMessage}
     * method before this method attempts to remove an element from the queue.
     * This will cause {@link ArrayBlockingQueue#remove() remove} to throw an
     * exception, which is safe to ignore.
     *
     * Unlike {@link #addToRecvQueue(Message) addToRecvQueue} this method does
     * not need to be synchronized since there is only one thread that inserts
     * an element in the queue and another thread that reads from the queue.
     *
     * @param queue
     *          Reference to the Queue
     * @param buffer
     *          Reference to the buffer to be inserted in the queue
     */
    private void addToSendQueue(ArrayBlockingQueue<ByteBuffer> queue,
          ByteBuffer buffer) {
        if (queue.remainingCapacity() == 0) {
            try {
                queue.remove();
            } catch (NoSuchElementException ne) {
                // element could be removed by poll()
                LOG.debug("Trying to remove from an empty " +
                        "Queue. Ignoring exception " + ne);
            }
        }
        try {
            queue.add(buffer);
        } catch (IllegalStateException ie) {
            // This should never happen
            LOG.fatal("Unable to insert an element in the queue " + ie);
        }
    }

    /**
     * Returns true if queue is empty.
     * @param queue
     *          Reference to the queue
     * @return
     *      true if the specified queue is empty
     */
    private boolean isSendQueueEmpty(ArrayBlockingQueue<ByteBuffer> queue) {
        return queue.isEmpty();
    }

    /**
     * Retrieves and removes buffer at the head of this queue,
     * waiting up to the specified wait time if necessary for an element to
     * become available.
     *
     * {@link ArrayBlockingQueue#poll(long, java.util.concurrent.TimeUnit)}
     */
    private ByteBuffer pollSendQueue(ArrayBlockingQueue<ByteBuffer> queue,
          long timeout, TimeUnit unit) throws InterruptedException {
       return queue.poll(timeout, unit);
    }

    /**
     * Inserts an element in the {@link #recvQueue}. If the Queue is full, this
     * methods removes an element from the head of the Queue and then inserts
     * the element at the tail of the queue.
     *
     * This method is synchronized to achieve fairness between two threads that
     * are trying to insert an element in the queue. Each thread checks if the
     * queue is full, then removes the element at the head of the queue, and
     * then inserts an element at the tail. This three-step process is done to
     * prevent a thread from blocking while inserting an element in the queue.
     * If we do not synchronize the call to this method, then a thread can grab
     * a slot in the queue created by the second thread. This can cause the call
     * to insert by the second thread to fail.
     * Note that synchronizing this method does not block another thread
     * from polling the queue since that synchronization is provided by the
     * queue itself.
     *
     * @param msg
     *          Reference to the message to be inserted in the queue
     */
    public void addToRecvQueue(Message msg) {
        synchronized(recvQueue) {
            if (recvQueue.remainingCapacity() == 0) {
                try {
                    recvQueue.remove();
                } catch (NoSuchElementException ne) {
                    // element could be removed by poll()
                     LOG.debug("Trying to remove from an empty " +
                         "recvQueue. Ignoring exception " + ne);
                }
            }
            try {
                recvQueue.add(msg);
            } catch (IllegalStateException ie) {
                // This should never happen
                LOG.fatal("Unable to insert element in the recvQueue " + ie);
            }
        }
    }

    /**
     * Retrieves and removes a message at the head of this queue,
     * waiting up to the specified wait time if necessary for an element to
     * become available.
     *
     * {@link ArrayBlockingQueue#poll(long, java.util.concurrent.TimeUnit)}
     */
    public Message pollRecvQueue(long timeout, TimeUnit unit)
       throws InterruptedException {
       return recvQueue.poll(timeout, unit);
    }

}
