#! /bin/bash
#######################################################################
# Copyright 2012 VMware, Inc.  All rights reserved.                   #
#######################################################################

#Script used to uninstall VSA Cluster Service on Linux

# Includes
THIS_DIR=`dirname $0`
source $THIS_DIR/common.sh

# Functions

print_usage() {
echo "Usage: ${0##*/} [options]

Uninstalls VMware Cluster Service.
This script must be run as root.

-h | --help                        Print this help.
-k | --keepuser                    If specified the uninstall script will not remove the user account. By default the user
                                   account is removed.
-s | --silent                      If specified the uninstall will not ask the user to confirm the deletion of the directories
                                   and the user account.
-v | --verbose                     Print verbose information.
-D | --debug                       Print all commands and their arguments as they are executed (set -x)."
}

# Removes firewall exceptions for ports specified in the ini file.
# Arguments $1 - a comma separated list of ports.
remove_firewall_exceptions() {
    ports=$1

    for port in ${ports//,/ }
    do
        print_verbose "Removing firewall exception for port $port."
        iptables -D INPUT -p tcp --dport $port -j ACCEPT 2>/dev/null
    done

    linux_distribution=$(get_linux_distribution)

    # Make the firewall changes persistent
    if [ "$linux_distribution" == "RedHat" ]
    then
        /sbin/service iptables save
    fi

    return 0
}

# MAIN Function
main() {
    # Stop VMVCS service
    myerr=0
    if [ -x "$ROOTDIR/bin/vmvcs" ]
    then
        "$ROOTDIR/bin/vmvcs" stop
        myerr=$?
    fi

    if [ $myerr != 0 ]
    then
        print_msg "Unable to stop Cluster Service."
    fi

    # Uninstall VMVCS service deamon process.
    myerr=0
    if [ -x "$ROOTDIR/bin/vmvcs" ]
    then
        "$ROOTDIR/bin/vmvcs" remove
        myerr=$?
    fi

    if [ $myerr != 0 ]
    then
        print_msg "Unable to uninstall Cluster Service Linux deamon process."
    fi

    # Get list of ports
    ports=$(get_list_of_ports "$ROOTDIR/setup/ports.ini")
    if [ $? != 0 ]
    then
        print_msg "Unable to obtain the list of ports used by VCS."
        return 1
    fi

    # Remove firewall exceptions
    remove_firewall_exceptions "$ports"
    if [ $? != 0 ]
    then
        return 1
    fi

    # Remove the product folder
    remove_directory "$ROOTDIR" "$SILENT"

    # Remove user account if requested
    if [ "$REMOVEUSER" == "1" ]
    then
        remove_useraccount "$VCSUSER" "$SILENT"
        if [ $? != 0 ]
        then
            print_msg "Unable to delete $VCSUSER user account. Continue."
        fi
    fi

    return 0
}

################### ENTRY POINT #######################

# Get the fully qualified directory name where this script is located
SOURCE="${BASH_SOURCE[0]}"
while [ -h "$SOURCE" ] ; do SOURCE="$(readlink "$SOURCE")"; done
# Directory where this script is located (.../VSAClusterService-x.x/setup)
SRCDIR="$( cd -P "$( dirname "$SOURCE" )" && pwd )"
ROOTDIR=`dirname "$SRCDIR"`

if [[ "X$ROOTDIR" == "X" || "X$SRCDIR" == "X" ]]
then
    print_msg "Unable to identify VCS installation location"
    return 1
fi

ver=`cat "$ROOTDIR/conf/vcs.version"`
print_msg "Uninstalling VMWare VSA cluster service $ver"

# Check if all required commands exists
CHKCMDS="sudo xargs gettext iptables sed useradd passwd chage chpasswd userdel netstat tr cut"
which $CHKCMDS > /dev/null
if [ $? != 0 ]
then
     print_msg "Missing required commands $CHKCMDS from $PATH"
     exit 1
fi

# Check if supported architecutre
arch=`uname -m`
if [ "$arch" != "x86_64" ]
then
    print_msg "Unsupported architecture $arch. VCS can be installed only on 64-bit distributions."
    exit 1
fi

# ---Defaults: should not be changed
VCSUSER="vmwarevcsadmin"
if [ "X$VMWAREVCSADMIN" != "X" ]
then
    VCSUSER=$VMWAREVCSADMIN
fi

VERBOSE=0
DEBUG=0
REMOVEUSER=1
SILENT=0

while [ "X$1" != "X" ]; do
        case "$1" in
                -h|--help)
                        print_usage
                        exit 1
                        ;;
                -k|--keepuser)
                        REMOVEUSER=0
                        shift
                        ;;
                -s|--silent)
                        SILENT=1
                        shift
                        ;;
                -v|--verbose)
                        VERBOSE=1
                        shift
                        ;;
                -D|--debug)
                        DEBUG=1
                        shift
                        ;;
                *)
                        print_usage
                        exit 1
                        ;;
        esac
done

# Check if root
if [ `id | sed 's/^uid=//;s/(.*$//'` != "0" ] ; then
    print_err "Must be root to install VCS"
    exit 1
fi

if [ $DEBUG == 1 ]
then
    set -x
fi

print_verbose "Arguments: ROOTDIR=$ROOTDIR REMOVEUSER=$REMOVEUSER VERBOSE=$VERBOSE DEBUG=$DEBUG VCSUSER=$VCSUSER"
main
print_msg "The VMware VSA Cluster Service has been uninstalled."
exit $?
