/* Portions of this file are under the following license:
 * ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2003 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Copyright (c) 1990, 1993
 *      The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */


#define DEFAULT_LOG_FORMAT "%h %l %u %t \"%r\" %>s %b"

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"          /* For REMOTE_NAME */
#include "http_log.h"

module MODULE_VAR_EXPORT rt_module;

/*
 * multi_log_state is our per-(virtual)-server configuration. We store
 * an array of the logs we are going to use, each of type config_log_state.
 * If a default log format is given by LogFormat, store in default_format
 * (backward compat. with mod_log_config).  We also store for each virtual
 * server a pointer to the logs specified for the main server, so that if this
 * vhost has no logs defined, we can use the main server's logs instead.
 *
 * So, for the main server, rt_logs contains a list of the log files
 * and server_rt_logs in empty. For a vhost, server_rt_logs
 * points to the same array as rt_logs in the main server, and
 * rt_logs points to the array of logs defined inside this vhost,
 * which might be empty.
 */

 static const char GIF_IMG[] = {'G', 'I', 'F', '8', '9', 'a', 0x01, 0x00, 0x01,
                                0x00, 0x80, 0x00, 0x00, 0xdb, 0xdf, 0xef, 0x00,
                                0x00, 0x00, 0x21, 0xf9, 0x04, 0x01, 0x00, 0x00,
                                0x00, 0x00, 0x2c, 0x00, 0x00, 0x00, 0x00, 0x01,
                                0x00, 0x01, 0x00, 0x00, 0x02, 0x02, 0x44, 0x01,
                                0x00, 0x3b};

typedef struct {
    array_header *rt_logs;
    array_header *server_rt_logs;

    array_header *eu_logs;
    array_header *server_eu_logs;
} multi_log_state;

/*
 * config_log_state holds the status of a single log file. fname might
 * be NULL, which means this module does no logging for this
 * request. format might be NULL, in which case the default_format
 * from the multi_log_state should be used, or if that is NULL as
 * well, use the CLF. log_fd is -1 before the log file is opened and
 * set to a valid fd after it is opened.
 */

typedef struct {
    char *fname;
    int log_fd;
} config_log_state;

typedef struct {
    long long usec;
} req_time;


#define USEC_FROM_TV(tv) ((long long)tv.tv_sec * 1000000ll) + tv.tv_usec

static const char *log_request_uri(request_rec *r)
{
    return ap_escape_logitem(r->pool, r->uri);
}

static const char *log_request_time(request_rec *r)
{
    struct timeval tv;
    long long usec;
    req_time *start = ap_get_module_config(r->request_config, &rt_module);
    char tstr[MAX_STRING_LEN];

    if (gettimeofday(&tv, NULL) != 0) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, r,
                     "could not get time of day.");
        return OK;
    }

    usec = (long long)USEC_FROM_TV(tv);

    /* We have to divide by 1000 here, because the RT log is expected to be
     * in milliseconds, but gettimeofday is giving us nanoseconds.  So, we
     * do the conversion here, so that RT doesn't get dates in 35393.
     */
    ap_snprintf(tstr, sizeof(tstr), " %qu %qu ", start->usec / 1000, 
                usec - start->usec);

    return ap_pstrdup(r->pool, tstr);
}

/*****************************************************************
 *
 * Actually logging.
 */

static char *format_integer(pool *p, int i)
{
    return ap_psprintf(p, "%d", i);
}

static char *pfmt(pool *p, int i)
{
    if (i <= 0) {
        return "-";
    }
    else {
        return format_integer(p, i);
    }
}

static int config_log_transaction(request_rec *r, config_log_state *cls)
{
    char *str, *s;
    const char **strs;
    int *strl;
    request_rec *orig;
    int i;
    int len = 0;

    if (cls->fname == NULL) {
        return DECLINED;
    }

    orig = r;
    while (orig->prev) {
        orig = orig->prev;
    }
    while (r->next) {
        r = r->next;
    }

    str = ap_pstrcat(r->pool, r->unparsed_uri, log_request_time(r),
                     pfmt(r->pool, r->status), " ",
                     r->connection->remote_ip, "\n", NULL);
    len = strlen(str);

    write(cls->log_fd, str, len);
    return OK;
}

long long apr_strtoi64(const char *nptr, char **endptr, int base)
{
    const char *s;
    long long acc;
    long long val;
    int neg, any;
    char c;

    /*
     * Skip white space and pick up leading +/- sign if any.
     * If base is 0, allow 0x for hex and 0 for octal, else
     * assume decimal; if base is already 16, allow 0x.
     */
    s = nptr;
    do {
        c = *s++;
    } while (ap_isspace(c));
    if (c == '-') {
        neg = 1;
        c = *s++;
    } else {
        neg = 0;
        if (c == '+')
            c = *s++;
    }
    if ((base == 0 || base == 16) &&
            c == '0' && (*s == 'x' || *s == 'X')) {
        c = s[1];
        s += 2;
        base = 16;
    }
    if (base == 0)
        base = c == '0' ? 8 : 10;
    acc = any = 0;
    if (base < 2 || base > 36) {
        errno = EINVAL;
        if (endptr != NULL)
            *endptr = (char *)(any ? s - 1 : nptr);
        return acc;
    }

    /* The classic bsd implementation requires div/mod operators
     * to compute a cutoff.  Benchmarking proves that iss very, very
     * evil to some 32 bit processors.  Instead, look for underflow
     * in both the mult and add/sub operation.  Unlike the bsd impl,
     * we also work strictly in a signed int64 word as we haven't
     * implemented the unsigned type in win32.
     *
     * Set 'any' if any `digits' consumed; make it negative to indicate
     * overflow.
     */
    val = 0;
    for ( ; ; c = *s++) {
        if (c >= '0' && c <= '9')
            c -= '0';
        else if (c >= 'A' && c <= 'Z')
            c -= 'A' - 10;
        else if (c >= 'a' && c <= 'z')
            c -= 'a' - 10;
        else
            break;
        if (c >= base)
            break;
        val *= base;
        if ( (any < 0)  /* already noted an over/under flow - short circuit */
                || (neg && (val > acc || (val -= c) > acc)) /* underflow */
                || (val < acc || (val += c) < acc)) {       /* overflow */
            any = -1;   /* once noted, over/underflows never go away */
            break;
        } else {
            acc = val;
            any = 1;
        }
    }

    if (any < 0) {
        acc = neg ? INT64_MIN : INT64_MAX;
        errno = ERANGE;
    } else if (!any) {
        errno = EINVAL;
    }
    if (endptr != NULL)
        *endptr = (char *)(any ? s - 1 : nptr);
    return (acc);
}

static int eu_log_transaction(request_rec *r, config_log_state *cls)
{
    const char *qstr;
    const char *timestr;
    char *vhoststr;
    char *endstr;
    char str[HUGE_STRING_LEN];
    int len = 0;
    long long orig_time;
    req_time *start = ap_get_module_config(r->request_config, &rt_module);

    if (cls->fname == NULL) {
        return DECLINED;
    }

    if (strcmp(r->uri, "/CAM_blank.gif") || (!r->args)) {
        return DECLINED;
    }
    
    qstr = r->args;
    timestr = strchr(qstr, 't') + 2;
    vhoststr = strchr(timestr, '&');

    orig_time = apr_strtoi64(timestr, &endstr, 10);

    ap_snprintf(str, HUGE_STRING_LEN, "%s %qd %qd %d %s\n", vhoststr + 3,
                orig_time / 1000, start->usec - orig_time, r->status,
                r->connection->remote_ip);
    len = strlen(str);

    write(cls->log_fd, str, len);
    return OK;
}

static int multi_log_transaction(request_rec *r)
{
    multi_log_state *mls = ap_get_module_config(r->server->module_config,
						&rt_module);
    config_log_state *clsarray;
    int i;

    /*
     * Log this transaction..
     */
    if (mls->rt_logs->nelts) {
        clsarray = (config_log_state *) mls->rt_logs->elts;
        for (i = 0; i < mls->rt_logs->nelts; ++i) {
            config_log_state *cls = &clsarray[i];

            config_log_transaction(r, cls);
        }
    }
    else if (mls->server_rt_logs) {
        clsarray = (config_log_state *) mls->server_rt_logs->elts;
        for (i = 0; i < mls->server_rt_logs->nelts; ++i) {
            config_log_state *cls = &clsarray[i];

            config_log_transaction(r, cls);
        }
    }

    if (mls->eu_logs->nelts) {
        clsarray = (config_log_state *) mls->eu_logs->elts;
        for (i = 0; i < mls->eu_logs->nelts; ++i) {
            config_log_state *cls = &clsarray[i];

            eu_log_transaction(r, cls);
        }
    }
    else if (mls->server_eu_logs) {
        clsarray = (config_log_state *) mls->server_eu_logs->elts;
        for (i = 0; i < mls->server_eu_logs->nelts; ++i) {
            config_log_state *cls = &clsarray[i];

            eu_log_transaction(r, cls);
        }
    }

    return OK;
}

/*****************************************************************
 *
 * Module glue...
 */

static void *make_config_log_state(pool *p, server_rec *s)
{
    multi_log_state *mls;

    mls = (multi_log_state *) ap_palloc(p, sizeof(multi_log_state));
    mls->rt_logs = ap_make_array(p, 1, sizeof(config_log_state));
    mls->server_rt_logs = NULL;

    mls->eu_logs = ap_make_array(p, 1, sizeof(config_log_state));
    mls->server_eu_logs = NULL;

    return mls;
}

static const char *add_rt_log(cmd_parms *cmd, void *dummy, char *fn)
{
    const char *err_string = NULL;
    multi_log_state *mls = ap_get_module_config(cmd->server->module_config,
						&rt_module);
    config_log_state *cls;

    cls = (config_log_state *) ap_push_array(mls->rt_logs);

    cls->fname = fn;
    cls->log_fd = -1;

    return err_string;
}

static const char *add_eu_log(cmd_parms *cmd, void *dummy, char *fn)
{
    const char *err_string = NULL;
    multi_log_state *mls = ap_get_module_config(cmd->server->module_config,
						&rt_module);
    config_log_state *cls;

    cls = (config_log_state *) ap_push_array(mls->eu_logs);

    cls->fname = fn;
    cls->log_fd = -1;

    return err_string;
}

static const command_rec config_log_cmds[] =
{
    {"RTLog", add_rt_log, NULL, RSRC_CONF, TAKE1,
     "a file name for the response-time log."},
    {"EndUserLog", add_eu_log, NULL, RSRC_CONF, TAKE1,
     "a file name for the response-time log."},
    {NULL}
};

static config_log_state *open_config_log(server_rec *s, pool *p,
                                         config_log_state *cls)
{
    if (cls->log_fd > 0) {
        return cls;             /* virtual config shared w/main server */
    }

    if (cls->fname == NULL) {
        return cls;             /* Leave it NULL to decline.  */
    }

    if (*cls->fname == '|') {
        piped_log *pl;

        pl = ap_open_piped_log(p, cls->fname + 1);
        if (pl == NULL) {
            exit(1);
        }
        cls->log_fd = ap_piped_log_write_fd(pl);
    }
    else {
        char *fname = ap_server_root_relative(p, cls->fname);
        if ((cls->log_fd = ap_popenf(p, fname, O_WRONLY | O_APPEND | O_CREAT, 
                              S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)) < 0) {
            ap_log_error(APLOG_MARK, APLOG_ERR, s,
                         "could not open transfer log file %s.", fname);
            exit(1);
        }
    }

    return cls;
}

static config_log_state *open_multi_logs(server_rec *s, pool *p)
{
    int i;
    multi_log_state *mls = ap_get_module_config(s->module_config,
                                             &rt_module);
    config_log_state *clsarray;
    const char *dummy;

    if (mls->rt_logs->nelts) {
        clsarray = (config_log_state *) mls->rt_logs->elts;
        for (i = 0; i < mls->rt_logs->nelts; ++i) {
            config_log_state *cls = &clsarray[i];

            cls = open_config_log(s, p, cls);
        }
    }
    else if (mls->server_rt_logs) {
        clsarray = (config_log_state *) mls->server_rt_logs->elts;
        for (i = 0; i < mls->server_rt_logs->nelts; ++i) {
            config_log_state *cls = &clsarray[i];

            cls = open_config_log(s, p, cls);
        }
    }

    if (mls->eu_logs->nelts) {
        clsarray = (config_log_state *) mls->eu_logs->elts;
        for (i = 0; i < mls->eu_logs->nelts; ++i) {
            config_log_state *cls = &clsarray[i];

            cls = open_config_log(s, p, cls);
        }
    }
    else if (mls->server_eu_logs) {
        clsarray = (config_log_state *) mls->server_eu_logs->elts;
        for (i = 0; i < mls->server_eu_logs->nelts; ++i) {
            config_log_state *cls = &clsarray[i];

            cls = open_config_log(s, p, cls);
        }
    }
    return NULL;
}

static void init_config_log(server_rec *s, pool *p)
{
    /* First, do "physical" server, which gets default log fd and format
     * for the virtual servers, if they don't override...
     */

    open_multi_logs(s, p);

    /* Then, virtual servers */

    for (s = s->next; s; s = s->next) {
        open_multi_logs(s, p);
    }
}

static int get_time(request_rec *r)
{
    req_time *rt;
    struct timeval tv;

    rt = ap_palloc(r->pool, sizeof(*rt));

    if (gettimeofday(&tv, NULL) != 0) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, r,
                     "could not get time of day.");
        rt->usec = -1;
        return OK;
    }

    rt->usec = (long long)USEC_FROM_TV(tv);

    ap_set_module_config(r->request_config, &rt_module, rt);
    return OK;
}

static int rt_handler(request_rec *r)
{
    int errstatus; 

    if ((errstatus = ap_discard_request_body(r)) != OK) {
        return errstatus;
    }

    r->allowed |= (1 << M_GET) | (1 << M_OPTIONS);

    if (r->method_number != M_GET) {
        return DECLINED;
    }

    if (strcmp(r->uri, "/CAM_blank.gif")) {
        return DECLINED;
    }

    ap_send_http_header(r);

    ap_rwrite(GIF_IMG, sizeof(GIF_IMG), r);
    return OK;
}

static const handler_rec rt_handlers[] =
{
    {"image/gif", rt_handler},
    {NULL}
};

module MODULE_VAR_EXPORT rt_module =
{
    STANDARD_MODULE_STUFF,
    init_config_log,            /* initializer */
    NULL,                       /* create per-dir config */
    NULL,                       /* merge per-dir config */
    make_config_log_state,      /* server config */
    NULL,                       /* merge server config */
    config_log_cmds,            /* command table */
    rt_handlers,                /* handlers */
    NULL,                       /* filename translation */
    NULL,                       /* check_user_id */
    NULL,                       /* check auth */
    NULL,                       /* check access */
    NULL,                       /* type_checker */
    NULL,                       /* fixups */
    multi_log_transaction,      /* logger */
    NULL,                       /* header parser */
    NULL,                       /* child_init */
    NULL,                       /* child_exit */
    get_time                    /* post read-request */
};
