#ifdef XP_WIN32
#define NSAPI_PUBLIC __declspec(dllexport)
#else /* !XP_WIN32 */
#define NSAPI_PUBLIC
#endif /* !XP_WIN32 */

#include "nsapi.h"

/* Netscape doesn't allow you to log the time_taken for a specific request,
 * but the RT feature of CAM requires that information, so we have to create
 * a SAF to do it for us.  This SAF will log very specific information:
 *
 *    URI Response_code start_time time_taken
 *
 * The URI will include the query string.  start_time is logged in a standard
 * time format ([05/Mar/2003:11:46:51 -0800]).  time_taken is in milliseconds.
 *
 * Usage For iPlanet-4.x:
 * At the beginning of obj.conf:
 *    Init fn=load-modules shlib=cam_rt.<ext> funcs=cam-rt-init,cam-rt-log,cam-start-time
 *    Init fn=cam-rt-init file=/foo/bar/baz
 * Inside an object in obj.conf:
 *    AuthTrans fn=cam-start-time
 *    AddLog fn=cam-rt-log
 *
 * Usage For iPlanet-6.x:
 * At the end of magus.conf:
 *    Init fn=load-modules shlib=cam_rt.<ext> funcs=cam-rt-init,cam-rt-log,cam-start-time
 *    Init fn=cam-rt-init file=/foo/bar/baz
 * Inside an object in obj.conf:
 *    AuthTrans fn=cam-start-time
 *    AddLog fn=cam-rt-log
 *
 * <ext> = so on UNIX
 * <ext> = dll on NT.
 */

#include "base/daemon.h" /* daemon_atrestart */
#include "base/file.h"   /* system_fopenWA, system_fclose */
#include "base/util.h"   /* sprintf */
#include <sys/time.h>

#define CAM_NAME "CAM_START_TIME"
#define DEFAULT_REQUEST_TIME_SIZE 32

static const char month_snames[12][4] =
{
    "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

/* File descriptor to be shared between the processes */
static SYS_FILE logfd = SYS_ERROR_FD;

#ifdef __cplusplus
extern "C"
#endif
NSAPI_PUBLIC void cam_terminate(void *parameter)
{
    system_fclose(logfd);
    logfd = SYS_ERROR_FD;
}

#ifdef __cplusplus
extern "C"
#endif
NSAPI_PUBLIC int cam_rt_init(pblock *pb, Session *sn, Request *rq)
{
    /* Parameter */
    char *fn = pblock_findval("file", pb);

    if(!fn) {
        pblock_nvinsert("error", "cam-rt-init: please supply a file name", pb);
        return REQ_ABORTED;
    }
    logfd = system_fopenWA(fn);
    if(logfd == SYS_ERROR_FD) {
        pblock_nvinsert("error", "cam-rt-init: please supply a file name", pb);
        return REQ_ABORTED;
    }
    /* Close log file when server is restarted */
    daemon_atrestart(cam_terminate, NULL);
    return REQ_PROCEED;
}

/* Yes, doing this in the Auth phase sucks, but we have no choice.
 * NSAPI just doesn't hook into the server as often as it should, so we are
 * stuck getting our first time very late in the process.  This is okay, because
 * times for RT are relative to each other, and all Netscape times will have
 * this fudge factor.
 */
#ifdef __cplusplus
extern "C"
#endif
NSAPI_PUBLIC int cam_start_time(pblock *pb, Session *sn, Request *rq)
{
    struct timeval tv;
    /* The NSPI funcs are just going to copy this string, so we don't need to 
     * be exact with the number of chars.
     */
    char str[25];
    long long t;

    gettimeofday(&tv, NULL);
    t = (long long)tv.tv_sec * 1000000 + (long long)tv.tv_usec;
    util_sprintf(str, "%lld", t);
    pblock_nvinsert(CAM_NAME, str, rq->reqpb);
    
    return REQ_NOACTION;
}

#ifdef __cplusplus
extern "C"
#endif
NSAPI_PUBLIC int cam_rt_log(pblock *pb, Session *sn, Request *rq)
{
    char *uri = pblock_findval("uri", rq->reqpb);
    char *query = pblock_findval("query", rq->reqpb);
    char stime[DEFAULT_REQUEST_TIME_SIZE];
    char logmsg[8192];
    int len;
    struct timeval tv;
    struct tm tm;
    long long begin_time;
    long long end_time;
    char sign;
    char *temp;
    int timz;

    localtime_r(&rq->req_start, &tm);
    timz = tm.tm_gmtoff;
    if (timz < 0) {
        timz = -timz;
        sign = '-';
    }
    else {
        sign = '+';
    }

    util_snprintf(stime, DEFAULT_REQUEST_TIME_SIZE,
                  "[%02d/%s/%d:%02d:%02d:%02d %c%.2d%.2d]",
                  tm.tm_mday, month_snames[tm.tm_mon], tm.tm_year + 1900, 
                  tm.tm_hour, tm.tm_min, tm.tm_sec, sign, timz / (60*60), 
                  timz %(60*60));
    
    gettimeofday(&tv, NULL);
    end_time = (long long)tv.tv_sec * 1000000 + (long long)tv.tv_usec;
    temp = pblock_findval(CAM_NAME, rq->reqpb);
    begin_time = atoll(temp);

    if (query != NULL) {
        len = util_sprintf(logmsg, "%s%s %lld %lld %d\n", 
                           uri, query, end_time / 1000,
                           end_time - begin_time, rq->status_num);
    }
    else {
        len = util_sprintf(logmsg, "%s %lld %lld %d\n", 
                           uri, end_time / 1000,
                           end_time - begin_time, rq->status_num);
    }

    len = strlen(logmsg);
    /* The atomic version uses locking to prevent interference */
    system_fwrite_atomic(logfd, logmsg, len);

    return REQ_NOACTION;
}
