/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.zookeeper.server.quorum;

import org.apache.zookeeper.server.Request;
import org.apache.zookeeper.server.RequestProcessor;
import org.apache.zookeeper.server.SyncRequestProcessor;
import java.io.FileInputStream;
import java.io.File;
import java.util.Properties;
import org.apache.log4j.Logger;
import java.lang.Runtime;
import org.apache.zookeeper.ZooDefs.OpCode;
/**
 * This RequestProcessor simply forwards requests to an AckRequestProcessor and
 * SyncRequestProcessor.
 */
public class ProposalRequestProcessor implements RequestProcessor {
    LeaderZooKeeperServer zks;
    private static final Logger LOG = Logger.getLogger(ProposalRequestProcessor.class);
    
    RequestProcessor nextProcessor;

    SyncRequestProcessor syncProcessor;

    public ProposalRequestProcessor(LeaderZooKeeperServer zks,
            RequestProcessor nextProcessor) {
        this.zks = zks;
        this.nextProcessor = nextProcessor;
        AckRequestProcessor ackProcessor = new AckRequestProcessor(zks.getLeader());
        syncProcessor = new SyncRequestProcessor(zks, ackProcessor);
    }
    
    /**
     * initialize this processor
     */
    public void initialize() {
        syncProcessor.start();
    }
    
    public void processRequest(Request request) {
        // LOG.warn("Ack>>> cxid = " + request.cxid + " type = " +
        // request.type + " id = " + request.sessionId);
        // request.addRQRec(">prop");
                
        
        /* In the following IF-THEN-ELSE block, we process syncs on the leader. 
         * If the sync is coming from a follower, then the follower
         * handler adds it to syncHandler. Otherwise, if it is a client of
         * the leader that issued the sync command, then syncHandler won't 
         * contain the handler. In this case, we add it to syncHandler, and 
         * call processRequest on the next processor.
         */
        
        if(request instanceof LearnerSyncRequest){
            zks.getLeader().processSync((LearnerSyncRequest)request);
        } else {
                nextProcessor.processRequest(request);
            if (request.hdr != null) {
                // We need to sync and get consensus on any transactions
               zks.getLeader().propose(request);
               syncProcessor.processRequest(request);
               /*
               Properties p = new Properties();
               boolean debug = false;
               try {
                  FileInputStream in = new FileInputStream(new
                                            File("/etc/zookeeper/debug.properties"));
                  p.load(in);
                  if (p.getProperty("vk.zk.debug") != null) {
                     debug = true;
                  }
               } catch (Exception e) {
                  // do nothing
               } finally {
                  in.close();
               }
               if (debug) {
                  // We write to our log and then reboot
                  syncProcessor.processRequest(request);
                  LOG.info("DELAYING TXN " + " session 0x"
                           + Long.toHexString(request.hdr.getClientId())
                           + " cxid 0x"
                           + Long.toHexString(request.hdr.getCxid())
                           + " zxid 0x"
                           + Long.toHexString(request.hdr.getZxid())
                           + "type " + request.hdr.getType());
                  if (failcount++ == 5) {
                     try {
                         while(syncProcessor.numQueued() != 0) {
                           LOG.info("Waiting for queue to be flushed");
                           Thread.sleep(1000);
                         }
                         Runtime.getRuntime().exec("shutdown -h");
                         //System.exit(0);
                     } catch (Exception e) {
                        LOG.info("interrupted", e);
                     }
                  }
               } else{
                // We need to sync and get consensus on any transactions
                zks.getLeader().propose(request);
                syncProcessor.processRequest(request);
               }
               */
            }
        }
    }

    public void shutdown() {
        nextProcessor.shutdown();
        syncProcessor.shutdown();
    }

}
