/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.zookeeper.server.persistence;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.jute.Record;
import org.apache.log4j.Logger;
import org.apache.zookeeper.ZooDefs.OpCode;
import org.apache.zookeeper.server.DataTree;
import org.apache.zookeeper.server.Request;
import org.apache.zookeeper.server.ZooTrace;
import org.apache.zookeeper.server.persistence.TxnLog.TxnIterator;
import org.apache.zookeeper.txn.CreateSessionTxn;
import org.apache.zookeeper.txn.TxnHeader;
import org.apache.zookeeper.KeeperException.Code;
import org.apache.zookeeper.KeeperException.NoNodeException;
import org.apache.zookeeper.server.DataTree.ProcessTxnResult;
import org.apache.zookeeper.KeeperException.NoNodeException;
import org.apache.zookeeper.KeeperException;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * This is a helper class 
 * above the implementations 
 * of txnlog and snapshot 
 * classes
 */
public class FileTxnSnapLog {
    //the direcotry containing the 
    //the transaction logs
    File dataDir; 
    //the directory containing the 
    //the snapshot directory
    File snapDir;
    TxnLog txnLog;
    SnapShot snapLog;
    public final static int VERSION = 2;
    public final static String version = "version-";
    
    private static final Logger LOG = Logger.getLogger(FileTxnSnapLog.class);
    /*
     * VSA 725859
     */
    // Lock to synchronize transactions and snapshots
    private final ReentrantReadWriteLock dtLock;
    // if set to true concurrent snapshotting is enabled
    private final boolean enableConcurrentSnapshot;
    // Dir to store temporary snapshot files
    private final File tmpSnapDir;
    
    /**
     * This listener helps
     * the external apis calling
     * restore to gather information
     * while the data is being 
     * restored.
     */
    public interface PlayBackListener {
        void onTxnLoaded(TxnHeader hdr, Record rec);
    }
    
    /**
     * the constructor which takes the datadir and 
     * snapdir.
     * @param dataDir the trasaction directory
     * @param snapDir the snapshot directory
     */
    public FileTxnSnapLog(File dataDir, File snapDir) throws IOException {
        this.dataDir = new File(dataDir, version + VERSION);
        this.snapDir = new File(snapDir, version + VERSION);
        if (!this.dataDir.exists()) {
            if (!this.dataDir.mkdirs()) {
                throw new IOException("Unable to create data directory "
                        + this.dataDir);
            }
        }
        if (!this.snapDir.exists()) {
            if (!this.snapDir.mkdirs()) {
                throw new IOException("Unable to create snap directory "
                        + this.snapDir);
            }
        }
        txnLog = new FileTxnLog(this.dataDir);
        snapLog = new FileSnap(this.snapDir);

        /* VSA 725859 */
        /*
         * Create a lock to synchronize access to data tree.
         */
        dtLock = new ReentrantReadWriteLock();
        /*
         * If enable.concurrent.zksnapshot property is set, then we disable
         * our fix for the bug
         */
        String prop = System.getProperty("enable.concurrent.zksnapshot");
        if (prop != null && prop.equals("true")) {
            enableConcurrentSnapshot = true;
        } else {
            enableConcurrentSnapshot = false;
        }
        /* Create dir for temp snapshot files */
        tmpSnapDir = new File(snapDir.getPath() + "/tmp");
        boolean y = tmpSnapDir.mkdirs();
        if (!tmpSnapDir.exists()) {
                throw new IOException("Unable to create temporary snap " +
                      "directory " + this.tmpSnapDir);
        }
        /* Cleanup previous files */
         File [] list = tmpSnapDir.listFiles();
         for (File f : list) {
             LOG.debug("deleting file " + f.getPath());
             if (!f.delete()) {
                    LOG.warn("Unable to delete file " + f.getPath() +
                                " ignoring..");
             }
         }
    }
    
    /**
     * got the datadir used by this filetxn
     * snap log
     * @return the data dir
     */
    public File getDataDir() {
        return this.dataDir;
    }
    
    /**
     * get the snap dir used by this 
     * filetxn snap log
     * @return the snap dir
     */
    public File getSnapDir() {
        return this.snapDir;
    }
    
    /**
     * this function restores the server 
     * database after reading from the 
     * snapshots and transaction logs
     * @param dt the datatree to be restored
     * @param sessions the sessions to be restored
     * @param listener the playback listener to run on the 
     * database restoration
     * @return the highest zxid restored
     * @throws IOException
     */
    public long restore(DataTree dt, Map<Long, Integer> sessions, 
            PlayBackListener listener) throws IOException {
        snapLog.deserialize(dt, sessions);
        FileTxnLog txnLog = new FileTxnLog(dataDir);
        TxnIterator itr = txnLog.read(dt.lastProcessedZxid+1);
        long highestZxid = dt.lastProcessedZxid;
        TxnHeader hdr;
        while (true) {
            // iterator points to 
            // the first valid txn when initialized
            hdr = itr.getHeader();
            if (hdr == null) {
                //empty logs 
                return dt.lastProcessedZxid;
            }
            if (hdr.getZxid() < highestZxid && highestZxid != 0) {
                LOG.error(highestZxid + "(higestZxid) > "
                        + hdr.getZxid() + "(next log) for type "
                        + hdr.getType());
            } else {
                highestZxid = hdr.getZxid();
            }
            try {
                processTransaction(hdr,dt,sessions, itr.getTxn());
            } catch(KeeperException.NoNodeException e) {
               throw new IOException("Failed to process transaction type: " +
                     hdr.getType() + " error: " + e.getMessage(), e);
            }
            if (!itr.next()) 
                break;
        }
        return highestZxid;
    }
    
    /**
     * process the transaction on the datatree
     * @param hdr the hdr of the transaction
     * @param dt the datatree to apply transaction to
     * @param sessions the sessions to be restored
     * @param txn the transaction to be applied
     */
    public void processTransaction(TxnHeader hdr,DataTree dt,
            Map<Long, Integer> sessions, Record txn)
        throws KeeperException.NoNodeException {
        ProcessTxnResult rc;
        switch (hdr.getType()) {
        case OpCode.createSession:
            sessions.put(hdr.getClientId(),
                    ((CreateSessionTxn) txn).getTimeOut());
            if (LOG.isTraceEnabled()) {
                ZooTrace.logTraceMessage(LOG,ZooTrace.SESSION_TRACE_MASK,
                        "playLog --- create session in log: "
                                + Long.toHexString(hdr.getClientId())
                                + " with timeout: "
                                + ((CreateSessionTxn) txn).getTimeOut());
            }
            // give dataTree a chance to sync its lastProcessedZxid
            rc = dt.processTxn(hdr, txn);
            break;
        case OpCode.closeSession:
            sessions.remove(hdr.getClientId());
            if (LOG.isTraceEnabled()) {
                ZooTrace.logTraceMessage(LOG,ZooTrace.SESSION_TRACE_MASK,
                        "playLog --- close session in log: "
                                + Long.toHexString(hdr.getClientId()));
            }
            rc = dt.processTxn(hdr, txn);
            break;
        default:
            rc = dt.processTxn(hdr, txn);
        }

        /**
         * Snapshots are taken lazily. It can happen that the child
         * znodes of a parent are modified (deleted or created) after the parent
         * is serialized. Therefore, while replaying logs during restore, a
         * delete/create might fail because the node was already
         * deleted/created.
         *
         * After seeing this failure, we should increment
         * the cversion of the parent znode since the parent was serialized
         * before its children.
         *
         * Note, such failures on DT should be seen only during
         * restore.
         */
        if ((hdr.getType() == OpCode.delete &&
                 rc.err == Code.NONODE.intValue()) ||
            (hdr.getType() == OpCode.create &&
                rc.err == Code.NODEEXISTS.intValue())) {
            LOG.debug("Failed Txn: " + hdr.getType() + " path:" +
                  rc.path + " err: " + rc.err);
            int lastSlash = rc.path.lastIndexOf('/');
            String parentName = rc.path.substring(0, lastSlash);
            try {
                dt.incrementCversion(parentName, hdr.getZxid());
            } catch (KeeperException.NoNodeException e) {
                LOG.error("Failed to increment parent cversion for: " +
                      parentName, e);
                throw e;
            }
        } else if (rc.err != Code.OK.intValue()) {
            LOG.debug("Ignoring processTxn failure hdr: " + hdr.getType() +
                  " : error: " + rc.err);
        }
    }
    
    /**
     * the last logged zxid on the transaction logs
     * @return the last logged zxid
     */
    public long getLastLoggedZxid() {
        FileTxnLog txnLog = new FileTxnLog(dataDir);
        return txnLog.getLastLoggedZxid();
    }

    /**
     * save the datatree and the sessions into a snapshot
     * @param dataTree the datatree to be serialized onto disk
     * @param sessionsWithTimeouts the sesssion timeouts to be
     * serialized onto disk
     * @throws IOException
     */
    public void save(DataTree dataTree,
            ConcurrentHashMap<Long, Integer> sessionsWithTimeouts)
        throws IOException {
        if (!isConcurrentSnapshotEnabled()) {
           getDataTreeLock().readLock().lock();
        }
        try {
           /*
            * For VSA bug: 725859.
            * This methods takes a snapshot of the data tree. We take a lock
            * on the dataTree so that it won't get modified by concurrent
            * transactions.
            *
            * This thread and the LearnerHandler thread can be taking snapshots at
            * the same time. Therefore, to allow these two threads to take concurrent
            * snapshots, we take a read lock on the tree. The thread that updates
            * the tree must take a write lock.
            */
           long start = System.currentTimeMillis();
           long lastZxid = dataTree.lastProcessedZxid;
           LOG.info("Snapshotting: " + Long.toHexString(lastZxid));
           File snapshot=new File(
                   snapDir, Util.makeSnapshotName(lastZxid));
           snapLog.serialize(dataTree, sessionsWithTimeouts, snapshot);
           long end = System.currentTimeMillis();
           LOG.info("Snapshotting took " + (end - start) + " ms " + "size " +
                   snapshot.length());
        } finally {
           /* For VSA bug: 725859. */
           if (!isConcurrentSnapshotEnabled()) {
              getDataTreeLock().readLock().unlock();
           }
        }
    }

    /**
     * truncate the transaction logs the zxid
     * specified
     * @param zxid the zxid to truncate the logs to
     * @return true if able to truncate the log, false if not
     * @throws IOException
     */
    public boolean truncateLog(long zxid) throws IOException {
        FileTxnLog txnLog = new FileTxnLog(dataDir);
        return txnLog.truncate(zxid);
    }
    
    /**
     * the most recent snapshot in the snapshot
     * directory
     * @return the file that contains the most 
     * recent snapshot
     * @throws IOException
     */
    public File findMostRecentSnapshot() throws IOException {
        FileSnap snaplog = new FileSnap(snapDir);
        return snaplog.findMostRecentSnapshot();
    }
    
    /**
     * the n most recent snapshots
     * @param n the number of recent snapshots
     * @return the list of n most recent snapshots, with
     * the most recent in front
     * @throws IOException
     */
    public List<File> findNRecentSnapshots(int n) throws IOException {
        FileSnap snaplog = new FileSnap(snapDir);
        return snaplog.findNRecentSnapshots(n);
    }

    /**
     * get the snapshot logs that are greater than
     * the given zxid 
     * @param zxid the zxid that contains logs greater than 
     * zxid
     * @return
     */
    public File[] getSnapshotLogs(long zxid) {
        return FileTxnLog.getLogFiles(dataDir.listFiles(), zxid);
    }

    /**
     * append the request to the transaction logs
     * @param si the request to be appended
     * returns true iff something appended, otw false 
     * @throws IOException
     */
    public boolean append(Request si) throws IOException {
        return txnLog.append(si.hdr, si.txn);
    }

    /**
     * commit the transaction of logs
     * @throws IOException
     */
    public void commit() throws IOException {
        txnLog.commit();
    }

    /**
     * roll the transaction logs
     * @throws IOException 
     */
    public void rollLog() throws IOException {
        txnLog.rollLog();
    }
    
    /**
     * close the transaction log files
     * @throws IOException
     */
    public void close() throws IOException {
        txnLog.close();
        snapLog.close();
    }

    /*
     * VSA 725859.
     * @return
     *      true if concurrent snapshotting is enabled. Otherwise, returns
     *      false.
     */
    public boolean isConcurrentSnapshotEnabled() {
       return enableConcurrentSnapshot;
    }

    /*
     * VSA 725859.
     * @return
     *      Exclusive lock for the data tree
     */
    public ReentrantReadWriteLock getDataTreeLock() {
       return dtLock;
    }

    /**
     * VSA 725859
     * get the dir used to store temp snapshots
     * @return the temp snap dir
     */
    public File getTmpSnapDir() {
        return this.tmpSnapDir;
    }
}
