﻿/*
 * Copyright 2014, 2016 VMware, Inc.  All rights reserved.
 */

namespace vmware.samples.common
{
    using System;
    using System.ServiceModel;
    using System.ServiceModel.Channels;
    using vmware.lookupservice;

    /// <summary>
    /// Lookup service connection.
    /// </summary>
    public class LsConnection
    {
        private string lsUrl;
        private LsPortTypeClient lsPortType;
        private LookupServiceContent serviceContent;

        public LsConnection(string lsUrl)
        {
            this.lsUrl = lsUrl;

            var binding = GetCustomBinding();
            var address = new EndpointAddress(lsUrl);
            lsPortType = new LsPortTypeClient(binding, address);

            var serviceInstanceRef = new ManagedObjectReference();
            serviceInstanceRef.type = "LookupServiceInstance";
            serviceInstanceRef.Value = "ServiceInstance";
			System.Net.ServicePointManager.SecurityProtocol |= System.Net.SecurityProtocolType.Tls12;
            serviceContent = lsPortType.RetrieveServiceContent(
                serviceInstanceRef);
        }

        public string GetServiceEndpointUrl(string mgmtNodeId, string productType,
            string serviceType, string endpointType, string endpointProtocol)
        {
            var services = GetRegistrationInfos(mgmtNodeId, productType,
                serviceType, endpointType, endpointProtocol);
            if (services != null && services.Length != 0 &&
                services[0] != null && services[0].serviceEndpoints != null &&
                services[0].serviceEndpoints.Length != 0)
            {
                return services[0].serviceEndpoints[0].url;
            }

            var errorMsg = string.Format("Could not find endpoint URL for " +
                "service '{0}' and protocol '{1}'", serviceType, endpointType);
            throw new Exception(errorMsg);
        }

        public LookupServiceRegistrationInfo[] GetRegistrationInfos(
            string mgmtNodeId, string productType,
            string serviceType, string endpointType, string endpointProtocol)
        {
            var filterServiceType =
                new LookupServiceRegistrationServiceType();
            filterServiceType.product = productType;
            filterServiceType.type = serviceType;

            var filterEndpointType =
                    new LookupServiceRegistrationEndpointType();
            filterEndpointType.protocol = endpointProtocol;
            filterEndpointType.type = endpointType;

            var filterCriteria = new LookupServiceRegistrationFilter();
            filterCriteria.serviceType = filterServiceType;
            filterCriteria.endpointType = filterEndpointType;
            filterCriteria.nodeId = mgmtNodeId;

            var registrationInfos = lsPortType.List(
                serviceContent.serviceRegistration, filterCriteria);
            return registrationInfos;
        }

        public void PrintAllServices()
        {
            var services = lsPortType.List(serviceContent.serviceRegistration,
                new LookupServiceRegistrationFilter());

            foreach (var service in services)
            {
                Console.WriteLine("Product: " + service.serviceType.product);
                Console.WriteLine("Service: " + service.serviceType.type);
                foreach (var endpoint in service.serviceEndpoints)
                {
                    Console.WriteLine("   Endpoint protocol: " +
                        endpoint.endpointType.protocol);
                    Console.WriteLine("   Endpoint URL: " +
                        endpoint.endpointType.protocol);
                }
            }
        }

        private static Binding GetCustomBinding()
        {
            var customBinding = new CustomBinding();

            var textMessageEncoding = new TextMessageEncodingBindingElement();
            textMessageEncoding.MessageVersion = MessageVersion.Soap11;

            var transport = new HttpsTransportBindingElement();
            transport.MaxReceivedMessageSize = 2147483647;

            customBinding.Elements.Add(textMessageEncoding);
            customBinding.Elements.Add(transport);

            return customBinding;
        }

        public override string ToString()
        {
            return string.Format("LS++ Connection ({0})", lsUrl);
        }
    }
}
