﻿/*
 * Copyright 2014, 2016 VMware, Inc.  All rights reserved.
 */

namespace vmware.samples.common
{
    using System;
    using System.Linq;
    using vmware.vapi.util.security;

    /// <summary>
    /// Infrastructure node services helper.
    /// </summary>
    public class PlatformServiceController
    {
        public SsoConnection SsoConnection { get; private set; }
        public LsConnection LsConnection { get; private set; }

        public void Connect(string lookupServiceUrl, string ssoUserName,
            string ssoPassword)
        {
            Uri uri = null;
            bool valid =
                Uri.TryCreate(lookupServiceUrl, UriKind.Absolute, out uri) &&
                (uri.Scheme == Uri.UriSchemeHttp ||
                uri.Scheme == Uri.UriSchemeHttps);

            if (!valid)
            {
                throw new ArgumentException("Invalid lookup service URL: " +
                    lookupServiceUrl);
            }

            Console.WriteLine("Using Platform Service Controller: " + uri.Host);

            // connect to Lookup Service
            LsConnection = new LsConnection(lookupServiceUrl);
            Console.WriteLine(LsConnection);

            // connect to SSO and get a SAML token
            var ssoUrl = FindSsoUrl();
            SsoConnection = new SsoConnection(ssoUrl);
            SsoConnection.AcquireSamlToken(ssoUserName, ssoPassword,
                SamlTokenConfirmationType.BEARER);
            Console.WriteLine(SsoConnection);
        }

        private string FindSsoUrl()
        {
            return LsConnection.GetServiceEndpointUrl(null, "com.vmware.cis",
                "cs.identity", "com.vmware.cis.cs.identity.sso", "wsTrust");
        }

        public string FindVapiUrl(string mgmtNodeId)
        {
            return LsConnection.GetServiceEndpointUrl(mgmtNodeId,
                "com.vmware.cis", "cs.vapi", "com.vmware.vapi.endpoint",
                "vapi.json.https.public");
        }

        public string FindVimUrl(string mgmtNodeId)
        {
            return LsConnection.GetServiceEndpointUrl(mgmtNodeId,
                "com.vmware.cis", "vcenterserver", "com.vmware.vim", "vmomi");
        }

        public string FindVimPbmUrl(string mgmtNodeId)
        {
            return LsConnection.GetServiceEndpointUrl(mgmtNodeId,
                "com.vmware.vim.sms", "sms", "com.vmware.vim.pbm", "https");
        }

        public string GetManagementNodeId(ref string mgmtNodeName)
        {
            var services = LsConnection.GetRegistrationInfos(null,
                "com.vmware.cis", "vcenterserver", "com.vmware.vim", "vmomi");

            if (services != null && services.Length > 0)
            {
                if (string.IsNullOrWhiteSpace(mgmtNodeName))
                {
                    // if management node name is not specified and there is
                    // only one management node, return that node
                    if (services.Length == 1)
                    {
                        // get management node name
                        mgmtNodeName = services[0].serviceAttributes.
                            FirstOrDefault(attr => attr.key ==
                                "com.vmware.vim.vcenter.instanceName").value;
                        return services[0].nodeId;
                    }
                    throw new Exception("There is more than one " +
                        "management node, specify the desired node name.");
                }

                // find management node id using name
                mgmtNodeName = System.Net.Dns.GetHostEntry(mgmtNodeName).HostName;
                foreach (var service in services)
                {
                    foreach (var serviceAttribute in service.serviceAttributes)
                    {
                        if (serviceAttribute.key.Equals(
                            "com.vmware.vim.vcenter.instanceName",
                            StringComparison.CurrentCultureIgnoreCase) &&
                            mgmtNodeName.Equals(serviceAttribute.value,
                            StringComparison.CurrentCultureIgnoreCase))
                        {
                            return service.nodeId;
                        }
                    }
                }
            }
            throw new Exception(string.Format(
                    "Could not find management node '{0}'", mgmtNodeName));
        }
    }
}
