﻿/*
 * Copyright 2015, 2016 VMware, Inc.  All rights reserved.
 */

namespace vmware.samples.common
{
    using CommandLine;
    using CommandLine.Text;

    /// <summary>
    /// Samples common implementation.
    /// </summary>
    public abstract class SamplesBase
    {
        /**
         * Common options accross all samples.
         * Sample specific options are provided by the sample itself.
         */

        [Option("ls-url", HelpText = "Lookup service URL.",
            Required = true)]
        public virtual string LsUrl { get; set; }

        [Option("mgmt-node-name", HelpText =
            "Management node's instance name. Ex: IP/FQDN of the node.",
            Required = false)]
        public virtual string MgmtNodeName { get; set; }

        [Option("sso-username", HelpText = "SSO username.", Required = true)]
        public virtual string SsoUserName { get; set; }

        [Option("sso-password", HelpText = "SSO password.", Required = true)]
        public virtual string SsoPassword { get; set; }

        [Option("cleardata", HelpText = "Specify this option to undo all"
            + "persistent results of running the sample", Required = false)]
        public virtual bool ClearData { get; set; }

        /// <summary>
        /// Management node services helper.
        /// </summary>
        public ManagementNode ServiceManager { get; private set; }

        [HelpOption]
        public string GetUsage()
        {
            var help = new HelpText();
            help.AddDashesToOption = true;
            help.AddPreOptionsLine("Usage: vSphereAutomationSamples.exe " +
                this.GetType().Name + " [options]");
            help.AddOptions(this);
            return help;
        }

        /// <summary>
        /// Connects to service endpoints.
        /// </summary>
        public void Connect()
        {
            ServiceManager = new ManagementNode(this);
            ServiceManager.Connect(MgmtNodeName);
        }

        /// <summary>
        /// Runs the sample. Each sample will provider its own implementation
        /// for this method.
        /// </summary>
        public abstract void Run();

        public abstract void Cleanup();

        /// <summary>
        /// Disconnects from service endpoints.
        /// </summary>
        public void Disconnect()
        {
            if(ClearData)
            {
                Cleanup();
            }
            ServiceManager.Disconnect();
        }
    }
}
