﻿/*
 * Copyright 2014, 2016 VMware, Inc.  All rights reserved.
 */

namespace vmware.samples.tagging
{
    using CommandLine;
    using System;
    using System.Collections.Generic;
    using vmware.cis.tagging;
    using vmware.vapi.std;
    using vmware.samples.common;

    /// <summary>
    /// Demonstrates tagging a Cluster.
    /// </summary>
    public class TagCluster : SamplesBase
    {
        private Tag tagService;
        private Category categoryService;
        private TagAssociation tagAssociation;
        private string categoryId, tagId;
        DynamicID clusterId;
        private Boolean tagAttached;

        [Option(
            "cluster-name",
            HelpText = "Name of a Cluster to be tagged.",
            Required = true)]
        public string ClusterName { get; set; }

        public override void Run()
        {
            var tagName = RandomIdGenerator.GetRandomString("Tag-");
            var tagDesc = "Sample tag";
            var categoryName = RandomIdGenerator.GetRandomString("Cat-");
            var categoryDesc = "Sample category";

            // create services
            this.tagService =
                ServiceManager.VapiConnection.GetService<Tag>();
            this.categoryService =
                ServiceManager.VapiConnection.GetService<Category>();
            this.tagAssociation =
                ServiceManager.VapiConnection.GetService<TagAssociation>();

            // create a category
            this.categoryId = CreateCategory(categoryService, categoryName,
                categoryDesc, CategoryModel.Cardinality.MULTIPLE);
            Console.WriteLine("Created category '{0}'", categoryName);

            // create a tag
            this.tagId = CreateTag(tagService, tagName, tagDesc, categoryId);
            Console.WriteLine("Created tag '{0}'", tagName);

            // get the Cluster Moref
            var clusterMoref = VimHelper.GetCluster(ClusterName,
                ServiceManager.VimConnection);
            Console.WriteLine("Got Cluster '{0}'", ClusterName);

            // convert the Moref to vAPI Dynamic ID
            this.clusterId = new DynamicID();
            this.clusterId.SetType(clusterMoref.type);
            this.clusterId.SetId(clusterMoref.Value);

            // tag the Cluster with the newely created tag
            this.tagAssociation.Attach(this.tagId, this.clusterId);
            if (this.tagAssociation.ListAttachedTags(
                this.clusterId).Contains(this.tagId))
            {
                Console.WriteLine("Cluster '{0}' tagged with '{1}'",
                    ClusterName, tagName);
                this.tagAttached = true;
            }
            else
            {
                throw new Exception(string.Format(
                    "Could not tag Cluster '{0}' with '{1}'",
                    ClusterName, tagName));
            }

        }

        public override void Cleanup()
        {
            // detach the Tag from the Cluster
            if(this.tagAttached)
            {
                this.tagAssociation.Detach(tagId, this.clusterId);
                Console.WriteLine("Cluster '{0}' untagged", ClusterName);
            }


            if (this.tagId != null)
            {
                this.tagService.Delete(tagId);
            }

            if (this.categoryId != null)
            {
                this.categoryService.Delete(categoryId);
            }
        }
        private string CreateCategory(
            Category categoryService,
            string name,
            string description,
            CategoryModel.Cardinality cardinality)
        {
            var createSpec = new CategoryTypes.CreateSpec();
            createSpec.SetName(name);
            createSpec.SetDescription(description);
            createSpec.SetCardinality(cardinality);
            createSpec.SetAssociableTypes(new HashSet<String>());
            return categoryService.Create(createSpec);
        }

        private string CreateTag(
            Tag taggingService,
            string name,
            string description,
            string categoryId)
        {
            var createSpec = new TagTypes.CreateSpec();
            createSpec.SetName(name);
            createSpec.SetDescription(description);
            createSpec.SetCategoryId(categoryId);
            return taggingService.Create(createSpec);
        }
    }
}
