"""
REST Handler for WSGI application
"""

__author__ = 'VMware, Inc.'
__copyright__ = 'Copyright (c) 2015-2016 VMware, Inc.  All rights reserved.'

import collections
import decimal
import json
import logging
import six
import werkzeug

from vmware.vapi.core import ExecutionContext
from vmware.vapi.data.serializers.cleanjson import DataValueConverter
from vmware.vapi.data.type import Type as DataType
from vmware.vapi.data.value import (
    data_value_factory, ListValue, StructValue,
    StringValue, OptionalValue, VoidValue, SecretValue)
from vmware.vapi.lib.constants import (OPERATION_INPUT, MAP_ENTRY)
from vmware.vapi.lib.context import create_default_application_context
from vmware.vapi.lib.load import dynamic_import
from vmware.vapi.protocol.client.local_connector import get_local_connector
from vmware.vapi.security.session import REST_SESSION_ID_KEY
from vmware.vapi.server.rest_rules import RoutingRuleGenerator
from vmware.vapi.settings import config
from vmware.vapi.settings.sections import REST
from vmware.vapi.stdlib.client.factories import StubConfigurationFactory

from com.vmware.vapi.metadata.metamodel_client import (
    Component, Structure, Type, Enumeration, GenericInstantiation)

logger = logging.getLogger(__name__)

# Mapping from HTTP method to operation name. The actual
# operation name might change based on ID parameters in the URL.
# Since GET is mapped to both list() and get(<>) operations,
# this map will return list as operation name and if the HTTP
# request has identifier arguments, then 'get' operation
# identifier should be used instead of 'list'.
http_method_map = {
    'GET': 'list',
    'PATCH': 'update',
    'DELETE': 'delete',
    'POST': 'create',
    'PUT': 'set',
    'HEAD': 'get'
}

# Mapping from vAPI operation identifier to HTTP method
vapi_method_map = dict((v, k) for k, v in six.iteritems(http_method_map))


class MetadataStore(object):
    """
    Helper class to process the metamodel metadata and
    provide a convenient way to access the data.
    """
    def __init__(self, api_provider):
        """
        Initialize MetadataStore

        :type  api_provider: :class:`vmware.vapi.core.ApiProvider`
        :param api_provider: ApiProvider to get the metadata
        """
        self.structure_map = {}
        self.enumeration_map = {}
        self.service_map = {}
        self._build(api_provider)

    def _build(self, api_provider):
        """
        Get the metamodel metadata and process it

        :type  api_provider: :class:`vmware.vapi.core.ApiProvider`
        :param api_provider: ApiProvider to get the metadata
        """
        local_connector = get_local_connector(api_provider)
        stub_config = StubConfigurationFactory.new_std_configuration(
            local_connector)
        component_svc = Component(stub_config)
        components = component_svc.list()
        for component_id in components:
            component_data = component_svc.get(component_id)
            self._process_component_info(component_data.info)

    def _process_component_info(self, component_info):
        """
        Process the metamodel component information. Scan the packages
        for services, structures and enumerations and store into dictionaries
        for fast lookup.

        :type  component_info: :class:`com.vmware.vapi.metadata.metamodel_client.ComponentInfo`
        :param component_info: Metamodel component information to be processed
        """
        for package_info in six.itervalues(component_info.packages):
            for structure_id, structure_info in six.iteritems(package_info.structures):
                self._process_structure_info(structure_id, structure_info)
            for service_id, service_info in six.iteritems(package_info.services):
                self._process_service_info(service_id, service_info)
            for enumeration_id, enumeration_info in six.iteritems(package_info.enumerations):
                self.enumeration_map[enumeration_id] = enumeration_info

    def _process_service_info(self, service_id, service_info):
        """
        Process the metamodel service information. Scan the services
        for operations, structures and enumerations and store into dictionaries
        for fast lookup.

        :type  service_id: :class:`str`
        :param service_id: Identifier of the service.
        :type  service_info: :class:`com.vmware.vapi.metadata.metamodel_client.ServiceInfo`
        :param service_info: Metamodel service information to be processed
        """
        for structure_id, structure_info in six.iteritems(service_info.structures):
            self._process_structure_info(structure_id, structure_info)
        for enumeration_id, enumeration_info in six.iteritems(service_info.enumerations):
            self.enumeration_map[enumeration_id] = enumeration_info
        operation_map = {}
        for operation_id, operation_info in six.iteritems(service_info.operations):
            param_map = collections.OrderedDict()
            for param_info in operation_info.params:
                param_map[param_info.name] = param_info
            operation_map[operation_id] = param_map
        self.service_map[service_id] = operation_map

    def _process_structure_info(self, structure_id, structure_info):
        """
        Process the metamodel structure information. Scan the structures for
        for fields and enumerations and store into dictionaries
        for fast lookup.

        :type  structure_id: :class:`str`
        :param structure_id: Identifier of the structure.
        :type  structure_info: :class:`com.vmware.vapi.metadata.metamodel_client.StructureInfo`
        :param structure_info: Metamodel structure information to be processed
        """
        field_map = {}
        for field_info in structure_info.fields:
            field_map[field_info.name] = field_info
        self.structure_map[structure_id] = field_map
        for enumeration_id, enumeration_info in six.iteritems(structure_info.enumerations):
            self.enumeration_map[enumeration_id] = enumeration_info


class URLValueDeserializer(object):
    """
    Deserialize the parameters provided in the HTTP query string
    into a dictionary.

    For example:
        /rest/vmodl/test/uber/rest/filter?
            struct.string_field=string&
            struct.boolean_field=True&
            struct.long_field=10&
            struct.struct_field.string_field=string&
            struct.struct_field.long_field=10&
            struct.optional_string=string&
            struct.list_string.1=string1&
            struct.list_string.2=string2&
            struct.list_list_long.1.1=11&
            struct.list_list_long.1.2=12&
            struct.list_list_long.2.1=21&
            struct.list_list_long.2.2=22&
            struct.map_simple.1.key=stringkey&
            struct.map_simple.1.value=stringvalue&
            struct.map_struct.1.key=stringkey&
            struct.map_struct.1.value.string_field=string&
            struct.map_struct.1.value.long_field=10

    1. Top level members of the request complex type are referenced by their
       name.
    2. Referencing nested complex structures will use "." and concatenation of
       names.
    3. Whenever arrays of given type are required we will use param.n notation
       to identify the instance number n of the object.

    Deserialized version of this query string will be:
    {
        "struct": {
            "string_field": "string",
            "boolean_field": True,
            "long_field": 10,
            "struct_field": {
                "long_field": 10,
                "string_field": "string"
            },
            "optional_string": "string",
            "list_string": [
                "string1",
                "string2"
            ],
            "list_list_long": [
                [
                    11,
                    12
                ],
                [
                    21,
                    22
                ]
            ],
            "map_simple": [
                {
                    "key": "stringkey",
                    "value": "stringvalue"
                }
            ],
            "map_struct": [
                {
                    "key": "stringkey",
                    "value": {
                        "long_field": 10,
                        "string_field": "string"
                    }
                }
            ]
        }
    }

    """
    @staticmethod
    def deserialize(query_string):
        """
        Deserialize the given query string into a python dictionary.

        :type  query_string: :class:`str`
        :param query_string: HTTP query string containing parameters.
        :rtype: :class:`dict` of :class:`str` and :class:`object`
        :return: Python dictionary deserialized from query string.
        """
        query_string_dict = {}

        query_string = query_string.decode('utf-8')
        for item in query_string.split('&'):
            # Parse only the query parameters
            if '=' not in item:
                continue

            k, v = item.split('=')
            tokens = k.split('.')

            #
            # The core idea here is to split the tokens in
            # a.b.c.d type of string and go from left to right.
            # At each step, the next token will be either put
            # at a key in the dictionary or element in a list if
            # the token is a digit.
            #
            # So, at each token, we should look ahead the next token,
            # and if the next token is a digit, then create an empty
            # list or return an existing list where it should be added
            # as element. If the next token is a non-digit, then create
            # a new dictionary or return an existing dictionary where
            # the token should be added as a key.
            #

            current_value = query_string_dict
            tokens_length = len(tokens)
            for i, token in enumerate(tokens):

                # Lookahead the next token
                next_token = None
                if i + 1 < tokens_length:
                    next_token = tokens[i + 1]

                # Next token is the last token
                if next_token is None:
                    if isinstance(current_value, list):
                        current_value.append(v)  # pylint: disable=E1101
                    elif isinstance(current_value, dict):
                        current_value[token] = v
                # Next token is not the last token
                else:
                    #
                    # If next token is a digit, create array as placeholder
                    # otherwise a dictionary
                    next_token_type = [] if next_token.isdigit() else {}

                    if isinstance(current_value, list):
                        index = int(token)
                        if index > len(current_value) + 1:
                            msg = ('Element with index %d is expected, but got %d' %
                                   (len(current_value) + 1, index))
                            logger.error(msg)
                            raise werkzeug.exceptions.BadRequest(msg)
                        elif index == len(current_value) + 1:
                            current_value.append(next_token_type)  # pylint: disable=E1101
                        next_value = current_value[index - 1]
                    elif isinstance(current_value, dict):
                        next_value = current_value.setdefault(
                            token, next_token_type)
                    current_value = next_value

        return query_string_dict


class DataValueDeserializer(object):
    """
    Convert from Python dictionary deserialized from a JSON object
    (or) from HTTP URL query string to DataValue.
    """
    _builtin_type_map = {
        Type.BuiltinType.VOID: DataType.VOID,
        Type.BuiltinType.BOOLEAN: DataType.BOOLEAN,
        Type.BuiltinType.LONG: DataType.INTEGER,
        Type.BuiltinType.DOUBLE: DataType.DOUBLE,
        Type.BuiltinType.STRING: DataType.STRING,
        Type.BuiltinType.BINARY: DataType.BLOB,
        Type.BuiltinType.SECRET: DataType.SECRET,
        Type.BuiltinType.DATE_TIME: DataType.STRING,
        Type.BuiltinType.ID: DataType.STRING,
        Type.BuiltinType.URI: DataType.STRING,
        Type.BuiltinType.ANY_ERROR: DataType.ANY_ERROR,
        Type.BuiltinType.DYNAMIC_STRUCTURE: DataType.DYNAMIC_STRUCTURE,
    }
    _builtin_native_type_map = {
        Type.BuiltinType.BOOLEAN: bool,
        Type.BuiltinType.LONG: int,
        Type.BuiltinType.DOUBLE: decimal.Decimal,
        Type.BuiltinType.STRING: str,
        Type.BuiltinType.BINARY: str,
        Type.BuiltinType.SECRET: str,
        Type.BuiltinType.DATE_TIME: str,
        Type.BuiltinType.ID: str,
        Type.BuiltinType.URI: str,
    }

    def __init__(self, metadata):
        """
        Initialize DataValueDeserializer

        :type  metadata: :class:`vmware.vapi.server.rest_handler.MetadataStore`
        :param metadata: Object that contains the metamodel metadata of
            all the services.
        """
        self._metadata = metadata
        self._category_map = {
            Type.Category.BUILTIN: self.visit_builtin,
            Type.Category.USER_DEFINED: self.visit_user_defined,
            Type.Category.GENERIC: self.visit_generic
        }
        self._generic_map = {
            GenericInstantiation.GenericType.OPTIONAL: self.visit_optional,
            GenericInstantiation.GenericType.LIST: self.visit_list,
            GenericInstantiation.GenericType.SET: self.visit_list,
            GenericInstantiation.GenericType.MAP: self.visit_map,
        }

    def visit_builtin(self, type_info, json_value):
        """
        Deserialize a primitive value

        :type  type_info: :class:`com.vmware.vapi.metadata.metamodel_client.Type`
        :param type_info: Metamodel type information
        :type  json_value: :class:`object`
        :param json_value: Value to be visited
        :rtype: :class:`vmware.vapi.data.value.DataValue`
        :return: DataValue created using the input
        """
        try:
            native_type = self._builtin_native_type_map[type_info.builtin_type]
            if native_type == str:
                native_value = six.text_type(json_value)
            elif native_type == int:
                native_value = int(json_value)
            elif native_type == decimal.Decimal:
                native_value = decimal.Decimal(json_value)
            elif native_type == bool:
                if isinstance(json_value, bool):
                    native_value = json_value
                elif json_value.lower() == 'true':
                    native_value = True
                elif json_value.lower() == 'false':
                    native_value = False
                else:
                    msg = 'Expected boolean value, but got %s' % json_value
                    logger.error(msg)
                    raise werkzeug.exceptions.BadRequest(msg)
            data_type = self._builtin_type_map[type_info.builtin_type]
            return data_value_factory(data_type, native_value)
        except KeyError:
            msg = ('Could not process the request, '
                   'builtin type %s is not supported' % type_info.builtin_type)
            logger.exception(msg)
            raise werkzeug.exceptions.InternalServerError(msg)

    def visit_user_defined(self, type_info, json_value):
        """
        Deserialize a user defined value

        :type  type_info: :class:`com.vmware.vapi.metadata.metamodel_client.Type`
        :param type_info: Metamodel type information
        :type  json_value: :class:`object`
        :param json_value: Value to be visited
        :rtype: :class:`vmware.vapi.data.value.StructValue` or
            :class:`vmware.vapi.data.value.StringValue`
        :return: DataValue created using the input
        """
        user_defined_type_info = type_info.user_defined_type
        if user_defined_type_info.resource_type == Enumeration.RESOURCE_TYPE:
            return StringValue(json_value)
        elif user_defined_type_info.resource_type == Structure.RESOURCE_TYPE:
            structure_info = self._metadata.structure_map[
                user_defined_type_info.resource_id]
            struct_value = StructValue(
                name=user_defined_type_info.resource_id)
            for field_name, field_value in six.iteritems(json_value):
                try:
                    field_info = structure_info[str(field_name)]
                    field_data_value = self.visit(field_info.type, field_value)
                    struct_value.set_field(field_name, field_data_value)
                except KeyError:
                    msg = 'Field %s is missing' % str(field_name)
                    logger.error(msg)
                    raise werkzeug.exceptions.BadRequest(msg)
            return struct_value
        else:
            msg = ('Could not process the request,'
                   'user defined type %s is not supported' %
                   user_defined_type_info.resource_type)
            logger.error(msg)
            raise werkzeug.exceptions.InternalServerError(msg)

    def visit_optional(self, type_info, json_value):
        """
        Deserialize an optional value

        :type  type_info: :class:`com.vmware.vapi.metadata.metamodel_client.Type`
        :param type_info: Metamodel type information
        :type  json_value: :class:`object`
        :param json_value: Value to be visited
        :rtype: :class:`vmware.vapi.data.value.OptionalValue`
        :return: DataValue created using the input
        """
        if json_value:
            element_value = self.visit(type_info.element_type, json_value)
            return OptionalValue(element_value)
        else:
            return OptionalValue()

    def visit_list(self, type_info, json_value):
        """
        Deserialize a list value

        :type  type_info: :class:`com.vmware.vapi.metadata.metamodel_client.Type`
        :param type_info: Metamodel type information
        :type  json_value: :class:`object`
        :param json_value: Value to be visited
        :rtype: :class:`vmware.vapi.data.value.ListValue`
        :return: DataValue created using the input
        """
        if not isinstance(json_value, list):
            msg = 'Excepted list, but got %s' % type(json_value).__name__
            logger.error(msg)
            raise werkzeug.exceptions.BadRequest(msg)
        return ListValue([
            self.visit(type_info.element_type, value) for value in json_value])

    def visit_map(self, type_info, json_value):
        """
        Deserialize a map value

        :type  type_info: :class:`com.vmware.vapi.metadata.metamodel_client.Type`
        :param type_info: Metamodel type information
        :type  json_value: :class:`object`
        :param json_value: Value to be visited
        :rtype: :class:`vmware.vapi.data.value.StructValue`
        :return: DataValue created using the input
        """
        if not isinstance(json_value, list):
            msg = 'Excepted list, but got %s' % type(json_value).__name__
            logger.error(msg)
            raise werkzeug.exceptions.BadRequest(msg)
        try:
            return ListValue(
                [StructValue(
                    name=MAP_ENTRY,
                    values={
                        'key': self.visit(type_info.map_key_type, value['key']),
                        'value': self.visit(type_info.map_value_type, value['value'])
                    })
                 for value in json_value])
        except KeyError as e:
            msg = 'Invalid Map input, missing %s' % str(e)
            logger.error(msg)
            raise werkzeug.exceptions.BadRequest(msg)

    def visit_generic(self, type_info, json_value):
        """
        Deserialize a list/optional/map value

        :type  type_info: :class:`com.vmware.vapi.metadata.metamodel_client.Type`
        :param type_info: Metamodel type information
        :type  json_value: :class:`object`
        :param json_value: Value to be visited
        :rtype: :class:`vmware.vapi.data.value.OptionalValue` or
            :class:`vmware.vapi.data.value.ListValue` or
            :class:`vmware.vapi.data.value.StructValue`
        :return: DataValue created using the input
        """
        try:
            generic_type_info = type_info.generic_instantiation
            generic_convert_method = self._generic_map[generic_type_info.generic_type]
            return generic_convert_method(generic_type_info, json_value)
        except KeyError:
            msg = ('Could not process the request, '
                   'generic type %s is not supported' % generic_type_info.generic_type)
            logger.exception(msg)
            raise werkzeug.exceptions.InternalServerError(msg)

    def visit(self, type_info, json_value):
        """
        Deserialize the given input using the metamodel type information

        :type  type_info: :class:`com.vmware.vapi.metadata.metamodel_client.Type`
        :param type_info: Metamodel type information
        :type  json_value: :class:`object`
        :param json_value: Value to be visited
        :rtype: :class:`vmware.vapi.data.value.DataValue` or
        :return: DataValue created using the input
        """
        convert_method = self._category_map[type_info.category]
        return convert_method(type_info, json_value)

    def generate_operation_input(self, service_id, operation_id, input_data):
        """
        This method generates a StructValue corresponding to the Python dict
        (deserialized from JSON) suitable as an input value for the specified
        operation.

        :type  service_id: :class:`str`
        :param service_id: Identifier of the service to be invoked.
        :type  operation_id: :class:`str`
        :param operation_id: Identifier of the operation to be invoked.
        :type  input_data: :class:`dict`
        :param input_data: Dictionary object that represents the deserialized
            json input.
        :rtype: :class:`vmware.vapi.data.value.DataValue` or
        :return: DataValue created using the input
        """
        operation_info = self._metadata.service_map[service_id][operation_id]
        fields = {}
        for param_name, param_value in six.iteritems(input_data):
            try:
                field_info = operation_info[str(param_name)]
                value = self.visit(field_info.type, param_value)
                fields[param_name] = value
            except KeyError:
                msg = 'Unexpected parameter %s ' % param_name
                logger.exception(msg)
                raise werkzeug.exceptions.BadRequest(msg)
        return StructValue(name=OPERATION_INPUT, values=fields)


class SecurityContextBuilder(object):
    """
    Helper class to build the appropriate security context based
    on the authentication information present in the request context.
    """
    def __init__(self):
        """
        Initialize SecurityContextBuilder
        """
        #
        # The order in the variable determines the order in which
        # the security parsers parse the request. The first parser
        # that can create a security context is used.
        #
        # If the request contains more than one kind of security
        # related information, the first one that gets parsed is used.
        # And if that information is invalid, 401 UNAUTHORIZED
        # is returned to the client. The other security information
        # that is not parsed is not taken into consideration.
        #
        # The user has to remove the invalid security context and invoke
        # the operation again.
        #
        parser_classes = config.get_value(
            section_name=REST,
            option_name='security_parsers',
            default_value=[])
        self._parsers = [dynamic_import(parser_class)()
                         for parser_class in parser_classes]

    def build(self, request):
        """
        Build the security context based on the authentication information
        in the request context

        :type  request: :class:`werkzeug.wrappers.Request`
        :param request: Request object
        :rtype: :class:`vmware.vapi.core.SecurityContext`
        :return: Security context
        """
        for parser in self._parsers:
            security_context = parser.build(request)
            if security_context:
                return security_context


class ResponseCookieBuilder(object):
    """
    Helper class to build the appropriate cookies to be set
    in the response.
    """
    def __init__(self):
        """
        Initialize ResponseCookieBuilder
        """
        self._session_methods = config.get_value(
            section_name=REST,
            option_name='session_methods',
            default_value=[])

    def build(self, service_id, operation_id, method_result):
        """
        Build the response cookie for the request

        :type  service_id: :class:`str`
        :param service_id: Identifier of the service to be invoked.
        :type  operation_id: :class:`str`
        :param operation_id: Identifier of the operation to be invoked.
        :type  method_result: :class:`vmware.vapi.core.MethodResult`
        :param method_result: MethodResult object to be serialized
        :rtype: :class:`dict` of :class:`str` and :class:`str`
        :return: Dictionary containing cookies that need to be set in the
            response.
        """
        session_method = '%s.%s' % (service_id, operation_id)
        if session_method in self._session_methods:
            if (isinstance(method_result.output, StringValue) or
                    isinstance(method_result.output, SecretValue)):
                session_id = method_result.output.value
            return {REST_SESSION_ID_KEY: session_id}


class RESTHandler(object):
    """
    Request handler that accept REST API calls and invoke the corresponding
    call on ApiProvider interface.
    """

    # Mapping of vAPI standard errors to HTTP error codes:
    # https://wiki.eng.vmware.com/VAPI/Specs/VMODL2toREST/Reference/REST-error-mapping
    _error_map = {
        'com.vmware.vapi.std.errors.already_exists': 400,
        'com.vmware.vapi.std.errors.already_in_desired_state': 400,
        'com.vmware.vapi.std.errors.feature_in_use': 400,
        'com.vmware.vapi.std.errors.internal_server_error': 500,
        'com.vmware.vapi.std.errors.invalid_argument': 400,
        'com.vmware.vapi.std.errors.invalid_configuration_change_for_current_state': 400,
        'com.vmware.vapi.std.errors.invalid_element_configuration': 400,
        'com.vmware.vapi.std.errors.invalid_element_type': 400,
        'com.vmware.vapi.std.errors.invalid_request': 400,
        'com.vmware.vapi.std.errors.not_found': 404,
        'com.vmware.vapi.std.errors.operation_not_allowed_in_current_state': 400,
        'com.vmware.vapi.std.errors.resource_busy': 400,
        'com.vmware.vapi.std.errors.resource_in_use': 400,
        'com.vmware.vapi.std.errors.resource_inaccessible': 400,
        'com.vmware.vapi.std.errors.service_not_found': 404,
        'com.vmware.vapi.std.errors.service_unavailable': 503,
        'com.vmware.vapi.std.errors.timed_out': 504,
        'com.vmware.vapi.std.errors.unable_to_allocate_resource': 400,
        'com.vmware.vapi.std.errors.unauthenticated': 401,
        'com.vmware.vapi.std.errors.unauthorized': 403,
        'com.vmware.vapi.std.errors.unimplemented': 501,
        'com.vmware.vapi.std.errors.unsupported': 400,
    }

    def __init__(self, api_provider):
        '''
        Initialize RESTHandler

        :type  api_provider: 'vmware.vapi.core.ApiProvider'
        :param api_provider: ApiProvider to be used to invoke the vAPI requests
        '''
        self._api_provider = api_provider
        metadata = MetadataStore(api_provider)
        self.rest_rules = RoutingRuleGenerator(metadata).rest_rules
        self.data_value_deserializer = DataValueDeserializer(metadata)
        self.security_context_builder = SecurityContextBuilder()
        self.response_cookie_builder = ResponseCookieBuilder()

    @staticmethod
    def _get_operation_id(http_method, query_string_dict, uri_parameters):
        """
        Figure out the operation identifier based on the HTTP method,
        the identifier arguments in the URL and the query string.

        :type  http_method: :class:`str`
        :param http_method: HTTP request method
        :type  query_string_dict: :class:`dict` or :class:`str` and :class:`object`
        :param query_string_dict: Decoded dictionary from the query string
        :type  uri_parameters: :class:`dict` of :class:`str` and :class:`object`
        :param uri_parameters: Arguments parsed from the HTTP URL
        :rtype: :class:`str`
        :return: Identifier of the operation to be invoked
        """
        operation_id = http_method_map[http_method]
        if uri_parameters and operation_id == 'list':
            ## TODO: Handle composite identifier case
            operation_id = 'get'
        action = query_string_dict.get('~action')
        if action:
            operation_id = action.replace('-', '_')
        return operation_id

    def _get_input_value(self, service_id, operation_id, input_data):
        """
        Generate the input DataValue for the given operation

        :type  service_id: :class:`str`
        :param service_id: Identifier of the service to be invoked.
        :type  operation_id: :class:`str`
        :param operation_id: Identifier of the operation to be invoked.
        :type  input_data: :class:`dict`
        :param input_data: Dictionary object that represents the deserialized
            json input.
        """
        try:
            return self.data_value_deserializer.generate_operation_input(
                service_id, operation_id, input_data)
        except Exception as e:
            msg = 'Cannot process request: %s' % str(e)
            logger.exception(msg)
            raise werkzeug.exceptions.BadRequest(msg)

    def _serialize_output(self, request, service_id, operation_id, method_result):
        """
        Serialize the MethodResult object

        :type  request: :class:`werkzeug.wrappers.Request`
        :param request: Request object
        :type  service_id: :class:`str`
        :param service_id: Identifier of the service to be invoked.
        :type  operation_id: :class:`str`
        :param operation_id: Identifier of the operation to be invoked.
        :type  method_result: :class:`vmware.vapi.core.MethodResult`
        :param method_result: MethodResult object to be serialized
        :rtype: :class:`tuple` of :class:`int`, :class:`str`, :class:`dict`
        :return: HTTP status code, serialized json output of the operation and
            a dictionary of response cookies.
        """
        if method_result.success():
            cookies = self.response_cookie_builder.build(
                service_id, operation_id, method_result)
            output = method_result.output
            json_output = None
            if not isinstance(output, VoidValue):
                if not isinstance(output, StructValue):
                    # If the output is not a struct or error value,
                    # box it with {'value':..} to make it a
                    # good json output
                    output = StructValue(name='output', values={
                        'value': output
                    })
                json_output = DataValueConverter.convert_to_json(output)
            status = 204 if request.method == 'DELETE' else 200
        else:
            cookies = None
            error = method_result.error
            json_output = DataValueConverter.convert_to_json(error)
            status = self._error_map.get(error.name, 500)
        return (status, json_output, cookies)

    def invoke(self, request, endpoint, uri_parameters):
        """
        Handle the REST API invocation

        :type  request: :class:`werkzeug.wrappers.Request`
        :param request: Request object
        :type  endpoint: :class:`str`
        :param endpoint: Identifier of the service to be invoked
        :type  uri_parameters: :class:`dict` of :class:`str` and :class:`object`
        :param uri_parameters: Arguments parsed from the HTTP URL
        :rtype: :class:`tuple` of :class:`int`, :class:`str`, :class:`dict`
        :return: HTTP status code, serialized json output of the operation and
            response cookies.
        """
        # Operation input is contributed by:
        # 1. URI parameters: Arguments parsed from HTTP URL
        # 2. Query parameters: Arguments parsed from the query string in the
        #               HTTP URL
        # 3. Request body: Arguments present in request body
        #
        # There is a definitive place for each argument, a particular argument
        # can be passed only using one of the above ways.
        #
        # Typically, main object identifiers are present in the base HTTP URL.
        # For GET operations, additional parameters can be provided in the query
        # string. For POST, PUT, PATCH, parameters can only be provided using
        # the body
        input_data = {}
        input_data.update(uri_parameters)
        query_string_flat_dict = werkzeug.urls.url_decode(
            request.query_string)

        if request.method == 'GET':
            query_string_parameters = URLValueDeserializer.deserialize(
                request.query_string)
            input_data.update(query_string_parameters)

        if request.method in ['PUT', 'PATCH', 'POST']:
            request_body_string = request.stream.read()
            if request_body_string:
                if isinstance(request_body_string, six.string_types):
                    input_data.update(
                        json.loads(request_body_string,
                                   parse_float=decimal.Decimal))
                elif isinstance(request_body_string, six.binary_type):
                    input_data.update(
                        json.loads(request_body_string.decode('utf-8'),
                                   parse_float=decimal.Decimal))

        # Create request context
        security_context = self.security_context_builder.build(request)
        application_context = create_default_application_context()
        ctx = ExecutionContext(
            application_context=application_context,
            security_context=security_context)

        service_id = endpoint
        operation_id = self._get_operation_id(
            request.method, query_string_flat_dict, uri_parameters)
        input_value = self._get_input_value(
            service_id, operation_id, input_data)
        method_result = self._api_provider.invoke(
            service_id, operation_id, input_value, ctx)
        return self._serialize_output(
            request, service_id, operation_id, method_result)
