"""
Configuration management for vAPI providers
"""

__author__ = 'VMware, Inc.'
__copyright__ = 'Copyright (c) 2015 VMware, Inc.  All rights reserved.'


# Singleton config variable
cfg = None


def get_value(section_name, option_name, default_value):
    """
    Get the configuration value for the given section and option.

    :type  section_name: :class:`str`
    :param section_name: Name of the section in the configuration file.
    :type  option_name: :class:`str`
    :param option_name: Name of the option in the configuration file.
    :type  default_value: :class:`object`
    :param default_value: Default value to be returned if section or option
        is not present in the configuration.
    :rtype: :class:`str` or :class:`list` of :class:`str` or :class:`object`
    :return: String or list of strings based on whether the value is
        comma separated or not. If the section or option is not present,
        the operation will return the default value provided
    """
    if cfg and cfg.has_section(section_name) and cfg.has_option(
            section_name, option_name):
        value = cfg.get(section_name, option_name)
        if ',' in value:
            tokens = value.replace('\n', '').replace('\\', '').split(',')
            return [token.strip() for token in tokens]
        else:
            return value
    else:
        return default_value
