/*

 * *****************************************************

 * Copyright VMware, Inc. 2010-2012.  All Rights Reserved.

 * *****************************************************

 *

 * DISCLAIMER. THIS PROGRAM IS PROVIDED TO YOU "AS IS" WITHOUT

 * WARRANTIES OR CONDITIONS # OF ANY KIND, WHETHER ORAL OR WRITTEN,

 * EXPRESS OR IMPLIED. THE AUTHOR SPECIFICALLY # DISCLAIMS ANY IMPLIED

 * WARRANTIES OR CONDITIONS OF MERCHANTABILITY, SATISFACTORY # QUALITY,

 * NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.

 */



package com.vmware.spbm.connection.helpers;



import java.util.Arrays;

import java.util.HashMap;

import java.util.List;

import java.util.Map;



import com.vmware.spbm.connection.Connection;

import com.vmware.spbm.connection.helpers.builders.ObjectSpecBuilder;

import com.vmware.spbm.connection.helpers.builders.PropertyFilterSpecBuilder;

import com.vmware.spbm.connection.helpers.builders.PropertySpecBuilder;

import com.vmware.vim25.DynamicProperty;

import com.vmware.vim25.InvalidCollectorVersionFaultMsg;

import com.vmware.vim25.InvalidPropertyFaultMsg;

import com.vmware.vim25.ManagedObjectReference;

import com.vmware.vim25.ObjectContent;

import com.vmware.vim25.ObjectSpec;

import com.vmware.vim25.ObjectUpdate;

import com.vmware.vim25.ObjectUpdateKind;

import com.vmware.vim25.PropertyChange;

import com.vmware.vim25.PropertyChangeOp;

import com.vmware.vim25.PropertyFilterSpec;

import com.vmware.vim25.PropertyFilterUpdate;

import com.vmware.vim25.PropertySpec;

import com.vmware.vim25.RetrieveOptions;

import com.vmware.vim25.RuntimeFaultFaultMsg;

import com.vmware.vim25.ServiceContent;

import com.vmware.vim25.UpdateSet;

import com.vmware.vim25.VimPortType;

import com.vmware.vim25.WaitOptions;



public class VCUtil {



   /**

    * Method to retrieve properties of a {@link ManagedObjectReference}

    *

    * @param entityMor

    *           {@link ManagedObjectReference} of the entity

    * @param props

    *           Array of properties to be looked up

    * @return Map of the property name and its corresponding value

    * @throws InvalidPropertyFaultMsg

    *            If a property does not exist

    * @throws RuntimeFaultFaultMsg

    */

   public static Map<String, Object> getEntityProps(Connection connection,

         ManagedObjectReference entityMor, String[] props)

         throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg {



      VimPortType vimPort = connection.getVimPort();

      ServiceContent serviceContent = connection.getVimServiceContent();



      final HashMap<String, Object> retVal = new HashMap<String, Object>();



      // Create PropertyFilterSpec using the PropertySpec and ObjectPec

      PropertyFilterSpec[] propertyFilterSpecs =

            { new PropertyFilterSpecBuilder().propSet(

            // Create Property Spec

                  new PropertySpecBuilder().all(Boolean.FALSE)

                        .type(entityMor.getType()).pathSet(props)).objectSet(

            // Now create Object Spec

                  new ObjectSpecBuilder().obj(entityMor)) };



      List<ObjectContent> oCont =

            vimPort.retrievePropertiesEx(serviceContent.getPropertyCollector(),

                  Arrays.asList(propertyFilterSpecs), new RetrieveOptions())

                  .getObjects();



      if (oCont != null) {

         for (ObjectContent oc : oCont) {

            List<DynamicProperty> dps = oc.getPropSet();

            for (DynamicProperty dp : dps) {

               retVal.put(dp.getName(), dp.getVal());

            }

         }

      }

      return retVal;

   }



   /**

    * Handle Updates for a single object. waits till expected values of

    * properties to check are reached Destroys the ObjectFilter when done.

    *

    * @param objmor

    *           MOR of the Object to wait for

    * @param filterProps

    *           Properties list to filter

    * @param endWaitProps

    *           Properties list to check for expected values these be properties

    *           of a property in the filter properties list

    * @param expectedVals

    *           values for properties to end the wait

    * @return true indicating expected values were met, and false otherwise

    * @throws RuntimeFaultFaultMsg

    * @throws InvalidPropertyFaultMsg

    * @throws InvalidCollectorVersionFaultMsg

    *

    */

   public static Object[] waitForTask(Connection connection,

         ManagedObjectReference objmor, String[] filterProps,

         String[] endWaitProps, Object[][] expectedVals)

         throws InvalidPropertyFaultMsg, RuntimeFaultFaultMsg,

         InvalidCollectorVersionFaultMsg {



      VimPortType vimPort = connection.connect().getVimPort();

      ServiceContent serviceContent =

            connection.connect().getVimServiceContent();





      // version string is initially null

      String version = "";

      Object[] endVals = new Object[endWaitProps.length];

      Object[] filterVals = new Object[filterProps.length];



      PropertyFilterSpec spec = propertyFilterSpec(objmor, filterProps);



      ManagedObjectReference filterSpecRef =

            vimPort.createFilter(serviceContent.getPropertyCollector(), spec,

                  true);



      boolean reached = false;



      UpdateSet updateset = null;

      List<PropertyFilterUpdate> filtupary = null;

      List<ObjectUpdate> objupary = null;

      List<PropertyChange> propchgary = null;

      while (!reached) {

         updateset =

               vimPort.waitForUpdatesEx(serviceContent.getPropertyCollector(),

                     version, new WaitOptions());

         if (updateset == null || updateset.getFilterSet() == null) {

            continue;

         }

         version = updateset.getVersion();



         // Make this code more general purpose when PropCol changes later.

         filtupary = updateset.getFilterSet();



         for (PropertyFilterUpdate filtup : filtupary) {

            objupary = filtup.getObjectSet();

            for (ObjectUpdate objup : objupary) {

               if (objup.getKind() == ObjectUpdateKind.MODIFY

                     || objup.getKind() == ObjectUpdateKind.ENTER

                     || objup.getKind() == ObjectUpdateKind.LEAVE) {

                  propchgary = objup.getChangeSet();

                  for (PropertyChange propchg : propchgary) {

                     updateValues(endWaitProps, endVals, propchg);

                     updateValues(filterProps, filterVals, propchg);

                  }

               }

            }

         }



         Object expctdval = null;

         // Check if the expected values have been reached and exit the loop

         // if done.

         // Also exit the WaitForUpdates loop if this is the case.

         for (int chgi = 0; chgi < endVals.length && !reached; chgi++) {

            for (int vali = 0; vali < expectedVals[chgi].length && !reached; vali++) {

               expctdval = expectedVals[chgi][vali];



               reached = expctdval.equals(endVals[chgi]) || reached;

            }

         }

      }



      // Destroy the filter when we are done.

      vimPort.destroyPropertyFilter(filterSpecRef);

      return filterVals;

   }



   private static void updateValues(String[] props, Object[] vals,

         PropertyChange propchg) {

      for (int findi = 0; findi < props.length; findi++) {

         if (propchg.getName().lastIndexOf(props[findi]) >= 0) {

            if (propchg.getOp() == PropertyChangeOp.REMOVE) {

               vals[findi] = "";

            } else {

               vals[findi] = propchg.getVal();

            }

         }

      }

   }



   private static PropertyFilterSpec propertyFilterSpec(

         ManagedObjectReference objmor, String[] filterProps) {

      PropertyFilterSpec spec = new PropertyFilterSpec();

      ObjectSpec oSpec = new ObjectSpec();

      oSpec.setObj(objmor);

      oSpec.setSkip(Boolean.FALSE);

      spec.getObjectSet().add(oSpec);



      PropertySpec pSpec = new PropertySpec();

      pSpec.getPathSet().addAll(Arrays.asList(filterProps));

      pSpec.setType(objmor.getType());

      spec.getPropSet().add(pSpec);

      return spec;

   }



}

