using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Text;
using Vim25Api;
using System.Xml;
using AppUtil;
namespace XVCvMotion
{
    ///<summary>
    ///This sample is used to relocate VM from one Virtual Center to another Virtual Center.
    ///</summary>
    ///<param name="vmname">Required: name of the virtual machine</param>
    ///<param name="remoteurl">Required: remote VC URL</param>
    ///<param name="ruser">Required: Remote VC username</param>
    ///<param name="rpassword">Required: Remote VC password</param>
    ///<param name="rthumbprint">Required: Thumbprint value for Destination VC</param>
    ///<param name="clustername">Required: Target Cluster</param>
    ///<param name="targetfolder">Optional: Target Folder</param>
    ///<remarks>
    ///Relocate VM across VC
    ///--url [webserviceurl]
    ///--username [username] --password [password] --vmname [myVM] --remoteurl [rurl] --ruser [ruser ] --rpassword [rpassword ]
    ///--rthumbprint [thumbprint] --clustername [clustername] --targetfolder [targetfolder]
    ///</remarks>
    class XVCvMotion
    {
        private static AppUtil.AppUtil cb = null;
        private static AppUtil.AppUtil cb_dest = null;

        private void MigrateVM()
        {
            String vmname = cb.get_option("vmname");
            String remoteurl = cb.get_option("remoteurl");
            String ruser = cb.get_option("ruser");
            String rpassword = cb.get_option("rpassword");
            String rthumbprint = cb.get_option("rthumbprint");
            String clustername = cb.get_option("clustername");
            String targetfolder = cb.get_option("targetfolder");

            PlacementResult placementResult;
            PlacementSpec placementSpec = new PlacementSpec();
            ManagedObjectReference vmMOR = null;
            ManagedObjectReference vmfolderMoRef = null;

            //Getting MOR of the VM to be Migrated
            Console.WriteLine("Searching for VM '" + vmname + "'..");
            vmMOR = cb._svcUtil.GetDecendentMoRef(null, "VirtualMachine", vmname);
            if (vmMOR != null)
            {
                Console.WriteLine("Found VM: " + vmname);
                VirtualMachineConfigSpec configSpec = new VirtualMachineConfigSpec();

                //Getting Properties of the VM that need to be Migrated in to an array of Objects.
                ObjectContent[] objcontent = cb.getServiceUtil().GetObjectProperties(null, vmMOR, new string[] { "config.cpuAllocation", "config.files", "config.hardware.device", "config.hardware.memoryMB", "config.hardware.numCPU", "config.memoryAllocation", "config.name", "config.swapPlacement", "config.version" });
                DynamicProperty[] properties = objcontent[0].propSet;
                Dictionary<String, Object> vmProperties = new Dictionary<string, object>();
                foreach (DynamicProperty dynamicProperty in properties)
                {
                    vmProperties.Add(dynamicProperty.name, dynamicProperty.val);
                }
                Console.WriteLine("Setting VirtualMachineConfigSpec properties--");
                configSpec.cpuAllocation = (ResourceAllocationInfo)vmProperties["config.cpuAllocation"];
                configSpec.files = (VirtualMachineFileInfo)vmProperties["config.files"];
                configSpec.memoryMBSpecified = true;
                configSpec.memoryMB = System.Convert.ToInt64(vmProperties["config.hardware.memoryMB"]);
                configSpec.numCPUsSpecified = true;
                configSpec.numCPUs = (int)vmProperties["config.hardware.numCPU"];
                configSpec.memoryAllocation = (ResourceAllocationInfo)vmProperties["config.memoryAllocation"];
                configSpec.name = (String)vmProperties["config.name"];
                configSpec.swapPlacement = (String)vmProperties["config.swapPlacement"];
                configSpec.version = (String)vmProperties["config.version"];

                //Setting devicechange Property of VirtualMachineConfigSpec
                VirtualDevice[] listvd = (VirtualDevice[])vmProperties["config.hardware.device"];
                VirtualDeviceConfigSpec[] vdconfigspecArray = new VirtualDeviceConfigSpec[listvd.Length];
                int position = 0;

                foreach (VirtualDevice device in listvd)
                {
                    VirtualDeviceConfigSpec vconfig = new VirtualDeviceConfigSpec();
                    vconfig.device = device;
                    vdconfigspecArray[position] = vconfig;
                    position++;
                }
                configSpec.deviceChange = vdconfigspecArray;

                //Setting placementSpec configSpec property
                placementSpec.configSpec = configSpec;

                //Getting the MOR for the Target
                Console.WriteLine("Looking for the Cluster defined on Destination VC");
                ManagedObjectReference clusterMOR = cb_dest._svcUtil.GetDecendentMoRef(null, "ClusterComputeResource", clustername);
                if (clusterMOR != null)
                {
                    Console.WriteLine("Found Cluster '" + clustername + "'on Destination vCenter!");
                    //To Get the Recommendation from the DRS
                    Console.WriteLine("Getting Recommendations from DRS for XVCvMotion--");
                    PlacementAction action = new PlacementAction();
                    try
                    {
                        placementResult = cb_dest._connection._service.PlaceVm(clusterMOR, placementSpec);
                        action = GetPlacementAction(placementResult);
                    }
                    catch (Exception e)
                    {
                        if (e.Message.Equals("vim.fault.InvalidState"))
                        {
                            Console.WriteLine("Check the Cluster setting and make sure that DRS is enabled");
                        }
                        else
                        {
                            Console.WriteLine(e.StackTrace);
                        }
                        return;
                    }
                    if (action != null)
                    {
                        if (targetfolder == null)
                        {
                            vmfolderMoRef = GetVMFolderMOR(clusterMOR, "Folder", "vm");
                        }
                        else
                        {
                            vmfolderMoRef = GetVMFolderMOR(clusterMOR, "Folder", targetfolder);
                        }
                        if (vmfolderMoRef != null)
                        {
                            //Using Target Virtual center details to initialize ServiceLocation instance
                            ServiceLocatorNamePassword slNamePassowrd = new ServiceLocatorNamePassword();
                            slNamePassowrd.username = ruser;
                            slNamePassowrd.password = rpassword;
                            ServiceLocator locator = new ServiceLocator();
                            locator.credential = slNamePassowrd;
                            locator.url = remoteurl;
                            locator.instanceUuid = cb_dest.getConnection()._sic.about.instanceUuid;
                            locator.sslThumbprint = rthumbprint;
                            VirtualMachineRelocateSpec relocateSpec = action.relocateSpec;
                            relocateSpec.service = locator;
                            relocateSpec.folder = vmfolderMoRef;

                            ManagedObjectReference taskMOR =
                                cb.getConnection()._service.RelocateVM_Task(vmMOR, relocateSpec, VirtualMachineMovePriority.defaultPriority);
                            String res = cb.getServiceUtil().WaitForTask(taskMOR);
                            if (res.Equals("sucess"))
                            {
                                Console.WriteLine("Relocation done successfully ");
                            }
                            else
                            {
                                Console.WriteLine("Error::  Relocation failed");
                            }
                        }
                        else
                        {
                            Console.WriteLine("Error:: Folder Not Found");
                        }

                    }

                    else
                    {
                        Console.WriteLine("Recommendations are not Correct");
                    }
                }

                else
                {
                    Console.WriteLine("No Cluster by the name of '" + clustername + "' found!");
                }
            }
            else
            {
                Console.WriteLine("No VM by the name of '" + vmname + "' found!");
            }
        }

        /// <summary>
        /// To Get the MOR of Folder where to place the VM
        /// </summary>
        /// <param name="name"> Name of Folder</param>
        /// <param name="type">Type :Folder</param>
        /// <param name="root">Starting point for the Search</param>
        /// <returns>MOR Of the Folder where to Place the VM</returns>
        public ManagedObjectReference GetVMFolderMOR(
           ManagedObjectReference root, string type, string name
        )
        {
            if (name == null || name.Length == 0)
            {
                return null;
            }

            string[][] typeinfo =
               new string[][] { new string[] { type,  "name", },
         };

            ObjectContent[] ocary =
               GetContentsRecursively(null, root, typeinfo, true);

            if (ocary == null || ocary.Length == 0)
            {
                return null;
            }

            ObjectContent oc = null;
            ManagedObjectReference mor = null;
            DynamicProperty[] propary = null;
            string propval = null;
            bool found = false;
            for (int oci = 0; oci < ocary.Length && !found; oci++)
            {
                oc = ocary[oci];
                mor = oc.obj;
                propary = oc.propSet;

                if ((type == null) || (type != null && cb_dest._svcUtil.typeIsA(type, mor.type)))
                {
                    if (propary.Length > 0)
                    {
                        propval = (string)propary[0].val;
                    }

                    found = propval != null && name.Equals(propval);
                    propval = null;
                }
            }

            if (!found)
            {
                mor = null;
            }

            return mor;
        }


        public ObjectContent[] GetContentsRecursively(
         ManagedObjectReference collector, ManagedObjectReference root,
         string[][] typeinfo, bool recurse
      )
        {
            if (typeinfo == null || typeinfo.Length == 0)
            {
                return null;
            }

            ManagedObjectReference usecoll = collector;
            if (usecoll == null)
            {
                usecoll = cb_dest._connection.PropCol;
            }

            ManagedObjectReference useroot = root;
            if (useroot == null)
            {
                useroot = cb_dest._connection.Root;
            }
            SelectionSpec[] selectionSpecs = null;
            if (recurse)
            {
                selectionSpecs = buildVMFolderTraversal();
            }

            PropertySpec[] propspecary = cb_dest._svcUtil.BuildPropertySpecArray(typeinfo);

            PropertyFilterSpec spec = new PropertyFilterSpec();
            spec.propSet = propspecary;
            spec.objectSet = new ObjectSpec[] { new ObjectSpec() };
            spec.objectSet[0].obj = useroot;
            spec.objectSet[0].skip = false;
            spec.objectSet[0].selectSet = selectionSpecs;

            ObjectContent[] retoc = cb_dest._svcUtil.retrievePropertiesEx(usecoll, new PropertyFilterSpec[] { spec });

            return retoc;
        }

        /// <summary>
        /// To create the traversal Spec to obtain the MOR of the Folder where to place the VM starting from ClusterMOR
        /// </summary>
        /// <returns>MOR</returns>
        public SelectionSpec[] buildVMFolderTraversal()
        {
            // DC -> VM Folder
            TraversalSpec dcToVmf = new TraversalSpec();
            dcToVmf.name = "dcToVmf";
            dcToVmf.type = "Datacenter";
            dcToVmf.path = "vmFolder";
            dcToVmf.skip = false;
            dcToVmf.skipSpecified = true;
            dcToVmf.selectSet = new SelectionSpec[] { new SelectionSpec() };
            dcToVmf.selectSet[0].name = "visitFolders";


            // CLUSTER -> Parent
            TraversalSpec crtopr = new TraversalSpec();
            crtopr.type = "ClusterComputeResource";
            crtopr.skipSpecified = true;
            crtopr.skip = false;
            crtopr.path = "parent";
            crtopr.name = "crtopr";
            crtopr.selectSet = new SelectionSpec[] { new SelectionSpec() };
            crtopr.selectSet[0].name = "foltopr";

            // Folder->Parent
            TraversalSpec foltopr = new TraversalSpec();
            foltopr.type = "Folder";
            foltopr.skipSpecified = true;
            foltopr.skip = false;
            foltopr.path = "parent";
            foltopr.name = "foltopr";
            foltopr.selectSet = new SelectionSpec[] { new SelectionSpec(), new SelectionSpec() };
            foltopr.selectSet[0].name = "foltopr";
            foltopr.selectSet[1].name = "dcToVmf";

            // Recurse thriugh the folders
            TraversalSpec visitFolders = new TraversalSpec();
            visitFolders.name = "visitFolders";
            visitFolders.type = "Folder";
            visitFolders.path = "childEntity";
            visitFolders.skip = false;
            visitFolders.skipSpecified = true;
            visitFolders.selectSet = new SelectionSpec[] { new SelectionSpec() };
            visitFolders.selectSet[0].name = "visitFolders";
            return new SelectionSpec[] { visitFolders, dcToVmf, foltopr, crtopr };
        }

        /// <summary>
        /// This method is to get the placement Action of Virtual machine.
        /// </summary>
        /// <param name="placementResult">PlacementResult is the class of the result returned by PlaceVm method which invokes DRS for recommendations for target hosts and datastores for placing a virtual machine and its virtual disks</param>
        /// <returns>PlacementAction</returns>
        public PlacementAction GetPlacementAction(PlacementResult placementResult)
        {
            ClusterRecommendation[] recommendatons = placementResult.recommendations;
            PlacementAction placementAction = null;
            int size = recommendatons.Length;
            bool actionOk = false;
            if (size > 0)
            {
                Console.WriteLine("Total number of recommendations are " + size);
                Console.WriteLine("Processing the xvcvmotion placement recommendations out of the recommendations received");
                foreach (ClusterRecommendation recommendation in recommendatons)
                {
                    if (recommendation.reason.Equals("xvmotionPlacement"))
                    {
                        ClusterAction[] actions = recommendation.action;
                        foreach (ClusterAction action in actions)
                        {
                            if (action is PlacementAction)
                            {
                                placementAction = (PlacementAction)action;
                                break;
                            }
                        }
                        if (placementAction != null)
                        {
                            if (placementAction.vm == null || placementAction.targetHost == null)
                            {
                                Console.WriteLine("Placement Action doesnot have a vm and targethost set");
                            }
                            else
                            {
                                if (placementAction.relocateSpec != null)
                                {
                                    actionOk = checkRelocateSpec(placementAction.relocateSpec);
                                    if (actionOk)
                                        break;
                                    else
                                        placementAction = null;
                                }
                            }
                        }
                        else
                        {
                            Console.WriteLine("Recommendation doesnot have a placement action");
                        }
                    }
                }
            }
            else
            {
                Console.WriteLine("No recommendations by DRS");
            }
            return placementAction;
        }

        //Create UserOptionSpec
        public static OptionSpec[] constructOptions()
        {
            OptionSpec[] useroptions = new OptionSpec[7];
            useroptions[0] = new OptionSpec("vmname", "String", 1,
                                            "Name of the virtual machine to be migrated"
                                            , null);
            useroptions[1] = new OptionSpec("remoteurl", "String", 1,
                                            "remote VC URL",
                                            null);
            useroptions[2] = new OptionSpec("ruser", "String", 1,
                                          "Remote VC username",
                                          null);
            useroptions[3] = new OptionSpec("rpassword", "String", 1,
                                         "Remote VC password",
                                         null);
            useroptions[4] = new OptionSpec("rthumbprint", "String", 1,
                                          "Thumbprint value for Destination VC",
                                          null);
            useroptions[5] = new OptionSpec("clustername", "String", 1,
                                          "Destination Cluster where to place the VM",
                                          null);
            useroptions[6] = new OptionSpec("targetfolder", "String", 0,
                                     "target folder where to place the VM",
                                     null);

            return useroptions;
        }

        /// <summary>
        /// The method is to check the relocateSpec is all set.
        /// </summary>
        /// <param name="relocateSpec">Specification for moving or copying a virtual machine to a different datastore or host</param>
        /// <returns>boolean</returns>
        bool checkRelocateSpec(VirtualMachineRelocateSpec relocateSpec)
        {
            bool check = false;
            if (relocateSpec.host != null)
            {
                if (relocateSpec.pool != null)
                {
                    if (relocateSpec.datastore != null)
                    {
                        check = true;
                    }
                    else
                    {
                        Console.WriteLine("RelocateSpec doesnot have a datastore");
                    }
                }
                else
                {
                    Console.WriteLine("RelocateSpec doesnot have a resource pool");
                }
            }
            else
            {
                Console.WriteLine("RelocateSpec doesnot have a host");
            }
            return check;
        }

        /// <summary>
        /// The Main Method that makes connection to Source and Destination VC, calls up MigrateVM method.
        /// </summary>
        /// <param name="args">the command line arguments</param>
        static void Main(string[] args)
        {
            String[] r_args = { "--url", "", "--username", "", "--password", "", "--disablesso", "true", "--ignorecert" };
            int index = 0;
            String source = null;
            foreach (string s in args)
            {

                if (s.Equals("--remoteurl"))
                {
                    r_args.SetValue(args.GetValue(index + 1), 1);
                }
                if (s.Equals("--ruser"))
                {
                    r_args.SetValue(args.GetValue(index + 1), 3);
                }
                if (s.Equals("--rpassword"))
                {
                    r_args.SetValue(args.GetValue(index + 1), 5);
                }
                 if (s.Equals("--url"))
                {
                  source = (String)args.GetValue(index +1);
                }

                index++;
            }
            XVCvMotion app = new XVCvMotion();
            cb = AppUtil.AppUtil.initialize("XVCvMotion", XVCvMotion.constructOptions(), args);
            cb_dest = AppUtil.AppUtil.initialize("XVCvMotion", null, r_args);
            try
            {
                //Connecting to the Source Virtual Center
                Console.WriteLine("Connecting to the Source vCenter - " + source);
                cb.connect();
                //Connecting to the Target Virtual Center
                Console.WriteLine("Connecting to the Target vCenter -" + r_args.GetValue(1));
                cb_dest.connect();
                app.MigrateVM();
                //Disconnecting
                Console.WriteLine("Disconnecting ...");
                cb.disConnect();
                cb_dest.disConnect();
            }
            catch (Exception e)
            {
                Console.WriteLine(e.Message.ToString());

            }

            Console.WriteLine("Press any key to exit: ");
            Console.Read();
        }
    }
}
