#!/bin/sh

MY_INSTANCE_BASE="$INSTANCE_BASE"
unset INSTANCE_BASE
export INSTANCE_BASE
INSTANCE_BASE="$MY_INSTANCE_BASE"

#function that prints out usage
usage () {
    PROG=`basename $0`
    echo "${PROG} instance_name cmd [options]"
    echo " "
    echo "  cmd is one of start | run | stop | restart | status"
    echo "    start             - starts a tomcat instance as a daemon process"
    echo "    run               - starts a tomcat instance as a foreground process"
    echo "    stop              - stops a running tomcat instance, forcing termination"
    echo "                        of the process if it does not exit gracefully within"
    echo "    restart           - restarts a running tomcat instance, forcing"
    echo "                        termination of the process if it does not exit"
    echo "                        gracefully within timeout of 30 seconds"
    echo "    status            - reports the status of a tomcat instance"
    echo " "
}

#find out the absolute path of the script
setupdir () {

  # handle softlinks
  SOURCE="$0"
  while [ -h "$SOURCE" ]; do
    PRGDIR="$( cd -P "$( dirname "$SOURCE" )" && pwd )"
    SOURCE="$(readlink "$SOURCE")"
    [[ $SOURCE != /* ]] && SOURCE="$PRGDIR/$SOURCE"
  done
  PRGDIR="$( cd -P "$( dirname "$SOURCE" )" && pwd )"

  if [ ! -r "$PRGDIR" ]; then
    echo "ERROR: Tomcat directory $PRGDIR does not exist or is not readable."
    exit 1
  fi
}

setupINSTANCE_NAMEandCOMMAND() {
  while [ $# -gt 0 ]; do
    case $1 in
      -* )
        shift
        ;;
      * )
        if [ $# -lt 2 ]; then
          echo "ERROR: Invalid number of arguments on command-line."
          usage
          exit 1
        fi
        INSTANCE_NAME=`echo $1 | sed 's/\/$//g'`
        COMMAND=$2
        break
        ;;
    esac
    shift
  done
}


setupCATALINA_BASE () {
  if [ "$COMMAND" = "create" ]; then
    if [ ! -x "$INSTANCE_BASE" ]; then
      echo "ERROR: Instance directory not writeable (${INSTANCE_BASE})."
      exit 1
    else
      return
    fi
  fi

  CATALINA_BASE=$INSTANCE_BASE/$INSTANCE_NAME

  if [ ! -r "$CATALINA_BASE" ]; then
    echo "ERROR: CATALINA_BASE directory $CATALINA_BASE does not exist or is not readable."
    exit 1
  fi
  if [ ! -d "$CATALINA_BASE" ]; then
    echo "ERROR: CATALINA_BASE $CATALINA_BASE is not a directory."
    exit 1
  fi

  CATALINA_BASE=`cd "$CATALINA_BASE"; pwd -P`
  INSTANCE_NAME=`basename "$CATALINA_BASE"`
}


setupTOMCAT_VERSION () {
  if [ -z "$TOMCAT_VER" ]; then
    if [ -r "$CATALINA_HOME"/conf/tomcat.version ]; then
      TOMCAT_VER=`cat "$CATALINA_HOME/conf/tomcat.version"`
    fi
  fi
}


setupCATALINA_HOME () {
  [ -z "$CATALINA_HOME" ] && CATALINA_HOME=`cd "$PRGDIR/../../" ; pwd -P`
  if [ ! -r "$CATALINA_HOME" ]; then
    echo "ERROR: CATALINA_HOME directory $CATALINA_HOME does not exist or is not readable."
    exit 1
  fi
}


isrunning() {
  #returns 0 if the process is running
  #returns 1 if the process is not running, or file was removed.
  #returns 2 if the process state is unknown
  if [ -f "$CATALINA_PID" ]; then
        # The process file exists, make sure the process is not running
        PID=`cat "$CATALINA_PID"`
        if [ -z $PID ]; then
            return 2;
        fi

        LINES="ps ww -p"
        LINES=`$LINES $PID 2> /dev/null | grep "\-Dcatalina.base=$CATALINA_BASE"`
        PIDRET=$?
        if [ $PIDRET -eq 0 ]; then
          export PID
          return 0;
        fi
        rm -f "$CATALINA_PID"
    fi
    return 1
}

instance_start() {
    isrunning
    if [ $? -eq 0 ]; then
      echo "ERROR: Instance is already running as PID=$PID."
      exit 1
    fi
    "$CATALINA_SCRIPT" start
    sleep 2
    isrunning
    evaluateStatusAndExit $?
}

instance_run() {
    isrunning
    if [ $? -eq 0 ]; then
      echo "ERROR: Instance is already running as PID=$PID."
      exit 1
    fi
    # catalina.sh won't create a PID file when using the run command
    if [ ! -f "$CATALINA_PID" ]; then
      echo $$ > "$CATALINA_PID"
    fi
    exec "$CATALINA_SCRIPT" run
}

instance_stop() {
    WAIT_FOR_SHUTDOWN=30
    isrunning
    RUNNING=$?
    if [ $RUNNING -eq 0 ]; then
        #tomcat process is running
        echo "Instance is running as PID=$PID, shutting down..."
        kill $PID
    elif [ $RUNNING -eq 2 ]; then
        echo "No action taken. Unable to stop Tomcat instance that was started as a foreground process. Use CTRL+C in the console of the process to stop the instance"
        return 2
    else
        #if the log file is not there then it is likely someone may have clean out the contents of the log directory, including the pid file
        #or they are trying to stop a new instance.
        CATALINA_LOG="${CATALINA_BASE}/logs/catalina.out"
        if [ -f "${CATALINA_LOG}" ]; then
            echo "Instance is not running. No action taken"
        else
            echo "Running state is unknown. Expected pid file '${CATALINA_PID}' was not found."
            echo "This could mean you are attempting to stop an instance which has never been started or the logs directory has been purged."
            return 2
        fi
        return 1
    fi
    isrunning
    if [ $? -eq 0 ]; then
        #process still exists
        echo "Instance is running PID=$PID, sleeping for up to $WAIT_FOR_SHUTDOWN seconds waiting for shutdown"
        i=0
        while [ $i -lt $WAIT_FOR_SHUTDOWN ]; do
            sleep 1
            isrunning
            if [ $? -eq 1 ]; then
                break
            fi
            i=`expr $i + 1`
        done
    fi

    # CAN_DELETE_PID = 0 is Yes, any other number is no.
    CAN_DELETE_PID=1
    isrunning
    if [ $? -eq 0 ];
    then
        echo "Instance is still running PID=$PID, forcing a shutdown"
        kill -9 $PID
        CAN_DELETE_PID=$?
    else
        echo "Instance shut down gracefully"
        CAN_DELETE_PID=0
    fi
    if [ ${CAN_DELETE_PID} -eq 0 ]; then
       if [ -f "$CATALINA_PID" ]; then
          rm -f "$CATALINA_PID"
       fi
    else
       echo "Unable to stop process PID=${PID}."
       exit 1
    fi
}

instance_restart() {
    instance_stop $@
    if [ $? -eq 0 ]; then
        instance_start
    fi
    exit $?
}

instance_status() {
    echo "Instance name:         ${INSTANCE_NAME}"
    echo "Tomcat version:        ${TOMCAT_VER}"
    echo "Tomcat Base:           ${CATALINA_BASE}"
    echo "Tomcat Home:           ${CATALINA_HOME}"

    isrunning
    evaluateStatusAndExit
}

evaluateStatusAndExit() {
    RUNNING=$?
    if [ $RUNNING -eq 0 ]; then
      echo "Status:                RUNNING as PID=$PID"
    elif [ $RUNNING -eq 2 ]; then
      echo "Status:                UNKNOWN due to Tomcat instance being started as a foreground process"
    else
      echo "Status:                NOT RUNNING"
    fi
    if [ -z "$1" ]; then
      exit 0
    else
      exit $1
    fi
}

# MAIN SCRIPT EXECUTION
setupdir "$@"
setupINSTANCE_NAMEandCOMMAND "$@"
if [ -z "$INSTANCE_NAME" ] || [ -z "$COMMAND" ]; then
    echo "ERROR: Instance name or command was not specified"
    usage
    exit 1
fi

setupCATALINA_BASE "$@"

. "$CATALINA_BASE/bin/setenv.sh"
CATALINA_PID="$CATALINA_BASE/logs/tomcat.pid"
setupCATALINA_HOME "$@"

setupTOMCAT_VERSION "$@"

CATALINA_SCRIPT="$CATALINA_HOME/bin/catalina.sh"

if [ -z "$LOGGING_CONFIG" ]; then
  if [ -r "$CATALINA_BASE"/conf/logging.properties ]; then
    LOGGING_CONFIG="-Djava.util.logging.config.file=$CATALINA_BASE/conf/logging.properties"
  else
    LOGGING_CONFIG="-Dnop"
  fi
fi

export CATALINA_HOME
export CATALINA_BASE
export CATALINA_PID
export CATALINA_SCRIPT
export LOGGING_CONFIG
export LOGGING_MANAGER
export INSTANCE_NAME

# Check cmd is expected
if [ "$COMMAND" != "start" ] && [ "$COMMAND" != "run" ] && [ "$COMMAND" != "stop" ] && [ "$COMMAND" != "restart" ] && [ "$COMMAND" != "status" ]; then
    echo " "
    echo "ERROR: Invalid command $COMMAND"
    echo " "
    usage
    exit 1
fi

#execute the specified action
instance_$COMMAND $@
