#!/bin/sh
# Script to create a CATALINA_BASE directory for your own tomcat

PROG=$(basename $0)

SOURCE="$0"
while [ -h "$SOURCE" ]; do
  PROG_DIR="$( cd -P "$( dirname "$SOURCE" )" && pwd )"
  SOURCE="$(readlink "$SOURCE")"
  [[ $SOURCE != /* ]] && SOURCE="$PROG_DIR/$SOURCE"
done
PROG_DIR="$( cd -P "$( dirname "$SOURCE" )" && pwd )"

usage() {
    read -r -d '' MSG << EOF
Usage: $PROG [-option <value>]
 *  --instance-user         The user to run the instance
 *  --instance-location     The directory where the instance will be created
 *  --instance-name         The name of the tomcat instance
    --provides              The value for "Provides" section of init info
    --should-start          The value for "Should-Start" section of init info
    --should-stop           The value for "Should-Stop" section of init info
    --short-descr           The value for "Short-Description" section of init info
    --description           The value for "Description" section of init info
    --chkconfig             The value for "chkconfig" section of init info
    -h, --help               Usage message
EOF

    echo "${MSG}"
    exit 1
}

# process the arguments
while [[ $# > 0 ]]; do
    key="$1"
    case $key in
        '--provides')
            export INIT_PROVIDES="$2"
            shift
            ;;
        '--should-start')
            export INIT_START="$2"
            shift
            ;;
        '--should-stop')
            export INIT_STOP="$2"
            shift
            ;;
        '--short-descr')
            export INIT_SHORT_DESCRIPTION="$2"
            shift
            ;;
        '--description')
            export INIT_DESCRIPTION="$2"
            shift
            ;;
        '--chkconfig')
            export INIT_CHKCONFIG="$2"
            shift
            ;;
        '--instance-user')
            export INSTANCE_USER="$2"
            shift
            ;;
        '--instance-location')
            export INSTANCE_LOCATION="$2"
            shift
            ;;
        '--instance-name')
            export INSTANCE_NAME="$2"
            shift
            ;;
        -h|--help)
            usage
            ;;
        *)
            echo Invalid option: $key >&2
            usage
        ;;
    esac
    shift
done

ERROR=""
test -z "$INSTANCE_NAME"      && ERROR="$ERROR\n    - Instance name"
test -z "$INSTANCE_LOCATION"  && ERROR="$ERROR\n    - Instance location"
test -z "$INSTANCE_USER"      && ERROR="$ERROR\n    - Instance user"

test -z "$ERROR"        || (echo -e "The following required option are not specified:$ERROR" >&2 && usage) || exit 1

# Fail if target directory does not exist
if [ ! -d "${INSTANCE_LOCATION}" ]; then
  echo "Error: Target directory: '${INSTANCE_LOCATION}' does not exist."
  exit 1
else
  echo "Tomcat instance template will be created in directory '${INSTANCE_LOCATION}'"
fi

TARGET="${INSTANCE_LOCATION}"/"${INSTANCE_NAME}"
mkdir -p "${TARGET}"

mkdir -p "${TARGET}/bin"
mkdir -p "${TARGET}/conf"
mkdir -p "${TARGET}/lib"
mkdir -p "${TARGET}/logs"
mkdir -p "${TARGET}/temp"
mkdir -p "${TARGET}/webapps"
mkdir -p "${TARGET}/work"

cp "${PROG_DIR}"/../template/* "${TARGET}"/bin
cp "${PROG_DIR}"/../../conf/tomcat.version "${TARGET}"/conf
chmod 0755 "${TARGET}"/bin/*.sh

INIT_D="${TARGET}"/bin/init.d.sh
sed -i -e "s|<INSTANCE_LOCATION>|${TARGET}|" "${INIT_D}"
sed -i -e "s|<INSTANCE_USER>|${INSTANCE_USER}|" "${INIT_D}"
sed -i -e "s|<INIT_CHKCONFIG>|${INIT_CHKCONFIG}|" "${INIT_D}"
sed -i -e "s|<INIT_PROVIDES>|${INIT_PROVIDES}|" "${INIT_D}"
sed -i -e "s|<INIT_START>|${INIT_START}|" "${INIT_D}"
sed -i -e "s|<INIT_STOP>|${INIT_STOP}|" "${INIT_D}"
sed -i -e "s|<INIT_SHORT_DESCRIPTION>|${INIT_SHORT_DESCRIPTION}|" "${INIT_D}"
sed -i -e "s|<INIT_DESCRIPTION>|${INIT_DESCRIPTION}|" "${INIT_D}"
sed -i -e "s|<INIT_CHKCONFIG>|${INIT_CHKCONFIG}|" "${INIT_D}"

echo "* New Tomcat instance created in ${TARGET}"
echo "* You might want to edit default configuration in ${TARGET}/conf"
