#!/usr/bin/env bash
TARGETDIR=${1:-.}
HOSTNAME=`uname -n`
OPENSSL_CONF_DIR=/opt/vmware/etc/ssl
OPENSSL_CONF=$OPENSSL_CONF_DIR/openssl.conf
TMP_DIR=/var/tmp/tmp-certs

trap "rm -rf $TMP_DIR; exit 0" 0 1 2 15

echo "Generating SSL certificates for sfcb in $TARGETDIR"
logger -- "Generating SSL certificates for sfcb in $TARGETDIR"

if [ ! -f $OPENSSL_CONF ]
then
    echo "WARNING: Unable to find openssl configuration file, using default"
    mkdir -p $OPENSSL_CONF_DIR
    cat > $OPENSSL_CONF <<EOF
[req]
distinguished_name=user_dn
x509_extensions=v3_req
prompt=no

[user_dn]
CN=WARNING: Config file not found

[v3_req]
# Extensions to add to a certificate request
subjectAltName = @alt_names
basicConstraints=CA:TRUE
basicConstraints=critical,CA:TRUE, pathlen:0

[alt_names]
DNS.1 = localhost
EOF
fi

#openssl tools uses path in RANDFILE to write out a seeding file, this fixes
#the "Unable to to write 'random state' message during key gen
RANDFILE=/root
export RANDFILE

mkdir -p $TMP_DIR;
chown root:root $TMP_DIR;
chmod 700 $TMP_DIR;

umask 0277; openssl req -x509 -days 10000 -newkey rsa:2048 \
   -nodes -config $OPENSSL_CONF   \
   -keyout $TMP_DIR/key.pem -out $TMP_DIR/cert.pem > /dev/null 2>&1

#
# A user needs to be in the vami group to read these certs. This
# is the same group they need to be in to log in to vami.
# The Studio UI has a checkbox for a user account called "VAMI Admin",
# and this checkbox will put that user into the vami group.
#

# server certs
cp $TMP_DIR/cert.pem $TARGETDIR/server.pem
cp $TMP_DIR/cert.pem $TARGETDIR/clist.pem
cp $TMP_DIR/key.pem $TARGETDIR/file.pem
chgrp vami $TARGETDIR/server.pem $TARGETDIR/file.pem $TARGETDIR/clist.pem
chmod 440 $TARGETDIR/server.pem $TARGETDIR/file.pem $TARGETDIR/clist.pem

# client certs
cp $TMP_DIR/cert.pem $TARGETDIR/client.pem
chgrp vami $TARGETDIR/client.pem
chmod 440 $TARGETDIR/client.pem

# export certificate on photon and ubuntu
if [ -d /etc/ssl/certs ]
then
    if [ -x /usr/sbin/update-ca-certificates ]; then
        #ubuntu, always need the fix, ca-certificates seems in minimal installation
        if [ -d  /usr/local/share/ca-certificates ]; then
            ln -s /opt/vmware/etc/sfcb/server.pem /usr/local/share/ca-certificates/vami_sfcb.crt
        fi
        #sles12, have ca issue when ca-certificates package is installed
        if [ -d  /etc/pki/trust/anchors ]; then
            ln -s /opt/vmware/etc/sfcb/server.pem /etc/pki/trust/anchors/vami_sfcb.pem
        fi
        update-ca-certificates
    else
    #photon, always need the fix
        hash=`openssl x509 -subject_hash -in /opt/vmware/etc/sfcb/server.pem  |head -1`
        ln -s /opt/vmware/etc/sfcb/server.pem /etc/ssl/certs/${hash}.pem
        cd /etc/ssl/certs;ln -s ${hash}.pem ${hash}.0
    fi
fi
